/********************************************************************** 
 Freeciv - Copyright (C) 1996 - A Kjeldberg, L Gregersen, P Unold
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
***********************************************************************/

#ifdef HAVE_CONFIG_H
#include <fc_config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

/* utility */
#include "bitvector.h"
#include "fcintl.h"
#include "log.h"
#include "mem.h"
#include "shared.h"
#include "support.h"

/* common */
#include "city.h"
#include "game.h"
#include "map.h"
#include "movement.h"
#include "packets.h"
#include "player.h"
#include "unitlist.h"

/* client */
#include "client_main.h"
#include "colors.h"
#include "control.h"
#include "climap.h"
#include "options.h"
#include "text.h"
#include "tilespec.h"

/* client/agents */
#include "cma_fec.h" 

/* client/gui-gtk-3.0 */
#include "citizensinfo.h"
#include "cityrep.h"
#include "cma_fe.h"
#include "dialogs.h"
#include "graphics.h"
#include "gui_main.h"
#include "gui_stuff.h"
#include "happiness.h"
#include "helpdlg.h"
#include "inputdlg.h"
#include "mapview.h"
#include "repodlgs.h"
#include "wldlg.h"

#include "citydlg.h"

#define CITYMAP_WIDTH MIN(512, canvas_width)
#define CITYMAP_HEIGHT (CITYMAP_WIDTH * canvas_height / canvas_width)
#define CITYMAP_SCALE ((double)CITYMAP_WIDTH / (double)canvas_width)

#define TINYSCREEN_MAX_HEIGHT (500 - 1)

struct city_dialog;

/* get 'struct dialog_list' and related function */
#define SPECLIST_TAG dialog
#define SPECLIST_TYPE struct city_dialog
#include "speclist.h"

#define dialog_list_iterate(dialoglist, pdialog) \
    TYPED_LIST_ITERATE(struct city_dialog, dialoglist, pdialog)
#define dialog_list_iterate_end  LIST_ITERATE_END

struct unit_node {
  GtkWidget *cmd;
  GtkWidget *pix;
};

/* get 'struct unit_node' and related function */
#define SPECVEC_TAG unit_node
#define SPECVEC_TYPE struct unit_node
#include "specvec.h"

#define unit_node_vector_iterate(list, elt) \
    TYPED_VECTOR_ITERATE(struct unit_node, list, elt)
#define unit_node_vector_iterate_end  VECTOR_ITERATE_END

enum { OVERVIEW_PAGE, MAP_PAGE, BUILDINGS_PAGE, WORKLIST_PAGE,
  HAPPINESS_PAGE, CMA_PAGE, TRADE_PAGE, MISC_PAGE
};

#define NUM_CITIZENS_SHOWN 30
#define NUM_INFO_FIELDS 12      /* number of fields in city_info */
#define NUM_PAGES 6             /* the number of pages in city dialog notebook 
                                 * (+1) if you change this, you must add an
                                 * entry to misc_whichtab_label[] */

/* minimal size for the city map scrolling windows*/
#define CITY_MAP_MIN_SIZE_X  200
#define CITY_MAP_MIN_SIZE_Y  150

struct city_map_canvas {
  GtkWidget *sw;
  GtkWidget *ebox;
  GtkWidget *darea;
};

struct city_dialog {
  struct city *pcity;

  GtkWidget *shell;
  GtkWidget *name_label;
  cairo_surface_t *map_canvas_store_unscaled;
  GtkWidget *notebook;

  GtkWidget *popup_menu;
  GtkWidget *citizen_pixmap;

  struct {
    struct city_map_canvas map_canvas;

    GtkWidget *production_bar;
    GtkWidget *production_combo;
    GtkWidget *buy_command;
    GtkWidget *improvement_list;

    GtkWidget *supported_units_frame;
    GtkWidget *supported_unit_table;

    GtkWidget *present_units_frame;
    GtkWidget *present_unit_table;

    struct unit_node_vector supported_units;
    struct unit_node_vector present_units;

    GtkWidget *info_ebox[NUM_INFO_FIELDS];
    GtkWidget *info_label[NUM_INFO_FIELDS];

    GtkListStore* change_production_store;
  } overview;

  struct {
    GtkWidget *production_label;
    GtkWidget *production_bar;
    GtkWidget *buy_command;
    GtkWidget *worklist;
  } production;

  struct {
    struct city_map_canvas map_canvas;

    GtkWidget *widget;
    GtkWidget *info_ebox[NUM_INFO_FIELDS];
    GtkWidget *info_label[NUM_INFO_FIELDS];
    GtkWidget *citizens;
  } happiness;

  struct cma_dialog *cma_editor;

  struct {
    GtkWidget *rename_command;
    GtkWidget *new_citizens_radio[3];
    GtkWidget *disband_on_settler;
    GtkWidget *whichtab_radio[NUM_PAGES];
    short block_signal;
  } misc;

  GtkWidget *buy_shell, *sell_shell;
  GtkTreeSelection *change_selection;
  GtkWidget *rename_shell, *rename_input;

  GtkWidget *show_units_command;
  GtkWidget *prev_command, *next_command;

  Impr_type_id sell_id;

  int cwidth;
};

static struct dialog_list *dialog_list;
static bool city_dialogs_have_been_initialised = FALSE;
static int canvas_width, canvas_height;
static int new_dialog_def_page = OVERVIEW_PAGE;
static int last_page = OVERVIEW_PAGE;

static bool low_citydlg;

/****************************************/

static void initialize_city_dialogs(void);
static void city_dialog_map_create(struct city_dialog *pdialog,
                                   struct city_map_canvas *map_canvas);
static void city_dialog_map_recenter(GtkWidget *map_canvas_sw);

static struct city_dialog *get_city_dialog(struct city *pcity);
static gboolean keyboard_handler(GtkWidget * widget, GdkEventKey * event,
				 struct city_dialog *pdialog);

static GtkWidget *create_city_info_table(struct city_dialog *pdialog,
    					 GtkWidget **info_ebox,
					 GtkWidget **info_label);
static void create_and_append_overview_page(struct city_dialog *pdialog);
static void create_and_append_map_page(struct city_dialog *pdialog);
static void create_and_append_buildings_page(struct city_dialog *pdialog);
static void create_and_append_worklist_page(struct city_dialog *pdialog);
static void create_and_append_happiness_page(struct city_dialog *pdialog);
static void create_and_append_cma_page(struct city_dialog *pdialog);
static void create_and_append_settings_page(struct city_dialog *pdialog);

static struct city_dialog *create_city_dialog(struct city *pcity);

static void city_dialog_update_title(struct city_dialog *pdialog);
static void city_dialog_update_citizens(struct city_dialog *pdialog);
static void city_dialog_update_information(GtkWidget **info_ebox,
					   GtkWidget **info_label,
                                           struct city_dialog *pdialog);
static void city_dialog_update_map(struct city_dialog *pdialog);
static void city_dialog_update_building(struct city_dialog *pdialog);
static void city_dialog_update_improvement_list(struct city_dialog
						*pdialog);
static void city_dialog_update_supported_units(struct city_dialog
					       *pdialog);
static void city_dialog_update_present_units(struct city_dialog *pdialog);
static void city_dialog_update_prev_next(void);

static void show_units_callback(GtkWidget * w, gpointer data);

static gboolean supported_unit_callback(GtkWidget * w, GdkEventButton * ev,
				        gpointer data);
static gboolean present_unit_callback(GtkWidget * w, GdkEventButton * ev,
				      gpointer data);
static gboolean supported_unit_middle_callback(GtkWidget * w,
					       GdkEventButton * ev,
					       gpointer data);
static gboolean present_unit_middle_callback(GtkWidget * w,
					     GdkEventButton * ev,
					     gpointer data);

static void unit_center_callback(GtkWidget * w, gpointer data);
static void unit_activate_callback(GtkWidget * w, gpointer data);
static void supported_unit_activate_close_callback(GtkWidget * w,
						   gpointer data);
static void present_unit_activate_close_callback(GtkWidget * w,
						 gpointer data);
static void unit_load_callback(GtkWidget * w, gpointer data);
static void unit_unload_callback(GtkWidget * w, gpointer data);
static void unit_sentry_callback(GtkWidget * w, gpointer data);
static void unit_fortify_callback(GtkWidget * w, gpointer data);
static void unit_disband_callback(GtkWidget * w, gpointer data);
static void unit_homecity_callback(GtkWidget * w, gpointer data);
static void unit_upgrade_callback(GtkWidget * w, gpointer data);

static gboolean citizens_callback(GtkWidget * w, GdkEventButton * ev,
			      gpointer data);
static gboolean button_down_citymap(GtkWidget * w, GdkEventButton * ev,
				    gpointer data);
static void draw_map_canvas(struct city_dialog *pdialog);

static void buy_callback(GtkWidget * w, gpointer data);
static void change_production_callback(GtkComboBox *combo,
                                       struct city_dialog *pdialog);

static void sell_callback(struct impr_type *pimprove, gpointer data);
static void sell_callback_response(GtkWidget *w, gint response, gpointer data);

static void impr_callback(GtkTreeView *view, GtkTreePath *path,
			  GtkTreeViewColumn *col, gpointer data);

static void rename_callback(GtkWidget * w, gpointer data);
static void rename_popup_callback(gpointer data, gint response,
                                  const char *input);
static void set_cityopt_values(struct city_dialog *pdialog);
static void cityopt_callback(GtkWidget * w, gpointer data);
static void misc_whichtab_callback(GtkWidget * w, gpointer data);

static void city_destroy_callback(GtkWidget *w, gpointer data);
static void close_city_dialog(struct city_dialog *pdialog);
static void close_callback(GtkWidget *w, gpointer data);
static void switch_city_callback(GtkWidget * w, gpointer data);

/****************************************************************
  Called to set the dimensions of the city dialog, both on
  startup and if the tileset is changed.
*****************************************************************/
static void init_citydlg_dimensions(void)
{
  canvas_width = get_citydlg_canvas_width();
  canvas_height = get_citydlg_canvas_height();
}

/****************************************************************
  Initialize stuff needed for city dialogs
*****************************************************************/
static void initialize_city_dialogs(void)
{
  fc_assert_ret(!city_dialogs_have_been_initialised);

  dialog_list = dialog_list_new();
  init_citydlg_dimensions();
  if (screen_height() <= TINYSCREEN_MAX_HEIGHT) {
    low_citydlg = TRUE;
  } else {
    low_citydlg = FALSE;
  }

  city_dialogs_have_been_initialised = TRUE;
}

/****************************************************************
  Called when the tileset changes.
*****************************************************************/
void reset_city_dialogs(void)
{
  if (!city_dialogs_have_been_initialised) {
    return;
  }

  init_citydlg_dimensions();

  dialog_list_iterate(dialog_list, pdialog) {
    /* There's no reasonable way to resize a GtkPixcomm, so we don't try.
       Instead we just redraw the overview within the existing area.  The
       player has to close and reopen the dialog to fix this. */
    city_dialog_update_map(pdialog);
  } dialog_list_iterate_end;

  popdown_all_city_dialogs();
}

/****************************************************************
  Return city dialog of the given city, or NULL is it doesn't
  already exist
*****************************************************************/
static struct city_dialog *get_city_dialog(struct city *pcity)
{
  if (!city_dialogs_have_been_initialised) {
    initialize_city_dialogs();
  }

  dialog_list_iterate(dialog_list, pdialog) {
    if (pdialog->pcity == pcity)
      return pdialog;
  }
  dialog_list_iterate_end;
  return NULL;
}

/***************************************************************************
  Redraw map canvas on expose.
****************************************************************************/
static gboolean canvas_exposed_cb(GtkWidget *w, cairo_t *cr,
                                  gpointer data)
{
  struct city_dialog *pdialog = data;

  cairo_scale(cr, CITYMAP_SCALE, CITYMAP_SCALE);
  cairo_set_source_surface(cr, pdialog->map_canvas_store_unscaled, 0, 0);
  if (!gtk_widget_get_sensitive(pdialog->overview.map_canvas.ebox)) {
    cairo_paint_with_alpha(cr, 0.5);
  } else {
    cairo_paint(cr);
  }

  return TRUE;
}

/***************************************************************************
  Create a city map widget; used in the overview and in the happiness page.
****************************************************************************/
static void city_dialog_map_create(struct city_dialog *pdialog,
                                   struct city_map_canvas *map_canvas)
{
  GtkWidget *sw, *ebox, *darea;

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_min_content_width(GTK_SCROLLED_WINDOW(sw),
                                            CITYMAP_WIDTH);
  gtk_scrolled_window_set_min_content_height(GTK_SCROLLED_WINDOW(sw),
                                             CITYMAP_HEIGHT);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
                                      GTK_SHADOW_NONE);

  ebox = gtk_event_box_new();
  gtk_widget_set_halign(ebox, GTK_ALIGN_CENTER);
  gtk_widget_set_valign(ebox, GTK_ALIGN_CENTER);
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(ebox), FALSE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(sw), ebox);

  darea = gtk_drawing_area_new();
  gtk_widget_add_events(darea, GDK_BUTTON_PRESS_MASK);
  gtk_widget_set_size_request(darea, CITYMAP_WIDTH, CITYMAP_HEIGHT);
  g_signal_connect(ebox, "button-press-event",
                   G_CALLBACK(button_down_citymap), pdialog);
  g_signal_connect(darea, "draw",
                   G_CALLBACK(canvas_exposed_cb), pdialog);
  gtk_container_add(GTK_CONTAINER(ebox), darea);

  /* save all widgets for the city map */
  map_canvas->sw = sw;
  map_canvas->ebox = ebox;
  map_canvas->darea = darea;
}

/****************************************************************
  Center city dialog map.
*****************************************************************/
static void city_dialog_map_recenter(GtkWidget *map_canvas_sw) {
  GtkAdjustment *adjust = NULL;
  gdouble value;

  fc_assert_ret(map_canvas_sw != NULL);

  adjust = gtk_scrolled_window_get_hadjustment(
    GTK_SCROLLED_WINDOW(map_canvas_sw));
  value = (gtk_adjustment_get_lower(adjust)
    + gtk_adjustment_get_upper(adjust)
    - gtk_adjustment_get_page_size(adjust)) / 2;
  gtk_adjustment_set_value(adjust, value);
  gtk_adjustment_value_changed(adjust);

  adjust = gtk_scrolled_window_get_vadjustment(
    GTK_SCROLLED_WINDOW(map_canvas_sw));
  value = (gtk_adjustment_get_lower(adjust)
    + gtk_adjustment_get_upper(adjust)
    - gtk_adjustment_get_page_size(adjust)) / 2;
  gtk_adjustment_set_value(adjust, value);
  gtk_adjustment_value_changed(adjust);
}

/****************************************************************
  Refresh city dialog of the given city
*****************************************************************/
void real_city_dialog_refresh(struct city *pcity)
{
  struct city_dialog *pdialog = get_city_dialog(pcity);

  log_debug("CITYMAP_WIDTH:  %d", CITYMAP_WIDTH);
  log_debug("CITYMAP_HEIGHT: %d", CITYMAP_HEIGHT);
  log_debug("CITYMAP_SCALE:  %.3f", CITYMAP_SCALE);

  if (city_owner(pcity) == client.conn.playing) {
    city_report_dialog_update_city(pcity);
    economy_report_dialog_update();
  }

  if (!pdialog)
    return;

  city_dialog_update_title(pdialog);
  city_dialog_update_citizens(pdialog);
  city_dialog_update_information(pdialog->overview.info_ebox,
				 pdialog->overview.info_label, pdialog);
  city_dialog_update_map(pdialog);
  city_dialog_update_building(pdialog);
  city_dialog_update_improvement_list(pdialog);
  city_dialog_update_supported_units(pdialog);
  city_dialog_update_present_units(pdialog);

  if (!client_has_player() || city_owner(pcity) == client_player()) {
    bool have_present_units = (unit_list_size(pcity->tile->units) > 0);

    refresh_worklist(pdialog->production.worklist);

    if (!low_citydlg) {
      city_dialog_update_information(pdialog->happiness.info_ebox,
                                     pdialog->happiness.info_label, pdialog);
    }
    refresh_happiness_dialog(pdialog->pcity);
    if (game.info.citizen_nationality) {
      citizens_dialog_refresh(pdialog->pcity);
    }

    if (!client_is_observer()) {
      refresh_cma_dialog(pdialog->pcity, REFRESH_ALL);
    }

    gtk_widget_set_sensitive(pdialog->show_units_command,
			     can_client_issue_orders() &&
			     have_present_units);
  } else {
    /* Set the buttons we do not want live while a Diplomat investigates */
    gtk_widget_set_sensitive(pdialog->show_units_command, FALSE);
  }
}

/****************************************************************
  Refresh city dialogs of unit's homecity and city where unit
  currently is.
*****************************************************************/
void refresh_unit_city_dialogs(struct unit *punit)
{
  struct city *pcity_sup, *pcity_pre;
  struct city_dialog *pdialog;

  pcity_sup = game_city_by_number(punit->homecity);
  pcity_pre = tile_city(unit_tile(punit));

  if (pcity_sup && (pdialog = get_city_dialog(pcity_sup)))
    city_dialog_update_supported_units(pdialog);

  if (pcity_pre && (pdialog = get_city_dialog(pcity_pre)))
    city_dialog_update_present_units(pdialog);
}

/****************************************************************
popup the dialog 10% inside the main-window 
*****************************************************************/
void real_city_dialog_popup(struct city *pcity)
{
  struct city_dialog *pdialog;

  if (!(pdialog = get_city_dialog(pcity))) {
    pdialog = create_city_dialog(pcity);
  }

  gtk_window_present(GTK_WINDOW(pdialog->shell));

  /* center the city map(s); this must be *after* the city dialog was drawn
   * else the size information is missing! */
  city_dialog_map_recenter(pdialog->overview.map_canvas.sw);
  if (pdialog->happiness.map_canvas.sw) {
    city_dialog_map_recenter(pdialog->happiness.map_canvas.sw);
  }
}

/****************************************************************
  Return whether city dialog for given city is open
*****************************************************************/
bool city_dialog_is_open(struct city *pcity)
{
  return get_city_dialog(pcity) != NULL;
}

/****************************************************************
popdown the dialog 
*****************************************************************/
void popdown_city_dialog(struct city *pcity)
{
  struct city_dialog *pdialog = get_city_dialog(pcity);

  if (pdialog) {
    close_city_dialog(pdialog);
  }
}

/****************************************************************
popdown all dialogs
*****************************************************************/
void popdown_all_city_dialogs(void)
{
  if (!city_dialogs_have_been_initialised) {
    return;
  }

  while (dialog_list_size(dialog_list)) {
    close_city_dialog(dialog_list_get(dialog_list, 0));
  }
  dialog_list_destroy(dialog_list);

  city_dialogs_have_been_initialised = FALSE;
}

/**************************************************************************
  Keyboard handler for city dialog
**************************************************************************/
static gboolean keyboard_handler(GtkWidget * widget, GdkEventKey * event,
				 struct city_dialog *pdialog)
{
  if (event->state & GDK_CONTROL_MASK) {
    switch (event->keyval) {
    case GDK_KEY_Left:
      gtk_notebook_prev_page(GTK_NOTEBOOK(pdialog->notebook));
      return TRUE;

    case GDK_KEY_Right:
      gtk_notebook_next_page(GTK_NOTEBOOK(pdialog->notebook));
      return TRUE;

    default:
      break;
    }
  }

  return FALSE;
}

/**************************************************************************
  Destroy info popup dialog when button released
**************************************************************************/
static gboolean show_info_button_release(GtkWidget *w, GdkEventButton *ev,
					 gpointer data)
{
  gtk_grab_remove(w);
  gdk_device_ungrab(ev->device, ev->time);
  gtk_widget_destroy(w);
  return FALSE;
}

enum { FIELD_FOOD, FIELD_SHIELD, FIELD_TRADE, FIELD_GOLD, FIELD_LUXURY,
       FIELD_SCIENCE, FIELD_GRANARY, FIELD_GROWTH, FIELD_CORRUPTION,
       FIELD_WASTE, FIELD_POLLUTION, FIELD_ILLNESS
};

/****************************************************************
  Popup info dialog
*****************************************************************/
static gboolean show_info_popup(GtkWidget *w, GdkEventButton *ev,
    				gpointer data)
{
  struct city_dialog *pdialog = g_object_get_data(G_OBJECT(w), "pdialog");

  if (ev->button == 1) {
    GtkWidget *p, *label, *frame;
    char buf[1024];
    
    switch (GPOINTER_TO_UINT(data)) {
    case FIELD_FOOD:
      get_city_dialog_output_text(pdialog->pcity, O_FOOD, buf, sizeof(buf));
      break;
    case FIELD_SHIELD:
      get_city_dialog_output_text(pdialog->pcity, O_SHIELD,
				  buf, sizeof(buf));
      break;
    case FIELD_TRADE:
      get_city_dialog_output_text(pdialog->pcity, O_TRADE, buf, sizeof(buf));
      break;
    case FIELD_GOLD:
      get_city_dialog_output_text(pdialog->pcity, O_GOLD, buf, sizeof(buf));
      break;
    case FIELD_SCIENCE:
      get_city_dialog_output_text(pdialog->pcity, O_SCIENCE,
				  buf, sizeof(buf));
      break;
    case FIELD_LUXURY:
      get_city_dialog_output_text(pdialog->pcity, O_LUXURY,
				  buf, sizeof(buf));
      break;
    case FIELD_POLLUTION:
      get_city_dialog_pollution_text(pdialog->pcity, buf, sizeof(buf));
      break;
    case FIELD_ILLNESS:
      get_city_dialog_illness_text(pdialog->pcity, buf, sizeof(buf));
      break;
    default:
      return TRUE;
    }
    
    p = gtk_window_new(GTK_WINDOW_POPUP);
    gtk_widget_set_name(p, "Freeciv");
    gtk_container_set_border_width(GTK_CONTAINER(p), 2);
    gtk_window_set_position(GTK_WINDOW(p), GTK_WIN_POS_MOUSE);

    frame = gtk_frame_new(NULL);
    gtk_container_add(GTK_CONTAINER(p), frame);

    label = gtk_label_new(buf);
    gtk_widget_set_name(label, "city_info_label");
    gtk_misc_set_padding(GTK_MISC(label), 4, 4);
    gtk_container_add(GTK_CONTAINER(frame), label);
    gtk_widget_show_all(p);

    gdk_device_grab(ev->device, gtk_widget_get_window(p),
                    GDK_OWNERSHIP_NONE, TRUE, GDK_BUTTON_RELEASE_MASK, NULL,
                    ev->time);
    gtk_grab_add(p);

    g_signal_connect_after(p, "button_release_event",
                           G_CALLBACK(show_info_button_release), NULL);
  }
  return TRUE;
}

/****************************************************************
 used once in the overview page and once in the happiness page
 **info_label points to the info_label in the respective struct
****************************************************************/
static GtkWidget *create_city_info_table(struct city_dialog *pdialog,
    					 GtkWidget **info_ebox,
					 GtkWidget **info_label)
{
  int i;
  GtkWidget *table, *label, *ebox;

  static const char *output_label[NUM_INFO_FIELDS] = { N_("Food:"),
    N_("Prod:"),
    N_("Trade:"),
    N_("Gold:"),
    N_("Luxury:"),
    N_("Science:"),
    N_("Granary:"),
    N_("Change in:"),
    N_("Corruption:"),
    N_("Waste:"),
    N_("Pollution:"),
    N_("Plague Risk:")
  };
  static bool output_label_done;

  table = gtk_grid_new();
  g_object_set(table, "margin", 4, NULL);

  intl_slist(ARRAY_SIZE(output_label), output_label, &output_label_done);

  for (i = 0; i < NUM_INFO_FIELDS; i++) {
    label = gtk_label_new(output_label[i]);
    switch (i) {
      case 2:
      case 5:
      case 7:
        gtk_widget_set_margin_bottom(label, 5);
        break;
      case 3:
      case 6:
      case 8:
        gtk_widget_set_margin_top(label, 5);
        break;
      default:
        break;
    }
    gtk_widget_set_margin_right(label, 5);
    gtk_widget_set_name(label, "city_label");	/* for font style? */
    gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
    gtk_grid_attach(GTK_GRID(table), label, 0, i, 1, 1);

    ebox = gtk_event_box_new();
    switch (i) {
      case 2:
      case 5:
      case 7:
        gtk_widget_set_margin_bottom(ebox, 5);
        break;
      case 3:
      case 6:
      case 8:
        gtk_widget_set_margin_top(ebox, 5);
        break;
      default:
        break;
    }
    gtk_event_box_set_visible_window(GTK_EVENT_BOX(ebox), FALSE);
    g_object_set_data(G_OBJECT(ebox), "pdialog", pdialog);
    g_signal_connect(ebox, "button_press_event",
	G_CALLBACK(show_info_popup), GUINT_TO_POINTER(i));
    info_ebox[i] = ebox;

    label = gtk_label_new("");
    info_label[i] = label;
    gtk_widget_set_name(label, "city_label");	/* ditto */
    gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);

    gtk_container_add(GTK_CONTAINER(ebox), label);

    gtk_grid_attach(GTK_GRID(table), ebox, 1, i, 1, 1);
  }

  gtk_widget_show_all(table);

  return table;
}

/****************************************************************
  Create main citydlg map
*****************************************************************/
static void create_citydlg_main_map(struct city_dialog *pdialog,
                                    GtkWidget *container)
{
  GtkWidget *frame;

  frame = gtk_frame_new(_("City map"));
  gtk_widget_set_size_request(frame, CITY_MAP_MIN_SIZE_X,
                              CITY_MAP_MIN_SIZE_Y);
  gtk_container_add(GTK_CONTAINER(container), frame);

  city_dialog_map_create(pdialog, &pdialog->overview.map_canvas);
  gtk_container_add(GTK_CONTAINER(frame), pdialog->overview.map_canvas.sw);
}

/****************************************************************
  Create improvements list
*****************************************************************/
static GtkWidget *create_citydlg_improvement_list(struct city_dialog *pdialog,
                                                  GtkWidget *vbox)
{
  GtkWidget *view;
  GtkListStore *store;
  GtkCellRenderer *rend;

  /* improvements */
  store = gtk_list_store_new(5, G_TYPE_POINTER, GDK_TYPE_PIXBUF,
                             G_TYPE_STRING, G_TYPE_INT, G_TYPE_BOOLEAN);

  view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
  gtk_widget_set_hexpand(view, TRUE);
  gtk_widget_set_vexpand(view, TRUE);
  g_object_unref(store);
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), FALSE);
  gtk_widget_set_name(view, "small_font");
  pdialog->overview.improvement_list = view;

  gtk_widget_set_tooltip_markup(view,
                                _("Press <b>ENTER</b> or double-click to sell an improvement."));

  rend = gtk_cell_renderer_pixbuf_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view), -1, NULL,
                                              rend, "pixbuf", 1, NULL);
  rend = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view), -1, NULL,
                                              rend, "text", 2,
                                              "strikethrough", 4, NULL);
  rend = gtk_cell_renderer_text_new();
  g_object_set(rend, "xalign", 1.0, NULL);
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(view), -1, NULL,
                                              rend, "text", 3,
                                              "strikethrough", 4, NULL);

  g_signal_connect(view, "row_activated", G_CALLBACK(impr_callback),
                   pdialog);

  return view;
}

/****************************************************************
                  **** Overview page ****
 +- GtkWidget *page ------------------------------------------+
 | +- GtkWidget *middle -----------+------------------------+ |
 | | City map                      |  Production            | |
 | +-------------------------------+------------------------+ |
 +------------------------------------------------------------+
 | +- GtkWidget *bottom -------+----------------------------+ |
 | | Info                      | +- GtkWidget *right -----+ | |
 | |                           | | supported units        | | |
 | |                           | +------------------------+ | |
 | |                           | | present units          | | |
 | |                           | +------------------------+ | |
 | +---------------------------+----------------------------+ |
 +------------------------------------------------------------+
*****************************************************************/
static void create_and_append_overview_page(struct city_dialog *pdialog)
{
  GtkWidget *page, *bottom;
  GtkWidget *hbox, *right, *vbox, *frame, *table;
  GtkWidget *label, *sw, *view, *bar, *production_combo;
  GtkCellRenderer *rend;
  GtkListStore *production_store;
  /* TRANS: Overview tab in city dialog */
  const char *tab_title = _("_Overview");
  int unit_height = tileset_unit_height(tileset);

  /* main page */
  page = gtk_grid_new();
  gtk_orientable_set_orientation(GTK_ORIENTABLE(page),
                                 GTK_ORIENTATION_VERTICAL);
  gtk_container_set_border_width(GTK_CONTAINER(page), 8);
  label = gtk_label_new_with_mnemonic(tab_title);
  gtk_notebook_append_page(GTK_NOTEBOOK(pdialog->notebook), page, label);

  if (!low_citydlg) {
    GtkWidget *middle;

    /* middle: city map, improvements */
    middle = gtk_grid_new();
    gtk_grid_set_column_spacing(GTK_GRID(middle), 6);
    gtk_container_add(GTK_CONTAINER(page), middle);

    /* city map */
    create_citydlg_main_map(pdialog, middle);

    /* improvements */
    vbox = gtk_grid_new();
    gtk_orientable_set_orientation(GTK_ORIENTABLE(vbox),
                                   GTK_ORIENTATION_VERTICAL);
    gtk_container_add(GTK_CONTAINER(middle), vbox);

    view = create_citydlg_improvement_list(pdialog, middle);

    label = g_object_new(GTK_TYPE_LABEL, "label", _("Production:"),
                         "xalign", 0.0, "yalign", 0.5, NULL);
    gtk_container_add(GTK_CONTAINER(vbox), label);

    hbox = gtk_grid_new();
    gtk_grid_set_column_spacing(GTK_GRID(hbox), 10);
    gtk_container_add(GTK_CONTAINER(vbox), hbox);

    production_store = gtk_list_store_new(4, GDK_TYPE_PIXBUF, G_TYPE_STRING,
                                          G_TYPE_INT, G_TYPE_BOOLEAN);
    pdialog->overview.change_production_store = production_store;

    production_combo =
      gtk_combo_box_new_with_model(GTK_TREE_MODEL(production_store));
    gtk_widget_set_hexpand(production_combo, TRUE);
    pdialog->overview.production_combo = production_combo;
    gtk_container_add(GTK_CONTAINER(hbox), production_combo);
    g_object_unref(production_store);
    g_signal_connect(production_combo, "changed",
                     G_CALLBACK(change_production_callback), pdialog);

    gtk_cell_layout_clear(GTK_CELL_LAYOUT(production_combo));
    rend = gtk_cell_renderer_pixbuf_new();
    gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(production_combo), rend, TRUE);
    gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(production_combo),
                                   rend, "pixbuf", 0, NULL);
    g_object_set(rend, "xalign", 0.0, NULL);

    rend = gtk_cell_renderer_text_new();
    gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(production_combo), rend, TRUE);
    gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(production_combo),
                                   rend, "text", 1, "strikethrough", 3, NULL);

    bar = gtk_progress_bar_new();
    gtk_progress_bar_set_show_text(GTK_PROGRESS_BAR(bar), TRUE);
    pdialog->overview.production_bar = bar;
    gtk_container_add(GTK_CONTAINER(production_combo), bar);
    gtk_combo_box_set_wrap_width(GTK_COMBO_BOX(production_combo), 3);

    gtk_progress_bar_set_text(GTK_PROGRESS_BAR(bar), _("%d/%d %d turns"));

    pdialog->overview.buy_command = gtk_stockbutton_new(GTK_STOCK_EXECUTE,
                                                        _("_Buy"));
    gtk_container_add(GTK_CONTAINER(hbox), pdialog->overview.buy_command);
    g_signal_connect(pdialog->overview.buy_command, "clicked",
                     G_CALLBACK(buy_callback), pdialog);

    label = g_object_new(GTK_TYPE_LABEL, "use-underline", TRUE,
                         "mnemonic-widget", view,
                         "label", _("I_mprovements:"),
                         "xalign", 0.0, "yalign", 0.5, NULL);
    gtk_container_add(GTK_CONTAINER(vbox), label);

    sw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
                                        GTK_SHADOW_ETCHED_IN);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                   GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_container_add(GTK_CONTAINER(vbox), sw);

    gtk_container_add(GTK_CONTAINER(sw), view);
  } else {
    pdialog->overview.buy_command = NULL;
    pdialog->overview.production_bar = NULL;
    pdialog->overview.production_combo = NULL;
    pdialog->overview.change_production_store = NULL;
  }

  /* bottom: info, units */
  bottom = gtk_grid_new();
  gtk_grid_set_column_spacing(GTK_GRID(bottom), 6);
  gtk_container_add(GTK_CONTAINER(page), bottom);

  /* info */
  frame = gtk_frame_new(_("Info"));
  gtk_container_add(GTK_CONTAINER(bottom), frame);

  table = create_city_info_table(pdialog,
                                 pdialog->overview.info_ebox,
                                 pdialog->overview.info_label);
  gtk_widget_set_halign(table, GTK_ALIGN_CENTER);
  gtk_widget_set_valign(table, GTK_ALIGN_CENTER);
  gtk_container_add(GTK_CONTAINER(frame), table);

  /* right: present and supported units (overview page) */
  right = gtk_grid_new();
  gtk_orientable_set_orientation(GTK_ORIENTABLE(right),
                                 GTK_ORIENTATION_VERTICAL);
  gtk_container_add(GTK_CONTAINER(bottom), right);

  pdialog->overview.supported_units_frame = gtk_frame_new("");
  gtk_container_add(GTK_CONTAINER(right),
                    pdialog->overview.supported_units_frame);
  pdialog->overview.present_units_frame = gtk_frame_new("");
  gtk_container_add(GTK_CONTAINER(right),
                    pdialog->overview.present_units_frame);

  /* supported units */
  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_hexpand(sw, TRUE);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_NEVER);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
                                      GTK_SHADOW_NONE);
  gtk_container_add(GTK_CONTAINER(pdialog->overview.supported_units_frame),
                                  sw);


  table = gtk_grid_new();
  gtk_grid_set_column_spacing(GTK_GRID(table), 2);
  gtk_widget_set_size_request(table, -1, unit_height);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(sw), table);

  gtk_container_set_focus_hadjustment(GTK_CONTAINER(table),
    gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(sw)));
  gtk_container_set_focus_vadjustment(GTK_CONTAINER(table),
    gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(sw)));

  pdialog->overview.supported_unit_table = table;
  unit_node_vector_init(&pdialog->overview.supported_units);

  /* present units */
  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_hexpand(sw, TRUE);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_NEVER);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
                                      GTK_SHADOW_NONE);
  gtk_container_add(GTK_CONTAINER(pdialog->overview.present_units_frame), sw);

  table = gtk_grid_new();
  gtk_grid_set_column_spacing(GTK_GRID(table), 2);
  gtk_widget_set_size_request(table, -1, unit_height);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(sw), table);

  gtk_container_set_focus_hadjustment(GTK_CONTAINER(table),
    gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(sw)));
  gtk_container_set_focus_vadjustment(GTK_CONTAINER(table),
    gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(sw)));

  pdialog->overview.present_unit_table = table;
  unit_node_vector_init(&pdialog->overview.present_units);

  /* show page */
  gtk_widget_show_all(page);
}

/****************************************************************
  Create map page for small screens
*****************************************************************/
static void create_and_append_map_page(struct city_dialog *pdialog)
{
  if (low_citydlg) {
    GtkWidget *page;
    GtkWidget *label;
    const char *tab_title = _("Citymap");

    page = gtk_grid_new();
    gtk_orientable_set_orientation(GTK_ORIENTABLE(page),
                                   GTK_ORIENTATION_VERTICAL);
    gtk_container_set_border_width(GTK_CONTAINER(page), 8);
    label = gtk_label_new_with_mnemonic(tab_title);
    gtk_notebook_append_page(GTK_NOTEBOOK(pdialog->notebook), page, label);

    create_citydlg_main_map(pdialog, page);

    gtk_widget_show_all(page);
  }
}

/****************************************************************
  Create buildings list page for small screens
*****************************************************************/
static void create_and_append_buildings_page(struct city_dialog *pdialog)
{
  if (low_citydlg) {
    GtkWidget *page;
    GtkWidget *label;
    GtkWidget *vbox;
    GtkWidget *view;
    const char *tab_title = _("Buildings");

    page = gtk_grid_new();
    gtk_orientable_set_orientation(GTK_ORIENTABLE(page),
                                   GTK_ORIENTATION_VERTICAL);
    gtk_container_set_border_width(GTK_CONTAINER(page), 8);
    label = gtk_label_new_with_mnemonic(tab_title);
    gtk_notebook_append_page(GTK_NOTEBOOK(pdialog->notebook), page, label);

    vbox = gtk_grid_new();
    gtk_orientable_set_orientation(GTK_ORIENTABLE(vbox),
                                   GTK_ORIENTATION_VERTICAL);
    gtk_container_add(GTK_CONTAINER(page), vbox);

    view = create_citydlg_improvement_list(pdialog, vbox);

    gtk_container_add(GTK_CONTAINER(vbox), view);

    gtk_widget_show_all(page);
  }
}

/****************************************************************
  Something dragged to worklist dialog
*****************************************************************/
static void
target_drag_data_received(GtkWidget *w, GdkDragContext *context,
			  gint x, gint y, GtkSelectionData *data,
			  guint info, guint time, gpointer user_data)
{
  struct city_dialog *pdialog = (struct city_dialog *) user_data;
  GtkTreeModel *model;
  GtkTreePath *path;

  if (NULL != client.conn.playing
      && city_owner(pdialog->pcity) != client.conn.playing) {
    gtk_drag_finish(context, FALSE, FALSE, time);
  }
    
  if (gtk_tree_get_row_drag_data(data, &model, &path)) {
    GtkTreeIter it;

    if (gtk_tree_model_get_iter(model, &it, path)) {
      cid cid;
      gtk_tree_model_get(model, &it, 0, &cid, -1);
      city_change_production(pdialog->pcity, cid_production(cid));
      gtk_drag_finish(context, TRUE, FALSE, time);
    }
    gtk_tree_path_free(path);
  }

  gtk_drag_finish(context, FALSE, FALSE, time);
}

/****************************************************************
                    **** Production Page **** 
*****************************************************************/
static void create_and_append_worklist_page(struct city_dialog *pdialog)
{
  const char *tab_title = _("P_roduction");
  GtkWidget *label = gtk_label_new_with_mnemonic(tab_title);
  GtkWidget *page, *hbox, *editor, *bar;

  page = gtk_grid_new();
  gtk_orientable_set_orientation(GTK_ORIENTABLE(page),
                                 GTK_ORIENTATION_VERTICAL);
  gtk_container_set_border_width(GTK_CONTAINER(page), 8);
  gtk_notebook_append_page(GTK_NOTEBOOK(pdialog->notebook), page, label);

  /* stuff that's being currently built */

  label = g_object_new(GTK_TYPE_LABEL,
		       "label", _("Production:"),
		       "xalign", 0.0, "yalign", 0.5, NULL);
  pdialog->production.production_label = label;
  gtk_container_add(GTK_CONTAINER(page), label);

  hbox = gtk_grid_new();
  g_object_set(hbox, "margin", 2, NULL);
  gtk_grid_set_column_spacing(GTK_GRID(hbox), 10);
  gtk_container_add(GTK_CONTAINER(page), hbox);

  /* The label is set in city_dialog_update_building() */
  bar = gtk_progress_bar_new();
  gtk_widget_set_hexpand(bar, TRUE);
  gtk_progress_bar_set_show_text(GTK_PROGRESS_BAR(bar), TRUE);
  pdialog->production.production_bar = bar;
  gtk_container_add(GTK_CONTAINER(hbox), bar);
  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(bar), _("%d/%d %d turns"));

  add_worklist_dnd_target(bar);
  g_signal_connect(bar, "drag_data_received",
		   G_CALLBACK(target_drag_data_received), pdialog);

  pdialog->production.buy_command = gtk_stockbutton_new(GTK_STOCK_EXECUTE,
						      _("_Buy"));
  gtk_container_add(GTK_CONTAINER(hbox), pdialog->production.buy_command);

  g_signal_connect(pdialog->production.buy_command, "clicked",
		   G_CALLBACK(buy_callback), pdialog);


  editor = create_worklist();
  g_object_set(editor, "margin", 6, NULL);
  reset_city_worklist(editor, pdialog->pcity);
  gtk_container_add(GTK_CONTAINER(page), editor);
  pdialog->production.worklist = editor;

  gtk_widget_show_all(page);
}

/***************************************************************************
                     **** Happiness Page ****
 +- GtkWidget *page ----------+-------------------------------------------+
 | +- GtkWidget *left ------+ | +- GtkWidget *right --------------------+ |
 | | Info                   | | | City map                              | |
 | +- GtkWidget *citizens --+ | +- GtkWidget pdialog->happiness.widget -+ |
 | | Citizens data          | | | Happiness                             | |
 | +------------------------+ | +---------------------------------------+ |
 +----------------------------+-------------------------------------------+
****************************************************************************/
static void create_and_append_happiness_page(struct city_dialog *pdialog)
{
  GtkWidget *page, *label, *table, *right, *left, *frame;
  const char *tab_title = _("Happ_iness");

  /* main page */
  page = gtk_grid_new();
  gtk_grid_set_column_spacing(GTK_GRID(page), 6);
  gtk_container_set_border_width(GTK_CONTAINER(page), 8);
  label = gtk_label_new_with_mnemonic(tab_title);
  gtk_notebook_append_page(GTK_NOTEBOOK(pdialog->notebook), page, label);

  /* left: info, citizens */
  left = gtk_grid_new();
  gtk_orientable_set_orientation(GTK_ORIENTABLE(left),
                                 GTK_ORIENTATION_VERTICAL);
  gtk_container_add(GTK_CONTAINER(page), left);

  if (!low_citydlg) {
    /* upper left: info */
    frame = gtk_frame_new(_("Info"));
    gtk_container_add(GTK_CONTAINER(left), frame);

    table = create_city_info_table(pdialog,
                                   pdialog->happiness.info_ebox,
                                   pdialog->happiness.info_label);
    gtk_widget_set_halign(table, GTK_ALIGN_CENTER);
    gtk_container_add(GTK_CONTAINER(frame), table);
  }

  /* lower left: citizens */
  if (game.info.citizen_nationality) {
    pdialog->happiness.citizens = gtk_grid_new();
    gtk_orientable_set_orientation(
        GTK_ORIENTABLE(pdialog->happiness.citizens),
        GTK_ORIENTATION_VERTICAL);
    gtk_container_add(GTK_CONTAINER(left), pdialog->happiness.citizens);
    gtk_container_add(GTK_CONTAINER(pdialog->happiness.citizens),
                       citizens_dialog_display(pdialog->pcity));
  }

  /* right: city map, happiness */
  right = gtk_grid_new();
  gtk_orientable_set_orientation(GTK_ORIENTABLE(right),
                                 GTK_ORIENTATION_VERTICAL);
  gtk_container_add(GTK_CONTAINER(page), right);

  if (!low_citydlg) {
    /* upper right: city map */
    frame = gtk_frame_new(_("City map"));
    gtk_widget_set_size_request(frame, CITY_MAP_MIN_SIZE_X,
                                CITY_MAP_MIN_SIZE_Y);
    gtk_container_add(GTK_CONTAINER(right), frame);

    city_dialog_map_create(pdialog, &pdialog->happiness.map_canvas);
    gtk_container_add(GTK_CONTAINER(frame), pdialog->happiness.map_canvas.sw);
  }

  /* lower right: happiness */
  pdialog->happiness.widget = gtk_grid_new();
  gtk_orientable_set_orientation(GTK_ORIENTABLE(pdialog->happiness.widget),
                                 GTK_ORIENTATION_VERTICAL);
  gtk_container_add(GTK_CONTAINER(right), pdialog->happiness.widget);
  gtk_container_add(GTK_CONTAINER(pdialog->happiness.widget),
                    get_top_happiness_display(pdialog->pcity));

  /* show page */
  gtk_widget_show_all(page);
}

/****************************************************************
            **** Citizen Management Agent (CMA) Page ****
*****************************************************************/
static void create_and_append_cma_page(struct city_dialog *pdialog)
{
  GtkWidget *page, *label;
  const char *tab_title = _("_Governor");

  page = gtk_grid_new();

  label = gtk_label_new_with_mnemonic(tab_title);

  gtk_notebook_append_page(GTK_NOTEBOOK(pdialog->notebook), page, label);

  pdialog->cma_editor = create_cma_dialog(pdialog->pcity, low_citydlg);
  gtk_container_add(GTK_CONTAINER(page), pdialog->cma_editor->shell);

  gtk_widget_show(page);
}

/****************************************************************
                    **** Misc. Settings Page **** 
*****************************************************************/
static void create_and_append_settings_page(struct city_dialog *pdialog)
{
  int i;
  GtkWidget *vbox2, *page, *frame, *label, *button;
  GtkSizeGroup *size;
  GSList *group;
  const char *tab_title = _("_Settings");

  static const char *new_citizens_label[] = {
    N_("Entertainers"),
    N_("Scientists"),
    N_("Taxmen")
  };

  static const char *disband_label = N_("Disband if build settler at size 1");

  static const char *misc_whichtab_label[NUM_PAGES] = {
    N_("Overview page"),
    N_("Production page"),
    N_("Happiness page"),
    N_("Governor page"),
    N_("This Settings page"),
    N_("Last active page")
  };

  static bool new_citizens_label_done;
  static bool misc_whichtab_label_done;

  /* initialize signal_blocker */
  pdialog->misc.block_signal = 0;


  page = gtk_grid_new();
  gtk_grid_set_column_spacing(GTK_GRID(page), 18);
  gtk_container_set_border_width(GTK_CONTAINER(page), 8);
  
  size = gtk_size_group_new(GTK_SIZE_GROUP_BOTH);
  
  label = gtk_label_new_with_mnemonic(tab_title);

  gtk_notebook_append_page(GTK_NOTEBOOK(pdialog->notebook), page, label);

  /* new_citizens radio */
  frame = gtk_frame_new(_("New citizens are"));
  gtk_grid_attach(GTK_GRID(page), frame, 0, 0, 1, 1);
  gtk_size_group_add_widget(size, frame);

  vbox2 = gtk_grid_new();
  gtk_orientable_set_orientation(GTK_ORIENTABLE(vbox2),
                                 GTK_ORIENTATION_VERTICAL);
  gtk_container_add(GTK_CONTAINER(frame), vbox2);

  intl_slist(ARRAY_SIZE(new_citizens_label), new_citizens_label,
             &new_citizens_label_done);

  group = NULL;
  for (i = 0; i < ARRAY_SIZE(new_citizens_label); i++) {
    button = gtk_radio_button_new_with_mnemonic(group, new_citizens_label[i]);
    pdialog->misc.new_citizens_radio[i] = button;
    gtk_container_add(GTK_CONTAINER(vbox2), button);
    g_signal_connect(button, "toggled",
		     G_CALLBACK(cityopt_callback), pdialog);
    group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));
  }

  /* next is the next-time-open radio group in the right column */
  frame = gtk_frame_new(_("Next time open"));
  gtk_grid_attach(GTK_GRID(page), frame, 1, 0, 1, 1);
  gtk_size_group_add_widget(size, frame);

  vbox2 = gtk_grid_new();
  gtk_orientable_set_orientation(GTK_ORIENTABLE(vbox2),
                                 GTK_ORIENTATION_VERTICAL);
  gtk_container_add(GTK_CONTAINER(frame), vbox2);

  intl_slist(ARRAY_SIZE(misc_whichtab_label), misc_whichtab_label,
             &misc_whichtab_label_done);
  
  group = NULL;
  for (i = 0; i < ARRAY_SIZE(misc_whichtab_label); i++) {
    button = gtk_radio_button_new_with_mnemonic(group, misc_whichtab_label[i]);
    pdialog->misc.whichtab_radio[i] = button;
    gtk_container_add(GTK_CONTAINER(vbox2), button);
    g_signal_connect(button, "toggled",
		     G_CALLBACK(misc_whichtab_callback), GINT_TO_POINTER(i));
    group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));
  }

  /* now we go back and fill the hbox rename */
  frame = gtk_frame_new(_("City"));
  gtk_widget_set_margin_top(frame, 12);
  gtk_widget_set_margin_bottom(frame, 12);
  gtk_grid_attach(GTK_GRID(page), frame, 0, 1, 1, 1);

  vbox2 = gtk_grid_new();
  gtk_orientable_set_orientation(GTK_ORIENTABLE(vbox2),
                                 GTK_ORIENTATION_VERTICAL);
  gtk_container_add(GTK_CONTAINER(frame), vbox2);

  button = gtk_button_new_with_mnemonic(_("R_ename..."));
  pdialog->misc.rename_command = button;
  gtk_container_add(GTK_CONTAINER(vbox2), button);
  g_signal_connect(button, "clicked",
		   G_CALLBACK(rename_callback), pdialog);

  gtk_widget_set_sensitive(button, can_client_issue_orders());
  
  /* the disband-if-size-1 button */
  button = gtk_check_button_new_with_mnemonic(_(disband_label));
  pdialog->misc.disband_on_settler = button;
  gtk_container_add(GTK_CONTAINER(vbox2), button);
  g_signal_connect(button, "toggled",
		   G_CALLBACK(cityopt_callback), pdialog);

  /* we choose which page to popup by default */
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
			       (pdialog->
				misc.whichtab_radio[new_dialog_def_page]),
			       TRUE);

  set_cityopt_values(pdialog);

  gtk_widget_show_all(page);

  if (new_dialog_def_page == (NUM_PAGES - 1)) {
    gtk_notebook_set_current_page(GTK_NOTEBOOK(pdialog->notebook),
				  last_page);
  } else {
    gtk_notebook_set_current_page(GTK_NOTEBOOK(pdialog->notebook),
				  new_dialog_def_page);
  }
}




/****************************************************************
                     **** Main City Dialog ****
 +----------------------------+-------------------------------+
 | GtkWidget *top: Citizens   | city name                     |
 +----------------------------+-------------------------------+
 | <notebook tab>                                             |
 +------------------------------------------------------------+
*****************************************************************/
static struct city_dialog *create_city_dialog(struct city *pcity)
{
  struct city_dialog *pdialog;

  GtkWidget *close_command;
  GtkWidget *vbox, *hbox, *cbox, *ebox;

  if (!city_dialogs_have_been_initialised) {
    initialize_city_dialogs();
  }

  pdialog = fc_malloc(sizeof(struct city_dialog));
  pdialog->pcity = pcity;
  pdialog->buy_shell = NULL;
  pdialog->sell_shell = NULL;
  pdialog->rename_shell = NULL;
  pdialog->happiness.map_canvas.sw = NULL;      /* make sure NULL if spy */
  pdialog->happiness.map_canvas.ebox = NULL;    /* ditto */
  pdialog->happiness.map_canvas.darea = NULL;   /* ditto */
  pdialog->happiness.citizens = NULL;           /* ditto */
  pdialog->cma_editor = NULL;
  pdialog->map_canvas_store_unscaled
    = cairo_image_surface_create(CAIRO_FORMAT_ARGB32,
            canvas_width, canvas_height);

  pdialog->shell = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(pdialog->shell), city_name(pcity));
  setup_dialog(pdialog->shell, toplevel);
  gtk_window_set_role(GTK_WINDOW(pdialog->shell), "city");

  g_signal_connect(pdialog->shell, "destroy",
		   G_CALLBACK(city_destroy_callback), pdialog);
  gtk_window_set_position(GTK_WINDOW(pdialog->shell), GTK_WIN_POS_MOUSE);
  gtk_widget_set_name(pdialog->shell, "Freeciv");

  gtk_widget_realize(pdialog->shell);

  /* keep the icon of the executable on Windows (see PR#36491) */
#ifndef WIN32_NATIVE
  {
    GdkPixbuf *pixbuf = sprite_get_pixbuf(get_icon_sprite(tileset, ICON_CITYDLG));

    /* Only call this after tileset_load_tiles is called. */
    gtk_window_set_icon(GTK_WINDOW(pdialog->shell), pixbuf);
    g_object_unref(pixbuf);
  }
#endif /* WIN32_NATIVE */

  /* Restore size of the city dialog. */
  gtk_window_set_default_size(GTK_WINDOW(pdialog->shell),
                              gui_gtk3_citydlg_xsize,
                              gui_gtk3_citydlg_ysize);

  pdialog->popup_menu = gtk_menu_new();

  vbox = gtk_dialog_get_content_area(GTK_DIALOG(pdialog->shell));
  hbox = gtk_grid_new();
  gtk_grid_set_column_homogeneous(GTK_GRID(hbox), TRUE);
  gtk_container_add(GTK_CONTAINER(vbox), hbox);

  /**** Citizens bar here ****/
  cbox = gtk_grid_new();
  gtk_container_add(GTK_CONTAINER(hbox), cbox);

  ebox = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(ebox), FALSE);
  gtk_container_add(GTK_CONTAINER(cbox), ebox);
  pdialog->citizen_pixmap =
      gtk_pixcomm_new(tileset_small_sprite_width(tileset)
                      * NUM_CITIZENS_SHOWN,
                      tileset_small_sprite_height(tileset));
  gtk_widget_add_events(pdialog->citizen_pixmap, GDK_BUTTON_PRESS_MASK);
  gtk_misc_set_padding(GTK_MISC(pdialog->citizen_pixmap), 2, 2);
  gtk_misc_set_alignment(GTK_MISC(pdialog->citizen_pixmap), 0.0f, 0.5f);
  gtk_container_add(GTK_CONTAINER(ebox), pdialog->citizen_pixmap);
  g_signal_connect(G_OBJECT(ebox), "button-press-event",
                   G_CALLBACK(citizens_callback), pdialog);

  /**** City name label here ****/
  pdialog->name_label = gtk_label_new(NULL);
  gtk_widget_set_hexpand(pdialog->name_label, TRUE);
  gtk_misc_set_alignment(GTK_MISC(pdialog->name_label), 0.0f, 0.5f);
  gtk_container_add(GTK_CONTAINER(hbox), pdialog->name_label);

  /**** -Start of Notebook- ****/

  pdialog->notebook = gtk_notebook_new();
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(pdialog->notebook),
			   GTK_POS_BOTTOM);
  gtk_container_add(GTK_CONTAINER(vbox), pdialog->notebook);

  create_and_append_overview_page(pdialog);
  create_and_append_map_page(pdialog);
  create_and_append_buildings_page(pdialog);
  create_and_append_worklist_page(pdialog);

  /* only create these tabs if not a spy */
  if (!client_has_player() || city_owner(pcity) == client_player()) {
    create_and_append_happiness_page(pdialog);
  }

  if (city_owner(pcity) == client_player()
      && !client_is_observer()) {
    create_and_append_cma_page(pdialog);
    create_and_append_settings_page(pdialog);
  } else {
    gtk_notebook_set_current_page(GTK_NOTEBOOK(pdialog->notebook),
                                  OVERVIEW_PAGE);
  }

  /**** End of Notebook ****/

  /* bottom buttons */

  pdialog->show_units_command =
	gtk_button_new_with_mnemonic(_("_List present units..."));
  gtk_container_add(GTK_CONTAINER(gtk_dialog_get_action_area(GTK_DIALOG(pdialog->shell))),
		    pdialog->show_units_command);
  gtk_button_box_set_child_secondary(GTK_BUTTON_BOX(gtk_dialog_get_action_area(GTK_DIALOG(pdialog->shell))),
      pdialog->show_units_command, TRUE);
  g_signal_connect(pdialog->show_units_command,
		   "clicked",
		   G_CALLBACK(show_units_callback), pdialog);

  pdialog->prev_command = gtk_stockbutton_new(GTK_STOCK_GO_BACK,
	_("_Prev city"));
  gtk_dialog_add_action_widget(GTK_DIALOG(pdialog->shell),
			       pdialog->prev_command, 1);

  pdialog->next_command = gtk_stockbutton_new(GTK_STOCK_GO_FORWARD,
	_("_Next city"));
  gtk_dialog_add_action_widget(GTK_DIALOG(pdialog->shell),
			       pdialog->next_command, 2);
  
  if (NULL == client.conn.playing
      || city_owner(pcity) != client.conn.playing) {
    gtk_widget_set_sensitive(pdialog->prev_command, FALSE);
    gtk_widget_set_sensitive(pdialog->next_command, FALSE);
  }

  close_command = gtk_dialog_add_button(GTK_DIALOG(pdialog->shell),
					GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);

  gtk_dialog_set_default_response(GTK_DIALOG(pdialog->shell),
	GTK_RESPONSE_CLOSE);

  g_signal_connect(close_command, "clicked",
		   G_CALLBACK(close_callback), pdialog);

  g_signal_connect(pdialog->prev_command, "clicked",
		   G_CALLBACK(switch_city_callback), pdialog);

  g_signal_connect(pdialog->next_command, "clicked",
		   G_CALLBACK(switch_city_callback), pdialog);

  /* some other things we gotta do */

  g_signal_connect(pdialog->shell, "key_press_event",
		   G_CALLBACK(keyboard_handler), pdialog);

  dialog_list_prepend(dialog_list, pdialog);

  real_city_dialog_refresh(pdialog->pcity);

  /* need to do this every time a new dialog is opened. */
  city_dialog_update_prev_next();

  gtk_widget_show_all(gtk_dialog_get_content_area(GTK_DIALOG(pdialog->shell)));
  gtk_widget_show_all(gtk_dialog_get_action_area(GTK_DIALOG(pdialog->shell)));

  gtk_window_set_focus(GTK_WINDOW(pdialog->shell), close_command);

  return pdialog;
}

/*********** Functions to update parts of the dialog ************/
/****************************************************************
  Update title of city dialog.
*****************************************************************/
static void city_dialog_update_title(struct city_dialog *pdialog)
{
  gchar *buf;
  const gchar *now;

  if (city_unhappy(pdialog->pcity)) {
    /* TRANS: city dialog title */
    buf = g_strdup_printf(_("<b>%s</b> - %s citizens - DISORDER"),
			  city_name(pdialog->pcity),
			  population_to_text(city_population(pdialog->pcity)));
  } else if (city_celebrating(pdialog->pcity)) {
    /* TRANS: city dialog title */
    buf = g_strdup_printf(_("<b>%s</b> - %s citizens - celebrating"),
			  city_name(pdialog->pcity),
			  population_to_text(city_population(pdialog->pcity)));
  } else if (city_happy(pdialog->pcity)) {
    /* TRANS: city dialog title */
    buf = g_strdup_printf(_("<b>%s</b> - %s citizens - happy"),
			  city_name(pdialog->pcity),
			  population_to_text(city_population(pdialog->pcity)));
  } else {
    /* TRANS: city dialog title */
    buf = g_strdup_printf(_("<b>%s</b> - %s citizens"),
			  city_name(pdialog->pcity),
			  population_to_text(city_population(pdialog->pcity)));

  }

  now = gtk_label_get_text(GTK_LABEL(pdialog->name_label));
  if (strcmp(now, buf) != 0) {
    gtk_window_set_title(GTK_WINDOW(pdialog->shell), city_name(pdialog->pcity));
    gtk_label_set_markup(GTK_LABEL(pdialog->name_label), buf);
  }

  g_free(buf);
}

/****************************************************************
  Update citizens in city dialog
*****************************************************************/
static void city_dialog_update_citizens(struct city_dialog *pdialog)
{
  enum citizen_category citizens[MAX_CITY_SIZE];
  int i, width, size, xpad;
  struct city *pcity = pdialog->pcity;
  int num_citizens = get_city_citizen_types(pcity, FEELING_FINAL, citizens);

  /* If there is not enough space we stack the icons. We draw from left to */
  /* right. width is how far we go to the right for each drawn pixmap. The */
  /* last icon is always drawn in full, and so has reserved                */
  /* tileset_small_sprite_width(tileset) pixels.                           */

  if (num_citizens > 1) {
    width = MIN(tileset_small_sprite_width(tileset),
		((NUM_CITIZENS_SHOWN - 1) * tileset_small_sprite_width(tileset)) /
		(num_citizens - 1));
  } else {
    width = tileset_small_sprite_width(tileset);
  }
  pdialog->cwidth = width;

  /* overview page */
  gtk_misc_get_padding(GTK_MISC(pdialog->citizen_pixmap), &xpad, NULL);
  gtk_pixcomm_clear(GTK_PIXCOMM(pdialog->citizen_pixmap));

  size = (num_citizens - 1) * width + tileset_small_sprite_width(tileset) +
         2 * xpad;
  gtk_widget_set_size_request(GTK_WIDGET(pdialog->citizen_pixmap), size, -1);

  for (i = 0; i < num_citizens; i++) {
    gtk_pixcomm_copyto(GTK_PIXCOMM(pdialog->citizen_pixmap),
                       get_citizen_sprite(tileset, citizens[i], i, pcity),
                       i * width, 0);
  }
}

/****************************************************************
  Update textual info fields in city dialog
*****************************************************************/
static void city_dialog_update_information(GtkWidget **info_ebox,
					   GtkWidget **info_label,
                                           struct city_dialog *pdialog)
{
  int i, illness = 0;
  char buf[NUM_INFO_FIELDS][512];
  struct city *pcity = pdialog->pcity;
  int granaryturns;
  GdkRGBA red = {1.0, 0, 0, 1.0};
  GdkRGBA *color;

  enum { FOOD, SHIELD, TRADE, GOLD, LUXURY, SCIENCE,
         GRANARY, GROWTH, CORRUPTION, WASTE, POLLUTION, ILLNESS
  };

  /* fill the buffers with the necessary info */
  fc_snprintf(buf[FOOD], sizeof(buf[FOOD]), "%3d (%+4d)",
              pcity->prod[O_FOOD], pcity->surplus[O_FOOD]);
  fc_snprintf(buf[SHIELD], sizeof(buf[SHIELD]), "%3d (%+4d)",
              pcity->prod[O_SHIELD] + pcity->waste[O_SHIELD],
              pcity->surplus[O_SHIELD]);
  fc_snprintf(buf[TRADE], sizeof(buf[TRADE]), "%3d (%+4d)",
              pcity->surplus[O_TRADE] + pcity->waste[O_TRADE],
              pcity->surplus[O_TRADE]);
  fc_snprintf(buf[GOLD], sizeof(buf[GOLD]), "%3d (%+4d)",
              pcity->prod[O_GOLD], pcity->surplus[O_GOLD]);
  fc_snprintf(buf[LUXURY], sizeof(buf[LUXURY]), "%3d",
              pcity->prod[O_LUXURY]);
  fc_snprintf(buf[SCIENCE], sizeof(buf[SCIENCE]), "%3d",
              pcity->prod[O_SCIENCE]);
  fc_snprintf(buf[GRANARY], sizeof(buf[GRANARY]), "%4d/%-4d",
              pcity->food_stock, city_granary_size(city_size_get(pcity)));

  granaryturns = city_turns_to_grow(pcity);
  if (granaryturns == 0) {
    /* TRANS: city growth is blocked.  Keep short. */
    fc_snprintf(buf[GROWTH], sizeof(buf[GROWTH]), _("blocked"));
  } else if (granaryturns == FC_INFINITY) {
    /* TRANS: city is not growing.  Keep short. */
    fc_snprintf(buf[GROWTH], sizeof(buf[GROWTH]), _("never"));
  } else {
    /* A negative value means we'll have famine in that many turns.
       But that's handled down below. */
    /* TRANS: city growth turns.  Keep short. */
    fc_snprintf(buf[GROWTH], sizeof(buf[GROWTH]),
                PL_("%d turn", "%d turns", abs(granaryturns)),
                abs(granaryturns));
  }
  fc_snprintf(buf[CORRUPTION], sizeof(buf[CORRUPTION]), "%4d",
              pcity->waste[O_TRADE]);
  fc_snprintf(buf[WASTE], sizeof(buf[WASTE]), "%4d",
              pcity->waste[O_SHIELD]);
  fc_snprintf(buf[POLLUTION], sizeof(buf[POLLUTION]), "%4d",
              pcity->pollution);
  if (!game.info.illness_on) {
    fc_snprintf(buf[ILLNESS], sizeof(buf[ILLNESS]), " -.-");
  } else {
    illness = city_illness_calc(pcity, NULL, NULL, NULL, NULL);
    /* illness is in tenth of percent */
    fc_snprintf(buf[ILLNESS], sizeof(buf[ILLNESS]), "%4.1f",
                (float)illness / 10.0);
  }

  /* stick 'em in the labels */
  for (i = 0; i < NUM_INFO_FIELDS; i++) {
    gtk_label_set_text(GTK_LABEL(info_label[i]), buf[i]);
  }

  /* 
   * Special style stuff for granary, growth and pollution below. The
   * "4" below is arbitrary. 3 turns should be enough of a warning.
   */
  color = (granaryturns > -4 && granaryturns < 0) ? &red : NULL;
  gtk_widget_override_color(info_label[GRANARY], GTK_STATE_FLAG_NORMAL, color);

  color = (granaryturns == 0 || pcity->surplus[O_FOOD] < 0) ? &red : NULL;
  gtk_widget_override_color(info_label[GROWTH], GTK_STATE_FLAG_NORMAL, color);

  /* someone could add the color &orange for better granularity here */

  color = (pcity->pollution >= 10) ? &red : NULL;
  gtk_widget_override_color(info_label[POLLUTION], GTK_STATE_FLAG_NORMAL, color);

  /* illness is in tenth of percent, i.e 100 != 10.0% */
  color = (illness >= 100) ? &red : NULL;
  gtk_widget_override_color(info_label[ILLNESS], GTK_STATE_FLAG_NORMAL, color);
}

/****************************************************************
  Update map display of city dialog
*****************************************************************/
static void city_dialog_update_map(struct city_dialog *pdialog)
{
  struct canvas store = FC_STATIC_CANVAS_INIT;

  store.surface = pdialog->map_canvas_store_unscaled;

  /* The drawing is done in three steps.
   *   1.  First we render to a pixmap with the appropriate canvas size.
   *   2.  Then the pixmap is rendered into a pixbuf of equal size.
   *   3.  Finally this pixbuf is composited and scaled onto the GtkImage's
   *       target pixbuf.
   */

  city_dialog_redraw_map(pdialog->pcity, &store);

  /* draw to real window */
  draw_map_canvas(pdialog);

  if (cma_is_city_under_agent(pdialog->pcity, NULL)) {
    gtk_widget_set_sensitive(pdialog->overview.map_canvas.ebox, FALSE);
    if (pdialog->happiness.map_canvas.ebox) {
      gtk_widget_set_sensitive(pdialog->happiness.map_canvas.ebox, FALSE);
    }
  } else {
    gtk_widget_set_sensitive(pdialog->overview.map_canvas.ebox, TRUE);
    if (pdialog->happiness.map_canvas.ebox) {
      gtk_widget_set_sensitive(pdialog->happiness.map_canvas.ebox, TRUE);
    }
  }
}

/****************************************************************
  Update what city is building and buy cost in city dialog
*****************************************************************/
static void city_dialog_update_building(struct city_dialog *pdialog)
{
  char buf[32], buf2[200];
  gdouble pct;

  GtkListStore* store;
  GtkTreeIter iter;
  struct universal targets[MAX_NUM_PRODUCTION_TARGETS];
  struct item items[MAX_NUM_PRODUCTION_TARGETS];
  int targets_used, item;
  struct city *pcity = pdialog->pcity;
  gboolean sensitive = city_can_buy(pcity);
  const char *descr = city_production_name_translation(pcity);
  int cost = city_production_build_shield_cost(pcity);

  if (pdialog->overview.buy_command != NULL) {
    gtk_widget_set_sensitive(pdialog->overview.buy_command, sensitive);
  }
  gtk_widget_set_sensitive(pdialog->production.buy_command, sensitive);

  /* Make sure build slots info is up to date */
  {
    int build_slots = city_build_slots(pcity);
    /* Only display extra info if more than one slot is available */
    if (build_slots > 1) {
      fc_snprintf(buf2, sizeof(buf2),
                  /* TRANS: never actually used with built_slots<=1 */
                  PL_("Production (up to %d unit per turn):",
                      "Production (up to %d units per turn):", build_slots),
                  build_slots);
      gtk_label_set_text(
        GTK_LABEL(pdialog->production.production_label), buf2);
    } else {
      gtk_label_set_text(
        GTK_LABEL(pdialog->production.production_label), _("Production:"));
    }
  }

  /* Update what the city is working on */
  get_city_dialog_production(pcity, buf, sizeof(buf));

  if (cost > 0) {
    pct = (gdouble) pcity->shield_stock / (gdouble) cost;
    pct = CLAMP(pct, 0.0, 1.0);
  } else {
    pct = 1.0;
  }

  if (pdialog->overview.production_bar != NULL) {
    fc_snprintf(buf2, sizeof(buf2), "%s%s\n%s", descr,
                worklist_is_empty(&pcity->worklist) ? "" : " (+)", buf);
    gtk_progress_bar_set_text(
      GTK_PROGRESS_BAR(pdialog->overview.production_bar), buf2);
    gtk_progress_bar_set_fraction(
      GTK_PROGRESS_BAR(pdialog->overview.production_bar), pct);
  }

  fc_snprintf(buf2, sizeof(buf2), "%s%s: %s", descr,
              worklist_is_empty(&pcity->worklist) ? "" : " (+)", buf);
  gtk_progress_bar_set_text(
    GTK_PROGRESS_BAR(pdialog->production.production_bar), buf2);
  gtk_progress_bar_set_fraction(
    GTK_PROGRESS_BAR(pdialog->production.production_bar), pct);

  if (pdialog->overview.production_combo != NULL) {
    gtk_combo_box_set_active(GTK_COMBO_BOX(pdialog->overview.production_combo),
                             -1);
  }

  store = pdialog->overview.change_production_store;
  if (store != NULL) {
    gtk_list_store_clear(pdialog->overview.change_production_store);

    targets_used
      = collect_eventually_buildable_targets(targets, pdialog->pcity, FALSE);  
    name_and_sort_items(targets, targets_used, items, FALSE, pcity);

    for (item = 0; item < targets_used; item++) {
      if (can_city_build_now(pcity, items[item].item)) {
        const char* name;
        struct sprite* sprite;
        GdkPixbuf *pix;
        struct universal target = items[item].item;
        bool useless;

        if (VUT_UTYPE == target.kind) {
          name = utype_name_translation(target.value.utype);
          sprite = get_unittype_sprite(tileset, target.value.utype,
                                       direction8_invalid(), TRUE);
          useless = FALSE;
        } else {
          name = improvement_name_translation(target.value.building);
          sprite = get_building_sprite(tileset, target.value.building);
          useless = is_improvement_redundant(pcity, target.value.building);
        }
        pix = sprite_get_pixbuf(sprite);
        gtk_list_store_append(store, &iter);
        gtk_list_store_set(store, &iter, 0, pix,
                           1, name, 3, useless,
                           2, (gint)cid_encode(items[item].item),-1);
        g_object_unref(G_OBJECT(pix));
      }
    }
  }

  /* work around GTK+ refresh bug. */
  if (pdialog->overview.production_bar != NULL) {
    gtk_widget_queue_resize(pdialog->overview.production_bar);
  }
  gtk_widget_queue_resize(pdialog->production.production_bar);
}

/****************************************************************
  Update list of improvements in city dialog
*****************************************************************/
static void city_dialog_update_improvement_list(struct city_dialog *pdialog)
{
  int total, item, targets_used;
  struct universal targets[MAX_NUM_PRODUCTION_TARGETS];
  struct item items[MAX_NUM_PRODUCTION_TARGETS];
  GtkTreeModel *model;
  GtkListStore *store;

  model =
    gtk_tree_view_get_model(GTK_TREE_VIEW(pdialog->overview.improvement_list));
  store = GTK_LIST_STORE(model);
  
  targets_used = collect_already_built_targets(targets, pdialog->pcity);
  name_and_sort_items(targets, targets_used, items, FALSE, pdialog->pcity);

  gtk_list_store_clear(store);  

  total = 0;
  for (item = 0; item < targets_used; item++) {
    GdkPixbuf *pix;
    GtkTreeIter it;
    int upkeep;
    struct sprite *sprite;
    struct universal target = items[item].item;

    fc_assert_action(VUT_IMPROVEMENT == target.kind, continue);
    /* This takes effects (like Adam Smith's) into account. */
    upkeep = city_improvement_upkeep(pdialog->pcity, target.value.building);
    sprite = get_building_sprite(tileset, target.value.building);

    pix = sprite_get_pixbuf(sprite);
    gtk_list_store_append(store, &it);
    gtk_list_store_set(store, &it,
		       0, target.value.building,
		       1, pix,
	2, items[item].descr,
	3, upkeep,
        4, is_improvement_redundant(pdialog->pcity, target.value.building),
	-1);
    g_object_unref(G_OBJECT(pix));

    total += upkeep;
  }
}

/****************************************************************
  Update list of supported units in city dialog
*****************************************************************/
static void city_dialog_update_supported_units(struct city_dialog *pdialog)
{
  struct unit_list *units;
  struct unit_node_vector *nodes;
  int n, m, i;
  gchar *buf;
  int free_unhappy = get_city_bonus(pdialog->pcity, EFT_MAKE_CONTENT_MIL);

  if (NULL != client.conn.playing
      && city_owner(pdialog->pcity) != client.conn.playing) {
    units = pdialog->pcity->client.info_units_supported;
  } else {
    units = pdialog->pcity->units_supported;
  }

  nodes = &pdialog->overview.supported_units;

  n = unit_list_size(units);
  m = unit_node_vector_size(nodes);

  if (m > n) {
    i = 0;
    unit_node_vector_iterate(nodes, elt) {
      if (i++ >= n) {
	gtk_widget_destroy(elt->cmd);
      }
    } unit_node_vector_iterate_end;

    unit_node_vector_reserve(nodes, n);
  } else {
    for (i = m; i < n; i++) {
      GtkWidget *cmd, *pix;
      struct unit_node node;
      int unit_height = tileset_tile_height(tileset) * 3 / 2;

      cmd = gtk_button_new();
      node.cmd = cmd;

      gtk_button_set_relief(GTK_BUTTON(cmd), GTK_RELIEF_NONE);
      gtk_widget_add_events(cmd,
	  GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);

      pix = gtk_pixcomm_new(tileset_full_tile_width(tileset), unit_height);
      node.pix = pix;

      gtk_container_add(GTK_CONTAINER(cmd), pix);

      gtk_grid_attach(GTK_GRID(pdialog->overview.supported_unit_table),
                      cmd, i, 0, 1, 1);
      unit_node_vector_append(nodes, node);
    }
  }

  i = 0;
  unit_list_iterate(units, punit) {
    struct unit_node *pnode;
    int happy_cost = city_unit_unhappiness(punit, &free_unhappy);

    pnode = unit_node_vector_get(nodes, i);
    if (pnode) {
      GtkWidget *cmd, *pix;

      cmd = pnode->cmd;
      pix = pnode->pix;

      put_unit_gpixmap(punit, GTK_PIXCOMM(pix));
      put_unit_gpixmap_city_overlays(punit, GTK_PIXCOMM(pix), punit->upkeep,
                                     happy_cost);

      g_signal_handlers_disconnect_matched(cmd,
	  G_SIGNAL_MATCH_FUNC,
	  0, 0, NULL, supported_unit_callback, NULL);

      g_signal_handlers_disconnect_matched(cmd,
	  G_SIGNAL_MATCH_FUNC,
	  0, 0, NULL, supported_unit_middle_callback, NULL);

      gtk_widget_set_tooltip_text(cmd, unit_description(punit));

      g_signal_connect(cmd, "button_press_event",
	  G_CALLBACK(supported_unit_callback),
	  GINT_TO_POINTER(punit->id));

      g_signal_connect(cmd, "button_release_event",
	  G_CALLBACK(supported_unit_middle_callback),
	  GINT_TO_POINTER(punit->id));

      if (city_owner(pdialog->pcity) != client.conn.playing) {
	gtk_widget_set_sensitive(cmd, FALSE);
      } else {
	gtk_widget_set_sensitive(cmd, TRUE);
      }

      gtk_widget_show(pix);
      gtk_widget_show(cmd);
    }
    i++;
  } unit_list_iterate_end;

  buf = g_strdup_printf(_("Supported units %d"), n);
  gtk_frame_set_label(GTK_FRAME(pdialog->overview.supported_units_frame), buf);
  g_free(buf);
}

/****************************************************************
  Update list of present units in city dialog
*****************************************************************/
static void city_dialog_update_present_units(struct city_dialog *pdialog)
{
  struct unit_list *units;
  struct unit_node_vector *nodes;
  int n, m, i;
  gchar *buf;

  if (NULL != client.conn.playing
      && city_owner(pdialog->pcity) != client.conn.playing) {
    units = pdialog->pcity->client.info_units_present;
  } else {
    units = pdialog->pcity->tile->units;
  }

  nodes = &pdialog->overview.present_units;

  n = unit_list_size(units);
  m = unit_node_vector_size(nodes);

  if (m > n) {
    i = 0;
    unit_node_vector_iterate(nodes, elt) {
      if (i++ >= n) {
	gtk_widget_destroy(elt->cmd);
      }
    } unit_node_vector_iterate_end;

    unit_node_vector_reserve(nodes, n);
  } else {
    for (i = m; i < n; i++) {
      GtkWidget *cmd, *pix;
      struct unit_node node;

      cmd = gtk_button_new();
      node.cmd = cmd;

      gtk_button_set_relief(GTK_BUTTON(cmd), GTK_RELIEF_NONE);
      gtk_widget_add_events(cmd,
	  GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);

      pix = gtk_pixcomm_new(tileset_full_tile_width(tileset),
                            tileset_full_tile_height(tileset));
      node.pix = pix;

      gtk_container_add(GTK_CONTAINER(cmd), pix);

      gtk_grid_attach(GTK_GRID(pdialog->overview.present_unit_table),
                      cmd, i, 0, 1, 1);
      unit_node_vector_append(nodes, node);
    }
  }

  i = 0;
  unit_list_iterate(units, punit) {
    struct unit_node *pnode;
    
    pnode = unit_node_vector_get(nodes, i);
    if (pnode) {
      GtkWidget *cmd, *pix;

      cmd = pnode->cmd;
      pix = pnode->pix;

      put_unit_gpixmap(punit, GTK_PIXCOMM(pix));

      g_signal_handlers_disconnect_matched(cmd,
	  G_SIGNAL_MATCH_FUNC,
	  0, 0, NULL, present_unit_callback, NULL);

      g_signal_handlers_disconnect_matched(cmd,
	  G_SIGNAL_MATCH_FUNC,
	  0, 0, NULL, present_unit_middle_callback, NULL);

      gtk_widget_set_tooltip_text(cmd, unit_description(punit));

      g_signal_connect(cmd, "button_press_event",
	  G_CALLBACK(present_unit_callback),
	  GINT_TO_POINTER(punit->id));

      g_signal_connect(cmd, "button_release_event",
	  G_CALLBACK(present_unit_middle_callback),
	  GINT_TO_POINTER(punit->id));

      if (city_owner(pdialog->pcity) != client.conn.playing) {
	gtk_widget_set_sensitive(cmd, FALSE);
      } else {
	gtk_widget_set_sensitive(cmd, TRUE);
      }

      gtk_widget_show(pix);
      gtk_widget_show(cmd);
    }
    i++;
  } unit_list_iterate_end;

  buf = g_strdup_printf(_("Present units %d"), n);
  gtk_frame_set_label(GTK_FRAME(pdialog->overview.present_units_frame), buf);
  g_free(buf);
}

/****************************************************************
 updates the sensitivity of the the prev and next buttons.
 this does not need pdialog as a parameter, since it iterates
 over all the open dialogs.
 note: we still need the sensitivity code in create_city_dialog()
 for the spied dialogs.
*****************************************************************/
static void city_dialog_update_prev_next(void)
{
  int count = 0;
  int city_number;

  if (NULL != client.conn.playing) {
    city_number = city_list_size(client.conn.playing->cities);
  } else {
    city_number = FC_INFINITY; /* ? */
  }

  /* the first time, we see if all the city dialogs are open */

  dialog_list_iterate(dialog_list, pdialog) {
    if (city_owner(pdialog->pcity) == client.conn.playing)
      count++;
  }
  dialog_list_iterate_end;

  if (count == city_number) {	/* all are open, shouldn't prev/next */
    dialog_list_iterate(dialog_list, pdialog) {
      gtk_widget_set_sensitive(pdialog->prev_command, FALSE);
      gtk_widget_set_sensitive(pdialog->next_command, FALSE);
    }
    dialog_list_iterate_end;
  } else {
    dialog_list_iterate(dialog_list, pdialog) {
      if (city_owner(pdialog->pcity) == client.conn.playing) {
	gtk_widget_set_sensitive(pdialog->prev_command, TRUE);
	gtk_widget_set_sensitive(pdialog->next_command, TRUE);
      }
    }
    dialog_list_iterate_end;
  }
}

/****************************************************************
  User has clicked show units
*****************************************************************/
static void show_units_callback(GtkWidget * w, gpointer data)
{
  struct city_dialog *pdialog = (struct city_dialog *) data;
  struct tile *ptile = pdialog->pcity->tile;

  if (unit_list_size(ptile->units))
    unit_select_dialog_popup(ptile);
}

/****************************************************************
  Set city menu position
*****************************************************************/
static void city_menu_position(GtkMenu *menu, gint *x, gint *y,
                               gboolean *push_in, gpointer data)
{
  GtkWidget *widget;
  GtkAllocation allocation;
  gint xpos;
  gint ypos;

  fc_assert_ret(GTK_IS_BUTTON(data));

  widget = GTK_WIDGET(data);

  gtk_widget_get_allocation(widget, &allocation);

  gdk_window_get_origin(gtk_widget_get_window(widget), &xpos, &ypos);

  xpos += allocation.x + allocation.width/2;
  ypos += allocation.y + allocation.height/2;

  *x = xpos;
  *y = ypos;
  *push_in = TRUE;
}

/****************************************************************
  Destroy widget -callback
*****************************************************************/
static void destroy_func(GtkWidget *w, gpointer data)
{
  gtk_widget_destroy(w);
}

/****************************************************************
Pop-up menu to change attributes of supported units
*****************************************************************/
static gboolean supported_unit_callback(GtkWidget * w, GdkEventButton * ev,
				        gpointer data)
{
  GtkWidget *menu, *item;
  struct city_dialog *pdialog;
  struct city *pcity;
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t) data);

  if (NULL != punit
   && NULL != (pcity = game_city_by_number(punit->homecity))
   && NULL != (pdialog = get_city_dialog(pcity))) {

    if (ev->type != GDK_BUTTON_PRESS || ev->button == 2 || ev->button == 3
	|| !can_client_issue_orders()) {
      return FALSE;
    }

    menu = pdialog->popup_menu;

    gtk_menu_popdown(GTK_MENU(menu));
    gtk_container_foreach(GTK_CONTAINER(menu), destroy_func, NULL);

    item = gtk_menu_item_new_with_mnemonic(_("Cen_ter"));
    g_signal_connect(item, "activate",
      G_CALLBACK(unit_center_callback),
      GINT_TO_POINTER(punit->id));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    
    item = gtk_menu_item_new_with_mnemonic(_("_Activate unit"));
    g_signal_connect(item, "activate",
      G_CALLBACK(unit_activate_callback),
      GINT_TO_POINTER(punit->id));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

    item = gtk_menu_item_new_with_mnemonic(_("Activate unit, _close dialog"));
    g_signal_connect(item, "activate",
      G_CALLBACK(supported_unit_activate_close_callback),
      GINT_TO_POINTER(punit->id));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

    item = gtk_menu_item_new_with_mnemonic(_("_Disband unit"));
    g_signal_connect(item, "activate",
      G_CALLBACK(unit_disband_callback),
      GINT_TO_POINTER(punit->id));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

    if (unit_has_type_flag(punit, UTYF_UNDISBANDABLE)) {
      gtk_widget_set_sensitive(item, FALSE);
    }

    gtk_widget_show_all(menu);

    gtk_menu_popup(GTK_MENU(menu), NULL, NULL,
      city_menu_position, w, ev->button, ev->time);


  }
  return TRUE;
}

/****************************************************************
Pop-up menu to change attributes of units, ex. change homecity.
*****************************************************************/
static gboolean present_unit_callback(GtkWidget * w, GdkEventButton * ev,
				      gpointer data)
{
  GtkWidget *menu, *item;
  struct city_dialog *pdialog;
  struct city *pcity;
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t) data);

  if (NULL != punit
   && NULL != (pcity = tile_city(unit_tile(punit)))
   && NULL != (pdialog = get_city_dialog(pcity))) {

    if (ev->type != GDK_BUTTON_PRESS || ev->button == 2 || ev->button == 3
	|| !can_client_issue_orders()) {
      return FALSE;
    }

    menu = pdialog->popup_menu;

    gtk_menu_popdown(GTK_MENU(menu));
    gtk_container_foreach(GTK_CONTAINER(menu), destroy_func, NULL);

    item = gtk_menu_item_new_with_mnemonic(_("_Activate unit"));
    g_signal_connect(item, "activate",
      G_CALLBACK(unit_activate_callback),
      GINT_TO_POINTER(punit->id));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

    item = gtk_menu_item_new_with_mnemonic(_("Activate unit, _close dialog"));
    g_signal_connect(item, "activate",
      G_CALLBACK(present_unit_activate_close_callback),
      GINT_TO_POINTER(punit->id));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

    item = gtk_menu_item_new_with_mnemonic(_("_Load unit"));
    g_signal_connect(item, "activate",
      G_CALLBACK(unit_load_callback),
      GINT_TO_POINTER(punit->id));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

    if (NULL == transporter_for_unit(punit)) {
      gtk_widget_set_sensitive(item, FALSE);
    }

    item = gtk_menu_item_new_with_mnemonic(_("_Unload unit"));
    g_signal_connect(item, "activate",
      G_CALLBACK(unit_unload_callback),
      GINT_TO_POINTER(punit->id));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

    if (!can_unit_unload(punit, unit_transport_get(punit))
        || !can_unit_exist_at_tile(punit, unit_tile(punit))) {
      gtk_widget_set_sensitive(item, FALSE);
    }

    item = gtk_menu_item_new_with_mnemonic(_("_Sentry unit"));
    g_signal_connect(item, "activate",
      G_CALLBACK(unit_sentry_callback),
      GINT_TO_POINTER(punit->id));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

    if (punit->activity == ACTIVITY_SENTRY
	|| !can_unit_do_activity(punit, ACTIVITY_SENTRY)) {
      gtk_widget_set_sensitive(item, FALSE);
    }

    item = gtk_menu_item_new_with_mnemonic(_("_Fortify unit"));
    g_signal_connect(item, "activate",
      G_CALLBACK(unit_fortify_callback),
      GINT_TO_POINTER(punit->id));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

    if (punit->activity == ACTIVITY_FORTIFYING
	|| !can_unit_do_activity(punit, ACTIVITY_FORTIFYING)) {
      gtk_widget_set_sensitive(item, FALSE);
    }

    item = gtk_menu_item_new_with_mnemonic(_("_Disband unit"));
    g_signal_connect(item, "activate",
      G_CALLBACK(unit_disband_callback),
      GINT_TO_POINTER(punit->id));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

    if (unit_has_type_flag(punit, UTYF_UNDISBANDABLE)) {
      gtk_widget_set_sensitive(item, FALSE);
    }

    item = gtk_menu_item_new_with_mnemonic(_("Set _Home City"));
    g_signal_connect(item, "activate",
      G_CALLBACK(unit_homecity_callback),
      GINT_TO_POINTER(punit->id));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    gtk_widget_set_sensitive(item, can_unit_change_homecity_to(punit, pcity));

    item = gtk_menu_item_new_with_mnemonic(_("U_pgrade unit"));
    g_signal_connect(item, "activate",
      G_CALLBACK(unit_upgrade_callback),
      GINT_TO_POINTER(punit->id));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

    if (!can_client_issue_orders()
	|| NULL == can_upgrade_unittype(client.conn.playing, unit_type(punit))) {
      gtk_widget_set_sensitive(item, FALSE);
    }

    gtk_widget_show_all(menu);

    gtk_menu_popup(GTK_MENU(menu), NULL, NULL,
      city_menu_position, w, ev->button, ev->time);
  }
  return TRUE;
}

/****************************************************************
 if user middle-clicked on a unit, activate it and close dialog
*****************************************************************/
static gboolean present_unit_middle_callback(GtkWidget * w,
					     GdkEventButton * ev,
					     gpointer data)
{
  struct city_dialog *pdialog;
  struct city *pcity;
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t) data);

  if (NULL != punit
   && NULL != (pcity = tile_city(unit_tile(punit)))
   && NULL != (pdialog = get_city_dialog(pcity))
   && can_client_issue_orders()) {

    if (ev->button == 3) {
      unit_focus_set(punit);
    } else if (ev->button == 2) {
      unit_focus_set(punit);
      close_city_dialog(pdialog);
    }
  }

  return TRUE;
}

/****************************************************************
 if user middle-clicked on a unit, activate it and close dialog
*****************************************************************/
static gboolean supported_unit_middle_callback(GtkWidget * w,
					       GdkEventButton * ev,
					       gpointer data)
{
  struct city_dialog *pdialog;
  struct city *pcity;
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t) data);

  if (NULL != punit
   && NULL != (pcity = game_city_by_number(punit->homecity))
   && NULL != (pdialog = get_city_dialog(pcity))
   && can_client_issue_orders()) {

    if (ev->button == 3) {
      unit_focus_set(punit);
    } else if (ev->button == 2) {
      unit_focus_set(punit);
      close_city_dialog(pdialog);
    }
  }

  return TRUE;
}

/****************************************************************
  User has requested centering to unit
*****************************************************************/
static void unit_center_callback(GtkWidget * w, gpointer data)
{
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t)data);

  if (NULL != punit) {
    center_tile_mapcanvas(unit_tile(punit));
  }
}

/****************************************************************
  User has requested unit activation
*****************************************************************/
static void unit_activate_callback(GtkWidget * w, gpointer data)
{
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t)data);

  if (NULL != punit) {
    unit_focus_set(punit);
  }
}

/****************************************************************
  User has requested some supported unit to be activated and
  city dialog to be closed
*****************************************************************/
static void supported_unit_activate_close_callback(GtkWidget * w,
						   gpointer data)
{
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t)data);

  if (NULL != punit) {
    struct city *pcity =
      player_city_by_number(client_player(), punit->homecity);

    unit_focus_set(punit);
    if (NULL != pcity) {
      struct city_dialog *pdialog = get_city_dialog(pcity);

      if (NULL != pdialog) {
	close_city_dialog(pdialog);
      }
    }
  }
}

/****************************************************************
  User has requested some present unit to be activated and
  city dialog to be closed
*****************************************************************/
static void present_unit_activate_close_callback(GtkWidget * w,
						 gpointer data)
{
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t)data);

  if (NULL != punit) {
    struct city *pcity = tile_city(unit_tile(punit));

    unit_focus_set(punit);
    if (NULL != pcity) {
      struct city_dialog *pdialog = get_city_dialog(pcity);

      if (NULL != pdialog) {
	close_city_dialog(pdialog);
      }
    }
  }
}

/****************************************************************
  User has requested unit to be loaded to transport
*****************************************************************/
static void unit_load_callback(GtkWidget * w, gpointer data)
{
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t)data);

  if (NULL != punit) {
    request_unit_load(punit, NULL);
  }
}

/****************************************************************
  User has requested unit to be unloaded from transport
*****************************************************************/
static void unit_unload_callback(GtkWidget * w, gpointer data)
{
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t)data);

  if (NULL != punit) {
    request_unit_unload(punit);
  }
}

/****************************************************************
  User has requested unit to be sentried
*****************************************************************/
static void unit_sentry_callback(GtkWidget * w, gpointer data)
{
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t)data);

  if (NULL != punit) {
    request_unit_sentry(punit);
  }
}

/****************************************************************
  User has requested unit to be fortified
*****************************************************************/
static void unit_fortify_callback(GtkWidget * w, gpointer data)
{
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t)data);

  if (NULL != punit) {
    request_unit_fortify(punit);
  }
}

/****************************************************************
  User has requested unit to be disbanded
*****************************************************************/
static void unit_disband_callback(GtkWidget * w, gpointer data)
{
  struct unit_list *punits;
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t)data);

  if (NULL == punit) {
    return;
  }

  punits = unit_list_new();
  unit_list_append(punits, punit);
  popup_disband_dialog(punits);
  unit_list_destroy(punits);
}

/****************************************************************
  User has requested unit to change homecity to city where it
  currently is
*****************************************************************/
static void unit_homecity_callback(GtkWidget * w, gpointer data)
{
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t)data);

  if (NULL != punit) {
    request_unit_change_homecity(punit);
  }
}

/****************************************************************
  User has requested unit to be upgraded
*****************************************************************/
static void unit_upgrade_callback(GtkWidget *w, gpointer data)
{
  struct unit_list *punits;
  struct unit *punit =
    player_unit_by_number(client_player(), (size_t)data);

  if (NULL == punit) {
    return;
  }

  punits = unit_list_new();
  unit_list_append(punits, punit);
  popup_upgrade_dialog(punits);
  unit_list_destroy(punits);
}

/*** Callbacks for citizen bar, map funcs that are not update ***/
/****************************************************************
Somebody clicked our list of citizens. If they clicked a specialist
then change the type of him, else do nothing.
*****************************************************************/
static gboolean citizens_callback(GtkWidget *w, GdkEventButton *ev,
                                  gpointer data)
{
  struct city_dialog *pdialog = data;
  struct city *pcity = pdialog->pcity;
  int citnum, tlen, len;

  if (!can_client_issue_orders()) {
    return FALSE;
  }

  tlen = tileset_small_sprite_width(tileset);
  len = (city_size_get(pcity) - 1) * pdialog->cwidth + tlen;
  if (ev->x > len) {
    /* no citizen that far to the right */
    return FALSE;
  }
  citnum = MIN(city_size_get(pcity) - 1, ev->x / pdialog->cwidth);

  city_rotate_specialist(pcity, citnum);

  return TRUE;
}

/**************************************************************************
  User has pressed button on citymap
**************************************************************************/
static gboolean button_down_citymap(GtkWidget * w, GdkEventButton * ev,
				    gpointer data)
{
  struct city_dialog *pdialog = data;
  int canvas_x, canvas_y, city_x, city_y;

  if (!can_client_issue_orders()) {
    return FALSE;
  }

  canvas_x = ev->x * (double)canvas_width / (double)CITYMAP_WIDTH;
  canvas_y = ev->y * (double)canvas_height / (double)CITYMAP_HEIGHT;

  if (canvas_to_city_pos(&city_x, &city_y,
                         city_map_radius_sq_get(pdialog->pcity),
                         canvas_x, canvas_y)) {
    city_toggle_worker(pdialog->pcity, city_x, city_y);
  }

  return TRUE;
}

/****************************************************************
  Set map canvas to be drawn
*****************************************************************/
static void draw_map_canvas(struct city_dialog *pdialog)
{
  gtk_widget_queue_draw(pdialog->overview.map_canvas.darea);
  if (pdialog->happiness.map_canvas.darea) { /* in case of spy */
    gtk_widget_queue_draw(pdialog->happiness.map_canvas.darea);
  }
}

/********* Callbacks for Buy, Change, Sell, Worklist ************/
/****************************************************************
  User has answered buy cost dialog
*****************************************************************/
static void buy_callback_response(GtkWidget *w, gint response, gpointer data)
{
  struct city_dialog *pdialog = data;

  if (response == GTK_RESPONSE_YES) {
    city_buy_production(pdialog->pcity);
  }
  gtk_widget_destroy(w);
}

/****************************************************************
  User has clicked buy-button
*****************************************************************/
static void buy_callback(GtkWidget *w, gpointer data)
{
  GtkWidget *shell;
  struct city_dialog *pdialog = data;
  const char *name = city_production_name_translation(pdialog->pcity);
  int value = city_production_buy_gold_cost(pdialog->pcity);
  char buf[1024];

  if (!can_client_issue_orders()) {
    return;
  }

  fc_snprintf(buf, ARRAY_SIZE(buf), PL_("Treasury contains %d gold.",
                                        "Treasury contains %d gold.",
                                        client_player()->economic.gold),
              client_player()->economic.gold);

  if (value <= client_player()->economic.gold) {
    shell = gtk_message_dialog_new(NULL,
        GTK_DIALOG_DESTROY_WITH_PARENT,
        GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
        /* TRANS: Last %s is pre-pluralised "Treasury contains %d gold." */
        PL_("Buy %s for %d gold?\n%s",
            "Buy %s for %d gold?\n%s", value),
        name, value, buf);
    setup_dialog(shell, pdialog->shell);
    gtk_window_set_title(GTK_WINDOW(shell), _("Buy It!"));
    gtk_dialog_set_default_response(GTK_DIALOG(shell), GTK_RESPONSE_NO);
    g_signal_connect(shell, "response", G_CALLBACK(buy_callback_response),
	pdialog);
    gtk_window_present(GTK_WINDOW(shell));
  } else {
    shell = gtk_message_dialog_new(NULL,
        GTK_DIALOG_DESTROY_WITH_PARENT,
        GTK_MESSAGE_INFO, GTK_BUTTONS_CLOSE,
        /* TRANS: Last %s is pre-pluralised "Treasury contains %d gold." */
        PL_("%s costs %d gold.\n%s",
            "%s costs %d gold.\n%s", value),
        name, value, buf);
    setup_dialog(shell, pdialog->shell);
    gtk_window_set_title(GTK_WINDOW(shell), _("Buy It!"));
    g_signal_connect(shell, "response", G_CALLBACK(gtk_widget_destroy),
      NULL);
    gtk_window_present(GTK_WINDOW(shell));
  }
}

/****************************************************************************
  Callback for the dropdown production menu.
****************************************************************************/
static void change_production_callback(GtkComboBox *combo,
                                       struct city_dialog *pdialog)
{
  GtkTreeIter iter;
  cid cid;

  if (can_client_issue_orders()
      && gtk_combo_box_get_active_iter(combo, &iter)) {
    gtk_tree_model_get(gtk_combo_box_get_model(combo), &iter, 2, &cid, -1);
    city_change_production(pdialog->pcity, cid_production(cid));
  }
}

/****************************************************************
  User has clicked sell-button
*****************************************************************/
static void sell_callback(struct impr_type *pimprove, gpointer data)
{
  GtkWidget *shl;
  struct city_dialog *pdialog = (struct city_dialog *) data;
  pdialog->sell_id = improvement_number(pimprove);
  int price;
  
  if (!can_client_issue_orders()) {
    return;
  }

  if (test_player_sell_building_now(client.conn.playing, pdialog->pcity,
                                    pimprove) != TR_SUCCESS) {
    return;
  }

  price = impr_sell_gold(pimprove);
  shl = gtk_message_dialog_new(NULL,
    GTK_DIALOG_DESTROY_WITH_PARENT,
    GTK_MESSAGE_QUESTION,
    GTK_BUTTONS_YES_NO,
    PL_("Sell %s for %d gold?",
        "Sell %s for %d gold?", price),
    city_improvement_name_translation(pdialog->pcity, pimprove), price);
  setup_dialog(shl, pdialog->shell);
  pdialog->sell_shell = shl;
  
  gtk_window_set_title(GTK_WINDOW(shl), _("Sell It!"));
  gtk_window_set_position(GTK_WINDOW(shl), GTK_WIN_POS_CENTER_ON_PARENT);

  g_signal_connect(shl, "response",
		   G_CALLBACK(sell_callback_response), pdialog);
  
  gtk_window_present(GTK_WINDOW(shl));
}

/****************************************************************
  User has responded to sell price dialog
*****************************************************************/
static void sell_callback_response(GtkWidget *w, gint response, gpointer data)
{
  struct city_dialog *pdialog = data;

  if (response == GTK_RESPONSE_YES) {
    city_sell_improvement(pdialog->pcity, pdialog->sell_id);
  }
  gtk_widget_destroy(w);
  
  pdialog->sell_shell = NULL;
}

/****************************************************************
 this is here because it's closely related to the sell stuff
*****************************************************************/
static void impr_callback(GtkTreeView *view, GtkTreePath *path,
			  GtkTreeViewColumn *col, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter it;
  GdkWindow *win;
  GdkDeviceManager *manager;
  GdkModifierType mask;
  struct impr_type *pimprove;

  model = gtk_tree_view_get_model(view);

  if (!gtk_tree_model_get_iter(model, &it, path)) {
    return;
  }

  gtk_tree_model_get(model, &it, 0, &pimprove, -1);

  win = gdk_get_default_root_window();
  manager = gdk_display_get_device_manager(gdk_window_get_display(win));

  gdk_window_get_device_position(win, 
                                 gdk_device_manager_get_client_pointer(manager),
                                 NULL, NULL, &mask);

  if (!(mask & GDK_CONTROL_MASK)) {
    sell_callback(pimprove, data);
  } else {
    if (is_great_wonder(pimprove)) {
      popup_help_dialog_typed(improvement_name_translation(pimprove), HELP_WONDER);
    } else {
      popup_help_dialog_typed(improvement_name_translation(pimprove), HELP_IMPROVEMENT);
    }
  }
}

/******* Callbacks for stuff on the Misc. Settings page *********/
/****************************************************************
  Called when Rename button pressed
*****************************************************************/
static void rename_callback(GtkWidget * w, gpointer data)
{
  struct city_dialog *pdialog;

  pdialog = (struct city_dialog *) data;

  pdialog->rename_shell = input_dialog_create(GTK_WINDOW(pdialog->shell),
                                              /*"shellrenamecity" */
                                              _("Rename City"),
                                              _("What should we rename the city to?"),
                                              city_name(pdialog->pcity),
                                              rename_popup_callback, pdialog);
}

/****************************************************************
  Called when user has finished with "Rename City" popup
*****************************************************************/
static void rename_popup_callback(gpointer data, gint response,
                                  const char *input)
{
  struct city_dialog *pdialog = data;

  if (pdialog) {
    if (response == GTK_RESPONSE_OK) {
      city_rename(pdialog->pcity, input);
    } /* else CANCEL or DELETE_EVENT */

    pdialog->rename_shell = NULL;
  }
}

/****************************************************************
 Sets which page will be set on reopen of dialog
*****************************************************************/
static void misc_whichtab_callback(GtkWidget * w, gpointer data)
{
  new_dialog_def_page = GPOINTER_TO_INT(data);
}

/**************************************************************************
City options callbacks
**************************************************************************/
static void cityopt_callback(GtkWidget * w, gpointer data)
{
  struct city_dialog *pdialog = (struct city_dialog *) data;

  if (!can_client_issue_orders()) {
    return;
  }

  if(!pdialog->misc.block_signal){
    struct city *pcity = pdialog->pcity;
    bv_city_options new_options;

    fc_assert(CITYO_LAST == 3);

    BV_CLR_ALL(new_options);
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pdialog->misc.disband_on_settler))) {
      BV_SET(new_options, CITYO_DISBAND);
    }
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pdialog->misc.new_citizens_radio[1]))) {
      BV_SET(new_options, CITYO_NEW_EINSTEIN);
    }
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pdialog->misc.new_citizens_radio[2]))) {
      BV_SET(new_options, CITYO_NEW_TAXMAN);
    }

    dsend_packet_city_options_req(&client.conn, pcity->id,new_options);
  }
}

/**************************************************************************
 refresh the city options (auto_[land, air, sea, helicopter] and 
 disband-is-size-1) in the misc page.
**************************************************************************/
static void set_cityopt_values(struct city_dialog *pdialog)
{
  struct city *pcity = pdialog->pcity;

  pdialog->misc.block_signal = 1;

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pdialog->misc.disband_on_settler),
			       is_city_option_set(pcity, CITYO_DISBAND));

  if (is_city_option_set(pcity, CITYO_NEW_EINSTEIN)) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
				 (pdialog->misc.new_citizens_radio[1]), TRUE);
  } else if (is_city_option_set(pcity, CITYO_NEW_TAXMAN)) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
				 (pdialog->misc.new_citizens_radio[2]), TRUE);
  } else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
				 (pdialog->misc.new_citizens_radio[0]), TRUE);
  }
  pdialog->misc.block_signal = 0;
}

/*************** Callbacks for: Close, Prev, Next. **************/
/****************************************************************
  User has clicked rename city-button
*****************************************************************/
static void close_callback(GtkWidget *w, gpointer data)
{
  close_city_dialog((struct city_dialog *) data);
}

/****************************************************************
  User has closed rename city dialog
*****************************************************************/
static void city_destroy_callback(GtkWidget *w, gpointer data)
{
  struct city_dialog *pdialog;

  pdialog = (struct city_dialog *) data;

  gtk_widget_hide(pdialog->shell);

  if (game.info.citizen_nationality) {
    citizens_dialog_close(pdialog->pcity);
  }
  close_happiness_dialog(pdialog->pcity);
  close_cma_dialog(pdialog->pcity);

  /* Save size of the city dialog. */
  gui_gtk3_citydlg_xsize = CLIP(GUI_GTK3_CITYDLG_MIN_XSIZE,
                                gtk_widget_get_allocated_width(pdialog->shell),
                                GUI_GTK3_CITYDLG_MAX_XSIZE);
  gui_gtk3_citydlg_ysize = CLIP(GUI_GTK3_CITYDLG_MIN_XSIZE,
                                gtk_widget_get_allocated_height(pdialog->shell),
                                GUI_GTK3_CITYDLG_MAX_XSIZE);

  last_page =
      gtk_notebook_get_current_page(GTK_NOTEBOOK(pdialog->notebook));

  if (pdialog->popup_menu) {
    gtk_widget_destroy(pdialog->popup_menu);
  }

  dialog_list_remove(dialog_list, pdialog);

  unit_node_vector_free(&pdialog->overview.supported_units);
  unit_node_vector_free(&pdialog->overview.present_units);

  if (pdialog->buy_shell) {
    gtk_widget_destroy(pdialog->buy_shell);
  }
  if (pdialog->sell_shell) {
    gtk_widget_destroy(pdialog->sell_shell);
  }
  if (pdialog->rename_shell) {
    gtk_widget_destroy(pdialog->rename_shell);
  }

  cairo_surface_destroy(pdialog->map_canvas_store_unscaled);

  free(pdialog);

  /* need to do this every time a new dialog is closed. */
  city_dialog_update_prev_next();
}

/************************************************************************
  Close city dialog
*************************************************************************/
static void close_city_dialog(struct city_dialog *pdialog)
{
  gtk_widget_destroy(pdialog->shell);
}

/************************************************************************
  Callback for the prev/next buttons. Switches to the previous/next
  city.
*************************************************************************/
static void switch_city_callback(GtkWidget *w, gpointer data)
{
  struct city_dialog *pdialog = (struct city_dialog *) data;
  int i, j, dir, size;
  struct city *new_pcity = NULL;

  if (NULL == client.conn.playing) {
    return;
  }

  size = city_list_size(client.conn.playing->cities);

  fc_assert_ret(city_dialogs_have_been_initialised);
  fc_assert_ret(size >= 1);
  fc_assert_ret(city_owner(pdialog->pcity) == client.conn.playing);

  if (size == 1) {
    return;
  }

  /* dir = 1 will advance to the city, dir = -1 will get previous */
  if (w == pdialog->next_command) {
    dir = 1;
  } else if (w == pdialog->prev_command) {
    dir = -1;
  } else {
    /* Always fails. */
    fc_assert_ret(w == pdialog->next_command
                  || w == pdialog->prev_command);
    dir = 1;
  }

  for (i = 0; i < size; i++) {
    if (pdialog->pcity == city_list_get(client.conn.playing->cities, i)) {
      break;
    }
  }

  fc_assert_ret(i < size);

  for (j = 1; j < size; j++) {
    struct city *other_pcity = city_list_get(client.conn.playing->cities,
					     (i + dir * j + size) % size);
    struct city_dialog *other_pdialog = get_city_dialog(other_pcity);

    fc_assert_ret(other_pdialog != pdialog);
    if (!other_pdialog) {
      new_pcity = other_pcity;
      break;
    }
  }

  if (!new_pcity) {
    /* Every other city has an open city dialog. */
    return;
  }

  /* cleanup happiness dialog */
  if (game.info.citizen_nationality) {
    citizens_dialog_close(pdialog->pcity);
  }
  close_happiness_dialog(pdialog->pcity);

  pdialog->pcity = new_pcity;

  /* reinitialize happiness, and cma dialogs */
  if (game.info.citizen_nationality) {
    gtk_container_add(GTK_CONTAINER(pdialog->happiness.citizens),
                      citizens_dialog_display(pdialog->pcity));
  }
  gtk_container_add(GTK_CONTAINER(pdialog->happiness.widget),
                    get_top_happiness_display(pdialog->pcity));
  if (!client_is_observer()) {
    fc_assert(pdialog->cma_editor != NULL);
    pdialog->cma_editor->pcity = new_pcity;
  }

  reset_city_worklist(pdialog->production.worklist, pdialog->pcity);

  can_slide = FALSE;
  center_tile_mapcanvas(pdialog->pcity->tile);
  can_slide = TRUE;
  if (!client_is_observer()) {
    set_cityopt_values(pdialog);  /* need not be in real_city_dialog_refresh */
  }

  real_city_dialog_refresh(pdialog->pcity);

  /* recenter the city map(s) */
  city_dialog_map_recenter(pdialog->overview.map_canvas.sw);
  if (pdialog->happiness.map_canvas.sw) {
    city_dialog_map_recenter(pdialog->happiness.map_canvas.sw);
  }
}
