//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Instruction.hpp,v 1.5 2004/01/26 18:23:42 delpinux Exp $

#ifndef _INSTRUCTION_HPP_
#define _INSTRUCTION_HPP_

#include <list>
#include <string>
#include <Expression.hpp>
#include <BooleanExpression.hpp>

#include <SceneExpression.hpp>
#include <Variable.hpp>
#include <OstreamExpression.hpp>

#include <FileDescriptor.hpp>
#include <FieldExpression.hpp>

class Instruction
{
public:
  enum Type {
    list,
    ifStatement,
    doWhileStatement,
    whileStatement,
    forStatement,
    exitStatement,

    declaration,
    affectation,
    increment,
    decrement,
    evaluation,

    output,
    Using,
    exec,
    plot,
    save,
    none
  };

private:
  const Instruction::Type __type;

public:
  virtual void execute() = 0;

  Instruction(const Instruction::Type& t)
    : __type(t)
  {
    ;
  }

  Instruction(const Instruction& I)
    : __type(I.__type)
  {
    ;
  }

  virtual ~Instruction()
  {
    ;
  }
};

class InstructionList
  : public Instruction
{
public:
  typedef std::list<ReferenceCounting<Instruction> > InstructionSet;
  InstructionSet __instructions;

public:
  void execute()
  {
    for(InstructionSet::iterator i = __instructions.begin();
	i != __instructions.end(); ++i) {
      (*(*i)).execute();
    }
  }

  void add(ReferenceCounting<Instruction> i)
  {
    __instructions.push_back(i);
  }

  InstructionList()
    : Instruction(Instruction::list)
  {
    ;
  }

  InstructionList(const InstructionList& I)
    : Instruction(I),
      __instructions(I.__instructions)
  {
    ;
  }

  ~InstructionList()
  {
    ;
  }
};

class InstructionIfStatement
  : public Instruction
{
public:
  ReferenceCounting<BooleanExpression> __booleanExpression;
  ReferenceCounting<Instruction> __firstStatement;
  ReferenceCounting<Instruction> __secondStatement;

public:
  void execute()
  {
    (*__booleanExpression).execute();
    if ((*__booleanExpression).boolValue()) {
      (*__firstStatement).execute();
    } else {
      (*__secondStatement).execute();
    }
  }

  InstructionIfStatement(ReferenceCounting<BooleanExpression> b,
			 ReferenceCounting<Instruction> first,
			 ReferenceCounting<Instruction> second)
    : Instruction(Instruction::ifStatement),
      __booleanExpression(b),
      __firstStatement(first),
      __secondStatement(second)
  {
    ;
  }

  InstructionIfStatement(const InstructionIfStatement& I)
    : Instruction(I),
      __booleanExpression(I.__booleanExpression),
      __firstStatement(I.__firstStatement),
      __secondStatement(I.__secondStatement)
  {
    ;
  }

  ~InstructionIfStatement()
  {
    ;
  }
};

class InstructionDoWhileStatement
  : public Instruction
{
public:
  ReferenceCounting<Instruction> __statement;
  ReferenceCounting<BooleanExpression> __booleanExpression;

public:
  void execute()
  {
    do {
      (*__statement).execute();
      (*__booleanExpression).execute();
    } while ((*__booleanExpression).boolValue());
  }

  InstructionDoWhileStatement(ReferenceCounting<Instruction> statement,
			      ReferenceCounting<BooleanExpression> b)
    : Instruction(Instruction::doWhileStatement),
      __statement(statement),
      __booleanExpression(b)
  {
    ;
  }

  InstructionDoWhileStatement(const InstructionDoWhileStatement& I)
    : Instruction(I),
      __statement(I.__statement),
      __booleanExpression(I.__booleanExpression)
  {
    ;
  }

  ~InstructionDoWhileStatement()
  {
    ;
  }
};

class InstructionForStatement
  : public Instruction
{
public:
  ReferenceCounting<Instruction> __instruction1;
  ReferenceCounting<BooleanExpression> __booleanExpression;
  ReferenceCounting<Instruction> __instruction2;
  ReferenceCounting<Instruction> __statement;

public:
  void execute()
  {
    for ((*__instruction1).execute();
	 (*__booleanExpression).execute(),
	   (*__booleanExpression).boolValue();
	 (*__instruction2).execute())
      {
	(*__statement).execute();
      }
  }

  InstructionForStatement(ReferenceCounting<Instruction> a,
			  ReferenceCounting<BooleanExpression> b,
			  ReferenceCounting<Instruction> c,
			  ReferenceCounting<Instruction> statement)
    : Instruction(Instruction::forStatement),
      __instruction1(a),
      __booleanExpression(b),
      __instruction2(c),
      __statement(statement)
  {
    ;
  }

  InstructionForStatement(const InstructionForStatement& I)
    : Instruction(I),
      __instruction1(I.__instruction1),
      __booleanExpression(I.__booleanExpression),
      __instruction2(I.__instruction2),
      __statement(I.__statement)
  {
    ;
  }

  ~InstructionForStatement()
  {
    ;
  }
};

class InstructionWhileStatement
  : public Instruction
{
public:
  ReferenceCounting<Instruction> __statement;
  ReferenceCounting<BooleanExpression> __booleanExpression;

public:
  void execute()
  {
    (*__booleanExpression).execute();
    while ((*__booleanExpression).boolValue()) {
      (*__statement).execute();
      (*__booleanExpression).execute();
    }
  }

  InstructionWhileStatement(ReferenceCounting<BooleanExpression> b,
			      ReferenceCounting<Instruction> statement)
    : Instruction(Instruction::whileStatement),
      __statement(statement),
      __booleanExpression(b)
  {
    ;
  }

  InstructionWhileStatement(const InstructionWhileStatement& I)
    : Instruction(I),
      __statement(I.__statement),
      __booleanExpression(I.__booleanExpression)
  {
    ;
  }

  ~InstructionWhileStatement()
  {
    ;
  }
};

class InstructionOutput
  : public Instruction
{
private:
  ReferenceCounting<OstreamExpression> __ostream;
  ReferenceCounting<Expression> __expression;

public:
  void execute()
  {
    (*__ostream).execute();
    (*__expression).execute();
    (*__ostream) << (*__expression);
  }

  InstructionOutput(ReferenceCounting<OstreamExpression> o,
		    ReferenceCounting<Expression> e)
    : Instruction(Instruction::output),
      __ostream(o),
      __expression(e)
  {
    ;
  }

  InstructionOutput(const InstructionOutput& I)
    : Instruction(I),
      __ostream(I.__ostream),
      __expression(I.__expression)    
  {
    ;
  }

  ~InstructionOutput()
  {
    ;
  }
};

/**
 * @file   Instruction.hpp
 * @author Stephane Del Pino
 * @date   Sun Feb  9 17:24:28 2003
 * 
 * @brief  Mother class for saving instruction
 * 
 * 
 */

class InstructionSave
  : public Instruction
{
protected:
  ReferenceCounting<FileDescriptor>  __fileDescriptor;
  ReferenceCounting<StringExpression>  __fileName;
  ReferenceCounting<MeshExpression> __mesh;
  
public:

  InstructionSave(ReferenceCounting<FileDescriptor> descriptor,
		  ReferenceCounting<StringExpression> fileName,
		  ReferenceCounting<MeshExpression> mesh);

  InstructionSave(const InstructionSave& I);

  ~InstructionSave();
};

/**
 * @file   Instruction.hpp
 * @author Stephane Del Pino
 * @date   Mon Jan 26 17:40:26 2004
 * 
 * @brief  
 * 
 * 
 */

class InstructionExec
  : public Instruction
{
protected:
  ReferenceCounting<StringExpression> __command;
  
public:
  void execute();

  InstructionExec(ReferenceCounting<StringExpression> command);

  InstructionExec(const InstructionExec& I);

  ~InstructionExec();
};

/**
 * @file   Instruction.hpp
 * @author Stephane Del Pino
 * @date   Sun Feb  9 17:24:55 2003
 * 
 * @brief  Saves meshes in files according to a given format
 * 
 * 
 */

class InstructionSaveMesh
  : public InstructionSave
{
public:
  void execute();

  InstructionSaveMesh(ReferenceCounting<FileDescriptor> descriptor,
		      ReferenceCounting<StringExpression> fileName,
		      ReferenceCounting<MeshExpression> mesh);

  InstructionSaveMesh(const InstructionSaveMesh& I);

  ~InstructionSaveMesh();
};

/**
 * @file   Instruction.hpp
 * @author Stephane Del Pino
 * @date   Sun Feb  9 17:25:15 2003
 * 
 * @brief  Saves functions to files according to given format
 * 
 */

class InstructionSaveFunction
  : public InstructionSave
{
private:
  ReferenceCounting<FunctionExpression> __function;

public:
  void execute();

  InstructionSaveFunction(ReferenceCounting<FileDescriptor> descriptor,
			  ReferenceCounting<StringExpression> fileName,
			  ReferenceCounting<FunctionExpression> function,
			  ReferenceCounting<MeshExpression> mesh);

  InstructionSaveFunction(const InstructionSaveFunction& I);

  ~InstructionSaveFunction();
};

/**
 * @file   Instruction.hpp
 * @author Stephane Del Pino
 * @date   Sun Feb  9 17:25:44 2003
 * 
 * @brief  Saves fields of function according to a given format.
 * 
 * 
 */

class InstructionSaveField
  : public InstructionSave
{
private:
  ReferenceCounting<FieldExpression> __field;

public:
  void execute();

  InstructionSaveField(ReferenceCounting<FileDescriptor> descriptor,
		       ReferenceCounting<StringExpression> fileName,
		       ReferenceCounting<FieldExpression> field,
		       ReferenceCounting<MeshExpression> mesh);

  InstructionSaveField(const InstructionSaveField& I);

  ~InstructionSaveField();
};


/**
 * @class InstructionPlot
 * 
 * This class is designed to plot data using graphic libraries
 *
 * @author Stephane Del Pino
 */

class InstructionPlot
  : public Instruction
{
private:
  ReferenceCounting<MeshExpression> __mesh;

public:
  /** 
   * Expression::execute() specialization
   * 
   */
  void execute();

  /** 
   * 
   * Constructor
   * 
   * @return 
   */  
  InstructionPlot(ReferenceCounting<MeshExpression> mesh);

  /** 
   * Copy Constructor
   * 
   */
  InstructionPlot(const InstructionPlot& __instruction);

  /** 
   * Destructor
   * 
   */
  ~InstructionPlot();
};


/*!
  \class InstructionNone

  This class is used for istructions which does nothing

  \author Stephane Del Pino
 */
class InstructionNone
  : public Instruction
{
public:
  void execute()
  {
    ;
  }

  InstructionNone()
    : Instruction(Instruction::none)
  {
    ;
  }

  InstructionNone(const InstructionNone& I)
    : Instruction(I)
  {
    ;
  }

  ~InstructionNone()
  {
    ;
  }
};

#include <VariableList.hpp>
extern VariableList variableList;

template <typename DataType, typename VarType>
class InstructionDeclaration
  : public Instruction
{
private:
  const std::string __name;
  ReferenceCounting<DataType> __expression;

public:
  void execute()
  {
    (*__expression).execute();
    Variable* variable
      = variableList.find(__name);
    ReferenceCounting<VarType> V
      = dynamic_cast<VarType*>(variable);
    (*V) = __expression;
  }

  InstructionDeclaration(const std::string name,
			 ReferenceCounting<DataType> e)
    : Instruction(Instruction::declaration),
      __name(name),
      __expression(e)
  {
    ReferenceCounting<Variable> a
      = new VarType(__name, __expression);
    variableList.add(a);
  }

  InstructionDeclaration(const InstructionDeclaration<DataType,VarType>& I)
    : Instruction(I),
      __name(I.__name),
      __expression(I.__expression)
  {
    ;
  }

  ~InstructionDeclaration()
  {
    ;
  }
};

template<typename ExpType,
	 typename DataType>
class InstructionAffectation
  : public Instruction
{
private:
  ReferenceCounting<DataType> __variable;
  ReferenceCounting<ExpType> __expression;

public:
  void execute()
  {
    (*__expression).execute();
    (*__variable) = __expression;
  }

  InstructionAffectation(ReferenceCounting<DataType> v,
			 ReferenceCounting<ExpType> e)
    : Instruction(Instruction::affectation),
      __variable(v),
      __expression(e)
  {
    ;
  }

  InstructionAffectation(const InstructionAffectation<DataType, ExpType>& I)
    : Instruction(I),
      __variable(I.__variable),
      __expression(I.__expression)
  {
    ;
  }

  ~InstructionAffectation()
  {
    ;
  }
};

class InstructionRealExpressionIncrement
  : public Instruction
{
private:
  ReferenceCounting<RealVariable> __variable;

public:
  void execute()
  {
    real_t r = (*(*__variable).expression()).realValue();
    (*__variable) = new RealExpressionValue(r+1);
  }

  InstructionRealExpressionIncrement(ReferenceCounting<RealVariable> v)
    : Instruction(Instruction::increment),
      __variable(v)
  {
    ;
  }

  InstructionRealExpressionIncrement(const InstructionRealExpressionIncrement& I)
    : Instruction(I),
      __variable(I.__variable)
  {
    ;
  }

  ~InstructionRealExpressionIncrement()
  {
    ;
  }
};

class InstructionRealExpressionDecrement
  : public Instruction
{
private:
  ReferenceCounting<RealVariable> __variable;

public:
  void execute()
  {
    real_t r = (*(*__variable).expression()).realValue();
    (*__variable) = new RealExpressionValue(r-1);
  }

  InstructionRealExpressionDecrement(ReferenceCounting<RealVariable> v)
    : Instruction(Instruction::decrement),
      __variable(v)
  {
    ;
  }

  InstructionRealExpressionDecrement(const InstructionRealExpressionDecrement& I)
    : Instruction(I),
      __variable(I.__variable)
  {
    ;
  }

  ~InstructionRealExpressionDecrement()
  {
    ;
  }
};

class FunctionExpression;
class FunctionExpressionFEM;
template<>
class InstructionAffectation<FunctionExpressionFEM, FunctionVariable>
  : public Instruction
{
private:
  ReferenceCounting<FunctionVariable> __variable;
  ReferenceCounting<FunctionExpression> __expression;

public:
  void execute();

  InstructionAffectation(ReferenceCounting<FunctionVariable> v,
			 ReferenceCounting<FunctionExpression> e)
    : Instruction(Instruction::affectation),
      __variable(v),
      __expression(e)
  {
    ;
  }

  InstructionAffectation(const InstructionAffectation<FunctionExpressionFEM,
			                              FunctionVariable>& I)
    : Instruction(I),
      __variable(I.__variable),
      __expression(I.__expression)
  {
    ;
  }

  ~InstructionAffectation()
  {
    ;
  }
};


template<typename ExpType>
class InstructionEvaluation
  : public Instruction
{
private:
  ReferenceCounting<ExpType> __expression;

public:
  void execute()
  {
    (*__expression).execute();
  }

  InstructionEvaluation(ReferenceCounting<ExpType> e)
    : Instruction(Instruction::evaluation),
      __expression(e)
  {
    ;
  }

  InstructionEvaluation(const InstructionEvaluation<ExpType>& I)
    : Instruction(I),
      __expression(I.__expression)
  {
    ;
  }

  ~InstructionEvaluation()
  {
    ;
  }
};


class InstructionUsingScene
  : public Instruction
{
private:
  ReferenceCounting<SceneExpression> __sceneExpression;

public:
  void execute();

  InstructionUsingScene(ReferenceCounting<SceneExpression> e);

  InstructionUsingScene(const InstructionUsingScene& I);

  ~InstructionUsingScene();
};

class InstructionExit
  : public Instruction
{
private:
  ReferenceCounting<RealExpression> __realExpression;

public:
  void execute()
  {
    if (__realExpression != 0) {
      (*__realExpression).execute();
      int i = static_cast<int>((*__realExpression).realValue());
      std::exit(i);
    }
    std::exit(0);
  }

  InstructionExit()
    : Instruction(Instruction::exitStatement),
      __realExpression(0)
  {
    ;
  }

  InstructionExit(ReferenceCounting<RealExpression> e)
    : Instruction(Instruction::exitStatement),
      __realExpression(e)
  {
    ;
  }

  InstructionExit(const InstructionExit& I)
    : Instruction(I),
      __realExpression(I.__realExpression)
  {
    ;
  }

  ~InstructionExit()
  {
    ;
  }
};


#endif // _INSTRUCTION_HPP_

