//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Domain.hpp,v 1.6 2003/10/22 17:08:28 delpinux Exp $


#ifndef DOMAIN_HPP
#define DOMAIN_HPP

class Scene;

#include <ReferenceCounting.hpp>
#include <Object.hpp>
#include <TinyVector.hpp>

#include <map>

/**
 * @file   Domain.hpp
 * @author Stephane Del Pino
 * @date   Fri Nov 15 18:32:48 2002
 * 
 * @brief This class describes a domain of \f$ R^3\f$ which is the
 * defined by the union of the interior and the exterior of sets of
 * Objects.
 * 
 */

class Domain
{
private:
  /** This boolean describes the Domain this way:
   * \li false: the domain is a subset of \f$ R^3\f$.
   * \li true:  the domain is \f$ R^3\f$
   */
  bool __isR3;

  ReferenceCounting<Object>
  __objects;			/**< The shape of the domain */

  const Scene& __scene;		/**< The POV-Ray scene */

  std::map<TinyVector<3, real_t>, size_t>
  __povToReference;		/**< The POV-Ray to internal
				   references association*/

  void __buildReferenceAssociation(const Object& o);

  const bool __inShape(const TinyVector<3>& X) const
  {
    return (*__objects).inside(X);
  }

public:
  /** 
   * Returns true if the considered domain is \f$\setR^3\f$
   * 
   * @return true if \f$\Omega=\setR^3\f$
   */
  const bool& isR3() const
  {
    return __isR3;
  }

  size_t reference(const TinyVector<3, real_t>& ref) const
  {
    std::map<TinyVector<3, real_t>, size_t>::const_iterator i
      = __povToReference.find(ref);
    if (i == __povToReference.end()) {
      fferr(0) << "\nYou provided the reference " << ref
	       << " that is not used to define\n";
	fferr(0) << "your computational domain. The possible references are\n";
      for (std::map<TinyVector<3, real_t>, size_t>::const_iterator j
	     = __povToReference.begin();
	   j != __povToReference.end(); ++j) {
	fferr(0) << '\t' << j->first << '\n';
      }
      std::exit(1);
    }
    return (*i).second;
  }

  /**
   * Outputs the Domain \a D in the \p std::ostream \a os.
   */
  friend std::ostream& operator<<(std::ostream& os,
				  const Domain& D)
  {
    os << *D.__objects;
    return os;
  }

  /** 
   * Returns true if X is inside the domain
   * 
   * @param X 
   * 
   * @return 
   */
  const bool inside(const TinyVector<3>& X) const
  {
    return (__isR3 || __inShape(X));
  }

  /** 
   * read-only access to the scene
   * 
   * 
   * @return __scene
   */
  const Scene& scene() const
  {
    return __scene;
  }

  /** 
   * Returns the shape if defined...
   * 
   * @return *__objects
   */
  const Object& object() const
  {
    if (__objects == 0) {
      fferr(0) << __FILE__ << ':' << __LINE__
	       << ": Trying to access Object in case of R^3 domain\n";
      std::exit(1);
    }

    return (*__objects);
  }

  /** 
   * Sets the shape of the domain
   * 
   * @param objects 
   */
  void setObjects(ReferenceCounting<Object> objects)
  {
    __isR3 = false;
    __objects = objects;
    ffout(2) << "\tReference translation (POV-Ray -> FEM)\n";
    this->__buildReferenceAssociation(*__objects);
    ffout(2) << "\tDone\n";
  }

  /** 
   * Constructor
   * 
   * @param S 
   */
  Domain(const Scene& S)
    : __isR3(true),
      __objects(0),
      __scene(S)
  {
    ;
  }

  /** 
   * Copy constructor
   * 
   * @param D 
   */
  Domain(const Domain& D)
    : __isR3(D.__isR3),
      __objects(D.__objects),
      __scene(D.__scene)
  {
    ;
  }

  /** 
   * Destructor
   * 
   */
  ~Domain()
  {
    ;
  }
};

#endif // DOMAIN_HPP

