//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Mesh.hpp,v 1.10 2004/04/15 15:19:51 delpinux Exp $

#ifndef MESH_HPP
#define MESH_HPP

#include <Vector.hpp>
#include <vector>

#include <ReferenceCounting.hpp>

#include <VerticesSet.hpp>

#include <Cell.hpp>
#include <Edge.hpp>

#include <string>

template <typename MeshType>
class MeshExtractor;

class Mesh
{
public:
  typedef enum {
    cartesianHexahedraMesh,
    hexahedraMesh,
    tetrahedraMesh,
    surfaceMeshTriangles,
    surfaceMeshQuadrangles,
  } Type;

  // Family
  enum {
    volume,
    surface
  };

private:
  //! Mesh extractors are friends
  template <typename MeshType> friend class MeshExtractor;

  //! The type of the mesh
  const Type __type;

protected:

  template <typename MeshType,
	    typename CellType = typename MeshType::ElementGeometry>
  class T_iterator
  {
  private:
    MeshType* __mesh;
    CellType* __iterator;
    size_t __number;

  public:
    typedef enum {
      Begin,
      End
    } Position;

    const size_t& number() const
    {
      return __number;
    }

    inline CellType& operator*()
    {
      return *__iterator;
    }

    inline CellType* pointer() const
    {
      return __iterator;
    }

    inline bool end() const
    {
      return (__number >= __mesh->numberOfCells());
    }

    inline bool operator<(const T_iterator<MeshType, CellType>& i) const
    {
      return (__number < i.__number);
    }

    const T_iterator<MeshType, CellType>&
    operator=(const T_iterator<MeshType, CellType>& i)
    {
      __mesh = i.__mesh;
      __iterator = i.__iterator;
      __number = i.__number;

      return *this;
    }

    const T_iterator<MeshType, CellType>&
    operator=(CellType* iterator)
    {
      __iterator = iterator;
      __number = static_cast<size_t>(__iterator-&(__mesh->cell(0)));

      return *this;
    }

    inline T_iterator<MeshType, CellType> operator++(int)
    {
      T_iterator<MeshType, CellType> i = *this;
      ++(*this);
      return i;
    }

    inline T_iterator<MeshType, CellType> operator++()
    {
      for(;;) {
	++__number;
	if (this->end()) {
	  break;
	}
	++__iterator;
	if (not(__iterator->isFictitious())) {
	  break;
	}
      }
      return *this;
    }

    T_iterator(MeshType& m,
	       T_iterator::Position position = T_iterator::Begin)
      : __mesh(&m),
	__number(0)
    {
      if ((m.numberOfCells() == 0)
	  or (position == T_iterator<MeshType, CellType>::End)) {
	__number = __mesh->numberOfCells();
      } else {
	__iterator = & m.cell(0);
	if (__iterator->isFictitious()) {
	  ++(*this);
	}
      }
    }

    T_iterator(MeshType& m,
	       const size_t& cellNumber)
      : __mesh(&m),
	__number(cellNumber)
    {
      if (cellNumber < m.numberOfCells()) {
	__iterator = & m.cell(cellNumber);
      }
    }

    T_iterator(const T_iterator<MeshType, CellType>& i)
      : __mesh(i.__mesh),
	__iterator(i.__iterator),
	__number(i.__number)
    {
      ;
    }

    ~T_iterator()
    {
      ;
    }
  };

  //! Container for vertices.
  ReferenceCounting<VerticesSet> __verticesSet;

  /*! Container for edges.
    \remark using std::vector because it is extensible ...
   */
  std::vector<Edge> __edges;

  //! Copies of meshes are almost forbidden
  Mesh(const Mesh& M)
    : __type(M.__type),
      __verticesSet(M.__verticesSet),
      __edges(M.__edges)
  {
    ;
  }

public:

  //! Writes the mesh to the OpenDX Mesh format
  virtual void plotOpenDX(std::ostream& fmesh,
			  const std::string& CR) const = 0;

  //! The type of the mesh.
  const Mesh::Type& type() const
  {
    return __type;
  }

  //! Returns the number of the vertex V in the list
  inline size_t vertexNumber(const Vertex& V) const
  {
    return (*__verticesSet).number(V);
  }

  //! Returns the number of the Edge E in the list
  inline size_t edgeNumber(const Edge& E) const
  {
    return (&E - &__edges[0]);
  }

  //! Read-only access to the number of vertices.
  inline const size_t& numberOfVertices() const
  {
    return (*__verticesSet).numberOfVertices();
  }

  //! Change the size of the vertices container.
  inline void setNumberOfVertices(const size_t& size)
  {
    return (*__verticesSet).setNumberOfVertices(size);
  }

  virtual const size_t& numberOfCells() const = 0;

  //! Read-only access to the number of edges.
  inline size_t nbEdges() const
  {
    return __edges.size();
  }

  inline ReferenceCounting<VerticesSet> verticesSet()
  {
    return __verticesSet;
  }


  inline const Vertex& vertex(const size_t& i) const
  {
    return (*__verticesSet)[i];
  }

  inline Vertex& vertex(const size_t& i)
  {
    return (*__verticesSet)[i];
  }

  //! Read-only access to the Edge \a i.
  inline const Edge& edge(const size_t& i) const
  {
    assert (i<__edges.size());
    return __edges[i];
  }

  //! Access to the Edge \a i.
  inline Edge& edge(const size_t& i)
  {
    assert (i<__edges.size());
    return __edges[i];
  }

  Mesh(const Mesh::Type& type, const size_t& nbV)
    : __type(type),
      __verticesSet(new VerticesSet(nbV))
  {
    ;
  }

  Mesh(const Mesh::Type& type, ReferenceCounting<VerticesSet> vertices)
    : __type(type),
      __verticesSet(vertices)
  {
    ;
  }

  virtual ~Mesh()
  {
    ;
  }
};

#endif // MESH_HPP

