//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: UserFunction.hpp,v 1.5 2004/04/15 15:21:08 delpinux Exp $

#ifndef _USER_FUNCTION_
#define _USER_FUNCTION_

#include <string>
#include <vector>
#include <cmath>

#include <StreamCenter.hpp>

#include <TinyVector.hpp>
#include <ReferenceCounting.hpp>

class Mesh;

class Structured3DMesh;

/*!
  \class UserFunction
  This class is the base class for \em user \em functions definition.
  
  \author Stphane Del Pino
*/
class UserFunction
{
 protected:
  //! Allows the definition of basic functions such as \f$ x, y\f$ or \f$ z\f$.
  real_t (*f)(const real_t x, const real_t y, const real_t z);

 public:
  //! operator= : allow affectation of a C function. 
  const UserFunction& operator = (real_t(*func)(const real_t x,
						const real_t y,
						const real_t z))
  {
    f = func;
    return *this;
  }

  /*! Unary minus for UserFunctions
    \todo Make this function virtual.
   */
  UserFunction operator- () const;

  //! operator= : allow affectation of a UserFunction.
  const UserFunction& operator = (const UserFunction& F)
  {
    f = F.f;
    return *this;
  }

  //! Ealuates the function at position (\a x0, \a x1, \a x2).
  inline real_t operator()(const real_t& x0,
			   const real_t& x1,
			   const real_t& x2) const
  {
    return this->operator()(TinyVector<3, real_t>(x0, x1, x2));
  }

  //! operator() : evaluate a UserFunction at position \a X.
  virtual real_t operator()(const TinyVector<3>& X) const
  {
    return f(X[0],X[1],X[2]);
  }

  //! The default constructor does nothing (used for inheritence).
  UserFunction()
  {
    ;
  }

  //! Constructs a UserFunction using a C function.
  UserFunction(real_t(*func)(const real_t x, const real_t y, const real_t z))
    : f(func)
  {
    ;
  }

  //! Copy Constructor.
  UserFunction(const UserFunction& F)
    : f(F.f)
  {
    ;
  }

  //! Destructor.
  virtual ~UserFunction()
  {
    ;
  }
};


class UnaryMinusUserFunction
  : public UserFunction
{
private:
  ConstReferenceCounting<UserFunction> __u;

public:
  real_t operator()(const TinyVector<3>& X) const
  {
    return (-(*__u)(X));
  }

  UnaryMinusUserFunction(const UnaryMinusUserFunction& U)
    : __u(U.__u)
  {
    ;
  }

  UnaryMinusUserFunction(ConstReferenceCounting<UserFunction> U)
    : __u(U)
  {
    ;
  }

  ~UnaryMinusUserFunction()
  {
    ;
  }
};

/*!
  \class UserFunctionLanguage
  
  \author Stphane Del Pino
*/
class FunctionExpression;
class UserFunctionLanguage
  : public UserFunction
{
private:
  ReferenceCounting<FunctionExpression> __function;

public:
  FunctionExpression& functionExpression()
  {
    return *__function;
  }

  void gradient(TinyVector<3>& gradient,
		const TinyVector<3>& X) const;

  real_t operator()(const TinyVector<3>& X) const;

  UserFunctionLanguage(UserFunctionLanguage& U);

  UserFunctionLanguage(ReferenceCounting<FunctionExpression>);

  ~UserFunctionLanguage();
};

/*!
  \class ConstUserFunction
  This class allows to define \em contant functions.
  
  \author Stphane Del Pino
*/
class ConstUserFunction
  : public UserFunction
{
private:
  //! The constant value.
  const real_t d;
public:

  //! operator() : evaluate a ConstUserFunction at position \a X.
  real_t operator()(const TinyVector<3>& X) const
  {
    return d;
  }

  //! Constructs the ConstUserFunction using the value \a val.
  ConstUserFunction(const real_t& val)
    : d(val)
  {
    ;
  }

  //! Destructor.
  ~ConstUserFunction()
  {
    ;
  }
};

/*!
  \class AddUserFunction
  This class allows to define \em sum of UserFunction.
  
  \author Stphane Del Pino
*/
class AddUserFunction
  : public UserFunction
{
private:
  //! The first function of the sum.
  ConstReferenceCounting<UserFunction> F;

  //! The second function of the sum.
  ConstReferenceCounting<UserFunction> G;
public:

  //! operator(): Evaluate the function sum at position \a X.
  real_t operator()(const TinyVector<3>& X) const
  {
    return (*F)(X) + (*G)(X);
  }

  //! Constructs the AddUserFunction using 2 UserFunction pointers.
  AddUserFunction(const UserFunction* F1,
		  const UserFunction* F2)
    : F(F1),
      G(F2)
  {
    ;
  }

  //! Destructor
  ~AddUserFunction()
  {
    ;
  }
};

/*!
  \class SubUserFunction
  This class allows to define \em difference of UserFunction.
  
  \author Stphane Del Pino
*/
class SubUserFunction
  : public UserFunction
{
 private:
  //! The first UserFunction of the difference.
  ConstReferenceCounting<UserFunction> F;

  //! The second UserFunction of the difference.
  ConstReferenceCounting<UserFunction> G;
 public:

  //! operator(): Evaluate the difference of \a F and \a G at position \a X.
  real_t operator()(const TinyVector<3>& X) const
  {
    return (*F)(X)-(*G)(X);
  }

  //! Constructs the SubUserFunction using 2 UserFunction pointers.
  SubUserFunction(const UserFunction* F1,
		  const UserFunction* F2)
    : F(F1),
      G(F2)
  {
    ;
  }

  //! Destructor.
  ~SubUserFunction()
  {
    ;
  }
};

/*!
  \class MulUserFunction
  This class allows to define \em multiplication of UserFunction.
  
  \author Stphane Del Pino
*/
class MulUserFunction
  : public UserFunction
{
 private:
  //! The first UserFunction of the multiplication.
  ConstReferenceCounting<UserFunction> F;

  //! The second UserFunction of the multiplication.
  ConstReferenceCounting<UserFunction> G;
 public:

  //! Evaluation of \a F \a G at \a X.
  real_t operator()(const TinyVector<3>& X) const {
    return (*F)(X) * (*G)(X);
  }

  //! Constructs the MulUserFunction using 2 UserFunction pointers.
  MulUserFunction(const UserFunction* F1,
		  const UserFunction* F2)
    : F(F1),
      G(F2)
  {
    ;
  }

  //! Destructor.
  ~MulUserFunction()
  {
    ;
  }
};

/*!
  \class DivUserFunction
  This class allows to define \em division of UserFunction.
  
  \author Stphane Del Pino
*/
class DivUserFunction
  : public UserFunction
{
private:
  //! The first UserFunction of the division.
  ConstReferenceCounting<UserFunction> F;

  //! The second UserFunction of the division.
  ConstReferenceCounting<UserFunction> G;
public:

  //! Evaluation of \f$ \frac{F}{G} \f$ at \a X.
  real_t operator()(const TinyVector<3>& X) const
  {
    return (*F)(X)/(*G)(X);
  }

  //! Constructs the DivUserFunction using 2 UserFunction pointers.
  DivUserFunction(const UserFunction* F1,
		  const UserFunction* F2)
    : F(F1),
      G(F2)
  {
    ;
  }

  //! Destructor
  ~DivUserFunction()
  {
    ;
  }
};

/*!
  \class CompUserFunction
  This class allows to define the \em composition of a UserFunction by a C function.
  
  \author Stphane Del Pino
*/
class CompUserFunction
  : public UserFunction
{
private:
  //! The C function.
  real_t (*F)(const real_t x);

  //! The UserFunction
  ConstReferenceCounting<UserFunction> G;

public:

  //! Evaluates \f$ F\circ G \f$ at \a X.
  real_t operator()(const TinyVector<3>& X) const
  {
    return (*F)((*G)(X));
  }

  //! Constructs the CompUserFunction using a C function and a UserFunction.
  CompUserFunction(real_t(*func1)(const real_t x),
		   const UserFunction* UG)
    : F(func1),
      G(UG)
  {
    ;
  }

  //! Destructor.
  ~CompUserFunction()
  {
    ;
  }

};


/*!
  \class PowerUserFunction
  This class allows to define a function of the form \f$ f^g \f$.
  It is the user responsability to check that it is well defined.
  
  \author Stphane Del Pino
*/
class PowerUserFunction
  : public UserFunction
{
private:
  //! The first function.
  ConstReferenceCounting<UserFunction> F;

  //! The second function.
  ConstReferenceCounting<UserFunction> G;

public:

  //! Evaluation of \f$ F^G \f$ at point \a X.
  real_t operator()(const TinyVector<3>& X) const
  {
    return std::pow((*F)(X),(*G)(X));
  }

  /*!
    Constructs a PowerUserFunction using 2 UserFunction
    \par example: \f$ x^y \d$
  */
  PowerUserFunction(const UserFunction* UF,
		    const UserFunction* UG)
    : F(UF),
      G(UG)
  {
    ;
  }

  //! Destructor.
  ~PowerUserFunction()
  {
    ;
  }
};

// Comparison functions

/*!
  \class OneGTUserFunction
  This class allows to define the \em characteristic function
  of the set defined by \f$ f>g \f$
  
  \author Stphane Del Pino
*/
class OneGTUserFunction
  : public UserFunction
{
private:
  //! The first UserFunction
  ConstReferenceCounting<UserFunction> f;

  //! The second UserFunction
  ConstReferenceCounting<UserFunction> g;

public:

  //! Evaluates \f$ 1_{f>g} \f$ at point \a X.
  real_t operator()(const TinyVector<3>& X) const
  {
    return (real_t)((*f)(X)>(*g)(X));
  }

  //! Constructs a OneGTUserFunction using 2 UserFunction.
  OneGTUserFunction(const UserFunction* UF, const UserFunction* UG)
    : f(UF),
      g(UG)
  {
    ;
  }

  //! Destructor.
  ~OneGTUserFunction()
  {
    ;
  }
};


/*!
  \class OneGEUserFunction
  This class allows to define the \em characteristic function
  of the set defined by \f$ f\geq g \f$
  
  \author Stphane Del Pino
*/
class OneGEUserFunction
  : public UserFunction
{
 private:
  //! the first UserFunction
  ConstReferenceCounting<UserFunction> f;

  //! the second UserFunction
  ConstReferenceCounting<UserFunction> g;
 public:

  //! Evaluates \f$ 1_{f\geq g} \f$ at point \a X.
  real_t operator()(const TinyVector<3>& X) const
  {
    return (real_t)((*f)(X)>=(*g)(X));
  }

  //! Constructs a OneGEUserFunction using 2 UserFunction.
  OneGEUserFunction(const UserFunction* UF, const UserFunction* UG)
    : f(UF),
      g(UG)
  {
    ;
  }

  //! Destructor.
  ~OneGEUserFunction()
  {
    ;
  }
};

/*!
  \class OneLTUserFunction
  This class allows to define the \em characteristic function
  of the set defined by \f$ f < g \f$
  
  \author Stphane Del Pino
*/
class OneLTUserFunction
  : public UserFunction
{
 private:
  //! the first UserFunction
  ConstReferenceCounting<UserFunction> f;

  //! the second UserFunction
  ConstReferenceCounting<UserFunction> g;
 public:

  //! Evaluates \f$ 1_{f < g} \f$ at point \a X.
  real_t operator()(const TinyVector<3>& X) const {
    return (real_t)((*f)(X)<(*g)(X));
  }

  //! Constructs OneLTUserFunction using 2 UserFunction.
  OneLTUserFunction(const UserFunction* UF, const UserFunction* UG)
    : f(UF),
      g(UG)
  {
    ;
  }

  //! Destructor
  ~OneLTUserFunction()
  {
    ;
  }
};

/*!
  \class OneLEUserFunction
  This class allows to define the \em characteristic function
  of the set defined by \f$ f \leq g \f$
  
  \author Stphane Del Pino
*/
class OneLEUserFunction
  : public UserFunction
{
 private:
  //! the first UserFunction
  ConstReferenceCounting<UserFunction> f;

  //! the second UserFunction
  ConstReferenceCounting<UserFunction> g;

 public:
  //! Evaluates \f$ 1_{f \leq g} \f$ at point \a X.
  real_t operator() (const TinyVector<3>& X) const
  {
    return (real_t)((*f)(X)<=(*g)(X));
  }

  //! Constructs a OneLEUserFunction using 2 UserFunction
  OneLEUserFunction(const UserFunction* UF, const UserFunction* UG)
    : f(UF),
      g(UG)
  {
    ;
  }

  //! Destructor.
  ~OneLEUserFunction() {}
};

/*!
  \class OneEQUserFunction
  This class allows to define the \em characteristic function
  of the set defined by \f$ f = g \f$
  
  \author Stphane Del Pino
*/
class OneEQUserFunction
  : public UserFunction
{
 private:
  //! the first UserFunction
  ConstReferenceCounting<UserFunction> f;

  //! the second UserFunction
  ConstReferenceCounting<UserFunction> g;
 public:

  //! Evaluates \f$ 1_{f = g} \f$ at point \a X.
  real_t operator() (const TinyVector<3>& X) const
  {
    return (real_t)(std::abs((*f)(X)-(*g)(X))<1E-6);
  }

  //! Constructs a OneEQUserFunction using 2 UserFunction.
  OneEQUserFunction(const UserFunction* UF, const UserFunction* UG)
    : f(UF),
      g(UG)
  {
    ;
  }

  //! Destructor.
  ~OneEQUserFunction()
  {
    ;
  }
};

/*!
  \class OneNEUserFunction
  This class allows to define the \em characteristic function
  of the set defined by \f$ f \neq g \f$
  
  \author Stphane Del Pino
*/
class OneNEUserFunction
  : public UserFunction
{
 private:
  //! the first UserFunction
  ConstReferenceCounting<UserFunction> f;

  //! the second UserFunction
  ConstReferenceCounting<UserFunction> g;

 public:
  //! Evaluates \f$ 1_{f \neq g} \f$ at point \a X.
  real_t operator() (const TinyVector<3>& X) const
  {
    return (real_t)(std::abs((*f)(X)-(*g)(X))>=1E-6);
  }

  //! Constructs a OneNEUserFunction using 2 UserFunction.
  OneNEUserFunction(const UserFunction* UF, const UserFunction* UG)
    : f(UF),
      g(UG)
  {
    ;
  }

  //! Destructor.
  ~OneNEUserFunction()
  {
    ;
  }
};

/*!
  \class OneANDUserFunction
  Allows to define the boolean operation \p and  for characteristic function.
  here return 1 if \f$X \in {x, f(x)=1} \cap \{x, g(x)=1\} \f$, else returns 0.
  \bug Should be called only for Boolean UserFunction (characteristic functions)
  \author Stphane Del Pino
*/
class OneANDUserFunction
  : public UserFunction
{
 private:
  //! the first UserFunction
  ConstReferenceCounting<UserFunction> f;

  //! the second UserFunction
  ConstReferenceCounting<UserFunction> g;
 public:

  //! Evaluates \f$ \{x, f(x)=1\} \cap \{x, g(x)=1\} \f$ at point \a X.
  real_t operator() (const TinyVector<3>& X) const
  {
    return (*f)(X)*(*g)(X);
  }

  //! Constructs OneANDUserFunction using 2 UserFunction.
  OneANDUserFunction(const UserFunction* UF, const UserFunction* UG)
    : f(UF),
      g(UG)
  {
    ;
  }

  // Destructor.
  ~OneANDUserFunction()
  {
    ;
  }
};

/*!
  \class OneORUserFunction
  Allows to define the boolean operation \p and  for characteristic function.
  here return 1 if \f$X \in {x, f(x)=1} \cup \{x, g(x)=1\} \f$, else returns 0.
  \bug Should be called only for Boolean UserFunction (characteristic functions)
  \author Stphane Del Pino
*/
class OneORUserFunction
  : public UserFunction
{
private:
  //! the first UserFunction
  ConstReferenceCounting<UserFunction> f;

  //! the second UserFunction
  ConstReferenceCounting<UserFunction> g;

public:

  //! Evaluates \f$ {x, f(x)=1} \cup \{x, g(x)=1\} \f$ at point \a X.
  real_t operator() (const TinyVector<3>& X) const
  {
    const real_t fxyz = (*f)(X);
    const real_t gxyz = (*g)(X);
    return fxyz+gxyz - fxyz*gxyz; // has f and g values are 1 or 0
				  // result will be 1 or 0 !
  }

  //! Constructs OneORUserFunction using 2 UserFunction.
  OneORUserFunction(const UserFunction* UF, const UserFunction* UG)
    : f(UF),
      g(UG)
  {
    ;
  }

  //! Destructor.
  ~OneORUserFunction()
  {
    ;
  }
};

#include <Vector.hpp>

class FEMUserFunction
  : public UserFunction
{
 private:
  //! The Mesh on which lives the function.
  ReferenceCounting<Mesh> __mesh;

  //! The vector of vertices values.
  Vector<real_t>__values;

 public:

  //! Evaluates the FEMUserFunction at point \a X.
  real_t operator()(const TinyVector<3>& V) const;

  //! Access to the value at the ith vertex of the mesh \a M.
  inline real_t& operator[](const size_t& i)
  {
    return __values[i];
  }

  //! Read only Access to the value at the ith vertex of the mesh \a M.
  inline const real_t& operator[](const size_t& i) const
  {
    return __values[i];
  }

  //! Affects a \a Vector to the FEMUserFunction.
  void operator=(Vector<real_t>& u);

  //! Read-only access to the mesh pointed by \a M.
  const Mesh& mesh() const
  {
    return (*__mesh);
  }

  //! Constructs a FEMUserFunction using a pointer on a Mesh and a UserFunction
  FEMUserFunction(ReferenceCounting<Mesh> pmesh, const UserFunction& F);

  //! Constructs a FEMUserFunction using a pointer on a Mesh and a real_t
  FEMUserFunction(ReferenceCounting<Mesh> pmesh, const real_t d);

  /*!
    Constructs a FEMUserFunction using a pointer on a Mesh and a Vector of data.
    \warning Will have to change this for the vectorial case.
  */
  FEMUserFunction(ReferenceCounting<Mesh> pmesh, const Vector<real_t>& F);

  //! Copy Constructor.
  FEMUserFunction(const FEMUserFunction& F);

  //! Destructor.
  ~FEMUserFunction();
};

class FEM0UserFunction
  : public UserFunction
{
 private:
  //! The Mesh on which lives the function.
  ReferenceCounting<Mesh> __mesh;

  //! The vector of cell values.
  Vector<real_t>__values;

 public:

  //! Evaluates the FEM0UserFunction at point \a X.
  real_t operator()(const TinyVector<3>& V) const;

  //! Access to the value at the ith vertex of the mesh \a M.
  inline real_t& operator[](const size_t& i)
  {
    return __values[i];
  }

  //! Read only Access to the value at the ith cell of the mesh \a M.
  inline const real_t& operator[](const size_t& i) const
  {
    return __values[i];
  }

  //! Affects a \a Vector to the FEM0UserFunction.
  void operator=(Vector<real_t>& u);

  //! Read-only access to the mesh pointed by \a M.
  const Mesh& mesh() const
  {
    return (*__mesh);
  }

  //! Constructs a FEM0UserFunction using a pointer on a Mesh and a UserFunction
  FEM0UserFunction(ReferenceCounting<Mesh> pmesh, const UserFunction& F);

  //! Constructs a FEMUserFunction using a pointer on a Mesh and a real_t
  FEM0UserFunction(ReferenceCounting<Mesh> pmesh, const real_t d);

  /*!
    Constructs a FEMUserFunction using a pointer on a Mesh and a Vector of data.
    \warning Will have to change this for the vectorial case.
  */
  FEM0UserFunction(ReferenceCounting<Mesh> pmesh, const Vector<real_t>& F);

  //! Copy Constructor.
  FEM0UserFunction(const FEM0UserFunction& F);

  //! Destructor.
  ~FEM0UserFunction();
};


class Scene;
class Object;

/*!
  \class OneUserFunction
  This class implemented characteristic function of Objects for a given reference:
  \f$ 1_{\cup_i o_i} \f$.
  
  \author Stphane Del Pino
*/
class OneUserFunction
  : public UserFunction
{
private:
  //! The list of objects for which 
  std::vector<Object*> o;
public:

  //! Evalutes \f$ 1_{\cup_i o_i} \f$ at point \a X.
  real_t operator()(const TinyVector<3>& X) const;

  //! Constructs a OneUserFunction using the POV-Ray reference \a r.
  OneUserFunction(TinyVector<3>& r);

  //! Destructor
  ~OneUserFunction()
  {
    ;
  }
};

/*!
  \class OneMeshUserFunction
  Used to define the characteristic function of a Structured3DMesh.

  \author Stphane Del Pino
*/
class OneMeshUserFunction
  : public UserFunction
{
private:
  //! The Structured3DMesh \a mesh.
  ConstReferenceCounting<Structured3DMesh> __mesh;
public:

  //! Evalutes the characteristic function of the \a mesh at point \a X
  real_t operator()(const TinyVector<3>& X) const;

  //! Constructs a OneMeshUserFunction using a \a mesh.
  OneMeshUserFunction(ConstReferenceCounting<Structured3DMesh> mesh);

  //! Destructor.
  ~OneMeshUserFunction();
};

/**
 * @class  OneDomainUserFunction
 * @author Stephane Del Pino
 * @date   Sun Sep 21 15:05:18 2003
 * 
 * @brief  return 1 if argument vertex is in the domain 0 if not.
 * 
 * 
 */
class Domain;
class OneDomainUserFunction
  : public UserFunction
{
private:
  ConstReferenceCounting<Domain> __domain; /**< The domain */
public:

  /** 
   * 
   * Returns 1 if \a x is inside the domain, else 0.
   * 
   * @return 1 or 0
   */
  real_t operator()(const TinyVector<3>& x) const;

  /** 
   * Constructor
   * 
   * @param domain the given domain
   * 
   */
  OneDomainUserFunction(ConstReferenceCounting<Domain> domain);

  /** 
   * Destructor
   */
  ~OneDomainUserFunction();
};

namespace UF {
  static ConstUserFunction zero(0);
}


#endif // _USER_FUNCTION_

