//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: VTKDriver.cpp,v 1.6 2003/12/26 18:38:03 uid68082 Exp $

#include <config.h>

#ifdef HAVE_LIBVTK

#include <MeshOfHexahedra.hpp>
#include <MeshOfTetrahedra.hpp>

#include <SurfaceMeshOfTriangles.hpp>
#include <SurfaceMeshOfQuadrangles.hpp>

#include <VTKDriver.hpp>

#include <vtkPointData.h>
#include <vtkCellData.h>
#include <vtkLODActor.h>
#include <vtkRenderer.h>
#include <vtkPolyDataMapper.h>
#include <vtkRenderWindow.h>
#include <vtkRenderWindowInteractor.h>
#include <vtkTriangle.h>
#include <vtkQuad.h>
#include <vtkHexahedron.h>
#include <vtkTetra.h>
#include <vtkUnstructuredGrid.h>
#include <vtkDataSetMapper.h>
#include <vtkDataSetToPolyDataFilter.h>
#include <vtkDoubleArray.h>

template <>
void VTKDriver::__plotMesh(const MeshOfTetrahedra& m)
{
  vtkPoints* vertices = vtkPoints::New();

  vertices->SetNumberOfPoints(m.numberOfVertices());

  for (size_t i=0; i<m.numberOfVertices(); ++i) {
    const Vertex& X = m.vertex(i);
    vertices->InsertPoint(i,X[0],X[1],X[2]);
  }

  vtkTetra* tetrahedron = vtkTetra::New();

  vtkUnstructuredGrid* aTetrahedraGrid = vtkUnstructuredGrid::New();
  aTetrahedraGrid->Allocate(m.numberOfCells(),1);

  for (size_t i=0; i<m.numberOfCells(); ++i) {
    const Tetrahedron& T = m.cell(i);
    for (size_t j=0; j<4; ++j)
      tetrahedron->GetPointIds()->SetId(j,m.vertexNumber(T(j)));

    aTetrahedraGrid->InsertNextCell(tetrahedron->GetCellType(),
				    tetrahedron->GetPointIds());
  }

  aTetrahedraGrid->SetPoints(vertices);

  vtkDoubleArray* values = vtkDoubleArray::New();
  values->Resize(m.numberOfVertices());
  for (size_t i=0; i<m.numberOfVertices(); ++i) {
    values->InsertNextValue(m.vertex(i)[0]);
  }

  aTetrahedraGrid->GetPointData()->SetScalars(values);

  // We create an instance of vtkPolyDataMapper to map the polygonal data 
  // into graphics primitives. We connect the output of the cone souece 
  // to the input of this mapper 

  vtkDataSetMapper* mapper = vtkDataSetMapper::New();
  mapper->SetInput(aTetrahedraGrid);


  // create an actor to represent the cone. The actor coordinates rendering of
  // the graphics primitives for a mapper. We set this actor's mapper to be
  // coneMapper which we created above.
  //
  vtkLODActor *anActor = vtkLODActor::New();
  anActor->SetMapper( mapper );
  anActor->SetNumberOfCloudPoints(1000);

  // Create the Renderer and assign actors to it. A renderer is like a
  // viewport. It is part or all of a window on the screen and it is
  // responsible for drawing the actors it has.  We also set the background
  // color here
  //
  vtkRenderer *ren1= vtkRenderer::New();
  ren1->AddActor( anActor );
  ren1->SetBackground( 0.1, 0.2, 0.4 );

  //
  // Finally we create the render window which will show up on the screen
  // We put our renderer into the render window using AddRenderer. We also
  // set the size to be 300 pixels by 300
  //
  vtkRenderWindow *renWin = vtkRenderWindow::New();
  renWin->AddRenderer( ren1 );
  renWin->SetSize( 300, 300 );

  vtkRenderWindowInteractor* iren = vtkRenderWindowInteractor::New();
  iren->SetRenderWindow(renWin);

  // now we loop over 360 degreeees and render the cone each time
  //
  // render the image
  renWin->Render();
  iren->Start();
  //
  // Free up any objects we created
  //
//   cone->Delete();
//   coneMapper->Delete();
//   coneActor->Delete();
//   ren1->Delete();
}

template<>
void VTKDriver::__plotMesh(const MeshOfHexahedra& m)
{
  vtkPoints* vertices = vtkPoints::New();

  vertices->SetNumberOfPoints(m.numberOfVertices());

  for (size_t i=0; i<m.numberOfVertices(); ++i) {
    const Vertex& X = m.vertex(i);
    vertices->InsertPoint(i,X[0],X[1],X[2]);
  }

  vtkHexahedron* hexahedron = vtkHexahedron::New();

  vtkUnstructuredGrid* anHexahedraGrid = vtkUnstructuredGrid::New();
  anHexahedraGrid->Allocate(m.numberOfCells(),1);

  for (size_t i=0; i<m.numberOfCells(); ++i) {
    const Hexahedron& H = m.cell(i);
    for (size_t j=0; j<8; ++j)
      hexahedron->GetPointIds()->SetId(j,m.vertexNumber(H(j)));

    anHexahedraGrid->InsertNextCell(hexahedron->GetCellType(),
				  hexahedron->GetPointIds());
  }

  anHexahedraGrid->SetPoints(vertices);

  vtkDoubleArray* values = vtkDoubleArray::New();
  values->Resize(m.numberOfVertices());
  for (size_t i=0; i<m.numberOfVertices(); ++i) {
    values->InsertNextValue(m.vertex(i)[0]);
  }

  anHexahedraGrid->GetPointData()->SetScalars(values);

  // We create an instance of vtkPolyDataMapper to map the polygonal data 
  // into graphics primitives. We connect the output of the cone souece 
  // to the input of this mapper 

  vtkDataSetMapper* mapper = vtkDataSetMapper::New();
  mapper->SetInput(anHexahedraGrid);


  // create an actor to represent the cone. The actor coordinates rendering of
  // the graphics primitives for a mapper. We set this actor's mapper to be
  // coneMapper which we created above.
  //
  vtkLODActor *anActor = vtkLODActor::New();
  anActor->SetMapper( mapper );
  anActor->SetNumberOfCloudPoints(1000);

  // Create the Renderer and assign actors to it. A renderer is like a
  // viewport. It is part or all of a window on the screen and it is
  // responsible for drawing the actors it has.  We also set the background
  // color here
  //
  vtkRenderer *ren1= vtkRenderer::New();
  ren1->AddActor( anActor );
  ren1->SetBackground( 0.1, 0.2, 0.4 );

  //
  // Finally we create the render window which will show up on the screen
  // We put our renderer into the render window using AddRenderer. We also
  // set the size to be 300 pixels by 300
  //
  vtkRenderWindow *renWin = vtkRenderWindow::New();
  renWin->AddRenderer( ren1 );
  renWin->SetSize( 300, 300 );

  vtkRenderWindowInteractor* iren = vtkRenderWindowInteractor::New();
  iren->SetRenderWindow(renWin);

  // now we loop over 360 degreeees and render the cone each time
  //
  // render the image
  renWin->Render();
  iren->Start();
  //
  // Free up any objects we created
  //
//   cone->Delete();
//   coneMapper->Delete();
//   coneActor->Delete();
//   ren1->Delete();
}

template<>
void VTKDriver::__plotMesh(const SurfaceMeshOfTriangles& s)
{
  vtkPoints* vertices = vtkPoints::New();

  vertices->SetNumberOfPoints(s.numberOfVertices());

  for (size_t i=0; i<s.numberOfVertices(); ++i) {
    const Vertex& X = s.vertex(i);
    vertices->InsertPoint(i,X[0],X[1],X[2]);
  }

  vtkTriangle* triangle = vtkTriangle::New();

  vtkUnstructuredGrid* aTriangleGrid = vtkUnstructuredGrid::New();
  aTriangleGrid->Allocate(s.numberOfCells(),1);

  for (size_t i=0; i<s.numberOfCells(); ++i) {
    const Triangle& T = s.cell(i);
    for (size_t j=0; j<3; ++j) {
      triangle->GetPointIds()->SetId(j,s.vertexNumber(T(j)));
    }

    aTriangleGrid->InsertNextCell(triangle->GetCellType(),
				  triangle->GetPointIds());
  }

  aTriangleGrid->SetPoints(vertices);

  vtkDoubleArray* values = vtkDoubleArray::New();
  values->Resize(s.numberOfCells());
  for (size_t i=0; i<s.numberOfCells(); ++i) {
    values->InsertNextValue(s.cell(i).reference()/4.);
  }

  aTriangleGrid->GetCellData()->SetScalars(values);

  // We create an instance of vtkPolyDataMapper to map the polygonal data 
  // into graphics primitives. We connect the output of the cone souece 
  // to the input of this mapper 

  vtkDataSetMapper* mapper = vtkDataSetMapper::New();
  mapper->SetInput(aTriangleGrid);


  // create an actor to represent the cone. The actor coordinates rendering of
  // the graphics primitives for a mapper. We set this actor's mapper to be
  // coneMapper which we created above.
  //
  vtkLODActor *anActor = vtkLODActor::New();
  anActor->SetMapper( mapper );
  anActor->SetNumberOfCloudPoints(10000);


  // Create the Renderer and assign actors to it. A renderer is like a
  // viewport. It is part or all of a window on the screen and it is
  // responsible for drawing the actors it has.  We also set the background
  // color here
  //
  vtkRenderer *ren1= vtkRenderer::New();
  ren1->AddActor( anActor );
  ren1->SetBackground( 0.1, 0.2, 0.4 );

  //
  // Finally we create the render window which will show up on the screen
  // We put our renderer into the render window using AddRenderer. We also
  // set the size to be 300 pixels by 300
  //
  vtkRenderWindow *renWin = vtkRenderWindow::New();
  renWin->AddRenderer( ren1 );
  renWin->SetSize( 300, 300 );

  vtkRenderWindowInteractor* iren = vtkRenderWindowInteractor::New();
  iren->SetRenderWindow(renWin);

  // now we loop over 360 degreeees and render the cone each time
  //
  // render the image
  renWin->Render();
  iren->Start();
  //
  // Free up any objects we created
  //
//   cone->Delete();
//   coneMapper->Delete();
//   coneActor->Delete();
//   ren1->Delete();
//   renWin->Delete();
}

template<>
void VTKDriver::__plotMesh(const SurfaceMeshOfQuadrangles& s)
{
  vtkPoints* vertices = vtkPoints::New();

  vertices->SetNumberOfPoints(s.numberOfVertices());

  for (size_t i=0; i<s.numberOfVertices(); ++i) {
    const Vertex& X = s.vertex(i);
    vertices->InsertPoint(i,X[0],X[1],X[2]);
  }

  vtkQuad* quad = vtkQuad::New();

  vtkUnstructuredGrid* aQuadGrid = vtkUnstructuredGrid::New();
  aQuadGrid->Allocate(s.numberOfCells(),1);

  for (size_t i=0; i<s.numberOfCells(); ++i) {
    const Quadrangle& Q = s.cell(i);
    for (size_t j=0; j<4; ++j) {
      quad->GetPointIds()->SetId(j,s.vertexNumber(Q(j)));
    }

    aQuadGrid->InsertNextCell(quad->GetCellType(),
			      quad->GetPointIds());
  }

  aQuadGrid->SetPoints(vertices);

  vtkDoubleArray* values = vtkDoubleArray::New();
  values->Resize(s.numberOfCells());
  for (size_t i=0; i<s.numberOfCells(); ++i) {
    values->InsertNextValue(s.cell(i).reference()/6.);
  }

  aQuadGrid->GetCellData()->SetScalars(values);

  // We create an instance of vtkPolyDataMapper to map the polygonal data 
  // into graphics primitives. We connect the output of the cone souece 
  // to the input of this mapper 

  vtkDataSetMapper* mapper = vtkDataSetMapper::New();
  mapper->SetInput(aQuadGrid);


  // create an actor to represent the cone. The actor coordinates rendering of
  // the graphics primitives for a mapper. We set this actor's mapper to be
  // coneMapper which we created above.
  //
  vtkLODActor *anActor = vtkLODActor::New();
  anActor->SetMapper( mapper );
  anActor->SetNumberOfCloudPoints(10000);


  // Create the Renderer and assign actors to it. A renderer is like a
  // viewport. It is part or all of a window on the screen and it is
  // responsible for drawing the actors it has.  We also set the background
  // color here
  //
  vtkRenderer *ren1= vtkRenderer::New();
  ren1->AddActor( anActor );
  ren1->SetBackground( 0.1, 0.2, 0.4 );

  //
  // Finally we create the render window which will show up on the screen
  // We put our renderer into the render window using AddRenderer. We also
  // set the size to be 300 pixels by 300
  //
  vtkRenderWindow *renWin = vtkRenderWindow::New();
  renWin->AddRenderer( ren1 );
  renWin->SetSize( 300, 300 );

  vtkRenderWindowInteractor* iren = vtkRenderWindowInteractor::New();
  iren->SetRenderWindow(renWin);

  // now we loop over 360 degreeees and render the cone each time
  //
  // render the image
  renWin->Render();
  iren->Start();
  //
  // Free up any objects we created
  //
//   cone->Delete();
//   coneMapper->Delete();
//   coneActor->Delete();
//   ren1->Delete();
//   renWin->Delete();
}


void VTKDriver::plotMesh(const Mesh& m)
{
  switch (m.type()) {
  case Mesh::hexahedraMesh: {
    this->__plotMesh(dynamic_cast<const MeshOfHexahedra&>(m));
    break;
  }
  case Mesh::tetrahedraMesh: {
    this->__plotMesh(dynamic_cast<const MeshOfTetrahedra&>(m));
    break;
  }
  case Mesh::surfaceMeshTriangles: {
    this->__plotMesh(dynamic_cast<const SurfaceMeshOfTriangles&>(m));
    break;
  }
  case Mesh::surfaceMeshQuadrangles: {
    this->__plotMesh(dynamic_cast<const SurfaceMeshOfQuadrangles&>(m));
    break;
  }
  default: {
    fferr(0) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
    std::exit(1);
  }
  }
}

#endif // HAVE_LIBVTK
