//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Variable.hpp,v 1.2 2003/10/18 16:05:27 delpinux Exp $

#ifndef _VARIABLE_HPP_
#define _VARIABLE_HPP_

#include <ReferenceCounting.hpp>
#include <string>

/*!
  \class Variable

  This class is the interface for all variables in the language.

  \author Stephane Del Pino
*/
class Variable
{
public:
  enum Type {
    real,
    vector3,
    function,
    testFuncion,
    mesh,
    domain,
    scene,
    unknown,
    unknownList
  };

private:
  //! The type of the variable
  const Variable::Type __type;

  //! the name of the variable
  const std::string __name;

public:
  const Variable::Type& type() const
  {
    return __type;
  }

  const std::string& name() const
  {
    return __name;
  }

  Variable(const std::string& name,
	   const Variable::Type& type)
    : __type(type),
      __name(name)
  {
    ;
  }

  Variable(const Variable& v)
    : __type(v.__type),
      __name(v.__name)
  {
    ;
  }

  virtual ~Variable()
  {
    ;
  }
};


/*!
  \class RealVariable

  This class allow manipulation of real_t variables.

  \author Stephane Del Pino
 */
class RealExpression;
class RealVariable
  : public Variable
{
private:
  ReferenceCounting<RealExpression> __expression;

public:
  const ReferenceCounting<RealExpression> expression() const;

  ReferenceCounting<RealExpression>
  operator=(ReferenceCounting<RealExpression> e);

  RealVariable(const std::string& name,
	       ReferenceCounting<RealExpression> expression);

  RealVariable(const RealVariable& rv);

  ~RealVariable();
};

/*!
  \class StringVariable

  This class allow manipulation of string variables.

  \author Stephane Del Pino
 */
class StringExpression;
class StringVariable
  : public Variable
{
private:
  ReferenceCounting<StringExpression> __expression;

public:
  const ReferenceCounting<StringExpression> expression() const;

  ReferenceCounting<StringExpression>
  operator=(ReferenceCounting<StringExpression> e);

  StringVariable(const std::string& name);

  StringVariable(const std::string& name,
		ReferenceCounting<StringExpression> expression);

  StringVariable(const StringVariable& rv);

  ~StringVariable();
};

/*!
  \class Vector3Variable

  This class allow manipulation of Vector3 variables.

  \author Stephane Del Pino
 */
class Vector3Expression;
class Vector3Variable
  : public Variable
{
private:
  ReferenceCounting<Vector3Expression> __expression;

public:
  const ReferenceCounting<Vector3Expression> expression() const;

  ReferenceCounting<Vector3Expression>
  operator=(ReferenceCounting<Vector3Expression> e);

  Vector3Variable(const std::string& name,
		  ReferenceCounting<Vector3Expression> expression);

  Vector3Variable(const Vector3Variable& rv);

  ~Vector3Variable();
};


/*!
  \class FunctionVariable

  This class allow manipulation of Function variables.

  \author Stephane Del Pino
 */
class FunctionExpression;
class FunctionVariable
  : public Variable
{
private:
  ReferenceCounting<FunctionExpression> __expression;

  // This is used to check if function variable is an Unknown
  bool __subscribed;

public:

  void subscribe()
  {
    __subscribed = true;
  }

  void unsubscribe()
  {
    __subscribed = false;
  }

  const bool& isSubscribed() const
  {
    return __subscribed;
  }

  ReferenceCounting<FunctionExpression> expression() const;

  ReferenceCounting<FunctionExpression>
  operator=(ReferenceCounting<FunctionExpression> e);

  FunctionVariable(const std::string& name,
		   ReferenceCounting<FunctionExpression> expression);

  FunctionVariable(const FunctionVariable& rv);

  ~FunctionVariable();
};

/*!
  \class TestFunctionVariable

  This class allow manipulation of Test Functions.

  \author Stephane Del Pino
 */
class TestFunctionVariable
  : public Variable
{
public:
  TestFunctionVariable(const std::string& name);

  TestFunctionVariable(const TestFunctionVariable& rv);

  ~TestFunctionVariable();
};



/*!
  \class MeshVariable

  This class allow manipulation of mesh variables.

  \author Stephane Del Pino
 */
class MeshExpression;
class MeshVariable
  : public Variable
{
private:
  ReferenceCounting<MeshExpression> __expression;

public:
  const ReferenceCounting<MeshExpression> expression() const;

  ReferenceCounting<MeshExpression>
  operator=(ReferenceCounting<MeshExpression> e);

  MeshVariable(const std::string& name);

  MeshVariable(const std::string& name,
	       ReferenceCounting<MeshExpression> expression);

  MeshVariable(const MeshVariable& rv);

  ~MeshVariable();
};

/*!
  \class SceneVariable

  This class allow manipulation of scene variables.

  \author Stephane Del Pino
 */
class SceneExpression;
class SceneVariable
  : public Variable
{
private:
  ReferenceCounting<SceneExpression> __expression;

public:
  const ReferenceCounting<SceneExpression> expression() const;

  ReferenceCounting<SceneExpression>
  operator=(ReferenceCounting<SceneExpression> e);

  SceneVariable(const std::string& name);

  SceneVariable(const std::string& name,
		ReferenceCounting<SceneExpression> expression);

  SceneVariable(const SceneVariable& rv);

  ~SceneVariable();
};


/*!
  \class DomainVariable

  This class allow manipulation of domain variables.

  \author Stephane Del Pino
 */
class DomainExpression;
class DomainVariable
  : public Variable
{
private:
  ReferenceCounting<DomainExpression> __expression;

public:
  const ReferenceCounting<DomainExpression> expression() const;

  ReferenceCounting<DomainExpression>
  operator=(ReferenceCounting<DomainExpression> e);

  DomainVariable(const std::string& name);

  DomainVariable(const std::string& name,
		 ReferenceCounting<DomainExpression> expression);

  DomainVariable(const DomainVariable& rv);

  ~DomainVariable();
};

/*!
  \class SolverOptionsVariable

  This class allow manipulation of solver variables.

  \author Stephane Del Pino
 */
class SolverOptionsExpression;
class SolverOptionsVariable
  : public Variable
{
private:
  ReferenceCounting<SolverOptionsExpression> __expression;

public:
  const ReferenceCounting<SolverOptionsExpression> expression() const;

  ReferenceCounting<SolverOptionsExpression>
  operator=(ReferenceCounting<SolverOptionsExpression> e);

  SolverOptionsVariable(const std::string& name);

  SolverOptionsVariable(const std::string& name,
			ReferenceCounting<SolverOptionsExpression> expression);

  SolverOptionsVariable(const SolverOptionsVariable& rv);

  ~SolverOptionsVariable();
};


/*!
  \class PDEProblemVariable

  This class allow manipulation of solver variables.

  \author Stephane Del Pino
 */
class PDEProblemExpression;
class PDEProblemVariable
  : public Variable
{
private:
  ReferenceCounting<PDEProblemExpression> __expression;

public:
  const ReferenceCounting<PDEProblemExpression> expression() const;

  ReferenceCounting<PDEProblemExpression>
  operator=(ReferenceCounting<PDEProblemExpression> e);

  PDEProblemVariable(const std::string& name);

  PDEProblemVariable(const std::string& name,
		     ReferenceCounting<PDEProblemExpression> expression);

  PDEProblemVariable(const PDEProblemVariable& rv);

  ~PDEProblemVariable();
};

/*!
  \class SolverVariable

  This class allow manipulation of solver variables.

  \author Stephane Del Pino
 */
class SolverExpression;
class SolverVariable
  : public Variable
{
private:
  ReferenceCounting<SolverExpression> __expression;

public:
  const ReferenceCounting<SolverExpression> expression() const;

  ReferenceCounting<SolverExpression>
  operator=(ReferenceCounting<SolverExpression> e);

  SolverVariable(const std::string& name);

  SolverVariable(const std::string& name,
		 ReferenceCounting<SolverExpression> expression);

  SolverVariable(const SolverVariable& rv);

  ~SolverVariable();
};
#endif // _VARIABLE_HPP_

