//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Quadrangle.hpp,v 1.9 2004/02/13 19:54:08 delpinux Exp $

// This is used to describe Surfacic Quadrangles Elements

#ifndef  QUADRANGLE_HPP
#define  QUADRANGLE_HPP

#include <cmath>
#include <Cell.hpp>

#include <TinyVector.hpp>
#include <Edge.hpp>

/**
 * @file   Quadrangle.hpp
 * @author Stphane Del Pino
 * @date   Sat Jul 27 17:33:58 2002
 * 
 * @brief  Describes quadrangles in 3d
 * 
 * @todo use numberOfEdges() in constructor instead of 4 when compiler allows it.
 */

class Quadrangle
  : public Cell
{
private:
  /**
   * The mother cell (ie: a 3d element which interacts with the quadrangle)
   * 
   */
  const Cell* __3Dcell;

public:
  enum {
    NumberOfVertices   = 4,	/**< number of vertices   */
    NumberOfNeighbours = 4,	/**< number of neighbours */
    NumberOfEdges      = 4	/**< number of edges      */
  };

  typedef Edge BorderType;

  /**
   * Access to the mother cell
   * 
   */
  const Cell& mother() const
  {
    assert(__3Dcell != 0);
    return *__3Dcell;
  }

  /**
   * Sets the mother cell
   * 
   */
  void setMother(const Cell* cell)
  {
    __3Dcell = cell;
  }

  /** 
   * Returns the "normal" to the face.  This is only a bad
   * approximation since the quadrangle may not be contained in a
   * plane.
   * 
   * @return 
   */
  const inline TinyVector<3> Normal() const
  {
    // IN This Function We Assume That The 4 Vertices Are Aligned.
    // Could be better to use the mean value ...

    const TinyVector<3>& v0 = *(__vertices[0]);
    const TinyVector<3>& v1 = *(__vertices[1]);
    //    const TinyVector<3>& v2 = *(__vertices[2]);
    const TinyVector<3>& v3 = *(__vertices[3]);

    const TinyVector<3> A(v1-v0);
    const TinyVector<3> B(v3-v0);
    TinyVector<3> N(A^B);
    const real_t n = Norm(N);
    N/=n;
    return N;
  }

  /** 
   * Access to the number of vertices of this Cell.
   * 
   * 
   * @return NumberOfVertices
   */
  size_t numberOfVertices() const
  {
    return NumberOfVertices;
  }

  /** 
   * Returns the type of the cell
   * 
   * @return Cell::quadrangle3d
   */
  virtual Cell::Type type() const
  {
    return Cell::quadrangle3d;
  }

  /** 
   * operator =
   * 
   * @param S 
   * 
   * @return 
   */
  inline const Quadrangle& operator=(const Quadrangle& S)
  {
    Cell::operator=(S);

    __3Dcell = S.__3Dcell;

    return *this;
  }

  /** 
   * Constructs the structure
   * 
   * 
   */
  Quadrangle()
    : Cell(NumberOfVertices),
      __3Dcell(0)
  {
    __volume = 0.;
  }

  /** 
   * Constructor: the quadrangle is defined by 4 vertices
   * 
   * @param V0 first vertex
   * @param V1 second vertex
   * @param V2 third vertex
   * @param V3 fourth vertex
   * @param a3Dcell a pointer to the 3D cell that contains this quadrangle as a face
   * @param reference the reference of this quadrangle (default is 0)
   *
   */
  Quadrangle(const Vertex& V0,
	     const Vertex& V1,
	     const Vertex& V2,
	     const Vertex& V3,
	     const size_t reference=0)
    : Cell(NumberOfVertices, reference)
  {
    __vertices[0] = (Vertex*)&V0;
    __vertices[1] = (Vertex*)&V1;
    __vertices[2] = (Vertex*)&V2;
    __vertices[3] = (Vertex*)&V3;

    const TinyVector<3>& v0 = *(__vertices[0]);
    const TinyVector<3>& v1 = *(__vertices[1]);
    const TinyVector<3>& v2 = *(__vertices[2]);
    const TinyVector<3>& v3 = *(__vertices[3]);

    const TinyVector<3> A(v1-v0);
    const TinyVector<3> B(v3-v0);

    const TinyVector<3> C(v1-v2);
    const TinyVector<3> D(v3-v2);

    __volume = 0.5*(Norm(A^B)+Norm(C^D));
  }

  /** 
   * Copy Constructor
   * 
   * @param sq 
   * 
   */
  Quadrangle(const Quadrangle& sq)
    : Cell(sq),
      __3Dcell(sq.__3Dcell)
  {
    ;
  }

  /** 
   * Destructor
   * 
   * 
   */
  ~Quadrangle()
  {
    ;
  }
};

#endif // QUADRANGLE_HPP

