//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: FFSyntaxHighlighter.hpp,v 1.1 2004/11/27 15:51:38 delpinux Exp $

#ifndef FF_SYNTAX_HIGHLIGHTER_HPP
#define FF_SYNTAX_HIGHLIGHTER_HPP

#include <FFLexer.hpp>

#include <qsyntaxhighlighter.h>
#include <qregexp.h>
#include <qstring.h>

/**
 * @file   FFSyntaxHighlighter.hpp
 * @author Stephane Del Pino
 * @date   Sat Nov 27 01:32:21 2004
 * 
 * @brief This class builds a simple syntax highlighter for ff3d
 * language files
 * 
 */
class FFSyntaxHighlighter
  : public QSyntaxHighlighter
{
private:
  QFont __defaultFont;
  QFont __keywordFont;
  QFont __commentFont;
  QFont __stringFont;

  QColor __defaultColor;
  QColor __keywordColor;
  QColor __commentColor;
  QColor __stringColor;

  enum ParagraphState
  {
    commonParagraph  =-2,
    commentParagraph = 1,
    stringParagraph  = 2
  };
  QRegExp __keywords;		/**< ff3d's keywords */

public:
  int highlightParagraph(const QString& text, int currentState)
  {
    // Reset default font
    setFormat(0, text.length(), __defaultFont, __defaultColor);

    int initialPosition=0;
    switch(currentState) {
    case commonParagraph:{
      break;
    }
    case stringParagraph: {
      int endPosition = text.find(QRegExp("\""),initialPosition);
      if (endPosition < 0) {
	setFormat(initialPosition, text.length(), __stringFont, __stringColor);
	return stringParagraph;
      } else {
	setFormat(initialPosition, endPosition-initialPosition+1, __stringFont, __stringColor);
	initialPosition = endPosition+1;
      }
      break;
    }
    case commentParagraph: {
      int endPosition = text.find(QRegExp("\\*/"),initialPosition);
      if (endPosition < 0) {
	setFormat(initialPosition, text.length(), __commentFont, __commentColor);
	return commentParagraph;
      } else {
	setFormat(initialPosition, endPosition-initialPosition+2, __commentFont, __commentColor);
	initialPosition = endPosition + 2;
      }
      break;
    }
    }

    // initial highlighting position can be none 0 (multiline comments or strings)
    int position = initialPosition;

    // Highlights keywords
    while ((position = text.find(__keywords,position)) != -1)    {
      setFormat(position, __keywords.matchedLength(), __keywordFont, __keywordColor);
      position += __keywords.matchedLength();
    }

    // Highlights comments and strings
    position = initialPosition;
    QRegExp commentOrString("(//.*$|/\\*.*|\".*)");

    while ((position = text.find(commentOrString, position)) != -1) {
      if (text[position]=='\"') {
	int endPosition = text.find(QRegExp("\""), position+1);
	  if (endPosition < 0) {
	    setFormat(position, commentOrString.matchedLength(), __stringFont, __stringColor);
	    return stringParagraph;
	  } else {
	    setFormat(position, endPosition-position+1, __stringFont, __stringColor);
	    position = endPosition+1;
	  }
      } else {
	if (text[position+1] == '/') {
	  setFormat(position, commentOrString.matchedLength(), __commentFont, __commentColor);
	  return commonParagraph;
	} else {
	  int endPosition = text.find(QRegExp("\\*/"),position+2);
	  if (endPosition < 0) {
	    setFormat(position, commentOrString.matchedLength(), __commentFont, __commentColor);
	    return commentParagraph;
	  } else {
	    setFormat(position, endPosition-position+2, __commentFont, __commentColor);
	    position = endPosition+2;
	  }
	}
      }
    }	    
    return commonParagraph;
  }

  /** 
   * Constructs a syntax highlighter for ff3d
   * 
   * @param textEdit the text editor to highlight
   * 
   */
  FFSyntaxHighlighter(QTextEdit* textEdit)
    : QSyntaxHighlighter(textEdit),
      __defaultFont(textEdit->currentFont()),
      __keywordFont(textEdit->currentFont()),
      __commentFont(textEdit->currentFont()),
      __stringFont(textEdit->currentFont()),
      __defaultColor(0,0,0),
      __keywordColor(0,0,255),
      __commentColor(150,150,150),
      __stringColor(50,155,155)
  {
    __commentFont.setItalic(true);
    __keywordFont.setBold(true);
    Lexer::KeyWordList keywordList;
    FFLexer::define(keywordList);
    QString keywordsRegExp("\\b(");
    const size_t numberOfKeyWords = keywordList.size();
    size_t n = 0;
    for (Lexer::KeyWordList::const_iterator i=keywordList.begin();
	 i != keywordList.end(); ++i,++n) {
      keywordsRegExp.append(i->first);
      if (n<numberOfKeyWords-1) keywordsRegExp.append("|");
    }
    keywordsRegExp.append(")\\b"); 
    __keywords = QRegExp(keywordsRegExp);
  }

  /** 
   * Destructor
   * 
   * 
   * @return 
   */
  virtual ~FFSyntaxHighlighter()
  {
    ;
  }
};


#endif // FF_SYNTAX_HIGHLIGHTER_HPP
