%{ /* -*- c++ -*- */

   /*
    *  This file is part of ff3d - http://www.freefem.org/ff3d
    *  Copyright (C) 2001, 2002, 2003 Stphane Del Pino
    * 
    *  This program is free software; you can redistribute it and/or modify
    *  it under the terms of the GNU General Public License as published by
    *  the Free Software Foundation; either version 2, or (at your option)
    *  any later version.
    * 
    *  This program is distributed in the hope that it will be useful,
    *  but WITHOUT ANY WARRANTY; without even the implied warranty of
    *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    *  GNU General Public License for more details.
    * 
    *  You should have received a copy of the GNU General Public License
    *  along with this program; if not, write to the Free Software Foundation,
    *  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
    * 
    *  $Id: parse.ff.yy,v 1.34 2005/01/08 18:35:15 delpinux Exp $
    * 
    */
   
#include <Types.hpp>

#include <stdlib.h>
#include <cmath>
#include <iostream>
#include <vector>

  int yyerror (const char*);
  int yyerror2(const char*);

#include<FFLexer.hpp>
  extern FFLexer* fflexer;

#define fflex fflexer->fflex
#define YYERROR_VERBOSE

#include <EmbededFunctions.hpp>

#include <Expression.hpp>
#include <RealExpression.hpp>
#include <StringExpression.hpp>

#include <BooleanExpression.hpp>
#include <Vector3Expression.hpp>
#include <FunctionExpression.hpp>

#include <UnknownListExpression.hpp>

#include <MeshExpression.hpp>
#include <SceneExpression.hpp>

#include <DomainExpression.hpp>
#include <InsideExpression.hpp>
#include <InsideListExpression.hpp>

#include <BoundaryConditionExpression.hpp>
#include <BoundaryExpression.hpp>

#include <BoundaryConditionListExpression.hpp>

#include <PDEOperatorExpression.hpp>
#include <PDEOperatorSumExpression.hpp>

#include <SolverOptionsExpression.hpp>
#include <SubOptionExpression.hpp>
#include <SubOptionListExpression.hpp>

#include <PDEProblemExpression.hpp>
#include <PDESystemExpression.hpp>

#include <PDEEquationExpression.hpp>

#include <SolverExpression.hpp>

#include <TestFunctionExpression.hpp>
#include <IntegratedExpression.hpp>
#include <IntegratedOperatorExpression.hpp>

#include <LinearExpression.hpp>
#include <MultiLinearExpression.hpp>
#include <MultiLinearFormExpression.hpp>

#include <VariationalFormulaExpression.hpp>
#include <VariationalProblemExpression.hpp>

#include <VariationalDirichletListExpression.hpp>

#include <Instruction.hpp>
#include <VariableList.hpp>

#include <ProblemExpression.hpp>
#include <OstreamExpression.hpp>
#include <OstreamExpressionList.hpp>

#include <ErrorHandler.hpp>
#include <Stringify.hpp>

  //! The set of instructions.
  extern std::vector<ReferenceCounting<Instruction> > iSet;
%}

%union {
  real_t value;
  int entier;

  Expression* expression;
  RealExpression* realExp;
  BooleanExpression* boolean;
  Vector3Expression* vector3;
  StringExpression* string;

  FunctionExpression* function;
  FunctionExpressionMeshReferences::ReferencesSet* reference;
  FieldExpression* field;

  MeshExpression* mesh;
  MeshExpressionSurface* surfacemesh;
  MeshExpressionStructured* structured3dmesh;
  MeshExpressionPeriodic::MappedReferencesList* mapreference;
  SceneExpression* scene;
  DomainExpression* domain;

  FunctionExpressionVariable* unknown;
  UnknownListExpression* unknownList;

  TestFunctionVariable* testFunction;
  TestFunctionExpressionList* testFunctionList;

  BoundaryConditionListExpression* boundaryConditionList;
  BoundaryConditionExpression* boundaryCondition;
  BoundaryExpression* boundary;

  PDEOperatorExpression* pdeOperatorExpression;
  PDEScalarOperatorExpressionOrderZero* zerothOrderOp;
  PDEVectorialOperatorExpressionOrderOne* vectorialFirstOrderOp;
  PDEScalarOperatorExpressionOrderOne* firstOrderOp;
  PDEVectorialOperatorExpressionOrderTwo* matricialSecondOrderOp;

  PDEOperatorSumExpression* pdeOperatorSumExpression;

  PDEEquationExpression* pdeEquation;

  PDEProblemExpression* pdeProblem;
  PDESystemExpression* pdeSystem;
  SolverOptionsExpression* solverOptions;

  OptionExpression* option;

  OstreamExpression * ostreamExpression;
  OstreamExpressionList * ostreamExpressionList;

  SolverOptionsExpression* solver;
  SubOptionExpression* subOption;
  SubOptionListExpression* subOptionList;

  Variable* variable;
  RealVariable* realVariable;
  Vector3Variable* vector3Variable;
  FunctionVariable* functionVariable;

  FunctionVariable* unknownVariable;

  StringVariable* stringVariable;

  MeshVariable* meshVariable;
  SceneVariable* sceneVariable;
  DomainVariable* domainVariable;
  InsideExpression* insideExpression;
  InsideListExpression* insideListExpression;

  Instruction* instruction;

  IntegratedExpression* integratedFunction;
  IntegratedOperatorExpression* integratedOperatorExp;

  LinearExpression* linearexp;

  MultiLinearExpression* multilinearexp;
  MultiLinearExpressionSum* multilinearexpsum;
  MultiLinearFormExpression* multilinearform;
  MultiLinearFormSumExpression* multilinearformsum;

  VariationalDirichletListExpression* variationalDirichletList;
  BoundaryConditionExpressionDirichlet* boundaryConditionDirichlet;

  VariationalFormulaExpression* variationalFormula;
  VariationalProblemExpression* variationalProblem;

  ProblemExpression* problem;

  char* str;
  const std::string* aString;

  FileDescriptor::FileType fileType;
  FileDescriptor::FormatType formatType;
}

%token <value> NUM "real_t value" /* real_t precision number */

/* Variables */
%token <realVariable> VARREALID "real_t variable"
%token <vector3Variable> VARVECTID "vector variable"
%token <functionVariable> VARFNCTID VARFEMFNCTID "function variable"
%token <meshVariable> MESHID

%token <sceneVariable> VARSCENEID "scene variable"
%token <domainVariable> VARDOMAINID "domain variable"
%token <unknownVariable> VARUNKNOWNID "unkown variable"
%token <testFunction> VARTESTFUNCTIONID "test function variable"

%type <integratedFunction> operatorFunctionExp
%type <integratedFunction> operatorOrFunctionExp

%type <integratedOperatorExp> elementarylinearexp

%type <linearexp> linearexp
%type <multilinearexp> multilinearexp
%type <multilinearexpsum> multilinearsumexp
%type <multilinearform> multiLinearFormExp
%type <multilinearformsum> multiLinearSumExp
%type <variationalFormula> variationalFormula

%type <expression> exp "expression"
%type <realExp> realexp  "double"
%type <vector3> vector3exp "vector"
%type <boolean> boolexp "bool"
%type <mesh> meshexp "mesh"
%type <surfacemesh> surfacemeshexp "surface mesh"
%type <structured3dmesh> structured3dmeshexp "structured 3d mesh"
%type <scene> sceneexp "scene"
%type <domain> domainexp "domain"
%type <insideExpression> inoutexp 
%type <insideListExpression> domaindefexp
%type <unknown> unknownexp "unknown"
%type <unknownList> unknownlistexp declareSolve "unknown list"

%type <testFunction> testfunctionexp "test function"
%type <testFunctionList> testFunctionsExp testfunctionlistexp "test function list"

%type <string> stringexp "string"
%type <string> strexp

%type <function> functionexp boolfunctionexp "function"
%type <reference> refexp "reference expression"
%type <mapreference> refmapexp "reference map expression"

%type <function> fdexp "function or real"

%type <field> fieldexp "field"

%type <boundary> singleBoundaryexp boundaryexp "boundary"

%type <pdeEquation> pdeexp "PDE"

%type <pdeOperatorExpression> pdeop "PDE operator"
%type <zerothOrderOp> zerothOrderOp "order zero PDE operator"
%type <firstOrderOp> firstOrderOp firstOrderOpSimple "order one PDE operator"
%type <vectorialFirstOrderOp> vectorialFirstOrderOp "order one vectorial PDE operator"
%type <matricialSecondOrderOp> matricialSecondOrderOp "order two PDE operator"

%type <pdeOperatorSumExpression> vectpdeop "vector PDE operator"

%type <function> rightHandSide "right hand side"

%type <problem> problemexp "problem"

%type <pdeSystem> pdesystemexp "pde system"
%type <variationalProblem> variationalexp "variational problem"

%type <variationalDirichletList> variationalDirichletList "Dirichlet boundary conditions"
%type <boundaryConditionDirichlet> variationalDirichlet "Dirichlet boundary condition"

%type <pdeProblem> pdeproblemexp "pde problem"

%type <instruction> instlist inst simpleinst otherinst declaration affectation
%type <instruction> block statement selectionStatement

%type <entier> face "face number"

%type <boundaryConditionList> boundaryConditionList "boundary condition list"
%type <boundaryCondition> boundaryCondition "boundary condition"

%type <ostreamExpression> ostreamexp "ostream"
%type <expression> ofluxexp
%type <ostreamExpressionList> ofluxexplist

%token TRANSFORM SIMPLIFY EXTRACT PERIODIC

%token XX "x"
%token YY "y"
%token ZZ "z"

%token VECTOR
%token FUNCTION "function"
%token DOUBLE "real"
%token MESH "mesh"
%token DOMAIN_ "domain" /* Cannot use DOMAIN (used by math.h) */

%token<str> STRING

/* Declaration keywords */
%token FEMFUNCTION STRUCTMESH SURFMESH  SCENE

/* This is use to specify global use
   (e.g.: if one has 2 scenes and wants one to be used has default).
 */
%token USING "using"
%token FINEMESH "finemesh"
%token COARSEMESH "coarsemesh"
// statements
%token IF ELSE DO WHILE FOR EXIT

%token LT "<"
%token GT ">"
%token LEQ "<="
%token GEQ ">="
%token EQ "=="
%token NE "!="
%token OR "or"
%token XOR "xor"
%token AND "and"
%token TRUE "true"
%token FALSE "false"
%token NOT "not"
%token LRAFTER "<<"
%token RRAFTER ">>"

%token DIV "div"
%token GRAD "grad"
%token DX DXOP "dx" /* SHOULD REMOVE ??OP VERSION WHICH IS AN ATROCITY */
%token DY DYOP "dy"
%token DZ DZOP "dz"

%token SOLVE "solve"

%token XMIN "xmin"
%token XMAX "xmax"
%token YMIN "ymin"
%token YMAX "ymax"
%token ZMIN "zmin"
%token ZMAX "zmax"

%token DNU CONVECT

%token INCR DECR

%token ONE "one"
%token REFERENCE "reference"
%token VERTICES "vertices"
%token ELEMENTS "elements"

%token <fileType> FILETYPE "file type"
%token <formatType> FILEFORMAT "file format"

%token SAVE "save"
%token PLOT "plot"
%token EXEC "exec"
%token READ "read"
%token TETRAHEDRIZE "tetrahedrize"

%token <str> OPTION
%type  <subOption> suboption
%type  <subOptionList> suboptionList

%type  <solverOptions> optionlist solveopts
%type  <option> option

%token INT INT1D INT2D INT3D MAXIMUM
%token EPSILON MAXITER
%token KEEPMATRIX
%token METHOD PENALTY PENALTYM FATBOUNDARY
%token KRYLOV CG BICG BICGSTAB ILUFACT MULTIGRID PSCR NONE
%token NU1 NU2 MU1 MU2 OMEGA
%token PRECOND DIAG ICHOL
%token PDEQ /* PDE is the name of the class */
%token TESTS
%token IN ON BY INSIDE OUTSIDE POV
%token COUT CERR
%token ABS SIN COS TAN ASIN ACOS ATAN SQRT EXP LOG

/* Variable, Function and User Function */
%token <aString> NAME

%right '='
%left '-' '+' OR XOR
%left '*' '/' AND LEQ GEQ EQ GT LT
%left NEG NOT       /* Negation--unary minus          */
%right '^' '(' ')'  /* Exponentiation and parenthesis */

%expect 8

%%

input:
  instlist
{
  iSet.push_back($1);
}
;

instlist:
/* empty */
{
  $$ = new InstructionList();
}
| instlist inst
{
  static_cast<InstructionList*>($1)->add($2);
  $$ = $1;
}
;

inst:
  simpleinst ';'
{
  $$ = $1;
}
| block
{
  $$ = $1;
}
| statement
{
  $$ = $1;
}
| otherinst
{
  $$ = $1;
}
;

simpleinst:

{
  $$ = new InstructionNone();
}
| declaration
{
  $$ = $1;
}
| affectation
{
  $$ = $1;
}
;

block:
  '{' instlist '}'
{
  $$ = new InstructionList();
  static_cast<InstructionList*>($$)->add($2);
}
;

statement:
  selectionStatement
{
  $$ = $1;
}
| DO inst WHILE '(' boolexp ')'
{
  $$ = new InstructionDoWhileStatement($2,$5);
}
| WHILE '(' boolexp ')' inst
{
  $$ = new InstructionWhileStatement($3,$5);
}
| FOR '(' simpleinst ';' boolexp ';' simpleinst ')' inst
{
  $$ = new InstructionForStatement($3,$5,$7,$9);
}
;

selectionStatement:
  IF '(' boolexp ')' inst
{
  ReferenceCounting<Instruction> I
    = new InstructionNone();
  $$ = new InstructionIfStatement($3,$5,I);
}
| IF '(' boolexp ')' inst ELSE inst
{
  $$ = new InstructionIfStatement($3,$5,$7);
}
;

declaration:
  DOUBLE NAME
{
  //! Default value is set to 0.
  ReferenceCounting<RealExpression> zero
    = new RealExpressionValue(0);
  $$ = new InstructionDeclaration<RealExpression, RealVariable>(*$2, zero);
}
| DOUBLE NAME '=' realexp
{
  $$ = new InstructionDeclaration<RealExpression, RealVariable>(*$2,$4);
}
| VECTOR NAME
{
  ReferenceCounting<RealExpression> zero
    = new RealExpressionValue(0);
  ReferenceCounting<Vector3Expression> zeroV
    = new Vector3ExpressionValue(zero,zero,zero);
  $$ = new InstructionDeclaration<Vector3Expression, Vector3Variable>(*$2,zeroV);
}
| VECTOR NAME '=' vector3exp
{
  $$ = new InstructionDeclaration<Vector3Expression, Vector3Variable>(*$2,$4);
}
| FUNCTION NAME
{
  ReferenceCounting<RealExpression> __zero
    = new RealExpressionValue(0.);

  ReferenceCounting<FunctionExpression> zero
    = new FunctionExpressionConstant(__zero);

  $$ = new InstructionDeclaration<FunctionExpression, FunctionVariable>(*$2,zero);
}
| FEMFUNCTION NAME '(' meshexp ')'
{
  ReferenceCounting<RealExpression> __zero
    = new RealExpressionValue(0.);

  ReferenceCounting<FunctionExpression> zero
    = new FunctionExpressionConstant(__zero);

  ReferenceCounting<FunctionExpression> f
    = new FunctionExpressionFEM($4, zero);

  $$ = new InstructionDeclaration<FunctionExpression, FunctionVariable>(*$2,f);
}
| FUNCTION NAME '=' fdexp
{
  $$ = new InstructionDeclaration<FunctionExpression, FunctionVariable>(*$2,$4);
}
| FEMFUNCTION NAME '(' meshexp ')' '=' fdexp
{
  ReferenceCounting<FunctionExpression> f
    = new FunctionExpressionFEM($4, $7);
  $$ = new InstructionDeclaration<FunctionExpression, FunctionVariable>(*$2,f);
}
| MESH NAME
{
  ReferenceCounting<MeshExpression> M = new MeshExpressionUndefined();
  $$ = new InstructionDeclaration<MeshExpression, MeshVariable>(*$2,M);
}
| MESH NAME '=' meshexp
{
  $$ = new InstructionDeclaration<MeshExpression, MeshVariable>(*$2,$4);
}
| SCENE NAME
{
  ReferenceCounting<SceneExpression> M = new SceneExpressionUndefined();
  $$ = new InstructionDeclaration<SceneExpression, SceneVariable>(*$2,M);
}
| SCENE NAME '=' sceneexp
{
  $$ = new InstructionDeclaration<SceneExpression, SceneVariable>(*$2,$4);
}
| DOMAIN_ NAME
{
  ReferenceCounting<DomainExpression> D
    = new DomainExpressionUndefined();
  $$ = new InstructionDeclaration<DomainExpression,
    DomainVariable>(*$2,D);
}
| DOMAIN_ NAME '=' domainexp
{
  $$ = new InstructionDeclaration<DomainExpression,
    DomainVariable>(*$2,$4);
}
;

affectation:
  VARREALID '=' realexp
{
  $$ = new InstructionAffectation<RealExpression, RealVariable>($1,$3);
}
| VARREALID INCR
{
  $$ = new InstructionRealExpressionIncrement($1);
}
| INCR VARREALID
{
  $$ = new InstructionRealExpressionIncrement($2);
}
| VARREALID DECR
{
  $$ = new InstructionRealExpressionDecrement($1);
}
| DECR VARREALID
{
  $$ = new InstructionRealExpressionDecrement($2);
}
| VARVECTID '=' vector3exp
{
  $$ = new InstructionAffectation<Vector3Expression, Vector3Variable>($1,$3);
}
| VARFNCTID '=' fdexp
{
  $$ = new InstructionAffectation<FunctionExpression, FunctionVariable>($1,$3);
}
| VARFEMFNCTID '=' fdexp
{
  $$ = new InstructionAffectation<FunctionExpressionFEM, FunctionVariable>($1,$3);
}
| MESHID '=' meshexp
{
  $$ = new InstructionAffectation<MeshExpression, MeshVariable>($1,$3);
}
| VARDOMAINID '=' domainexp
{
  $$ = new InstructionAffectation<DomainExpression, DomainVariable>($1,$3);
}
| VARSCENEID '=' sceneexp
{
  $$ = new InstructionAffectation<SceneExpression, SceneVariable>($1,$3);
}
;

declareSolve:
  SOLVE '(' unknownlistexp ')'
{
  UnknownListExpressionSet& U = *dynamic_cast<UnknownListExpressionSet*>($3);
  U.unknownsDeclare();
  $$ = $3;
}
;

otherinst:
  ostreamexp ofluxexplist ';'
{
  $$ = new InstructionOutput($1,$2);
}
| EXEC '(' stringexp ')' ';'
{
  $$ = new InstructionExec($3);
}
| PLOT '(' meshexp ')' ';'
{
  $$ = new InstructionPlot($3);
}
| PLOT '(' functionexp ',' meshexp ')' ';'
{
  $$ = new InstructionPlot($5, $3);
}
| SAVE '(' FILEFORMAT ',' stringexp ',' meshexp ')' ';'
{
  FileDescriptor* f = new FileDescriptor($3, FileDescriptor::Unix);
  $$ = new InstructionSaveMesh(f, $5, $7);
}
| SAVE '(' FILEFORMAT ',' stringexp ',' meshexp ',' FILETYPE ')' ';'
{
  FileDescriptor* f = new FileDescriptor($3, $9);
  $$ = new InstructionSaveMesh(f, $5, $7);
}
| SAVE '(' FILEFORMAT ',' stringexp ',' fdexp ',' meshexp ')' ';'
{
  FileDescriptor* f = new FileDescriptor($3, FileDescriptor::Unix);
  $$ = new InstructionSaveFunction(f, $5, $7, $9);
}
| SAVE '(' FILEFORMAT ',' stringexp ',' fdexp ',' meshexp ',' FILETYPE ')' ';'
{
  FileDescriptor* f = new FileDescriptor($3, $11);
  $$ = new InstructionSaveFunction(f, $5, $7, $9);
}
| SAVE '(' FILEFORMAT ',' stringexp ',' fieldexp ',' meshexp ')' ';'
{
  FileDescriptor* f = new FileDescriptor($3, FileDescriptor::Unix);
  $$ = new InstructionSaveField(f, $5, $7, $9);
}
| SAVE '(' FILEFORMAT ',' stringexp ',' fieldexp ',' meshexp ',' FILETYPE ')' ';'
{
  FileDescriptor* f = new FileDescriptor($3, $11);
  $$ = new InstructionSaveField(f, $5, $7, $9);
}
| declareSolve IN domainexp BY meshexp solveopts '{' problemexp '}'
{
  $$ = new InstructionList();

  ReferenceCounting<SolverExpressionSolveFDM> solver
    = new SolverExpressionSolveFDM($1, $3, $5, $6, $8);

  $$ = new InstructionEvaluation<SolverExpressionSolveFDM>(solver);

  UnknownListExpressionSet& U = *dynamic_cast<UnknownListExpressionSet*>($1);
  U.unknownsUndeclare();
}
| declareSolve IN meshexp solveopts '{' problemexp '}'
{
  $$ = new InstructionList();

  ReferenceCounting<SolverExpressionSolveFEM> solver
    = new SolverExpressionSolveFEM($1, $3, $4, $6);

  $$ = new InstructionEvaluation<SolverExpressionSolveFEM>(solver);

  UnknownListExpressionSet& U = *dynamic_cast<UnknownListExpressionSet*>($1);
  U.unknownsUndeclare();
}
| USING sceneexp ';'
{
  $$ = new InstructionUsingScene($2);
}
| COARSEMESH ';'
{
  $$ = new InstructionCoarseMesh(true);
}
| FINEMESH ';'
{
  $$ = new InstructionCoarseMesh(false);
}
| EXIT ';'
{
  $$ = new InstructionExit();
}
| EXIT '(' realexp ')' ';'
{
  $$ = new InstructionExit($3);  
}
;

fieldexp:
  '[' fdexp ',' fdexp ',' fdexp ']'
{
  $$ = new FieldExpression($2,$4,$6);
}
;

ostreamexp:
  COUT
{
  $$ = new OstreamExpression();
}
;

ofluxexplist:
{
  $$ = new OstreamExpressionList();
}
| ofluxexplist LRAFTER ofluxexp
{
  (*$$) << $3;
}
;

ofluxexp:
  stringexp
{
  $$ = $1;
}
| realexp
{
  $$ = $1;
}
;

meshexp:
  structured3dmeshexp
{
  $$ = $1;
}
| surfacemeshexp
{
  $$ = $1;
}
| MESHID
{
  $$ = new MeshExpressionVariable($1);
}
| READ '(' FILEFORMAT ',' stringexp ')'
{
  FileDescriptor* f = new FileDescriptor($3, FileDescriptor::Unix);
  $$ = new MeshExpressionRead(f, $5);
}
| TETRAHEDRIZE '(' meshexp ')'
{
  $$ = new MeshExpressionTetrahedrize($3);
}
| TETRAHEDRIZE '(' domainexp ',' meshexp ')'
{
  $$ = new MeshExpressionTetrahedrizeDomain($5, $3);
}
| EXTRACT '(' meshexp ',' realexp ')'
{
  $$ = new MeshExpressionExtract($3, $5);
}
| TRANSFORM '(' meshexp ',' fieldexp ')'
{
  $$ = new MeshExpressionTransform($3, $5);
}
| SIMPLIFY '(' meshexp ')'
{
  $$ = new MeshExpressionSimplify($3);
}
| PERIODIC '(' meshexp ',' refmapexp ')'
{
  $$ = new MeshExpressionPeriodic($3,$5);
}
;

refmapexp:
  realexp ':' realexp
{
  $$ = new MeshExpressionPeriodic::MappedReferencesList();
  (*$$).push_back(std::make_pair($1,$3));
}
| refmapexp ',' realexp ':' realexp
{
  $$ = $1;
  (*$$).push_back(std::make_pair($3,$5));
}
;


structured3dmeshexp:
  STRUCTMESH '(' vector3exp ',' vector3exp ',' vector3exp ')'
{
  $$ = new MeshExpressionStructured($3,$5,$7);
}
;

surfacemeshexp:
  SURFMESH '(' domainexp ',' meshexp ')'
{
  $$ = new MeshExpressionSurface($3,$5);
}
;

sceneexp:
  POV '(' stringexp ')'
{
  $$ = new SceneExpressionPOVRay($3);
}
| VARSCENEID
{
  $$ = new SceneExpressionVariable($1);
}
;

domainexp:
  DOMAIN_ '(' sceneexp ',' domaindefexp ')'
{
  $$ = new DomainExpressionSet($3, $5);
}
| VARDOMAINID
{
  $$ = new DomainExpressionVariable($1);
}
;

domaindefexp:
  inoutexp
{
  $$ = new InsideListExpressionLeaf($1);
}
| '(' domaindefexp ')'
{
  $$ = $2;
}
| NOT domaindefexp %prec NEG
{
  $$ = new InsideListExpressionNot($2);
}
| domaindefexp AND domaindefexp
{
  $$ = new InsideListExpressionAnd($1,$3);
}
| domaindefexp OR domaindefexp
{
  $$ = new InsideListExpressionOr($1,$3);
}
;

inoutexp:
  INSIDE '(' vector3exp')'
{
  $$ = new InsideExpression(true,$3);
}
| OUTSIDE '(' vector3exp')'
{
  $$ = new InsideExpression(false,$3);
}
;

solveopts:
{
  $$ = new SolverOptionsExpression();
}
| optionlist
{
  $$ = $1;
}
;

optionlist:
  option
{
  $$ = new SolverOptionsExpression();
  (*$$).add($1);
}
| optionlist ',' option
{
  $$ = $1;
  (*$$).add($3);
}
;

option:
  OPTION '(' suboptionList  ')'
{
  ReferenceCounting<StringExpression> S
    = new StringExpressionValue($1);
  $$ = new OptionExpression(S,$3);
  delete [] $1;
}
;

suboptionList:
  suboption
{
  $$ = new SubOptionListExpression;
  (*$$).add($1);
}
| suboption ',' suboptionList
{
  $$ = $3;
  (*$$).add($1);
}
;

suboption:
  OPTION '=' OPTION
{
  ReferenceCounting<StringExpression> S1
    = new StringExpressionValue($1);
  ReferenceCounting<StringExpression> S2
    = new StringExpressionValue($3);

  $$ = new SubOptionExpressionString(S1,S2);

  delete [] $1;
  delete [] $3;
}
| OPTION '=' realexp
{
  ReferenceCounting<StringExpression> S
    = new StringExpressionValue($1);
  $$ = new SubOptionExpressionReal(S,$3);

  delete [] $1;
}
;

problemexp:
  pdesystemexp
{
  $$ = $1;
}
| variationalexp
{
  $$ = $1;
}
;

variationalexp:
  testFunctionsExp
{
  (*fflexer).switchToContext(FFLexer::variationalContext);
  (*$1).subscribe();
}
  variationalFormula ';' variationalDirichletList
{
  $$ = new VariationalProblemExpression($3,$5,$1);

  (*$1).unSubscribe();
  (*fflexer).switchToContext(FFLexer::defaultContext);
}
;

testFunctionsExp:
  TESTS '(' testfunctionlistexp ')'
{
  $$ = $3;
}
;

testfunctionlistexp:
  testfunctionexp
{
  $$ = new TestFunctionExpressionList();
  (*$$).add($1);
}
| testfunctionlistexp ',' testfunctionexp
{
  $$ = $1;
  (*$$).add($3);
}
;

testfunctionexp:
  NAME
{
  std::string name(*$1);
  $$ = new TestFunctionVariable(name);
}
;

variationalFormula:
  multiLinearSumExp '=' NUM
{
  if ($3 != 0) {
    yyerror("Should be equal to 0");
  }
  $$ = new VariationalFormulaExpression($1);
}
| multiLinearSumExp '=' multiLinearSumExp
{
  (*$1) -= (*$3);
  $$ = new VariationalFormulaExpression($1);
}
;

multiLinearSumExp:
  multiLinearFormExp
{
  $$ = new MultiLinearFormSumExpression();
  (*$$).plus($1);
}
| multiLinearSumExp '+' multiLinearFormExp
{
  $$ = $1;
  (*$$).plus($3);
}
| multiLinearSumExp '-' multiLinearFormExp
{
  $$ = $1;
  (*$$).minus($3);
}
| '-' multiLinearFormExp %prec NEG
{
  $$ = new MultiLinearFormSumExpression();
  (*$$).minus($2);
}
;

multiLinearFormExp:
  INT '(' boundaryexp ')'  '(' multilinearsumexp ')'
{
  $$ = new MultiLinearFormExpression($6, MultiLinearFormExpression::twoD, $3);
}
| INT '(' multilinearsumexp ')'
{
  $$ = new MultiLinearFormExpression($3, MultiLinearFormExpression::threeD);
}
;

multilinearsumexp:
  multilinearexp
{
  $$ = new MultiLinearExpressionSum();
  (*$$).plus($1);
}
| multilinearsumexp '+'  multilinearexp
{
  $$ = $1;
  (*$$).plus($3);

}
| multilinearsumexp '-'  multilinearexp
{
  (*$$).minus($3);
}
| '-'  multilinearexp %prec NEG
{
  $$ = new MultiLinearExpressionSum();
  (*$$).minus($2);
}
;

multilinearexp:
  linearexp
{
  $$ = new MultiLinearExpression($1);
}
| multilinearexp '*' linearexp
{
  $$ = $1;
  (*$$).times($3);
}
| multilinearexp '*' functionexp
{
  $$ = $1;
  (*$$).times($3);
}
| multilinearexp '*' realexp
{
  $$ = $1;
  (*$$).times($3);
}
| '(' multilinearexp ')'
{
  $$ = $2;
}
;


linearexp:
  elementarylinearexp
{
  $$ = new LinearExpressionElementary($1);
}
| functionexp '*' elementarylinearexp
{
  $$ = new LinearExpressionElementaryTimesFunction($3,$1);
}
| realexp '*' elementarylinearexp
{
  $$ = new LinearExpressionElementaryTimesReal($3,$1);
}
;

elementarylinearexp:
  operatorFunctionExp
{
  $$ = new IntegratedOperatorExpressionOrderZero($1);
}
| GRAD '(' operatorOrFunctionExp ')'
{
  $$ = new IntegratedOperatorExpressionGrad($3);
}
| DXOP '(' operatorOrFunctionExp ')'
{
  $$ = new IntegratedOperatorExpressionDx($3);
}
| DYOP '(' operatorOrFunctionExp ')'
{
  $$ = new IntegratedOperatorExpressionDy($3);
}
| DZOP '(' operatorOrFunctionExp ')'
{
  $$ = new IntegratedOperatorExpressionDz($3);
}
;

operatorOrFunctionExp:
  operatorFunctionExp
{
  $$ = $1;
}
| functionexp
{
  $$ = new IntegratedExpressionFunctionExpression($1);
}
;

operatorFunctionExp:
  VARUNKNOWNID
{  
  $$ = new IntegratedExpressionUnknown($1);
}
| VARTESTFUNCTIONID
{
  $$ = new IntegratedExpressionTest($1);
}
;

// functionInOperatorExp:
//   GRAD '(' functionexp ')'
// {
// }
// | DXOP '(' functionexp ')'
// {
// }
// | DYOP '(' functionexp ')'
// {
// }
// | DZOP '(' functionexp ')' 
// {
// }
// ;


variationalDirichletList:
{
  $$ = new VariationalDirichletListExpression;
}

| variationalDirichletList variationalDirichlet ';'
{
  $$ = $1;
  (*$$).add($2);
}
;

variationalDirichlet:
  VARUNKNOWNID '=' fdexp ON boundaryexp
{
  $$ = new BoundaryConditionExpressionDirichlet($1,$3,$5);
}
;

pdesystemexp:
  pdeproblemexp
{
  $$ = new PDESystemExpression();
  (*$$).add($1);
}
| pdesystemexp pdeproblemexp
{
  $$ = $1;
  (*$$).add($2);
}
;

pdeproblemexp:
  PDEQ '(' VARUNKNOWNID ')' pdeexp ';' boundaryConditionList
{
  $$ = new PDEProblemExpressionDescription($3, $5, $7);
}
;

pdeexp:
  vectpdeop '='rightHandSide
{
  $$ = new PDEEquationExpression($1,$3);
}
;

rightHandSide:
  fdexp
{
  $$ = $1;
}
;

boundaryConditionList:
  /* No Boundary Conditions */
{
  $$ = new BoundaryConditionListExpressionSet();
}
| boundaryCondition ';' boundaryConditionList
{
  static_cast<BoundaryConditionListExpressionSet*>($3)->add($1);
  $$ = $3;
}
;

vectpdeop:
  pdeop
{
  $$ = new PDEOperatorSumExpression();
  (*$$).add($1);
}
| '(' vectpdeop ')'
{
  $$ = $2;
}
|  '-' vectpdeop %prec NEG
{
  (*$2).unaryMinus();
  $$ = $2;
}
|  vectpdeop '+' vectpdeop
{
  $$ = $3;
  (*$$).add($1);
}
| vectpdeop '-' vectpdeop
{
  $$ = $1;
  (*$$).minus($3);
}
;

pdeop:
  DIV '(' GRAD '(' VARUNKNOWNID ')' ')'
{
  RealExpression* One = new RealExpressionValue(1);
  FunctionExpression* OneFunction = new FunctionExpressionConstant(One);
  $$ = new PDEScalarOperatorExpressionDivMuGrad($5,OneFunction);
}
| DIV '(' realexp '*' GRAD '(' VARUNKNOWNID ')' ')'
{
  ReferenceCounting<FunctionExpression> F
    = new FunctionExpressionConstant($3);
  $$ = new PDEScalarOperatorExpressionDivMuGrad($7,F);
}
| DIV '(' functionexp '*' GRAD '(' VARUNKNOWNID ')' ')'
{
  $$ = new PDEScalarOperatorExpressionDivMuGrad($7,$3);
}
| matricialSecondOrderOp
{
  $$ = $1;
}
| firstOrderOp
{
  $$ = $1;
}
| zerothOrderOp
{
  $$ = $1;
}
;

zerothOrderOp:
  VARUNKNOWNID
{
  RealExpression* One = new RealExpressionValue(1);
  FunctionExpression* OneFunction = new FunctionExpressionConstant(One);
  $$ = new PDEScalarOperatorExpressionOrderZero($1,OneFunction);
}
| functionexp '*' VARUNKNOWNID
{
  $$ = new PDEScalarOperatorExpressionOrderZero($3,$1);
}
| realexp '*' VARUNKNOWNID
{
  ReferenceCounting<FunctionExpression> F
    = new FunctionExpressionConstant($1);
  $$ = new PDEScalarOperatorExpressionOrderZero($3,F);
}
;

firstOrderOp:
  firstOrderOpSimple
{
  $$ = $1;
}
| functionexp '*' firstOrderOp
{
  $$ = $3;
  (*$3) *= $1;
}
| realexp '*' firstOrderOp
{
  
  ReferenceCounting<FunctionExpression> F = new FunctionExpressionConstant($1);
  $$ = $3;
  (*$3) *= F;
}
;


firstOrderOpSimple:
  DX '(' VARUNKNOWNID ')'
{
  RealExpression* One = new RealExpressionValue(1);
  FunctionExpression* OneFunction = new FunctionExpressionConstant(One);
  $$ = new PDEScalarOperatorExpressionOrderOne($3,
					       OneFunction,
					       0,
					       0);
}
| DY '(' VARUNKNOWNID ')'
{
  RealExpression* One = new RealExpressionValue(1);
  FunctionExpression* OneFunction = new FunctionExpressionConstant(One);

  $$ = new PDEScalarOperatorExpressionOrderOne($3,
					       0,
					       OneFunction,
					       0);
}
| DZ '(' VARUNKNOWNID ')'
{
  RealExpression* One = new RealExpressionValue(1);
  FunctionExpression* OneFunction = new FunctionExpressionConstant(One);

  $$ = new PDEScalarOperatorExpressionOrderOne($3,
					       0,
					       0,
					       OneFunction);
}
;

matricialSecondOrderOp:
  DX '(' vectorialFirstOrderOp ')'
{
  $$ = new PDEVectorialOperatorExpressionOrderTwo();
  (*$$).add($3,0);
}
| DY '(' vectorialFirstOrderOp ')'
{
  $$ = new PDEVectorialOperatorExpressionOrderTwo();
  (*$$).add($3,1);
}
| DZ '(' vectorialFirstOrderOp ')'
{
  $$ = new PDEVectorialOperatorExpressionOrderTwo();
  (*$$).add($3,2);
}
;

vectorialFirstOrderOp:
  firstOrderOpSimple
{
  $$ = new PDEVectorialOperatorExpressionOrderOne();
  (*$$) += $1;
}
| '-' vectorialFirstOrderOp %prec NEG
{
  $$ = new PDEVectorialOperatorExpressionOrderOne();
  (*$$) -= $2;
}
| vectorialFirstOrderOp '+' vectorialFirstOrderOp
{
  $$ = $1;
  (*$$) += $3;
}
| vectorialFirstOrderOp '-' vectorialFirstOrderOp
{
  $$ = $1;
  (*$$) -= $3;
}
| functionexp '*' vectorialFirstOrderOp
{
  $$ = $3;
  (*$3) *= $1;
}
| realexp '*' vectorialFirstOrderOp
{
  $$ = $3;
  ReferenceCounting<FunctionExpression> f
    = new FunctionExpressionConstant($1);
  (*$3) *= f;
}
| '(' vectorialFirstOrderOp ')'
{
  $$ = $2;
}
;

unknownlistexp:
  unknownexp
{
  $$ = new UnknownListExpressionSet();
  static_cast<UnknownListExpressionSet*>($$)->add($1);
}
| unknownlistexp ',' unknownexp
{
  $$ = $1;
  static_cast<UnknownListExpressionSet*>($$)->add($3);
}
;

boundaryCondition: /* Dirichlet like boundary conditions */
  VARUNKNOWNID '=' fdexp ON boundaryexp
{
  $$ = new BoundaryConditionExpressionDirichlet($1,$3,$5);
}
/* Neumann like boundary conditions. The conormal is given! */
| DNU '(' VARUNKNOWNID ')' '=' fdexp ON boundaryexp
{
  $$ = new BoundaryConditionExpressionNeumann($3,$6,$8);
}
/* Fourrier like Boundary conditions */
| functionexp '*' VARUNKNOWNID '+' DNU '(' VARUNKNOWNID ')' '=' fdexp ON boundaryexp
{
  $$ = new BoundaryConditionExpressionFourrier($3,$1,$10,$12);
}
| VARUNKNOWNID '+' DNU '(' VARUNKNOWNID ')' '=' fdexp ON boundaryexp
{
  RealExpression* One = new RealExpressionValue(1);
  FunctionExpression* OneFunction = new FunctionExpressionConstant(One);
  $$ = new BoundaryConditionExpressionFourrier($5,OneFunction,$8,$10);
}
| DNU '(' VARUNKNOWNID ')' '+' VARUNKNOWNID '=' fdexp ON boundaryexp
{
  RealExpression* One = new RealExpressionValue(1);
  FunctionExpression* OneFunction = new FunctionExpressionConstant(One);
  $$ = new BoundaryConditionExpressionFourrier($3,OneFunction,$8,$10);
}
| DNU '(' VARUNKNOWNID ')' '+' fdexp '*' VARUNKNOWNID '=' fdexp ON boundaryexp
{
  $$ = new BoundaryConditionExpressionFourrier($3,$6,$10,$12);
}
| DNU '(' VARUNKNOWNID ')' '-' VARUNKNOWNID '=' fdexp ON boundaryexp
{
  RealExpression* One = new RealExpressionValue(-1);
  FunctionExpression* OneFunction = new FunctionExpressionConstant(One);
  $$ = new BoundaryConditionExpressionFourrier($3,OneFunction,$8,$10);
}
| DNU '(' VARUNKNOWNID ')' '-' fdexp '*' VARUNKNOWNID '=' fdexp ON boundaryexp
{
  FunctionExpression* MinusFDexp
 = new FunctionExpressionUnaryOperator<ExpressionUnaryMinus<FunctionExpression> >($6);
  $$ = new BoundaryConditionExpressionFourrier($3,MinusFDexp,$10,$12);
}
;

boundaryexp:
  singleBoundaryexp
{
  $$ = $1;
}
| boundaryexp ',' singleBoundaryexp
{
  if ((*$1).boundaryType() != BoundaryExpression::list) {
    BoundaryExpressionList* bl = new BoundaryExpressionList();
    bl = new BoundaryExpressionList();
    bl->add($1); bl->add($3);
    $$ = bl;
  } else {
    dynamic_cast<BoundaryExpressionList&>(*$1).add($3);
    $$ = $1;
  }
}
;

singleBoundaryexp:
  vector3exp 
{
  $$ = new BoundaryExpressionPOVRay($1);
}
| realexp 
{
  BoundaryExpressionReferences* b = new BoundaryExpressionReferences();
  b->add($1);
  $$ = b;
}
| MESHID
{
  switch ((*$1->expression()).typeOfMesh()) {
  case MeshExpression::structured: {
    BoundaryExpressionReferences* b = new BoundaryExpressionReferences(); 
    for (size_t i=0; i<6; ++i) {
      b->add(new RealExpressionValue(i));
    }
    $$ = b;
    break;
  }
  default: {
    ReferenceCounting<MeshExpression> m
      = new MeshExpressionVariable($1);
    $$ = new BoundaryExpressionSurfaceMesh(m);
    break;
  }
  }
}
| MESHID face
{
  BoundaryExpressionReferences* b = new BoundaryExpressionReferences(); 
  b->add(new RealExpressionValue($2));

  $$ = b;
}
;

face:
  XMIN
{
  $$ = 0;
}
| XMAX
{
  $$ = 1;
}
| YMIN
{
  $$ = 2;
}
| YMAX
{
  $$ = 3;
}
| ZMIN
{
  $$ = 4;
}
| ZMAX
{
  $$ = 5;
}
;

unknownexp:
  NAME
{
  std::string name(*$1);
  ReferenceCounting<RealExpression> zero
    = new RealExpressionValue(0);
  ReferenceCounting<FunctionExpression> F
    = new FunctionExpressionConstant(zero);
  ReferenceCounting<FunctionVariable> U
    = new FunctionVariable(name, F);
  $$ = new FunctionExpressionVariable(U);
}
| VARUNKNOWNID
{
  $$ = new FunctionExpressionVariable($1);
}
| VARFNCTID
{
  $$ = new FunctionExpressionVariable($1);
}
| VARFEMFNCTID
{
  $$ = new FunctionExpressionVariable($1);
}
;

fdexp:
  realexp
{
  $$ = new FunctionExpressionConstant($1);
}
| functionexp
{
  $$ = $1;
}
;

exp:
  realexp
{
  $$ = $1;
}
| boolexp
{
  $$ = $1;
}
| vector3exp
{
  $$ = $1;
}
| functionexp
{
  $$ = $1;
}
| meshexp
{
  $$ = $1;
}
| sceneexp
{
  $$ = $1;
}
| stringexp
{
  $$ = $1;
}
;

functionexp: 
  XX
{
  $$ = new FunctionExpressionValue(FunctionExpressionValue::x);
}
| YY
{
  $$ = new FunctionExpressionValue(FunctionExpressionValue::y);
}
| ZZ
{
  $$ = new FunctionExpressionValue(FunctionExpressionValue::z);
}
| functionexp '(' functionexp ',' functionexp ',' functionexp ')'
{
  $$ = new FunctionExpressionComposed($1,$3,$5,$7);
}
| functionexp '(' functionexp ',' functionexp ',' realexp ')'
{
  ReferenceCounting<FunctionExpression> F3 = new FunctionExpressionConstant($7);
  $$ = new FunctionExpressionComposed($1,$3,$5,F3);
}
| functionexp '(' functionexp ',' realexp ',' functionexp ')'
{
  ReferenceCounting<FunctionExpression> F2 = new FunctionExpressionConstant($5);
  $$ = new FunctionExpressionComposed($1,$3,F2,$7);
}
| functionexp '(' functionexp ',' realexp ',' realexp ')'
{
  ReferenceCounting<FunctionExpression> F2 = new FunctionExpressionConstant($5);
  ReferenceCounting<FunctionExpression> F3 = new FunctionExpressionConstant($7);
  $$ = new FunctionExpressionComposed($1,$3,F2,F3);
}
| functionexp '(' realexp ',' functionexp ',' functionexp ')'
{
  ReferenceCounting<FunctionExpression> F1 = new FunctionExpressionConstant($3);
  $$ = new FunctionExpressionComposed($1,F1,$5,$7);
}
| functionexp '(' realexp ',' functionexp ',' realexp ')'
{
  ReferenceCounting<FunctionExpression> F1 = new FunctionExpressionConstant($3);
  ReferenceCounting<FunctionExpression> F3 = new FunctionExpressionConstant($7);
  $$ = new FunctionExpressionComposed($1,F1,$5,F3);
}
| functionexp '(' realexp ',' realexp ',' functionexp ')'
{
  ReferenceCounting<FunctionExpression> F1 = new FunctionExpressionConstant($3);
  ReferenceCounting<FunctionExpression> F2 = new FunctionExpressionConstant($5);
  $$ = new FunctionExpressionComposed($1,F1,F2,$7);
}
| VARFNCTID
{
  $$ = new FunctionExpressionVariable($1);
}
| VARFEMFNCTID
{
  $$ = new FunctionExpressionVariable($1);
}
| ONE '(' boolfunctionexp ')'
{
  $$ = $3;
}
| REFERENCE '(' ELEMENTS ',' meshexp ',' refexp ')'
{
  $$ = new FunctionExpressionMeshElementsReferences($5, $7);
}
| REFERENCE '(' VERTICES ','  meshexp ',' refexp ')'
{
  $$ = new FunctionExpressionMeshVerticesReferences($5, $7);
}
| '(' functionexp ')'
{
  $$ = $2;
}
| READ '(' FILEFORMAT ',' stringexp ',' meshexp ')'
{
  FileDescriptor* f = new FileDescriptor($3, FileDescriptor::Unix);
  $$ = new FunctionExpressionRead(f, $5, $7);
}
| ONE '(' vector3exp ')'
{
  $$ = new FunctionExpressionCharacteristic($3);
}
| ONE '(' domainexp ')'
{
  $$ = new FunctionExpressionDomainCharacteristic($3);
}
| ONE '(' meshexp ')'
{
  $$ = new FunctionExpressionMeshCharacteristic($3);
}
| functionexp '+' functionexp
{
  $$ = new FunctionExpressionBinaryOperator<ExpressionPlus<FunctionExpression> >($1,$3);
}
| functionexp '-' functionexp
{
  $$ = new FunctionExpressionBinaryOperator<ExpressionMinus<FunctionExpression> >($1,$3);
}
| functionexp '*' functionexp
{
  $$ = new FunctionExpressionBinaryOperator<ExpressionMultiplies<FunctionExpression> >($1,$3);
}
| functionexp '/' functionexp
{
  $$ = new FunctionExpressionBinaryOperator<ExpressionDivides<FunctionExpression> >($1,$3);
}
| functionexp '^' realexp
{
  ReferenceCounting<FunctionExpression> g
    = new FunctionExpressionConstant($3);
  $$ = new FunctionExpressionBinaryOperator<ExpressionPower<FunctionExpression> >($1,g);
}
| functionexp '^' functionexp
{
  $$ = new FunctionExpressionBinaryOperator<ExpressionPower<FunctionExpression> >($1,$3);
}
| realexp '^' functionexp
{
  ReferenceCounting<FunctionExpression> f
    = new FunctionExpressionConstant($1);
  $$ = new FunctionExpressionBinaryOperator<ExpressionPower<FunctionExpression> >(f,$3);
}
| '-' functionexp %prec NEG
{
  $$ = new FunctionExpressionUnaryOperator<ExpressionUnaryMinus<FunctionExpression> >($2);
}
| DX '(' fdexp ')'
{
  $$ = new FunctionExpressionDx($3, FunctionExpressionDx::x);
}
| DY '(' fdexp ')'
{
  $$ = new FunctionExpressionDx($3, FunctionExpressionDx::y);
}
| DZ '(' fdexp ')'
{
  $$ = new FunctionExpressionDx($3, FunctionExpressionDx::z);
}
| CONVECT '(' fdexp ',' fdexp ',' fdexp ',' realexp ',' fdexp ')'
{
  $$ = new FunctionExpressionConvection($3,$5,$7,$9,$11);
}
| ABS '(' functionexp ')'
{
  $$ = new FunctionExpressionUnaryOperator<ExpressionStdFunction<FunctionExpression,
    std::abs> >($3);
}
| EXP '(' functionexp ')'
{
  $$ = new FunctionExpressionUnaryOperator<ExpressionStdFunction<FunctionExpression,
    std::exp> >($3);
}
| LOG '(' functionexp ')'
{
  $$ = new FunctionExpressionUnaryOperator<ExpressionStdFunction<FunctionExpression,
    std::log> >($3);
}
| SIN '(' functionexp ')'
{
  $$ = new FunctionExpressionUnaryOperator<ExpressionStdFunction<FunctionExpression,
    std::sin> >($3);
}
| COS '(' functionexp ')'
{
  $$ = new FunctionExpressionUnaryOperator<ExpressionStdFunction<FunctionExpression,
    std::cos> >($3);
}
| TAN '(' functionexp ')'
{
  $$ = new FunctionExpressionUnaryOperator<ExpressionStdFunction<FunctionExpression,
    std::tan> >($3);
}
| ASIN '(' functionexp ')'
{
  $$ = new FunctionExpressionUnaryOperator<ExpressionStdFunction<FunctionExpression,
    std::asin> >($3);
}
| ACOS '(' functionexp ')'
{
  $$ = new FunctionExpressionUnaryOperator<ExpressionStdFunction<FunctionExpression,
    std::acos> >($3);
}
| ATAN '(' functionexp ')'
{
  $$ = new FunctionExpressionUnaryOperator<ExpressionStdFunction<FunctionExpression,
    std::atan> >($3);
}
| SQRT '(' functionexp ')'
{
  $$ = new FunctionExpressionUnaryOperator<ExpressionStdFunction<FunctionExpression,
    std::sqrt> >($3);
}
/*
  Those declarations are to avoid conflicts using constants
  There might be a cleaner way to do it ?
*/
| realexp  '+' functionexp
{
  ReferenceCounting<FunctionExpression> C = new FunctionExpressionConstant($1);
  $$ = new FunctionExpressionBinaryOperator<ExpressionPlus<FunctionExpression> >(C,$3);
}
| functionexp '+' realexp
{
  ReferenceCounting<FunctionExpression> C = new FunctionExpressionConstant($3);
  $$ = new FunctionExpressionBinaryOperator<ExpressionPlus<FunctionExpression> >($1,C);
}
| realexp  '*' functionexp
{
  ReferenceCounting<FunctionExpression> C = new FunctionExpressionConstant($1);
  $$ = new FunctionExpressionBinaryOperator<ExpressionMultiplies<FunctionExpression> >(C,$3);
}
| functionexp '*' realexp
{
  ReferenceCounting<FunctionExpression> C = new FunctionExpressionConstant($3);
  $$ = new FunctionExpressionBinaryOperator<ExpressionMultiplies<FunctionExpression> >($1,C);
}
| realexp  '-' functionexp
{
  ReferenceCounting<FunctionExpression> C = new FunctionExpressionConstant($1);
  $$ = new FunctionExpressionBinaryOperator<ExpressionMinus<FunctionExpression> >(C,$3);
}
| functionexp '-' realexp
{
  ReferenceCounting<FunctionExpression> C = new FunctionExpressionConstant($3);
  $$ = new FunctionExpressionBinaryOperator<ExpressionMinus<FunctionExpression> >($1,C);
}
| realexp  '/' functionexp
{
  ReferenceCounting<FunctionExpression> C = new FunctionExpressionConstant($1);
  $$ = new FunctionExpressionBinaryOperator<ExpressionDivides<FunctionExpression> >(C,$3);
}
| functionexp '/' realexp
{
  ReferenceCounting<FunctionExpression> C = new FunctionExpressionConstant($3);
  $$ = new FunctionExpressionBinaryOperator<ExpressionDivides<FunctionExpression> >($1,C);
}
;

refexp:
  realexp ':' fdexp
{
  $$ = new FunctionExpressionMeshReferences::ReferencesSet($1, $3);
}
| refexp ',' realexp ':' fdexp
{
  $$ = $1;
  $$->add($3, $5);
}
;

vector3exp:
  LT realexp ',' realexp ',' realexp GT
{
  $$ = new Vector3ExpressionValue($2, $4, $6);
}
| VARVECTID
{
  $$ = new Vector3ExpressionVariable($1);
}
| '(' realexp ',' realexp ',' realexp ')'
{
  $$ = new Vector3ExpressionValue($2, $4, $6);
}
| vector3exp '+' vector3exp
{
  $$ = new Vector3ExpressionBinaryOperator<sum>($1,$3);
}
| vector3exp '-' vector3exp
{
  $$ = new Vector3ExpressionBinaryOperator<difference>($1, $3);
}
| realexp '*' vector3exp
{
  $$ = new Vector3ExpressionTimesScalar($3, $1);
}
| vector3exp '*' realexp
{
  $$ = new Vector3ExpressionTimesScalar($1, $3);
}
;

stringexp:
  strexp
{
  $$ = $1;
}
| stringexp '.' strexp
{
  $$ = new StringExpressionConcat($1,$3);
}
| stringexp '.' realexp
{
  $$ = new StringExpressionConcat($1,
				  new StringExpressionReal($3));
}
;

strexp:
  STRING
{
  std::string s($1);
  delete [] $1;
  $$ = new StringExpressionValue(s);
}
;

realexp:
  NUM
{
  $$ = new RealExpressionValue($1);
}
| VARREALID
{
  $$ = new RealExpressionVariable($1);
}
| functionexp '(' vector3exp ')'
{
  $$ = new RealExpressionFunctionEvaluate($1, $3);
}
| functionexp '(' realexp ',' realexp ',' realexp ')'
{
  $$ = new RealExpressionFunctionEvaluate($1, $3, $5, $7);
}
| INT '(' meshexp ')' '(' fdexp ')'
{
  $$ = new RealExpressionIntegrate($6,$3);
}
| MAXIMUM '(' meshexp ')' '(' fdexp ')'
{
  $$ = new RealExpressionMax($6,$3);
}
| '(' realexp ')'
{
  $$ = $2;
}
/* Birary Operators */
| realexp '+' realexp
{ 
  $$ = new RealExpressionBinaryOperator<sum>($1,$3);
}
| realexp '-' realexp
{
  $$ = new RealExpressionBinaryOperator<difference>($1,$3);
}
| realexp '*' realexp
{
  $$ = new RealExpressionBinaryOperator<product>($1,$3);
}
| realexp '/' realexp
{
  $$ = new RealExpressionBinaryOperator<division>($1,$3);
}
| realexp '^' realexp
{
  $$ = new RealExpressionBinaryOperator<std::pow>($1,$3);
}
/* Unary Operators */
| '-' realexp  %prec NEG
{
  $$ = new RealExpressionUnaryOperator<unaryMinus>($2);
}
| INCR VARREALID
{
  $$ = new RealExpressionPreIncrement(new RealExpressionVariable($2));
}
| VARREALID INCR
{
  $$ = new RealExpressionPostIncrement(new RealExpressionVariable($1));
}
| DECR VARREALID
{
  $$ = new RealExpressionPreDecrement(new RealExpressionVariable($2));
}
| VARREALID DECR
{
  $$ = new RealExpressionPostDecrement(new RealExpressionVariable($1));
}
| ABS  '(' realexp ')'
{
  $$ = new RealExpressionUnaryOperator<std::fabs>($3);
}
| EXP  '(' realexp ')'
{
  $$ = new RealExpressionUnaryOperator<std::exp>($3);
}
| LOG  '(' realexp ')'
{
  $$ = new RealExpressionUnaryOperator<std::log>($3);
}
| SIN  '(' realexp ')'
{
  $$ = new RealExpressionUnaryOperator<std::sin>($3);
}
| COS  '(' realexp ')'
{
  $$ = new RealExpressionUnaryOperator<std::cos>($3);
}
| TAN  '(' realexp ')'
{
  $$ = new RealExpressionUnaryOperator<std::tan>($3);
}
| ASIN '(' realexp ')'
{
  $$ = new RealExpressionUnaryOperator<std::asin>($3);
}
| ACOS '(' realexp ')'
{
  $$ = new RealExpressionUnaryOperator<std::acos>($3);
}
| ATAN '(' realexp ')'
{
  $$ = new RealExpressionUnaryOperator<std::atan>($3);
}
| SQRT '(' realexp ')'
{
  $$ = new RealExpressionUnaryOperator<std::sqrt>($3);
}
/* Casts */
| '(' DOUBLE ')' boolexp
{
  $$ = new RealExpressionBoolean($4);
}
;

boolexp:
  TRUE
{
  $$ = new BooleanExpressionValue(true);
}
| FALSE
{
  $$ = new BooleanExpressionValue(false);
}
| realexp GT realexp
{
  $$ = new BooleanExpressionCompareOperator<gt>($1,$3);
}
| realexp LT realexp
{
  $$ = new BooleanExpressionCompareOperator<lt>($1,$3);
}
| realexp LEQ realexp
{
  $$ = new BooleanExpressionCompareOperator<le>($1,$3);
}
| realexp GEQ realexp
{
  $$ = new BooleanExpressionCompareOperator<ge>($1,$3);
}
| realexp EQ  realexp
{
  $$ = new BooleanExpressionCompareOperator<eq>($1,$3);
}
| realexp NE  realexp
{
  $$ = new BooleanExpressionCompareOperator<ne>($1,$3);
}
| boolexp OR   boolexp
{
  $$ = new BooleanExpressionBinaryOperator<or_>($1,$3);
}
| boolexp XOR  boolexp
{
  $$ = new BooleanExpressionBinaryOperator<xor_>($1,$3);
}
| boolexp AND  boolexp
{
  $$ = new BooleanExpressionBinaryOperator<and_>($1,$3);
}
| '(' boolexp ')'
{
  $$ = $2;
}
| NOT boolexp
{
  $$ = new BooleanExpressionUnaryOperator<not_>($2);
}
;

boolfunctionexp:
  TRUE
{
  ReferenceCounting<BooleanExpression> b
    = new BooleanExpressionValue(true);
  $$ = new FunctionExpressionBooleanValue(b);
}
| FALSE
{
  ReferenceCounting<BooleanExpression> b
    = new BooleanExpressionValue(false);
  $$ = new FunctionExpressionBooleanValue(b);
}
| fdexp GT fdexp
{
  $$ = new FunctionExpressionBooleanCompare<gt>($1,$3);
}
| fdexp LT fdexp
{
  $$ = new FunctionExpressionBooleanCompare<lt>($1,$3);
}
| fdexp GEQ fdexp
{
  $$ = new FunctionExpressionBooleanCompare<ge>($1,$3);
}
| fdexp LEQ fdexp
{
  $$ = new FunctionExpressionBooleanCompare<le>($1,$3);
}
| fdexp EQ fdexp
{
  $$ = new FunctionExpressionBooleanCompare<eq>($1,$3);
}
| fdexp NE fdexp
{
  $$ = new FunctionExpressionBooleanCompare<ne>($1,$3);
}
| boolfunctionexp OR boolfunctionexp
{
  FunctionExpressionBoolean* b1
    = dynamic_cast<FunctionExpressionBoolean*>($1);

  FunctionExpressionBoolean* b2
    = dynamic_cast<FunctionExpressionBoolean*>($3);

  $$ = new FunctionExpressionBooleanBinaryOperator<or_>(b1,b2);
}
| boolfunctionexp XOR boolfunctionexp
{
  FunctionExpressionBoolean* b1
    = dynamic_cast<FunctionExpressionBoolean*>($1);

  FunctionExpressionBoolean* b2
    = dynamic_cast<FunctionExpressionBoolean*>($3);

  $$ = new FunctionExpressionBooleanBinaryOperator<xor_>(b1,b2);
}
| boolfunctionexp AND boolfunctionexp
{
  FunctionExpressionBoolean* b1
    = dynamic_cast<FunctionExpressionBoolean*>($1);

  FunctionExpressionBoolean* b2
    = dynamic_cast<FunctionExpressionBoolean*>($3);

  $$ = new FunctionExpressionBooleanBinaryOperator<and_>(b1,b2);
}
| '(' boolfunctionexp ')'
{
  $$ = $2;
}
| NOT boolfunctionexp
{
  FunctionExpressionBoolean* b
    = dynamic_cast<FunctionExpressionBoolean*>($2);

  $$ = new FunctionExpressionBooleanUnaryOperator<not_>(b);
}
;

/* End of grammar */
%%

int yyerror (const char * s)  /* Called by yyparse on error */ {
  throw ErrorHandler("PARSED FILE",fflexer->lineno(),
		     stringify(s)+"'"+fflexer->YYText()+"' unexpected",
		     ErrorHandler::compilation);
  return 0;
}

int yyerror2 (const char * s)  /* Called by yyparse on error */ {
  throw ErrorHandler("PARSED FILE",fflexer->lineno(),
		     stringify(s),
		     ErrorHandler::compilation);
  return 0;
}

