//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: DegreeOfFreedomSet.hpp,v 1.6 2005/01/29 20:20:33 delpinux Exp $

#ifndef DEGREE_OF_FREEDOM_SET_HPP
#define DEGREE_OF_FREEDOM_SET_HPP

#include <Vector.hpp>

#include <ReferenceCounting.hpp>
#include <ErrorHandler.hpp>

#include <cassert>
#include <cstddef>
/*!
  \class DegreeOfFreedomSet

  This class defines degrees of freedom sets in the case where all the
  unknown share the same sets of vertices.

  \author Stphane Del Pino
*/

class DegreeOfFreedomSet
{
public:
  typedef Vector<int> Correspondance;

private:
  ReferenceCounting<Correspondance> __correspondance;
  const size_t __numberOfVariables;
  const size_t __numberOfUsedVertices;

public:
  const size_t& numberOfUsedVertices() const
  {
    return __numberOfUsedVertices;
  }

  const size_t& numberOfVertices() const
  {
    return (*__correspondance).size();
  }

  const size_t& numberOfVariables() const
  {
    return __numberOfVariables;
  }

  size_t size() const
  {
    return __numberOfVariables * ((*__correspondance).size());
  }

  size_t operator()(const size_t& variableNumber,
		    const size_t& vertexNumber) const
  {
    assert((*__correspondance)[vertexNumber] != -1);
    return ((*__correspondance)[vertexNumber]*__numberOfVariables+variableNumber);
  }

  bool isDOFVertex(const size_t& vertexNumber) const
  {
    return ((*__correspondance)[vertexNumber] != -1);
  }

  DegreeOfFreedomSet(const size_t& numberOfVariables,
		     const size_t& numberOfUsedVertices,
		     ReferenceCounting<Correspondance> correspondance)
    : __correspondance(correspondance),
      __numberOfVariables(numberOfVariables),
      __numberOfUsedVertices(numberOfUsedVertices)
  {
    ;
  }

  ~DegreeOfFreedomSet()
  {
    ;
  }
};

#endif // DEGREE_OF_FREEDOM_SET_HPP

