//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: DegreeOfFreedomSetBuilder.hpp,v 1.2 2005/01/18 21:10:28 delpinux Exp $

#ifndef DEGREE_OF_FREEDOM_SET_BUILDER_HPP
#define DEGREE_OF_FREEDOM_SET_BUILDER_HPP

#include <Mesh.hpp>
#include <Domain.hpp>

#include <DegreeOfFreedomSet.hpp>
#include <ReferenceCounting.hpp>

/**
 * @file   DegreeOfFreedomSetBuilder.hpp
 * @author Stephane Del Pino
 * @date   Sun Jan 16 23:39:29 2005
 * 
 * @brief Builds degrees of freefom set according to the mesh, the
 * periodicity, the number of variables and eventually the
 * computational domain
 * 
 */

class DegreeOfFreedomSetBuilder
{
private:
  /// the degree of freedom set
  ReferenceCounting<DegreeOfFreedomSet> __degreeOfFreedomSet; /**<  */

  /** 
   * Builds the degrees of freedom set for non abstract mesh classes
   * in the case of fictitious domain methods
   * 
   * @param numberOfVariables the number of variables
   * @param mesh the non-abstract mesh type
   * @param domain the computational domain
   */
  template <typename MeshType>
  void __buildFictitious(const size_t& numberOfVariables,
			 const MeshType& mesh,
			 const Domain& domain);

public:
  /** 
   * Read only access to the degree of freedom set
   * 
   * @return the degree of freedom set
   */
  const DegreeOfFreedomSet& degreeOfFreedomSet() const
  {
    return *__degreeOfFreedomSet;
  }

  /** 
   * Builds the DegreeOfFreedomSet according to the number of
   * variable, the periodicity and the mesh
   * 
   * @param numberOfVariables the number of variables
   * @param mesh the mesh
   */
  DegreeOfFreedomSetBuilder(const size_t& numberOfVariables,
			    const Mesh& mesh);

  /** 
   * Builds the DegreeOfFreedomSet according to the number of
   * variables, the periodicity, the mesh and the domain. This
   * constructor is called in the case of fictitious domain methods
   * 
   * @param numberOfVariables the number of variables
   * @param mesh the mesh
   * @param domain the domain
   */
  DegreeOfFreedomSetBuilder(const size_t& numberOfVariables,
			    const Mesh& mesh,
			    const Domain& domain);

  /** 
   * Destructor
   * 
   */
  ~DegreeOfFreedomSetBuilder()
  {
    ;
  }
};

#endif // DEGREE_OF_FREEDOM_SET_BUILDER_HPP
