//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Triangle.hpp,v 1.12 2004/08/28 08:28:03 delpinux Exp $


#ifndef  TRIANGLE_HPP
#define  TRIANGLE_HPP

#include <TinyVector.hpp>
#include <Cell.hpp>

#include <Edge.hpp>

/**
 * @file   Triangle.hpp
 * @author Stphane Del Pino
 * @date   Sat Jul 27 17:32:51 2002
 * 
 * @brief  Describes triangles in 3D
 * 
 */

class Triangle
  : public Cell
{
private:
  /**
   * The mother cell (ie: a 3d element which interacts with the triangle)
   * 
   */
  const Cell* __3Dcell;

public:
  enum {
    NumberOfVertices   = 3,	/**< number of vertices   */
    NumberOfNeighbours = 3,	/**< number of neighbours */
    NumberOfEdges      = 3	/**< number of edges      */
  };

  typedef Edge BorderType;

  /**
   * Access to the mother cell
   * 
   */
  const Cell& mother() const
  {
    assert(__3Dcell!=0);
    return *__3Dcell;
  }

  /**
   * Sets the mother cell
   * 
   */
  void setMother(const Cell* cell)
  {
    __3Dcell = cell;
  }

  /** 
   * The number of vertices of a triangle
   * 
   * @return the number of vertices
   */
  size_t numberOfVertices() const
  {
    return NumberOfVertices;
  }

  /** 
   * Access to the type of this cell
   * 
   * @return Cell::triangle3d
   */
  Cell::Type type() const
  {
    return Cell::triangle3d;
  }

  /** 
   *  Find the vertex in the triangle
   * 
   * 
   * @return bool
   */
  inline bool find(const Vertex * P) {
    for(size_t i=0 ; i<NumberOfVertices ; ++i) {
      if(__vertices[i]==P) {
	return true;
      }
    }
    return false;
  }

  
  /** 
   *  Returns the normal to the face.
   * 
   * 
   * @return normal
   */
  const inline TinyVector<3> Normal() const
  {
    const TinyVector<3>& v0 = *(__vertices[0]);
    const TinyVector<3>& v1 = *(__vertices[1]);
    const TinyVector<3>& v2 = *(__vertices[2]);

    const TinyVector<3> A(v1-v0);
    const TinyVector<3> B(v2-v0);
    TinyVector<3> N(A^B);
    const real_t n = Norm(N);
    N/=n;
    return N;
  }

  /** 
   * Operator equal
   * 
   * @param S 
   * 
   * @return *this
   */
  inline const Triangle& operator = (const Triangle& S)
  {
    Cell::operator=(S);
    __3Dcell = S.__3Dcell;

    return *this;
  }

  /** 
   * Prepares the structures for the Cells
   * 
   * 
   */
  Triangle()
    : Cell(NumberOfVertices),
      __3Dcell(0)
  {
    ;
  }

  /** 
   * Access to the ith edge of the triangle
   * 
   * @param i the number of the Edge
   * 
   * @return the ith edge of the triangle
   */
  Edge edge(size_t i) const
  {
    assert(i<NumberOfVertices);
    return Edge(*__vertices[(i+1)%3], *__vertices[(i+2)%3]);
  }

  /** 
   * Constructs a triangle using 3 vertices.
   * 
   * @param v0 first vertex
   * @param v1 second vertex
   * @param v2 third vertex
   * @param reference the reference of the triangle (default is 0)
   *
   */
  Triangle(const Vertex& v0,
	   const Vertex& v1,
	   const Vertex& v2,
	   const size_t& reference=0)
    : Cell(NumberOfVertices, reference),
      __3Dcell(0)
  {
    __vertices[0] = (Vertex*)&v0;
    __vertices[1] = (Vertex*)&v1;
    __vertices[2] = (Vertex*)&v2;

    /// Now computes the volume (area) of the triangle
    const TinyVector<3>& X0 = *(__vertices[0]);
    const TinyVector<3>& X1 = *(__vertices[1]);
    const TinyVector<3>& X2 = *(__vertices[2]);

    const TinyVector<3> A(X1-X0);
    const TinyVector<3> B(X2-X0);

    __volume = 0.5*Norm(A^B);
  }


  /** 
   * Copy constructor
   * 
   * @param st 
   * 
   */
  Triangle(const Triangle& st)
    : Cell(st),
      __3Dcell(st.__3Dcell)
  {
    ;
  }

  /** 
   * Destructor
   * 
   * 
   */
  ~Triangle()
  {
    ;
  }

};

#endif // TRIANGLE_HPP

