//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001-2005 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: FEMFunction.hpp,v 1.3 2005/10/29 14:41:20 delpinux Exp $

#ifndef FEM_FUNCTION_HPP
#define FEM_FUNCTION_HPP

#include <UserFunction.hpp>
#include <DegreeOfFreedomPositionsSet.hpp>
#include <DegreeOfFreedomSetManager.hpp>

#include <FiniteElementTraits.hpp>

class FunctionExpression;

struct FEMFunctionBase
  : public UserFunction
{
protected:
  //! degree of freedom set
  const DegreeOfFreedomPositionsSet& __dofPositionsSet;

  //! The vector of vertices values.
  Vector<real_t> __values;

  const DiscretizationType::Type __type;

public:
  /** 
   * The type of the finite elememt function
   * 
   * @return __type
   */
  const DiscretizationType::Type& type() const
  {
    return __type;
  }

  const Vector<real_t>& values() const
  {
    return __values;
  }

  //! Access to the value at the ith degree of freedom
  inline real_t& operator[](const size_t& i)
  {
    return __values[i];
  }

  //! Read only Access to the value at the ith degree of freedom
  inline const real_t& operator[](const size_t& i) const
  {
    return __values[i];
  }

  //! Affects a \a Vector to the FEMUserFunction.
  void operator=(const Vector<real_t>& u)
  {
    assert (u.size() == __values.size());
    __values = u;
  }

  //! Affects a \a UserFunction to the FEMUserFunction.
  void operator=(const UserFunction& u)
  {
    for (size_t i=0; i<__values.size(); i++) {
      const TinyVector<3>& X = __dofPositionsSet.vertex(i);
      __values[i] = u(X);
    }
  }

  virtual TinyVector<3, real_t> gradient(const TinyVector<3>& V) const = 0;

  virtual real_t dx(const TinyVector<3>& V) const = 0;
  virtual real_t dy(const TinyVector<3>& V) const = 0;
  virtual real_t dz(const TinyVector<3>& V) const = 0;

  virtual void dump(std::ostream& os,
		    const std::string& CR) const = 0;

  FEMFunctionBase(const Mesh& mesh,
		  DiscretizationType::Type discretizationType)
    : __dofPositionsSet(DegreeOfFreedomSetManager::instance().getDOFPositionsSet(mesh,discretizationType)),
      __values(__dofPositionsSet.number()),
      __type(discretizationType)
  {
    ;
  }

  virtual ~FEMFunctionBase()
  {
    DegreeOfFreedomSetManager::instance().unsubscribe(__dofPositionsSet);
  }
};

template <typename MeshType,
	  typename FiniteElementTraits>
class FEMFunction
  : public FEMFunctionBase
{
private:
  typedef typename MeshType::CellType CellType;
  typedef typename FiniteElementTraits::Transformation Transformation;
  typedef typename FiniteElementTraits::Type FiniteElementType;

  const MeshType& __mesh;

  FEMFunction(const FEMFunction<MeshType, FiniteElementTraits>& f);
public:

  void dump(std::ostream& os, const std::string& CR) const
  {
    for (size_t i=0; i<__values.size(); ++i) {
      os << __values[i] << CR;
    }
  }


  //! Evaluates the FEMFunction at point \a X.
  real_t operator()(const TinyVector<3, real_t>& X) const
  {
    typename MeshType::const_iterator icell = __mesh.find(X);
    if (icell.end()) {
      return 0;
    }

    const CellType& K = *icell;
    Transformation T(K);

    TinyVector<3, real_t> Xhat;
    T.invertT(X, Xhat);

    real_t value = 0;
    for (size_t l=0; l<FiniteElementType::numberOfDegreesOfFreedom; ++l) {
      value += __values[__dofPositionsSet(icell.number(),l)]*FiniteElementType::instance().W(l,Xhat);
    }

    return value;
  }

  TinyVector<3, real_t> gradient(const TinyVector<3>& X) const
  {
    typename MeshType::const_iterator icell = __mesh.find(X);
    if (icell.end()) {
      return 0;
    }

    const CellType& K = *icell;
    Transformation T(K);

    TinyVector<3, real_t> Xhat;
    T.invertT(X, Xhat);

    TinyVector<3, real_t> referenceGradient = 0;
    for (size_t l=0; l<FiniteElementType::numberOfDegreesOfFreedom; ++l) {
      const real_t value = __values[__dofPositionsSet(icell.number(),l)];
      referenceGradient[0] += value*FiniteElementType::instance().dxW(l,Xhat);
      referenceGradient[1] += value*FiniteElementType::instance().dyW(l,Xhat);
      referenceGradient[2] += value*FiniteElementType::instance().dzW(l,Xhat);
    }

    TinyMatrix<3,3, real_t> J;
    {
      TinyVector<3, real_t> temp;

      T.dx(X,temp);
      for(size_t i=0; i<3; ++i) {
	J(0,i) = temp[i];
      }

      T.dy(X,temp);
      for(size_t i=0; i<3; ++i) {
	J(1,i) = temp[i];
      }

      T.dz(X,temp);
      for(size_t i=0; i<3; ++i) {
	J(2,i) = temp[i];
      }
    }
    // now we use 
    TinyVector<3, real_t> result;

    gaussPivot(J, referenceGradient, result);
    return result;
  }



  real_t dx(const TinyVector<3>& X) const
  {
    return gradient(X)[0];
  }

  real_t dy(const TinyVector<3>& X) const
  {
    return gradient(X)[1];
  }

  real_t dz(const TinyVector<3>& X) const
  {
    return gradient(X)[2];
  }

  //! Constructs a FEMUserFunction using a pointer on a Mesh and a UserFunction
  FEMFunction(const MeshType& mesh)
    : FEMFunctionBase(mesh, DiscretizationType::Type(FiniteElementTraits::DiscretizationType)),
      __mesh(mesh)
  {
    ;
  }

  //! Constructs a FEMUserFunction using a pointer on a Mesh and a UserFunction
  FEMFunction(const MeshType& mesh,
	      const UserFunction& F)
    : FEMFunctionBase(mesh, DiscretizationType::Type(FiniteElementTraits::DiscretizationType)),
      __mesh(mesh)
  {
    fferr(0) << __FILE__ << ':' << __LINE__
	     << ":warning: FEMFunction copy not optimized!\n";
    for (size_t i=0; i<__values.size(); i++) {
      const TinyVector<3>& X = __dofPositionsSet.vertex(i);
      __values[i] = F(X);
    }
  }

  //! Constructs a FEMUserFunction using a pointer on a Mesh and a real_t
  FEMFunction(const MeshType& mesh,
	      const real_t& d)
    : FEMFunctionBase(mesh, DiscretizationType::Type(FiniteElementTraits::DiscretizationType)),
      __mesh(mesh)
  {
    __values = d;
  }

  /*!
    Constructs a FEMUserFunction using a pointer on a Mesh and a Vector of data.
    \warning Will have to change this for the vectorial case.
  */
  FEMFunction(const MeshType& mesh,
	      const Vector<real_t>& F)
    : FEMFunctionBase(mesh, DiscretizationType::Type(FiniteElementTraits::DiscretizationType)),
      __mesh(mesh)
  {
    assert(__values.size() == F.size());
    for (size_t i=0; i<__values.size(); i++) {
      __values[i] = F[i];
    }
  }

  //! Destructor.
  ~FEMFunction()
  {
    ;
  }
};

#endif // FEM_FUNCTION_HPP
