#include <glib.h>

#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <dirent.h>

#include <libyahoo2/yahoo2.h>
#define USE_STRUCT_CALLBACKS //required for yahoo2_callbacks.h
#include <libyahoo2/yahoo2_callbacks.h>

#include <guile/gh.h>
#include <readline/readline.h>
#include <readline/history.h>

#include "freehoo.h"
#include "extension.h"
#include "messenger.h"
#include "yahoo-wrapper.h"
#include "fh-utils.h"
#include "interpreter.h"


struct connect_callback_data {
	yahoo_connect_callback callback;
	void * callback_data;
	int id;
};

extern YList * conferences;


/*
 * Name: fh_ext_yahoo_login_response
 * 	Called when the login process is complete
 * Params:
 * 	id   - the id that identifies the server connection
 * 	succ - enum yahoo_login_status
 * 	url  - url to reactivate account if locked
 */
void 
fh_ext_yahoo_login_response (int id, int succ, char *url)
{
  switch (succ)
    {
    case YAHOO_LOGIN_OK:
  // add GNU_ROBOT as buddy by default for every user :)
      if (get_setup_mode () == 1)
	yahoo_add_buddy ((get_fh_session ())->session_id, 
			 FH_ROBOT, 
			 FH_ROBOT_GROUP);
      return;
      
    case YAHOO_LOGIN_PASSWD:
      PRINTF_MESSAGE ("\nCould not log into Yahoo!.  Check your username/password\n");
      break;
      
    case YAHOO_LOGIN_LOCK:
      PRINTF_MESSAGE ("\nCould not log into Yahoo!.  Your account has been locked\nVisit [%s] to reactivate it\n", url);
      break;
      
    case YAHOO_LOGIN_DUPL:
      PRINTF_MESSAGE ("\nYou have been logged out of the yahoo!(duplicate login!!!)\n");
    }
  
  // (get_fh_session ())->status = YAHOO_STATUS_OFFLINE;

  set_terminal_attributes ();
  exit (EXIT_FAILURE);
}


/*
 * Name: fh_ext_yahoo_got_buddies
 * 	Called when the contact list is got from the server
 * Params:
 * 	id   - the id that identifies the server connection
 * 	buds - the buddy list
 */
void 
fh_ext_yahoo_got_buddies (int id, YList * buds)
{
  (get_fh_session ())->buddy_list = buds;

  build_fh_buddy_list ();
}


/*
 * Name: fh_ext_yahoo_got_ignore
 * 	Called when the ignore list is got from the server
 * Params:
 * 	id   - the id that identifies the server connection
 * 	igns - the ignore list
 */
void 
fh_ext_yahoo_got_ignore (int id, YList * igns)
{
  (get_fh_session ())->ignore_list = igns;
}


/*
 * Name: fh_ext_yahoo_status_changed
 * 	Called when remote user's status changes.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the handle of the remote user
 * 	stat - status code (enum yahoo_status)
 * 	msg  - the message if stat == YAHOO_STATUS_CUSTOM
 * 	away - whether the contact is away or not (YAHOO_STATUS_CUSTOM)
 */
void 
fh_ext_yahoo_status_changed (int id, char *who, int stat, char *msg, int away)
{
  fh_buddy *hoob = NULL;
  hoob = (fh_buddy *) g_hash_table_lookup (get_fh_buddy_list (), who);
  if (hoob == NULL)
    {
      hoob = (fh_buddy *) malloc (sizeof (fh_buddy));
  
      hoob->group = strdup (FH_DEFAULT_GROUP);
      hoob->id = strdup (who);
      hoob->real_name = NULL;
      hoob->status_msg = NULL;
      g_hash_table_insert (get_fh_buddy_list (), hoob->id, hoob);
    }

  if (hoob->status == YAHOO_STATUS_CUSTOM && hoob->status_msg)
    {
      free (hoob->status_msg);
      hoob->status_msg = NULL;
    }

  hoob->status = stat;
  
  if (stat == YAHOO_STATUS_CUSTOM)
    {
      if (get_status_mode ())
	{
	  if (msg)
	    {
	      PRINTF_MESSAGE ("%s is now [%s]\n", who, msg);
	    }
	  else
	    {
	      PRINTF_MESSAGE ("%s is now [Custom]\n", who);
	    }
	}
      if (msg)
	hoob->status_msg = strdup (msg);
      else
	hoob->status_msg = strdup ("Custom");
    }
  else
    {
      if (get_status_mode ())
      PRINTF_MESSAGE ("%s is now [%s]\n",
		      who, yahoo_status_code (stat));
    }
}


/*
 * Name: fh_ext_yahoo_got_im
 * 	Called when remote user sends you a message.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the handle of the remote user
 * 	msg  - the message - NULL if stat == 2
 * 	tm   - timestamp of message if offline
 * 	stat - message status - 0
 * 				1
 * 				2 == error sending message
 * 				5
 * 	utf8 - whether the message is encoded as utf8 or not
 */
void 
fh_ext_yahoo_got_im (int id, char *who, char *msg, long tm, int stat, int utf8)
{
  char *umsg = msg;

  if (stat == 2)
    {
      PRINTF_MESSAGE ("Error sending message to %s", who);
      return;
    }
  
  if(!msg)
    return;

  if(utf8)
    umsg = y_utf8_to_str(msg);
  
  msg = filter_message (msg);
  
  // AUTO-INSERT-MODE
  set_current_target_buddy (who);

  // invoke fh-message-receive-hook
  set_hook_return (0);
  scm_run_hook (get_message_receive_hook (),
		gh_list (gh_str02scm (who),
			 gh_str02scm (msg),
			 SCM_UNDEFINED));
  if (get_hook_return () == 1)
    return;

  if(tm)
    {
      char timestr[255];
      
      strncpy (timestr, ctime (&tm), sizeof (timestr));
      timestr[strlen (timestr) - 1] = '\0';
      
      PRINTF_MESSAGE ("[%s]%s -> %s\r\n", timestr, who, msg);
    }
  else
    PRINTF_MESSAGE ("%s -> %s\n", who, msg);
}


/*
 * Name: fh_ext_yahoo_got_conf_invite
 * 	Called when remote user sends you a conference invitation.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the user inviting you
 * 	room - the room to join
 * 	msg  - the message
 *      members - the initial members of the conference (null terminated list)
 */
void 
fh_ext_yahoo_got_conf_invite (int id, char *who, char *room, char *msg, YList *members)
{
  conf_room *cr;
  //conference room by the name available already
  if(find_conf_room_by_name_and_id(id, room)) 
    return;

  cr = calloc (1, sizeof(conf_room));
  if (!cr)
    {
      PRINTF_MESSAGE ("Unable to allocate memory\n");
      exit (-1);
    }
  cr->room_name = strdup (room);
  cr->host = strdup (who);
  cr->members = members;
  cr->id = id;

  conferences = y_list_append (conferences, cr);
  
  PRINTF_MESSAGE ("[%s] has invited you to a conference: <%s>\n", who, room);
  PRINTF_MESSAGE ("with the message: %s\n", msg);
  for(; members; members=members->next)
    PRINTF_MESSAGE ("\t[%s]\n", (char *)members->data);
  
}


/*
 * Name: fh_ext_yahoo_conf_userdecline
 * 	Called when someone declines to join the conference.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the user who has declined
 * 	room - the room
 * 	msg  - the declining message
 */
void 
fh_ext_yahoo_conf_userdecline (int id, char *who, char *room, char *msg)
{
  YList * l;
  conf_room * cr = find_conf_room_by_name_and_id (id, room);
  
  if (cr)
    for (l = cr->members; l; l=l->next)
      {
	char * w = l->data;
	if (!strcmp (w, who)) 
	  {
	    cr->members = y_list_remove_link (cr->members, l);
	    free(l->data);
	    free(l);
	    break;
	  }
      }
  
  PRINTF_MESSAGE ("[%s] declined the invitation to <%s>\n"
	  "with the message: %s\n", who, room, msg);
}


/*
 * Name: fh_ext_yahoo_conf_userjoin
 * 	Called when someone joins the conference.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the user who has joined
 * 	room - the room joined
 */
void 
fh_ext_yahoo_conf_userjoin (int id, char *who, char *room)
{
  conf_room * cr = find_conf_room_by_name_and_id (id, room);
  if (cr)
    {
      YList * l = NULL;
	for(l = cr->members; l; l=l->next) 
	  {
		char * w = l->data;
		if(!strcmp (w, who))
		  break;
	  }
	if(!l)
	  cr->members = y_list_append (cr->members, strdup(who));
    }

  PRINTF_MESSAGE ("[%s] joined the conference <%s>\n", who, room);

}


/*
 * Name: fh_ext_yahoo_conf_userleave
 * 	Called when someone leaves the conference.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the user who has left
 * 	room - the room left
 */
void 
fh_ext_yahoo_conf_userleave (int id, char *who, char *room)
{
  YList * l;
  conf_room * cr = find_conf_room_by_name_and_id (id, room);
  
  if (cr)
    for (l = cr->members; l; l=l->next) 
      {
	char * w = l->data;
	if (!strcmp (w, who))
	  {
	    cr->members = y_list_remove_link (cr->members, l);
	    free (l->data);
	    free (l);
	    break;
	  }
    }
  
  PRINTF_MESSAGE ("[%s] left the conference <%s>\n", who, room);
}


/*
 * Name: fh_ext_yahoo_conf_message
 * 	Called when someone messages in the conference.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the user who messaged
 * 	room - the room
 * 	msg  - the message
 */
void 
fh_ext_yahoo_conf_message (int id, char *who, char *room, char *msg, int utf8)
{
  char buffer[4096];
  sprintf (buffer, "[%s] %s", room, msg);
  //  PRINTF_MESSAGE ("%s [in %s] -> %s\n", who, room, msg);
  fh_ext_yahoo_got_im (id, who, buffer, 0, 0, utf8);

  // AUTO-INSERT-MODE
  set_current_target_buddy (room);
}

void fh_ext_yahoo_chat_cat_xml (int id, char *xml) 
{
  PRINTF_MESSAGE ("%s\n", xml);
}

static void print_chat_member (struct yahoo_chat_member *ycm) 
{
	PRINTF_MESSAGE ("%s (%s) ", ycm->id, ycm->alias);
	PRINTF_MESSAGE (" Age: %d Sex: ", ycm->age);
	if (ycm->attribs & YAHOO_CHAT_MALE) {
		PRINTF_MESSAGE ("Male");
	} else if (ycm->attribs & YAHOO_CHAT_FEMALE) {
		PRINTF_MESSAGE ("Female");
	} else {
		PRINTF_MESSAGE ("Unknown");
	}
	if (ycm->attribs & YAHOO_CHAT_WEBCAM) {
		PRINTF_MESSAGE (" with webcam");
	}

	PRINTF_MESSAGE ("  Location: %s\n", ycm->location);
}

void fh_ext_yahoo_chat_join (int id, char *room, char * topic, YList *members)
{
  PRINTF_MESSAGE ("You have joined the chatroom %s with topic %s\n", room, topic);

  while(members) {
    YList *n = members->next;
    
    PRINTF_MESSAGE ("\t");
    print_chat_member(members->data);
    PRINTF_MESSAGE ("\n");
    FREE (((struct yahoo_chat_member *)members->data)->id);
    FREE (((struct yahoo_chat_member *)members->data)->alias);
    FREE (((struct yahoo_chat_member *)members->data)->location);
    FREE (members->data);
    FREE (members);
    members=n;
  }
}

void fh_ext_yahoo_chat_userjoin (int id, char *room, struct yahoo_chat_member *who)
{
  print_chat_member(who);
  PRINTF_MESSAGE (" joined the chatroom %s\n", room);
  FREE (who->id);
  FREE (who->alias);
  FREE (who->location);
  FREE (who);
}

void fh_ext_yahoo_chat_userleave (int id, char *room, char *who)
{
  PRINTF_MESSAGE ("%s left the chatroom %s\n", who, room);
}

void fh_ext_yahoo_chat_message (int id, char *who, char *room, 
			     char *msg, int msgtype, int utf8)
{
  char * umsg = msg;
  char * charpos;
	
  if(utf8)
    umsg = y_utf8_to_str(msg);
  /* Remove escapes */
  charpos = umsg;
  while(*charpos) {
    if (*charpos == 0x1b) {
      *charpos = ' ';
    }
    charpos++;
  }
  
  if (msgtype == 2) {
    PRINTF_MESSAGE ("(in %s) %s %s\n", room, who, umsg);
  } else {
    PRINTF_MESSAGE ("(in %s) %s: %s\n", room, who, umsg);
  }
  
  if(utf8)
    FREE(umsg);
}

void fh_ext_yahoo_got_webcam_image(int id, const char *who, 
				const unsigned char *image, unsigned int image_size, 
				unsigned int real_size, unsigned int timestamp)
{
  // FIXME: refer sample_client.c and implement this feature
}

void fh_ext_yahoo_webcam_viewer(int id, char *who, int connect)
{
  // FIXME: refer sample_client.c and implement this feature
}

void fh_ext_yahoo_webcam_closed(int id, char *who, int reason)
{
  // FIXME: refer sample_client.c and implement this feature
}

void fh_ext_yahoo_webcam_data_request(int id, int send)
{
  // FIXME: refer sample_client.c and implement this feature
}

void fh_ext_yahoo_webcam_invite(int id, char *from)
{
  // FIXME: refer sample_client.c and implement this feature
}

void fh_ext_yahoo_webcam_invite_reply(int id, char *from, int accept)
{
  // FIXME: refer sample_client.c and implement this feature
}

/*
 * Name: fh_ext_yahoo_got_file
 * 	Called when someone sends you a file
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the user who sent the file
 * 	url  - the file url
 * 	expires  - the expiry date of the file on the server (timestamp)
 * 	msg  - the message
 * 	fname- the file name if direct transfer
 * 	fsize- the file size if direct transfer
 */
void
fh_ext_yahoo_got_file (int id, char *who, char *url, long expires, char *msg, char *fname, unsigned long fesize)
{

  char lfile[256], *ptr, *qtr;
  char buf[4096];
  unsigned long fsize;
  int bytes_read, fd = -1, lfd = -1;
  char filename[256] = "";

  if (msg && msg[0])
    {
      PRINTF_MESSAGE ("[%s] has sent a file [%s] with mesg [%s]\n", who, url, msg);
    }
  else
    {
      PRINTF_MESSAGE ("[%s] has sent a file [%s]\n", who, url);
    }

  //////////////////////////////////////////////////////  
  /* FIXME: broke while moving to newer libyahoo2-0.7.2
     please ask <ab@gnu.org.in>, he has comments on this
  fd = yahoo_get_url_handle(id, url, filename, &fsize);
  */
  return;
  //////////////////////////////////////////////////////
  
  if(fname)
    strcpy (filename, fname);
  else
    {
      /* filename is not filled properly by yahoo, So a round about way to
	 find filename */
      ptr = strstr (url, "/.tmp/");
      if(!ptr)
	{
	  PRINTF_MESSAGE ("Error in file download: Unable to get filename\n");
	  return;
	}
      ptr += strlen ("/.tmp/");
      qtr = filename;
      while (*ptr != '?')
	{
	  *qtr = *ptr;
	  ptr++;
	  qtr++;
	}
      *qtr = '\0';
    }
      
  
  if (fd < 0)
    {
      PRINTF_MESSAGE ("Error in file [%s] download\n", filename) ;
      return;
    }

  {
    DIR *download_dir;
  /* check if downloads dir exits. downloads dir is created only
  durning setup mode, so for users upgrading from 1.0.0 to higher, the
  dir needs to be created here */
    if (!(download_dir = opendir (get_downloads_directory ())))
      {/* ~/.freehoo/downloads/ does not exist, so create it */
	if(mkdir (get_downloads_directory (), FH_CONFIG_DIRECTORY_MODE) !=0)
	  {
	    perror ("creating download directory");
	    exit (EXIT_FAILURE);
	  }
      }
    else
      closedir (download_dir);
  }

  sprintf(lfile, "%s/%s", get_downloads_directory (), filename);
  
  lfd = open (lfile, O_CREAT | O_RDWR);
  if (lfd < 0)
    {
      PRINTF_MESSAGE ("Error in opening file [%s]\n", lfile) ;
      return;
    }
  while (fsize)
    {
      if((bytes_read = read (fd, buf, sizeof(buf))) < 0)
	{
	  PRINTF_MESSAGE ("Error in file [%s] download\n", filename) ;
	  break;
	}
      fsize -= bytes_read;
      if (!bytes_read)
	break;
      if(write (lfd, buf, bytes_read) != bytes_read)
	{
	  PRINTF_MESSAGE ("Error in writing file [%s]\n", lfile) ;
	  break;
	}
      
    }
  close (lfd);
  PRINTF_MESSAGE ("File saved as [%s]\n", lfile) ;
}


/*
 * Name: fh_ext_yahoo_contact_added
 * 	Called when a contact is added to your list
 * Params:
 * 	id   - the id that identifies the server connection
 * 	myid - the identity he was added to
 * 	who  - who was added
 * 	msg  - any message sent
 */
void 
fh_ext_yahoo_contact_added (int id, char *myid, char *who, char *msg)
{
  char buf[1024];

  PRINTF_MESSAGE ("%s has added you to their contact list.\n", who);
  PRINTF_MESSAGE ("Use \"?add %s\" to add %s to your contact list\n", who, who);
  PRINTF_MESSAGE ("or Use \"?reject %s [MSG]\" to reject %s.\n", who, who);
  
  if(msg)
    strcpy (buf, msg);
  else
    strcpy (buf, "");
  // invoke fh-contact-added-hook
  set_hook_return (0);
  scm_run_hook (get_contact_added_hook (),
		gh_list (gh_str02scm (who),
			 gh_str02scm (buf),
			 SCM_UNDEFINED));
}


/*
 * Name: fh_ext_yahoo_rejected
 * 	Called when a contact rejects your add
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - who rejected you
 * 	msg  - any message sent
 */
void 
fh_ext_yahoo_rejected (int id, char *who, char *msg)
{
  PRINTF_MESSAGE ("%s has rejected you%s%s\n", who, 
	  (msg ? " with the message:\n" : "."), 
	  (msg ? msg : ""));
}


/*
 * Name: fh_ext_yahoo_typing_notify
 * 	Called when remote user starts or stops typing.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the handle of the remote user
 * 	stat - 1 if typing, 0 if stopped typing
 */
void 
fh_ext_yahoo_typing_notify (int id, char *who, int stat)
{
  //   if (stat && TRUE)
  //      set_default_prompt as (prompt + [who]);
}


/*
 * Name: fh_ext_yahoo_game_notify
 * 	Called when remote user starts or stops a game.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	who  - the handle of the remote user
 * 	stat - 1 if game, 0 if stopped gaming
 */
void 
fh_ext_yahoo_game_notify (int id, char *who, int stat)
{
  if (stat)
    {
      PRINTF_MESSAGE ("%s started the game\n", who);
    }
  else
    {
      PRINTF_MESSAGE ("%s stoped the game\n", who);
    }
}


/*
 * Name: fh_ext_yahoo_mail_notify
 * 	Called when you receive mail, or with number of messages
 * Params:
 * 	id   - the id that identifies the server connection
 * 	from - who the mail is from - NULL if only mail count
 * 	subj - the subject of the mail - NULL if only mail count
 * 	cnt  - mail count - 0 if new mail notification
 */
void 
fh_ext_yahoo_mail_notify (int id, char *from, char *subj, int cnt)
{
  if (cnt)
    {
      PRINTF_MESSAGE ("You have %d mail%s\n", cnt, (cnt == 1 ? "." : "s."));
    }
  if (from && subj)
    {
      // Bug in libyahoo2 parsing. When we send mail to ourself and if
      // we immediately check the mail we get empty "from" and
      // "subject" and so we should not print the empty string.
      // -- Nagappan <nagappanal@yahoo.com>
      if (strlen (from) > 0 && strlen (subj) > 0 )
	PRINTF_MESSAGE ("You have mail from: <%s>, subject: <%s>.\n", 
			from, subj);
    }
}


/*
 * Name: fh_ext_yahoo_system_message
 * 	System message
 * Params:
 * 	id   - the id that identifies the server connection
 * 	msg  - the message
 */
void 
fh_ext_yahoo_system_message (int id, char *msg)
{
  PRINTF_MESSAGE ("Yahoo System Message: %s\n",
		  filter_message (msg));
}


/*
 * Name: fh_ext_yahoo_error
 * 	Called on error.
 * Params:
 * 	id   - the id that identifies the server connection
 * 	err  - the error message
 * 	fatal- whether this error is fatal to the connection or not
 */
void 
fh_ext_yahoo_error (int id, char *err, int fatal)
{
  PRINTF_MESSAGE ("Yahoo Error: %s\n", err);
  //  if (fatal)
  //    yahoo_logout ();
}


/*
 * Name: fh_ext_yahoo_log
 * 	Called to log a message.
 * Params:
 * 	fmt  - the printf formatted message
 * Returns:
 * 	0
 */
int 
fh_ext_yahoo_log (char *fmt, ...)
{
// Not implemented right now
//   va_list ap;
//   
//   va_start(ap, fmt);
//   
//   vfprintf(stderr, fmt, ap);
//   fflush(stderr);
//   va_end(ap);
  return 0;
}


// typedef enum {
// 	YAHOO_INPUT_READ = 1 << 0,
// 	YAHOO_INPUT_WRITE = 1 << 1,
// 	YAHOO_INPUT_EXCEPTION = 1 << 2
// } yahoo_input_condition;
/*
 * Name: fh_ext_yahoo_add_handler
 * 	Add a listener for the fd
 * Params:
 * 	id   - the id that identifies the server connection
 * 	fd   - the fd on which to listen
 * 	cond - the condition on which to call the callback
 */
void 
fh_ext_yahoo_add_handler (int id, int fd, yahoo_input_condition cond, void *data)
{
  (get_fh_session ())->session_id = id;
  (get_fh_session ())->socket_fd = fd;
  (get_fh_session ())->status = yahoo_current_status (id);
  (get_fh_session ())->data = data;
  
  /* call "fh-login-post-hook" when freehoo is ready to go
   */
  set_hook_return (0);
  scm_run_hook (get_login_post_hook (), gh_list (SCM_UNDEFINED));
  if (get_hook_return () == 1)
    return;
}


/*
 * Name: fh_ext_yahoo_remove_handler
 * 	Remove the listener for the fd
 * Params:
 * 	id   - the id that identifies the server connection
 * 	fd   - the fd on which to listen
 */
void 
fh_ext_yahoo_remove_handler (int id, int fd)
{
  // Not implemented right now
}


/*
 * Name: ext_yahoo_got_identities
 * 	Called when the contact list is got from the server
 * Params:
 * 	id   - the id that identifies the server connection
 * 	ids  - the identity list
 */
void fh_ext_yahoo_got_identities(int id, YList * ids)
{
}


void ext_yahoo_got_cookies(int id)
{
	/*yahoo_get_yab(id);*/
}

/*
 * Name: fh_ext_yahoo_connect
 * 	Connect to a host:port
 * Params:
 * 	host - the host to connect to
 * 	port - the port to connect on
 * Returns:
 * 	a unix file descriptor to the socket
 */
int 
fh_ext_yahoo_connect (char *host, int port)
{
  struct sockaddr_in serv_addr;
  struct hostent *server;
  int servfd;
  int res;
  char **p;
  
  if (!(server = gethostbyname (host))) 
    {
      fprintf (stderr, "Failed to look up server[%s:%d]\n", host, port);
      fprintf (stderr, "%d: %s\n", h_errno, hstrerror (h_errno));
      return -1;
    }
  
  if ((servfd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
    {
      fprintf (stderr, "Socket create error (%d): %s\n", errno, strerror (errno));
      return -1;
    }
  
  for (p = server->h_addr_list; *p; p++)
    {
      memset (&serv_addr, 0, sizeof (serv_addr));
      serv_addr.sin_family = AF_INET;
      memcpy (&serv_addr.sin_addr.s_addr, *p, server->h_length);
      serv_addr.sin_port = htons (port);
      
      res = -1;
      res = connect (servfd, (struct sockaddr *) &serv_addr, sizeof (serv_addr));
      
      if (res == 0)
	  return servfd;
    }
  return -1;
}

int 
fh_ext_yahoo_connect_async (int id, char *host, int port, 
		yahoo_connect_callback callback, void *data)
{
	struct sockaddr_in serv_addr;
	static struct hostent *server;
	int servfd;
	struct connect_callback_data * ccd;
	int error;

	if(!(server = gethostbyname(host))) {
		errno=h_errno;
		return -1;
	}

	if((servfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		return -1;
	}

	memset(&serv_addr, 0, sizeof(serv_addr));
	serv_addr.sin_family = AF_INET;
	memcpy(&serv_addr.sin_addr.s_addr, *server->h_addr_list, server->h_length);
	serv_addr.sin_port = htons(port);

	error = connect(servfd, (struct sockaddr *) &serv_addr, sizeof(serv_addr));

	//	LOG(("Trying to connect: fd:%d error:%d", servfd, error));
	if(!error) {
		callback(servfd, 0, data);
		return 0;
	} else if(error == -1 && errno == EINPROGRESS) {
		ccd = calloc(1, sizeof(struct connect_callback_data));
		ccd->callback = callback;
		ccd->callback_data = data;
		ccd->id = id;

		fh_ext_yahoo_add_handler(-1, servfd, YAHOO_INPUT_WRITE, ccd);
		return 1;
	} else {
		close(servfd);
		return -1;
	}
}


void register_callbacks ()
{
  static struct yahoo_callbacks yc;

  yc.ext_yahoo_login_response = fh_ext_yahoo_login_response;
  yc.ext_yahoo_got_buddies = fh_ext_yahoo_got_buddies;
  yc.ext_yahoo_got_ignore = fh_ext_yahoo_got_ignore;
  yc.ext_yahoo_got_identities = fh_ext_yahoo_got_identities;
  yc.ext_yahoo_got_cookies = ext_yahoo_got_cookies;
  yc.ext_yahoo_status_changed = fh_ext_yahoo_status_changed;
  yc.ext_yahoo_got_im = fh_ext_yahoo_got_im;
  yc.ext_yahoo_got_conf_invite = fh_ext_yahoo_got_conf_invite;
  yc.ext_yahoo_conf_userdecline = fh_ext_yahoo_conf_userdecline;
  yc.ext_yahoo_conf_userjoin = fh_ext_yahoo_conf_userjoin;
  yc.ext_yahoo_conf_userleave = fh_ext_yahoo_conf_userleave;
  yc.ext_yahoo_conf_message = fh_ext_yahoo_conf_message;

  yc.ext_yahoo_chat_cat_xml = fh_ext_yahoo_chat_cat_xml;
  yc.ext_yahoo_chat_join = fh_ext_yahoo_chat_join;
  yc.ext_yahoo_chat_userjoin = fh_ext_yahoo_chat_userjoin;
  yc.ext_yahoo_chat_userleave = fh_ext_yahoo_chat_userleave;
  yc.ext_yahoo_chat_message = fh_ext_yahoo_chat_message;
  yc.ext_yahoo_got_webcam_image = fh_ext_yahoo_got_webcam_image;
  yc.ext_yahoo_webcam_invite = fh_ext_yahoo_webcam_invite;
  yc.ext_yahoo_webcam_invite_reply = fh_ext_yahoo_webcam_invite_reply;
  yc.ext_yahoo_webcam_closed = fh_ext_yahoo_webcam_closed;
  yc.ext_yahoo_webcam_viewer = fh_ext_yahoo_webcam_viewer;
  yc.ext_yahoo_webcam_data_request = fh_ext_yahoo_webcam_data_request;
  
  yc.ext_yahoo_got_file = fh_ext_yahoo_got_file;
  yc.ext_yahoo_contact_added = fh_ext_yahoo_contact_added;
  yc.ext_yahoo_rejected = fh_ext_yahoo_rejected;
  yc.ext_yahoo_typing_notify = fh_ext_yahoo_typing_notify;
  yc.ext_yahoo_game_notify = fh_ext_yahoo_game_notify;
  yc.ext_yahoo_mail_notify = fh_ext_yahoo_mail_notify;
  yc.ext_yahoo_system_message = fh_ext_yahoo_system_message;
  yc.ext_yahoo_error = fh_ext_yahoo_error;
  yc.ext_yahoo_log = fh_ext_yahoo_log;
  yc.ext_yahoo_add_handler = fh_ext_yahoo_add_handler;
  yc.ext_yahoo_remove_handler = fh_ext_yahoo_remove_handler;
  yc.ext_yahoo_connect = fh_ext_yahoo_connect;
  yc.ext_yahoo_connect_async = fh_ext_yahoo_connect_async;

  yahoo_register_callbacks(&yc);
        
}

