/*
---------------------------------------------------------------------------
 $Id: protocol.c,v 1.5 2003/08/11 15:18:22 jfboud Exp $
---------------------------------------------------------------------------
Copyright (c) 2001-2003 Hexago Inc. All rights reserved.

     The contents of this file are subject to the Hexago
     Public License Version 1.0 (the "License"); you may not
     use this file except in compliance with the License.

     Software distributed under the License is distributed on
     an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either
     express or implied. See the License for the specific
     language governing rights and limitations under the
     License.

     The Original Code is _source code of the tunnel server
     protocol-client side_.

     The Initial Developer of the Original Code is Hexago .

     All Rights Reserved.

     Contributor(s): ______________________________________.

---------------------------------------------------------------------------
*/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>

#include "string.h"
#include "protocol.h"
#include "network.h"
#include "log.h"
#include "version.h"

/*
   query       = header payload
   header      = content-length: separator size CR LF
   separator   = space or tab
   space       = " "
   tab         = "\t"
   size        = 0..2147483647
   payload     = *OCTET

   response    = header return-code payload
   return-code = number separator msg CR LF
   number      = digit digit digit
   digit       = 0..9
   msg         = *ALPHANUM

*/

#define MAXNAME 1024

#define ProtocolVersion "VERSION=" TSP_VERSION "\r\n"

int tspConnect(char *server)
{
  int socket;
  char Buffer[MAXNAME], *srvname, *srvport;

  snprintf(Buffer, sizeof Buffer, "%s", server);
  srvname = strtok(Buffer, ":");
  if((srvport = strtok(NULL, ":"))==NULL) {
    srvport = SERVER_PORT;
  }

  if(atoi(srvport) <= 0) {
    LogPrintf(ELError, "tspConnect", "Service port not valid: %s", srvport);
    return 0;
  }

  if((socket = NetConnect(srvname, atoi(srvport))) < 0) {
    LogPrintf(ELError, "tspConnect", "Not able to connect to service port %s", srvport);
    return 0;
  }

  if (NetPrintf(socket, ProtocolVersion) == 0) {
    LogPrintf(ELError, "tspConnect", "Socket transmit error");
    NetClose(socket);
    return 0;
  }

  return socket;
}

void tspClose(int socket)
{
   NetClose(socket);
}

void tspClearPayload(tPayload *Payload)
{
  if (Payload->payload) {
    free(Payload->payload);
  }

  memset(Payload, 0, sizeof(tPayload));
}

int tspGetStatusCode(char *payload)
{
  if (payload)
     return atoi(payload);
  return(0);
}

char *tspGetStatusMsg(char *payload)
{
  static char Msg[1024], *ptr;
   
  if (!payload)
     return("");
  memset(Msg, 0, sizeof(Msg));

  if ((ptr = strchr(payload, '\n')) != NULL) {
    if (ptr - payload > sizeof Msg)
      ptr = payload + sizeof Msg;
    memcpy(Msg, payload, (int)(ptr-payload));
  } else {
    if ((ptr = strchr(payload, ' ')) != NULL) {
      snprintf(Msg, sizeof Msg, "%s", ptr+1); 
    } else {
      snprintf(Msg, sizeof Msg, "%s", payload); 
    }
  }

  return(Msg);
}

int
tspReceive(int Socket, tPayload *Payload)
{
  int BytesTotal = 0, BytesRead = 0, BytesLeft = 0;
  char Header[PROTOCOLMAXHEADER];

  if ((BytesRead = NetReadLine(Socket, Header, sizeof Header)) <= 0) {
    LogPrintf(ELError, "tspReceive", "Not able to read from server socket");
    return PROTOCOL_EREAD;
  }

  if (memcmp(Header, "Content-length:", 15)) {
    LogPrintf(ELError, "tspReceive", "Was expecting 'Content-length:', received %s", Header);
    return PROTOCOL_ERROR;
  }

  if (Payload->payload) free(Payload->payload);

  if ((Payload->size = atol(Header + 15)) <= 0L) {
    LogPrintf(ELError, "tspReceive", "Invalid payload size: %s", Header + 15);
    return PROTOCOL_ERROR;
  }

  if((Payload->payload = (char *)malloc((Payload->size) + 1)) == NULL) {
    LogPrintf(ELError, "tspReceive", "Memory allocation error for payload");
    return(PROTOCOL_EMEM);
  }

  BytesLeft = Payload->size;

  while(BytesLeft) {
    if((BytesRead = NetRead(Socket, (Payload->payload + BytesTotal), BytesLeft)) <= 0) {
      if (Payload->payload) free(Payload->payload);
      LogPrintf(ELError, "tspReceive", "Not able to read from server socket");
      return(PROTOCOL_EREAD);
    }
    BytesTotal += BytesRead;
    BytesLeft -= BytesRead;
    Payload->payload[BytesTotal] = 0; /* NULL terminate the Payload */
  }

  return PROTOCOL_OK;
}

int tspSend(int Socket, tPayload *Payload)
{
  if (NetPrintf(Socket, "Content-length: %ld\r\n", Payload->size) == 0) {
    LogPrintf(ELError, "tspSend", "Not able to write 'Content-length:...' to server socket");
    return 1;
  }

  if (NetWrite(Socket, Payload->payload, Payload->size) == 0) {
    LogPrintf(ELError, "tspSend", "Not able to write payload to server socket");
    return 1;
  }

  return 0;
}

char *tspAddPayloadString(tPayload *Payload, char *Addition)
{
  char *NewPayload;
         
  if(Addition) {

    if(Payload->PayloadCapacity == 0) {
      if((NewPayload = Payload->payload = (char *)malloc(PROTOCOLMAXPAYLOADCHUNK)) == NULL) {
        LogPrintf(ELError, "tspAddPayloadString", "Not able to allocate memory for payload");
	return NULL;
      }
      *Payload->payload = 0;
      Payload->PayloadCapacity = PROTOCOLMAXPAYLOADCHUNK;
    }

    if((Payload->size + strlen(Addition) + 1) > Payload->PayloadCapacity) {
      Payload->PayloadCapacity += PROTOCOLMAXPAYLOADCHUNK;

      if((NewPayload = (char *) malloc(Payload->PayloadCapacity)) == NULL) {
        LogPrintf(ELError, "tspAddPayloadString", "Not able to allocate memory for payload");
	return NULL;
      }

      memcpy(NewPayload, Payload->payload, Payload->size + 1);
      free(Payload->payload);
      Payload->payload = NewPayload;
    }

    strcat(Payload->payload, Addition);
    Payload->size += strlen(Addition);
  }

  return Payload->payload;
}

/*----- protocol.c ----------------------------------------------------------*/
