/*
   Does power come from fancy toys
   or does power come from the integrity of our walk?
*/

#include <sys/time.h>

#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>

#include <math.h>
#include <string.h>

#include <pthread.h>
#include <sched.h>
#include <sys/mman.h>

#include "fweelin_config.h"
#include "fweelin_audioio.h"
#include "fweelin_core_dsp.h"

#if USE_FLUIDSYNTH
#include "fweelin_fluidsynth.h"
#endif

// **************** SYSTEM LEVEL AUDIO

int AudioIO::process (nframes_t nframes, void *arg) {
  AudioIO *inst = static_cast<AudioIO *>(arg);

  // Get CPU load
  inst->cpuload = jack_cpu_load(inst->client);

  // Get buffers from jack
  AudioBuffers *ab = inst->app->getABUFS();
  for (int i = 0; i < ab->numins_ext; i++) {
    // Left/mono channel
    ab->ins[0][i] = 
      (sample_t *) jack_port_get_buffer (inst->iport[0][i], nframes);
    // Right channel
    ab->ins[1][i] = (inst->iport[1][i] != 0 ?
		     (sample_t *) 
		     jack_port_get_buffer (inst->iport[1][i], nframes) :
		     0);
  }
  for (int i = 0; i < ab->numouts; i++) {
    // Left/mono channel
    ab->outs[0][i] = 
      (sample_t *) jack_port_get_buffer (inst->oport[0][i], nframes);
    // Right channel
    ab->outs[1][i] = (inst->oport[1][i] != 0 ?
		      (sample_t *) 
		      jack_port_get_buffer (inst->oport[1][i], nframes) :
		      0);
  }

  if (inst->rp != 0) {
    if (nframes != inst->app->getBUFSZ()) {
      printf("AUDIO: We've got a problem, honey!--\n");
      printf("Audio buffer size has changed: %d->%d\n",
	     inst->app->getBUFSZ(),nframes);
      exit(1);
    }

    // Run through audio processors
    inst->rp->process(0, nframes, ab);
  }

  return 0;      
}

int AudioIO::srate_callback (nframes_t nframes, void *arg) {
  AudioIO *inst = static_cast<AudioIO *>(arg);
  printf ("AUDIO: Sample rate is now %d/sec\n", nframes);
  inst->srate = nframes;
  return 0;
}

void AudioIO::audio_shutdown (void *arg)
{
  printf("AUDIO: shutdown! exiting..\n");
  exit(1);
}

int AudioIO::activate (Processor *rp) {
  const char **ports;

  // Store the rootprocessor passed as beginning of signal chain
  this->rp = rp;

  // Start rolling audio through JACK server
  if (jack_activate (client)) {
    printf("AUDIO: ERROR: Cannot activate client!\n");
    return 1;
  }
  
  // Connect ports
  //AudioBuffers *ab = app->getABUFS();
  
  // INPUT
  // No longer connect audio ports because the mapping isn't clear with
  // multi stereoins/outs
  if ((ports = jack_get_ports (client, NULL, NULL, JackPortIsPhysical|JackPortIsOutput)) == NULL) {
    printf("AUDIO: ERROR: Cannot find any physical capture ports!\n");
    return 1;
  }

#if 0
  for (int i = 0; i < ab->numins_ext && ports[i] != 0; i++)
    if (jack_connect (client, ports[i], jack_port_name (iport[i]))) {
      printf("AUDIO: Cannot connect input port %d->%s!\n",i,
	     jack_port_name(iport[i]));
    }
#endif

  free(ports);
  
  // OUTPUT
  if ((ports = jack_get_ports (client, NULL, NULL, JackPortIsPhysical|JackPortIsInput)) == NULL) {
    printf("AUDIO: ERROR: Cannot find any physical playback ports!");
    return 1;
  }

#if 0
  for (int i = 0; i < ab->numouts && ports[i] != 0; i++)
    if (jack_connect (client, jack_port_name (oport[i]), ports[i])) {
      printf("AUDIO: Cannot connect output port %d->%s!\n",i,
	     jack_port_name(oport[i]));
    }
#endif

  free(ports);

  return 0;
}

nframes_t AudioIO::getbufsz() {
  return jack_get_buffer_size (client);
}

void AudioIO::close () {
  jack_client_close (client);

  delete[] iport[0];
  delete[] iport[1];
  delete[] oport[0];
  delete[] oport[1];

  printf("AUDIO: end\n");
}

int AudioIO::open () {
  // **** AUDIO startup
  
  // Try to become a client of the JACK server
  if ((client = jack_client_new ("FreeWheeling")) == 0) {
    fprintf (stderr, "AUDIO: ERROR: Jack server not running!\n");
    return 1;
  }
  
  /* tell the JACK server to call `process()' whenever
     there is work to be done.
  */  
  jack_set_process_callback (client, process, this);
  
  jack_nframes_t bufsz = jack_get_buffer_size (client);
  printf ("AUDIO: Audio buffer size is: %d\n", bufsz);

  /* tell the JACK server to call `srate_callback()' whenever
     the sample rate of the system changes.
  */
  jack_set_sample_rate_callback (client, srate_callback, this);
  
  /* tell the JACK server to call `jack_shutdown()' if
     it ever shuts down, either entirely, or if it
     just decides to stop calling us.
  */
  jack_on_shutdown (client, audio_shutdown, this);
  
  /* display the current sample rate. once the client is activated 
     (see below), you should rely on your own sample rate
     callback (see above) for this value.
  */
  srate = jack_get_sample_rate (client);
  printf ("AUDIO: Engine sample rate is %d\n", srate);

  // Set time scale
  timescale = (float) bufsz/srate;
  
  // Create buffers
  AudioBuffers *ab = app->getABUFS();

  printf("AUDIO: Using %d external inputs, %d total inputs\n",
	 ab->numins_ext,ab->numins);
  iport[0] = new jack_port_t *[ab->numins_ext];
  iport[1] = new jack_port_t *[ab->numins_ext];
  oport[0] = new jack_port_t *[ab->numouts];
  oport[1] = new jack_port_t *[ab->numouts];

  // Create ports
  char tmp[255];
  for (int i = 0; i < ab->numins_ext; i++) {
    char stereo = ab->IsStereoInput(i);
    snprintf(tmp,255,"in_%d%s",i+1,(stereo ? "L" : ""));
    iport[0][i] = jack_port_register(client, tmp, JACK_DEFAULT_AUDIO_TYPE, 
				     JackPortIsInput, 0);
    if (stereo) {
      snprintf(tmp,255,"in_%d%s",i+1,"R");
      iport[1][i] = jack_port_register(client, tmp, JACK_DEFAULT_AUDIO_TYPE, 
				       JackPortIsInput, 0);
    } else
      iport[1][i] = 0;
  }

  for (int i = 0; i < ab->numouts; i++) {
    char stereo = ab->IsStereoOutput(i);
    snprintf(tmp,255,"out_%d%s",i+1,(stereo ? "L" : ""));
    oport[0][i] = jack_port_register(client, tmp, JACK_DEFAULT_AUDIO_TYPE, 
				     JackPortIsOutput, 0);
    if (stereo) {
      snprintf(tmp,255,"out_%d%s",i+1,"R");
      oport[1][i] = jack_port_register(client, tmp, JACK_DEFAULT_AUDIO_TYPE, 
				       JackPortIsOutput, 0);
    } else
      oport[1][i] = 0;
  }
  
  return 0;
}
