/* 
   Truth has a power like electricity
   When we dance in Truth,
   it is infectious,
   unmistakable. 
*/

#include <sys/time.h>

#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>

#include <math.h>
#include <string.h>

#include <pthread.h>
#include <sched.h>
#include <sys/mman.h>

#include <X11/Xlib.h>

#include <glob.h>

#include "fweelin_core.h"
#include "fweelin_fluidsynth.h"

// *********** CORE

// Save loop
void Loop::Save(Fweelin *app) {
  // Queue save
  app->getLOOPMGR()->AddLoopToSaveQueue(this);
};

void LoopManager::AddToSaveQueue(Event *ev) {
  EventManager::QueueEvent(&savequeue,ev);
};

void LoopManager::AddLoopToSaveQueue(Loop *l) {
  if (!autosave && l->GetSaveStatus() == NO_SAVE) {
    LoopListEvent *ll = (LoopListEvent *) 
      Event::GetEventByType(T_EV_LoopList,1);
    ll->l = l;
	
    EventManager::QueueEvent(&savequeue,ll);
  }
};

void LoopManager::AddLoopToLoadQueue(char *filename, int index, float vol) {
  LoopListEvent *ll = (LoopListEvent *) 
    Event::GetEventByType(T_EV_LoopList,1);
  strcpy(ll->l_filename,filename);
  ll->l_idx = index;
  ll->l_vol = vol;
  
  EventManager::QueueEvent(&loadqueue,ll);
};

// Adds the loop with given filename to the loop browser br
void LoopManager::AddLoopToBrowser(Browser *br, char *filename) {
  char tmp[FWEELIN_OUTNAME_LEN];
  
  struct stat st;
  if (stat(filename,&st) == 0) {
    // Loop exists, use combination of time and md5 as name
    char *md5ptr = strrchr(filename,'-'),
      *extptr = strrchr(filename,'.');
    char md5short_1 = 'X',
      md5short_2 = 'X';
    if (md5ptr != 0 && extptr != 0 && 
	strlen(md5ptr+1) - strlen(extptr) == SAVEABLE_HASH_LENGTH*2) {
      md5short_1 = *(md5ptr+1);
      md5short_2 = *(md5ptr+2);
    }
    snprintf(tmp,FWEELIN_OUTNAME_LEN,"%s-%c%c %s",
	     FWEELIN_OUTPUT_LOOP_NAME,md5short_1,md5short_2,
	     ctime(&st.st_mtime));
    tmp[FWEELIN_OUTNAME_LEN-1] = '\0';
    tmp[strlen(tmp)-1] = '\0'; // Cut off return character

    /* struct tm *t = localtime(&st.st_mtime);
       snprintf(tmp,FWEELIN_OUTNAME_LEN,"%s-%02d/%02d %02d:%02d",
       FWEELIN_OUTPUT_LOOP_NAME,
       t->tm_mon,t->tm_mday,t->tm_hour,t->tm_min); */
    br->AddItem(new LoopBrowserItem(st.st_mtime,tmp,filename),1);
  }
};

// Populate the loop browser with any loops on disk
void LoopManager::SetupLoopBrowser() {
  Browser *br = app->getBROWSER(B_Loop);

  if (br != 0) {
    // Clear
    br->ClearAllItems();

    // Look for loops on disk 
    glob_t globbuf;
    char tmp[FWEELIN_OUTNAME_LEN];
    snprintf(tmp,FWEELIN_OUTNAME_LEN,"%s/%s*%s",
	     app->getCFG()->GetLibraryPath(),FWEELIN_OUTPUT_LOOP_NAME,
	     FWEELIN_OUTPUT_AUDIO_EXT);
    printf("BROWSER: (Loop) Scanning for loops in library: %s\n",tmp);
    if (glob(tmp, 0, NULL, &globbuf) == 0) {
      for (size_t i = 0; i < globbuf.gl_pathc; i++) {
	// printf("BROWSER: (Loop) Loop: %s\n",globbuf.gl_pathv[i]);
	AddLoopToBrowser(br,globbuf.gl_pathv[i]);
      }

      br->AddDivisions(FWEELIN_FILE_BROWSER_DIVISION_TIME);
      br->MoveToBeginning();
      globfree(&globbuf);
    }
  }
};

// Adds the scene with given filename to the scene browser br
void LoopManager::AddSceneToBrowser(Browser *br, char *filename) {
  char tmp[FWEELIN_OUTNAME_LEN];
  
  struct stat st;
  if (stat(filename,&st) == 0) {
    // Scene exists, use time + md5 as name
    char *md5ptr = strrchr(filename,'-'),
      *extptr = strrchr(filename,'.');
    char md5short_1 = 'X',
      md5short_2 = 'X';
    if (md5ptr != 0 && extptr != 0 && 
	strlen(md5ptr+1) - strlen(extptr) == SAVEABLE_HASH_LENGTH*2) {
      md5short_1 = *(md5ptr+1);
      md5short_2 = *(md5ptr+2);
    }
    snprintf(tmp,FWEELIN_OUTNAME_LEN,"%s-%c%c %s",
	     FWEELIN_OUTPUT_SCENE_NAME,md5short_1,md5short_2,
	     ctime(&st.st_mtime));
    tmp[FWEELIN_OUTNAME_LEN-1] = '\0';
    tmp[strlen(tmp)-1] = '\0'; // Cut off return character

    /* struct tm *t = localtime(&st.st_mtime);
       snprintf(tmp,FWEELIN_OUTNAME_LEN,"%s-%02d/%02d %02d:%02d",
       FWEELIN_OUTPUT_SCENE_NAME,
       t->tm_mon,t->tm_mday,t->tm_hour,t->tm_min); */
    br->AddItem(new SceneBrowserItem(st.st_mtime,tmp,filename),1);
  }
};

// Populate the scene browser with any scenes on disk
void LoopManager::SetupSceneBrowser() {
  Browser *br = app->getBROWSER(B_Scene);

  if (br != 0) {
    // Clear
    br->ClearAllItems();

    // Look for scenes on disk 
    glob_t globbuf;
    char tmp[FWEELIN_OUTNAME_LEN];
    snprintf(tmp,FWEELIN_OUTNAME_LEN,"%s/%s*%s",
	     app->getCFG()->GetLibraryPath(),FWEELIN_OUTPUT_SCENE_NAME,
	     FWEELIN_OUTPUT_DATA_EXT);
    printf("BROWSER: (Scene) Scanning for scenes in library: %s\n",tmp);
    if (glob(tmp, 0, NULL, &globbuf) == 0) {
      for (size_t i = 0; i < globbuf.gl_pathc; i++) {
	// printf("BROWSER: (Scene) Scene: %s\n",globbuf.gl_pathv[i]);
	AddSceneToBrowser(br,globbuf.gl_pathv[i]);
      }
      br->AddDivisions(FWEELIN_FILE_BROWSER_DIVISION_TIME);
      br->MoveToBeginning();
      globfree(&globbuf);
    }
  }
};

void LoopManager::ItemBrowsed(BrowserItem *i) {};
void LoopManager::ItemSelected(BrowserItem *i) {
  switch (i->GetType()) {
  case B_Loop: 
    printf("DISK: Load '%s'\n",((LoopBrowserItem *) i)->filename);
    LoadLoop(((LoopBrowserItem *) i)->filename,loadloopid,
	     newloopvol / GetOutputVolume());
    break;
    
  case B_Scene:
    printf("DISK: Load '%s'\n",((SceneBrowserItem *) i)->filename);
    LoadScene(((SceneBrowserItem *) i)->filename);
    break;

  default:
    break;
  }
};

LoopManager::LoopManager (Fweelin *app) : 
  savequeue(0), loadqueue(0), loadloopid(0), needs_saving_stamp(0),

  autosave(0), app(app), newloopvol(1.0), subdivide(1), curpulseindex(-1) {
  int mapsz = app->getTMAP()->GetMapSize();

  plist = new Processor *[mapsz];
  status = new LoopStatus[mapsz];
  waitactivate = new int[mapsz];
  waitactivate_vol = new float[mapsz];
  waitactivate_od = new char[mapsz];
  waitactivate_od_fb = new float[mapsz];

  numloops = 0;
  numrecordingloops = 0;
  
  lastrecidx = new int[LAST_REC_COUNT];
  memset(lastrecidx, 0, sizeof(int) * LAST_REC_COUNT);
  
  memset(plist, 0, sizeof(Processor *) * mapsz);
  int lst = T_LS_Off;
  memset(status, lst, sizeof(LoopStatus) * mapsz);
  memset(waitactivate, 0, sizeof(int) * mapsz);
  memset(waitactivate_vol, 0, sizeof(float) * mapsz);
  memset(waitactivate_od, 0, sizeof(char) * mapsz);
  memset(waitactivate_od_fb, 0, sizeof(float) * mapsz);
  memset(pulses, 0, sizeof(Pulse *) * MAX_PULSES);

  // Turn on block read/write managers for loading & saving loops
  bread = ::new BlockReadManager(0,this,app->getBMG(),
				 app->getCFG()->loop_peaksavgs_chunksize);
  bwrite = ::new BlockWriteManager(0,this,app->getBMG());
  app->getBMG()->AddManager(bread);
  app->getBMG()->AddManager(bwrite);

  // Listen for important events
  app->getEMG()->ListenEvent(this,0,T_EV_EndRecord);
  app->getEMG()->ListenEvent(this,0,T_EV_ToggleDiskOutput);
  app->getEMG()->ListenEvent(this,0,T_EV_SetAutoLoopSaving);
  app->getEMG()->ListenEvent(this,0,T_EV_SaveLoop);
  app->getEMG()->ListenEvent(this,0,T_EV_SaveScene);
  app->getEMG()->ListenEvent(this,0,T_EV_SetLoadLoopId);

  app->getEMG()->ListenEvent(this,0,T_EV_SlideMasterInVolume);
  app->getEMG()->ListenEvent(this,0,T_EV_SlideMasterOutVolume);
  app->getEMG()->ListenEvent(this,0,T_EV_SlideInVolume);
  app->getEMG()->ListenEvent(this,0,T_EV_SetMasterInVolume);
  app->getEMG()->ListenEvent(this,0,T_EV_SetMasterOutVolume);
  app->getEMG()->ListenEvent(this,0,T_EV_SetInVolume);
  app->getEMG()->ListenEvent(this,0,T_EV_ToggleInputRecord);

  app->getEMG()->ListenEvent(this,0,T_EV_DeletePulse);
  app->getEMG()->ListenEvent(this,0,T_EV_SelectPulse);
  app->getEMG()->ListenEvent(this,0,T_EV_TapPulse);
  app->getEMG()->ListenEvent(this,0,T_EV_SwitchMetronome);

  app->getEMG()->ListenEvent(this,0,T_EV_SetTriggerVolume);
  app->getEMG()->ListenEvent(this,0,T_EV_SlideLoopAmp);
  app->getEMG()->ListenEvent(this,0,T_EV_SetLoopAmp);
  app->getEMG()->ListenEvent(this,0,T_EV_TriggerLoop);
  app->getEMG()->ListenEvent(this,0,T_EV_MoveLoop);
  app->getEMG()->ListenEvent(this,0,T_EV_EraseLoop);
  app->getEMG()->ListenEvent(this,0,T_EV_EraseAllLoops);
  app->getEMG()->ListenEvent(this,0,T_EV_SlideLoopAmpStopAll);
};

LoopManager::~LoopManager() { 
  // Stop block read/write managers
  bread->End(0);
  bwrite->End();
  app->getBMG()->DelManager(bread);
  app->getBMG()->DelManager(bwrite);

  // Stop listening
  app->getEMG()->UnlistenEvent(this,0,T_EV_EndRecord);
  app->getEMG()->UnlistenEvent(this,0,T_EV_ToggleDiskOutput);
  app->getEMG()->UnlistenEvent(this,0,T_EV_SetAutoLoopSaving);
  app->getEMG()->UnlistenEvent(this,0,T_EV_SaveLoop);
  app->getEMG()->UnlistenEvent(this,0,T_EV_SaveScene);
  app->getEMG()->UnlistenEvent(this,0,T_EV_SetLoadLoopId);

  app->getEMG()->UnlistenEvent(this,0,T_EV_SlideMasterInVolume);
  app->getEMG()->UnlistenEvent(this,0,T_EV_SlideMasterOutVolume);
  app->getEMG()->UnlistenEvent(this,0,T_EV_SlideInVolume);
  app->getEMG()->UnlistenEvent(this,0,T_EV_SetMasterInVolume);
  app->getEMG()->UnlistenEvent(this,0,T_EV_SetMasterOutVolume);
  app->getEMG()->UnlistenEvent(this,0,T_EV_SetInVolume);
  app->getEMG()->UnlistenEvent(this,0,T_EV_ToggleInputRecord);

  app->getEMG()->UnlistenEvent(this,0,T_EV_DeletePulse);
  app->getEMG()->UnlistenEvent(this,0,T_EV_SelectPulse);
  app->getEMG()->UnlistenEvent(this,0,T_EV_TapPulse);
  app->getEMG()->UnlistenEvent(this,0,T_EV_SwitchMetronome);

  app->getEMG()->UnlistenEvent(this,0,T_EV_SetTriggerVolume);
  app->getEMG()->UnlistenEvent(this,0,T_EV_SlideLoopAmp);
  app->getEMG()->UnlistenEvent(this,0,T_EV_SetLoopAmp);
  app->getEMG()->UnlistenEvent(this,0,T_EV_TriggerLoop);
  app->getEMG()->UnlistenEvent(this,0,T_EV_MoveLoop);
  app->getEMG()->UnlistenEvent(this,0,T_EV_EraseLoop);
  app->getEMG()->UnlistenEvent(this,0,T_EV_EraseAllLoops);
  app->getEMG()->UnlistenEvent(this,0,T_EV_SlideLoopAmpStopAll);

  EventManager::DeleteQueue(savequeue);
  EventManager::DeleteQueue(loadqueue);

  // Let BMG know that we are ending
  app->getBMG()->RefDeleted((AutoWriteControl *) this);

  delete[] lastrecidx;

  delete[] plist; delete[] status; 
  delete[] waitactivate; delete[] waitactivate_vol; delete[] waitactivate_od;
  delete[] waitactivate_od_fb;
};

// Get length returns the length of any loop on the specified index
nframes_t LoopManager::GetLength(int index) {
  if (status[index] == T_LS_Recording) {
    // Ooh, we are recording on this index. Get the current length
    return ((RecordProcessor *) plist[index])->GetRecordedLength();
  }
  else {
    Loop *cur = app->getTMAP()->GetMap(index);
    if (cur != 0)
      return cur->blocks->GetTotalLen();
  }

  return 0;
}

// Get length returns the length of any loop on the specified index
// Rounded to its currently quantized length
// Or 0 if the loop has no pulse
nframes_t LoopManager::GetRoundedLength(int index) {
  if (status[index] == T_LS_Recording) {
    // Return 0 when recording
    return 0;

    // Ooh, we are recording on this index. Get the current length
    // return ((RecordProcessor *) plist[index])->GetRecordedLength();
  }
  else {
    Loop *cur = app->getTMAP()->GetMap(index);
    if (cur != 0)
      if (cur->pulse != 0)
	return cur->pulse->QuantizeLength(cur->blocks->GetTotalLen());
      else
	return 0; // cur->blocks->GetTotalLen();
  }
  
  return 0;
}

float LoopManager::GetPos(int index) {
  if (status[index] == T_LS_Recording)
    return 0.0;
  else {
    Loop *cur = app->getTMAP()->GetMap(index);
    if (cur != 0 && plist[index] != 0) {
      nframes_t playedlen = 0;
      if (status[index] == T_LS_Playing)
	playedlen = ((PlayProcessor *) plist[index])->GetPlayedLength();
      else if (status[index] == T_LS_Overdubbing)
	playedlen = ((RecordProcessor *) plist[index])->GetRecordedLength();

      if (cur->pulse == 0) 
	return (float) playedlen / cur->blocks->GetTotalLen();
      else {
	if (cur->pulse->QuantizeLength(cur->blocks->GetTotalLen()) == 0) {
	  printf("LoopManager: ERROR: Problem with quantize GetPos\n");
	  exit(1);
	}
	return (float) playedlen / 
	  cur->pulse->QuantizeLength(cur->blocks->GetTotalLen());
      }
    }
  }
  
  return 0.0;
}

// Get current # of samples into block chain with given index
nframes_t LoopManager::GetCurCnt(int index) {
  if (status[index] == T_LS_Recording)
    return 0;
  else {
    Loop *cur = app->getTMAP()->GetMap(index);
    if (cur != 0 && plist[index] != 0) {
      if (status[index] == T_LS_Playing)
	return ((PlayProcessor *) plist[index])->GetPlayedLength();
      else if (status[index] == T_LS_Overdubbing) 
	return ((RecordProcessor *) plist[index])->GetRecordedLength();
    }
  }
  
  return 0;
}

// Sets triggered volume on specified index
// If index is not playing, activates the index
void LoopManager::SetTriggerVol(int index, float vol) {
  if (status[index] == T_LS_Playing) 
    ((PlayProcessor *) plist[index])->SetPlayVol(vol);
  else if (status[index] == T_LS_Overdubbing) 
    ((RecordProcessor *) plist[index])->SetODPlayVol(vol);
}

// Gets trigger volume on specified index
// If index is not playing, returns 0
float LoopManager::GetTriggerVol(int index) {
  if (status[index] == T_LS_Playing)
    return ((PlayProcessor *) plist[index])->GetPlayVol();
  else if (status[index] == T_LS_Overdubbing) 
    return ((RecordProcessor *) plist[index])->GetODPlayVol();
  else
    return 0.0;
}

// Returns a loop with the specified index, if one exists
Loop *LoopManager::GetSlot(int index) {
  return app->getTMAP()->GetMap(index);
}

void LoopManager::AdjustOutputVolume(float adjust) {
  app->getRP()->AdjustOutputVolume(adjust);
}

void LoopManager::SetOutputVolume(float set) {
  app->getRP()->SetOutputVolume(set);
}

float LoopManager::GetOutputVolume() {
  return app->getRP()->GetOutputVolume();
}

void LoopManager::AdjustInputVolume(float adjust) {
  app->getRP()->AdjustInputVolume(adjust);
}

void LoopManager::SetInputVolume(float set) {
  app->getRP()->SetInputVolume(set);
}

float LoopManager::GetInputVolume() {
  return app->getRP()->GetInputVolume();
}

void LoopManager::SetLoopVolume(int index, float val) {
  Loop *lp = app->getTMAP()->GetMap(index);
  if (lp != 0) {
    // First, preprocess for smoothing
    Processor *p = GetProcessor(index);
    if (p != 0) 
      p->dopreprocess();

    lp->vol = val;
  }
}

float LoopManager::GetLoopVolume(int index) {
  Loop *lp = app->getTMAP()->GetMap(index);
  if (lp != 0)
    return lp->vol;
  else
    return 1.0;
}

void LoopManager::AdjustLoopVolume(int index, float adjust) {
  Loop *lp = app->getTMAP()->GetMap(index);
  if (lp != 0) {
    lp->dvol += adjust*app->getAUDIO()->GetTimeScale();
    if (lp->dvol < 0.0)
      lp->dvol = 0.0;
  }
}

void LoopManager::SetLoopdVolume(int index, float val) {
  Loop *lp = app->getTMAP()->GetMap(index);
  if (lp != 0)
    lp->dvol = val;
}

float LoopManager::GetLoopdVolume(int index) {
  Loop *lp = app->getTMAP()->GetMap(index);
  if (lp != 0)
    return lp->dvol;
  else
    return 1.0;
}

void LoopManager::SelectPulse(int pulseindex) {
  if (pulseindex == -1) {
    //printf("**Select: No pulse\n");
    curpulseindex = -1;
  } else if (pulseindex < 0 || pulseindex >= MAX_PULSES) {
    printf("CORE: Invalid pulse #%d, ignoring.\n",pulseindex);
    return;
  } else if (pulses[pulseindex] == 0) {
    //printf("New pulse[%d]: %d SUB: %d\n", pulseindex, lastindex, subdivide);
    CreatePulse(lastindex, pulseindex, subdivide);
  } else {
    //printf("Select pulse[%d]\n", pulseindex);
    curpulseindex = pulseindex;
    StripePulseOn(pulses[pulseindex]);
  }
}

// Saves scene XML data
void TriggerMap::Save(Fweelin *app) {
  if (GetSaveStatus() == NO_SAVE) {
    // Scene hash is generated from hash of all loops in the triggermap--
    // so start by saving all loops
    app->getLOOPMGR()->SetAutoLoopSaving(0);
    for (int i = 0; i < app->getCFG()->GetNumTriggers(); i++) 
      app->getLOOPMGR()->SaveLoop(i);

    // Now, we have to wait until all that saving is done.
    // Queue a scene marker event in the save queue
    app->getLOOPMGR()->
      AddToSaveQueue(Event::GetEventByType(T_EV_SceneMarker,1));
  }
};

void TriggerMap::GoSave(Fweelin *app) {
  // All loops in the scene are now hashed and saved

  // Begin our save by generating a scene hash from the loop hashes
  MD5_CTX md5gen;
  MD5_Init(&md5gen);
  for (int i = 0; i < mapsize; i++)
    if (map[i] != 0) {
      if (map[i]->GetSaveStatus() == SAVE_DONE)
	// Update scene hash with hash from this loop
	MD5_Update(&md5gen,map[i]->GetSaveHash(),SAVEABLE_HASH_LENGTH);
      else
	printf("DISK: WARNING: Loop %d not saved yet but scene about to be "
	       "saved!\n",i);
    }
  
  // Done- compute our final hash
  MD5_Final(GetSaveHash(),&md5gen);
  SetSaveStatus(SAVE_DONE);
    
  // Compose filenames & start writing
  char tmp[FWEELIN_OUTNAME_LEN];
  GET_SAVEABLE_HASH_TEXT(GetSaveHash());
  snprintf(tmp,FWEELIN_OUTNAME_LEN,"%s/%s-%s%s",
	   app->getCFG()->GetLibraryPath(),FWEELIN_OUTPUT_SCENE_NAME,
	   hashtext,FWEELIN_OUTPUT_DATA_EXT);
    
  struct stat st;
  printf("DISK: Opening '%s' for saving.\n",tmp);
  if (stat(tmp,&st) == 0) {
    printf("DISK: ERROR: MD5 collision while saving scene- file exists!\n");
  } else {
    // Save scene XML data
    xmlDocPtr ldat = xmlNewDoc((xmlChar *) "1.0");
    if (ldat != 0) {
      const static int XT_LEN = 10;
      char xmltmp[XT_LEN];
      
      // Scene
      ldat->children = xmlNewDocNode(ldat,0,
				     (xmlChar *) FWEELIN_OUTPUT_SCENE_NAME,0);

      // Loops
      for (int i = 0; i < mapsize; i++)
	if (map[i] != 0 && map[i]->GetSaveStatus() == SAVE_DONE) {
	  xmlNodePtr lp = xmlNewChild(ldat->children, 0, 
				      (xmlChar *) FWEELIN_OUTPUT_LOOP_NAME, 0);

	  // Loop index
	  snprintf(xmltmp,XT_LEN,"%d",i);
	  xmlSetProp(lp,(xmlChar *) "loopid",(xmlChar *) xmltmp);

	  // Loop hash (used to find the loop on disk)
	  unsigned char *sh = map[i]->GetSaveHash();
	  GET_SAVEABLE_HASH_TEXT(sh);
	  xmlSetProp(lp,(xmlChar *) "hash",(xmlChar *) hashtext);

	  // Loop volume
	  snprintf(xmltmp,XT_LEN,"%.5f",map[i]->vol);
	  xmlSetProp(lp,(xmlChar *) "volume",(xmlChar *) xmltmp);
	}

      xmlSaveFormatFile(tmp,ldat,1);
      xmlFreeDoc(ldat);

      // Add scene to browser so we can load it
      Browser *br = app->getBROWSER(B_Scene);
      if (br != 0) {
	app->getLOOPMGR()->AddSceneToBrowser(br,tmp);
	br->AddDivisions(FWEELIN_FILE_BROWSER_DIVISION_TIME);
      }

      printf("DISK: Close output.\n");
    }
  }
};

// If we are autosaving, we have to maintain a list of new loops to be saved
void LoopManager::CheckSaveMap() {
  if (needs_saving_stamp != app->getTMAP()->GetLastUpdate()) {
    //printf("Rebuild save map.\n");

    // No, rebuild!
    savequeue = EventManager::DeleteQueue(savequeue);

    // Scan for loops that haven't yet been saved, add them to our list
    TriggerMap *tmap = app->getTMAP();
    int mapsz = tmap->GetMapSize();
    for (int i = 0; i < mapsz; i++) {
      Loop *l = tmap->GetMap(i);
      if (l != 0 && l->GetSaveStatus() == NO_SAVE) {
	// Loop exists but not saved-- add to our map
	LoopListEvent *ll = (LoopListEvent *) 
	  Event::GetEventByType(T_EV_LoopList,1);
	ll->l = l;
	
	EventManager::QueueEvent(&savequeue,ll);
      }
    }

    // Now we've updated map
    needs_saving_stamp = tmap->GetLastUpdate();
  } else {
    //printf("Stamp match: %lf\n",needs_saving_stamp);
  }
}

// Saves loop XML data & prepares to save loop audio
void LoopManager::SetupSaveLoop(Loop *l, int l_idx, FILE **out, 
				AudioBlock **b, 
				AudioBlockIterator **i,
				nframes_t *len) {
  const static nframes_t LOOP_HASH_CHUNKSIZE = 10000;

  if (l->GetSaveStatus() == NO_SAVE) {
    // Now return blocks from this loop to save
    *b = l->blocks;
    if (l->pulse != 0)
      // Loop is syncronized to a pulse- quantize length
      *len = l->pulse->QuantizeLength(l->blocks->GetTotalLen());
    else
      // Take length from blocks
      *len = l->blocks->GetTotalLen();
    *i = 0;
    
    // Generate hash from audio data
    // *** Hopefully this won't take so long- or we may have to split it up
    // as we split up the write phase
    AudioBlockIterator *hashi = new AudioBlockIterator(l->blocks,
						       LOOP_HASH_CHUNKSIZE);
    MD5_CTX md5gen;
    MD5_Init(&md5gen);
    char go = 1;
    char stereo = l->blocks->IsStereo();
    
    do {
      nframes_t pos = hashi->GetTotalLength2Cur(),
	remaining = *len-pos;
      nframes_t num = MIN(LOOP_HASH_CHUNKSIZE,remaining);
      
      sample_t *ibuf[2];
      if (stereo) {
	// Stereo
	hashi->GetFragment(&ibuf[0],&ibuf[1]);
	MD5_Update(&md5gen,ibuf[0],sizeof(sample_t) * num);
	MD5_Update(&md5gen,ibuf[1],sizeof(sample_t) * num);
      } else {
	// Mono
	hashi->GetFragment(&ibuf[0],0);
	MD5_Update(&md5gen,ibuf[0],sizeof(sample_t) * num);
      }
      
      if (remaining <= LOOP_HASH_CHUNKSIZE) {
	// Finished encoding
	go = 0;
      } else
	hashi->NextFragment();
    } while (go);
    
    // Done- compute final hash
    MD5_Final(l->GetSaveHash(),&md5gen);
    l->SetSaveStatus(SAVE_DONE);
    delete hashi;
    
    // Compose filenames & start writing
    char tmp[FWEELIN_OUTNAME_LEN];
    GET_SAVEABLE_HASH_TEXT(l->GetSaveHash());
    snprintf(tmp,FWEELIN_OUTNAME_LEN,"%s/%s-%s%s",
	     app->getCFG()->GetLibraryPath(),FWEELIN_OUTPUT_LOOP_NAME,
	     hashtext,FWEELIN_OUTPUT_AUDIO_EXT);
    
    struct stat st;
    printf("DISK: Opening '%s' for saving.\n",tmp);
    if (stat(tmp,&st) == 0) {
      printf("DISK: ERROR: MD5 collision while saving loop- file exists!\n");

      *b = 0;
      *len = 0;
      *i = 0;
      if (*out != 0) {
	fclose(*out);
	*out = 0;
      }
    } else {
      // Go save!
      *out = fopen(tmp,"wb");
      if (*out == 0) {
	printf("DISK: ERROR: Couldn't open file! Does the folder exist and "
	       "do you have write permission?\n");
	*b = 0;
	*len = 0;
	*i = 0;
	if (*out != 0) {
	  fclose(*out);
	  *out = 0;
	}	
      } else {
	// Add loop to browser so we can load it
	Browser *br = app->getBROWSER(B_Loop);
	if (br != 0) {
	  AddLoopToBrowser(br,tmp);
	  br->AddDivisions(FWEELIN_FILE_BROWSER_DIVISION_TIME);
	}

	// Main file open, now save loop XML data
	snprintf(tmp,FWEELIN_OUTNAME_LEN,"%s/%s-%s%s",
		 app->getCFG()->GetLibraryPath(),FWEELIN_OUTPUT_LOOP_NAME,
		 hashtext,FWEELIN_OUTPUT_DATA_EXT);

	xmlDocPtr ldat = xmlNewDoc((xmlChar *) "1.0");
	if (ldat != 0) {
	  const static int XT_LEN = 10;
	  char xmltmp[XT_LEN];

	  ldat->children = xmlNewDocNode(ldat,0,
	    (xmlChar *) FWEELIN_OUTPUT_LOOP_NAME,0);

	  // # beats
	  snprintf(xmltmp,XT_LEN,"%ld",l->nbeats);
	  xmlSetProp(ldat->children,(xmlChar *) "nbeats",(xmlChar *) xmltmp);

	  // pulse length
	  if (l->pulse == 0)
	    xmlSetProp(ldat->children,(xmlChar *) "pulselen",(xmlChar *) "0");
	  else {
	    snprintf(xmltmp,XT_LEN,"%d",l->pulse->GetLength());
	    xmlSetProp(ldat->children,(xmlChar *) "pulselen",
		       (xmlChar *) xmltmp);
	  }
	  xmlSaveFormatFile(tmp,ldat,1);
	  xmlFreeDoc(ldat);
	}
      }
    }
  } else {
    printf("DISK: WARNING: Loop marked already saved.\n");

    *b = 0;
    *len = 0;
    *i = 0;
    if (*out != 0) {
      fclose(*out);
      *out = 0;
    }
  }
};

// Loads loop XML data & prepares to load loop audio
int LoopManager::SetupLoadLoop(FILE **in, Loop **new_loop, 
			       int l_idx, float l_vol, char *l_filename) {
  // Open up right file and begin loading
  char tmp[FWEELIN_OUTNAME_LEN];
  snprintf(tmp,FWEELIN_OUTNAME_LEN,"%s%s",
	   l_filename,FWEELIN_OUTPUT_AUDIO_EXT);
  printf("DISK: Open loop '%s'\n",tmp);

  *in = fopen(tmp,"rb");
  if (*in == 0) {
    printf("DISK: ERROR: Couldn't open loop '%s'!\n",tmp);
    return 1;
  }
  else {
    // Main file open, now load loop XML data
    snprintf(tmp,FWEELIN_OUTNAME_LEN,"%s%s",
	     l_filename,FWEELIN_OUTPUT_DATA_EXT);
    
    // Create loop data
    *new_loop = new Loop(0,0,1.0,l_vol,0);
    
    xmlDocPtr ldat = xmlParseFile(tmp);
    if (ldat == 0)
      printf("DISK: WARNING: Loop data '%s' invalid or missing!\n"
	     "I will load just the raw audio.\n",tmp);
    else {
      xmlNode *root = xmlDocGetRootElement(ldat);

      // MD5 hash identifies loop (from filename)
      char *md5_ptr = strrchr(l_filename,'-');
      if (md5_ptr != 0 && strlen(md5_ptr+1) == SAVEABLE_HASH_LENGTH*2) {
	md5_ptr++;
	(*new_loop)->SetSaveableHashFromText(md5_ptr);
	//GET_SAVEABLE_HASH_TEXT((*new_loop)->GetSaveHash());
	//printf("md5: %s\n",hashtext);
      } else
	printf("DISK: Loop filename '%s' missing MD5 hash!\n",l_filename);

      // # beats
      xmlChar *n = xmlGetProp(root, (const xmlChar *) "nbeats");
      if (n != 0) {
	(*new_loop)->nbeats = atoi((char *) n);
	xmlFree(n);
      }

      // pulse length
      if ((n = xmlGetProp(root, (const xmlChar *) "pulselen")) != 0) {
	int plen = atoi((char *) n);
	if (plen != 0)
	  // Loop should be syncronized to a pulse of given length-
	  (*new_loop)->pulse = CreatePulse(plen);
		
	xmlFree(n);
      }

      xmlFreeDoc(ldat);
      // Don't call cleanup because another thread may have xml open
      // xmlCleanupParser();
    }
  }

  return 0;
};

void LoopManager::GetWriteBlock(FILE **out, AudioBlock **b, 
				AudioBlockIterator **i,
				nframes_t *len) {
  // If we are autosaving, check that our list is up to date
  if (autosave)
    CheckSaveMap();

  // Do we have a loop to save?
  Event *cur = savequeue,
    *prev = 0;

  int l_idx = 0;  
  char go = 1, advance = 1;
  while (cur != 0 && go) {
    if (cur->GetType() == T_EV_LoopList) {
      // Loop in save queue- does it exist and is it ready to save?
      if ((l_idx = app->getTMAP()->SearchMap(((LoopListEvent *) cur)->l)) == -1
	  || GetStatus(l_idx) == T_LS_Overdubbing
	  || GetStatus(l_idx) == T_LS_Recording) {
	if (l_idx == -1) {
	  printf("DEBUG: Loop no longer exists- abort save!\n");
	  EventManager::RemoveEvent(&savequeue,prev,&cur);
	  advance = 0;
	}

	// If we are overdubbing or recording, just ignore this loop
	// we will come back to it
      } else {
	go = 0; // Found a suitable loop to save- stop!
	advance = 0;
      }
    } else {
      if (cur->GetType() == T_EV_SceneMarker) {
	// Scene marker in queue indicates we need to save a scene-
	if (cur == savequeue) {
	  // No loops are waiting to be saved.. go
          app->getTMAP()->GoSave(app);
	  EventManager::RemoveEvent(&savequeue,prev,&cur);
	  advance = 0;
	} 
      }
    }

    // Move to next item
    if (advance) {
      prev = cur; 
      cur = cur->next;
    }
    else 
      advance = 1;
  }

  if (cur != 0) {
    if (cur->GetType() != T_EV_LoopList) {
      printf("DISK: ERROR: LoopList event type mismatch!\n");
      EventManager::RemoveEvent(&savequeue,prev,&cur);
    } else {
      // Remove from list
      Loop *curl = ((LoopListEvent *) cur)->l;
      EventManager::RemoveEvent(&savequeue,prev,&cur);

      // Open up right files, save data & setup for audio save
      SetupSaveLoop(curl,l_idx,out,b,i,len);
    }
  } else {
    // No loops to save right now
    *b = 0;
    *len = 0;
    *i = 0;
    if (*out != 0) {
      fclose(*out);
      *out = 0;
    }
  }
}

// We receive calls periodically for loading of loops-
void LoopManager::GetReadBlock(FILE **in) {
  // Do we have a loop to load?
  Event *cur = loadqueue;
  if (cur != 0) {
    if (cur->GetType() == T_EV_LoopList) {
      // Open up right files, load data & setup for audio load
      LoopListEvent *ll = (LoopListEvent *) cur;
      if (SetupLoadLoop(in,&ll->l,ll->l_idx,ll->l_vol,ll->l_filename))
	// Not a loop or there was an error in loading
	EventManager::RemoveEvent(&loadqueue,0,&cur);
    } else
      // Not a loop- remove
      EventManager::RemoveEvent(&loadqueue,0,&cur);
  } else {
    // Nothing to load right now
    if (*in != 0) {
      printf("DISK: (Load) Nothing to load- close input!\n");
      fclose(*in);
      *in = 0;
    }
  }
}

void LoopManager::ReadComplete(AudioBlock *b) {
  // Add loop to triggermap and remove from load queue
  Event *cur = loadqueue;
  if (cur == 0 || cur->GetType() != T_EV_LoopList)
    printf("DISK: ERROR: Load list mismatch!\n");
  else {
    if (b == 0)
      printf("DISK: ERROR: .. during load!\n");
    else {
      LoopListEvent *ll = (LoopListEvent *) cur;

      // Put blocks into loop
      ll->l->blocks = b;
      
      // Add loop to our map
      DeleteLoop(ll->l_idx);
      app->getTMAP()->SetMap(ll->l_idx,ll->l);
      lastindex = ll->l_idx; // Set this so we can make a pulse from this loop
    }

    // And remove from load list
    EventManager::RemoveEvent(&loadqueue,0,&cur);
  }
}

void LoopManager::StripePulseOn(Pulse *pulse) {
  app->getBMG()->StripeBlockOn(pulse,app->getAMPEAKS(),
			       app->getAMPEAKSI());
  app->getBMG()->StripeBlockOn(pulse,app->getAUDIOMEM(),
			       app->getAUDIOMEMI());
}

void LoopManager::StripePulseOff(Pulse *pulse) {
  app->getBMG()->StripeBlockOff(pulse,app->getAMPEAKS());
  app->getBMG()->StripeBlockOff(pulse,app->getAUDIOMEM());
}

// Creates a pulse of the given length in the first available slot,
// if none already exists of the right length
Pulse *LoopManager::CreatePulse(nframes_t len) {
  // First, check to see if we have a pulse of the right length-
  int i;
  for (i = 0; i < MAX_PULSES && 
	 (pulses[i] == 0 || pulses[i]->GetLength() != len); i++);
  if (i < MAX_PULSES)
    // Found it, use this pulse
    return pulses[i];
  else {
    // No pulse found of right length-- create a new one
    for (i = 0; i < MAX_PULSES && pulses[i] != 0; i++);
    if (i < MAX_PULSES) {
      app->getRP()->AddChild(pulses[i] = 
			     new Pulse(app,len,0),
			     ProcessorItem::TYPE_HIPRIORITY);
      StripePulseOn(pulses[i]);
      curpulseindex = i;

      return pulses[i];
    } else
      // No space for a new pulse!
      return 0;
  }
};

// Create a time pulse around the specified index
// The length of the loop on the specified index becomes
// a time constant around which other loops center themselves
// subdivide the length of the loop by subdivide to get the core pulse
void LoopManager::CreatePulse(int index, int pulseindex, int sub) {
  Loop *cur = app->getTMAP()->GetMap(index);
  if (cur != 0 && (status[index] == T_LS_Off ||
		   status[index] == T_LS_Playing ||
		   status[index] == T_LS_Overdubbing)) {
    // Set pulse length based on loop length
    nframes_t len = GetLength(index);
    if (len != 0) {
      // Create iterator
      len /= sub; // Length subdivide
      nframes_t startpos = 0;
      // So set the starting pulse position based on where loop is playing
      if (status[index] == T_LS_Playing)
	startpos = ((PlayProcessor *) plist[index])->GetPlayedLength() % len;
      else if (status[index] == T_LS_Overdubbing)
	startpos = ((RecordProcessor *) plist[index])->
	  GetRecordedLength() % len;
      
      app->getRP()->AddChild(cur->pulse = pulses[pulseindex] = 
			     new Pulse(app,len,startpos),
			     ProcessorItem::TYPE_HIPRIORITY);
      StripePulseOn(cur->pulse);
      curpulseindex = pulseindex;
      cur->nbeats = sub; // Set # of beats in the loop

      // Now reconfigure processor on this index to be synced to the new pulse
      if (status[index] == T_LS_Playing) 
	((PlayProcessor *) plist[index])->SyncUp();
      else if (status[index] == T_LS_Overdubbing)
	((RecordProcessor *) plist[index])->SyncUp();
    }
  }
}

// Taps a pulse- starting at the downbeat- if newlen is nonzero, the pulse's
// length is adjusted to reflect the length between taps- and a new pulse
// is created if none exists
void LoopManager::TapPulse(int pulseindex, char newlen) {
  // If more than TIMEOUT_RATIO * the current pulse length
  // frames have passed since the last tap, a new length is not defined
  const static float TAP_NEWLEN_TIMEOUT_RATIO = 2.0,
    // Higher graduation makes tempo more stable against changes
    TAP_NEWLEN_GRADUATION = 0.5,
    // Tolerance for rejecting tap tempo changes- as fraction of current length
    TAP_NEWLEN_REJECT_TOLERANCE = 0.3;

  if (pulseindex >= 0 || pulseindex < MAX_PULSES) {
    Pulse *cur = pulses[pulseindex];
    if (cur == 0) {
      if (newlen) {
	// New pulse- tap now!- set zero length for now
	cur = pulses[pulseindex] = new Pulse(app,0,0);
	cur->stopped = 1;
	cur->prevtap = app->getRP()->GetSampleCnt();
	//cur->SwitchMetronome(1);
	app->getRP()->AddChild(cur,ProcessorItem::TYPE_HIPRIORITY);
	
	StripePulseOn(cur);
	curpulseindex = pulseindex;
      }
    } else {
      // Test position of axis
      char nextdownbeat = 0;
      if (cur->GetPct() >= 0.5)
	nextdownbeat = 1;

      // Old pulse- tap now!
      if (newlen) {
	// Redefine length from tap
	nframes_t oldlen = cur->GetLength(),
	  newtap = app->getRP()->GetSampleCnt(),
	  newlen = newtap - cur->prevtap;

	// .. only if the new length isn't outrageous
	if (oldlen < 64)
	  cur->SetLength(newlen);
	else if (newlen < oldlen * TAP_NEWLEN_TIMEOUT_RATIO) {
	  // 2nd outrageous length check
	  float ratio = (float) MIN(newlen,oldlen)/MAX(newlen,oldlen);
	  if (ratio > 1.0-TAP_NEWLEN_REJECT_TOLERANCE)
	    cur->SetLength((nframes_t) (oldlen*TAP_NEWLEN_GRADUATION +
					newlen*(1-TAP_NEWLEN_GRADUATION)));
	}

	cur->prevtap = newtap;
	//cur->SwitchMetronome(1);
	cur->stopped = 0;
      }

      if (nextdownbeat)
	// Tap to beginning- with wrap
	cur->Wrap();
      else
	// Tap to beginning- no wrap
	cur->SetPos(0);
    }
  } else
    printf("CORE: Invalid pulse #%d, ignoring.\n",pulseindex);
}

void LoopManager::SwitchMetronome(int pulseindex, char active) {
  if (pulseindex >= 0 || pulseindex < MAX_PULSES) {
    Pulse *cur = pulses[pulseindex];
    if (cur != 0)
      cur->SwitchMetronome(active);
    else
      printf("CORE: No pulse at #%d, ignoring.\n",pulseindex);
  } else
    printf("CORE: Invalid pulse #%d, ignoring.\n",pulseindex);
}

void LoopManager::DeletePulse(int pulseindex) {
  if (pulseindex < 0 || pulseindex >= MAX_PULSES) {
    printf("CORE: Invalid pulse #%d, ignoring.\n",pulseindex);
    return;
  }

  if (pulses[pulseindex] != 0) {
    // Stop striping beats from this pulse
    StripePulseOff(pulses[pulseindex]);

    // Erase all loops which are attached to this pulse-- or we'll have
    // references pointing to the deleted pulse
    int nt = app->getCFG()->GetNumTriggers();
    for (int i = 0; i < nt; i++)
      if (GetPulse(i) == pulses[pulseindex]) 
	DeleteLoop(i);

    // Erase this pulse
    app->getRP()->DelChild(pulses[pulseindex]);
    pulses[pulseindex] = 0;
  }
}

Pulse *LoopManager::GetPulse(int index) {
  Loop *lp = GetSlot(index);
  if (lp != 0)
    return lp->pulse;
  else
    return 0;
}

// Move the loop at specified index to another index
// only works if target index is empty
// returns 1 if success
int LoopManager::MoveLoop (int src, int tgt) {
  Loop *srloop = app->getTMAP()->GetMap(src);
  if (srloop != 0) {
    Loop *tgtloop = app->getTMAP()->GetMap(tgt);
    if (tgtloop == 0) {
      app->getTMAP()->SetMap(tgt,srloop);
      app->getTMAP()->SetMap(src,0);
      plist[tgt] = plist[src];
      plist[src] = 0;
      status[tgt] = status[src];
      status[src] = T_LS_Off;
      waitactivate[tgt] = waitactivate[src];
      waitactivate[src] = 0;
      waitactivate_vol[tgt] = waitactivate_vol[src];
      waitactivate_vol[src] = 0.;
      waitactivate_od[tgt] = waitactivate_od[src];
      waitactivate_od[src] = 0;
      waitactivate_od_fb[tgt] = waitactivate_od_fb[src];
      waitactivate_od_fb[src] = 0.;
      for (int i = 0; i < LAST_REC_COUNT; i++)	
	if (lastrecidx[i] == src)
	  lastrecidx[i] = tgt;

      if (lastindex == src)
	lastindex = tgt;
    }
    else 
      return 0;
  }
  else
    return 0;
  
  return 1;
}

// Delete the loop at the specified index..
// Not RT safe!
void LoopManager::DeleteLoop (int index) {
  Loop *lp = app->getTMAP()->GetMap(index);
  if (lp != 0) {
    // First, zero the map at the given index
    // To prevent anybody from attaching to the loop as we delete it
    app->getTMAP()->SetMap(index, 0);
  }

  if (plist[index] != 0) {
    // We have a processor on this loop! Stop it!
    if (status[index] == T_LS_Recording) {
      RecordProcessor *recp = (RecordProcessor *) plist[index];
      recp->AbortRecording();
      AudioBlock *recblk = recp->GetFirstRecordedBlock();
      if (recblk != 0) {
	app->getBMG()->RefDeleted(recblk);
	recblk->DeleteChain();
	if (lp != 0)
	  lp->blocks = 0;
      }

      numrecordingloops--;
    } else if (status[index] == T_LS_Overdubbing)
      numrecordingloops--;

    // Remove the record/play processor
    app->getRP()->DelChild(plist[index]);
    plist[index] = 0;
    status[index] = T_LS_Off;
    waitactivate[index] = 0;
    waitactivate_vol[index] = 0.;
    waitactivate_od[index] = 0;
    waitactivate_od_fb[index] = 0.;
  }
    
  if (lp != 0) {
    if (lp->blocks != 0) {
      // Notify any blockmanagers working on this loop's audio to end!
      app->getBMG()->RefDeleted(lp->blocks);
      lp->blocks->DeleteChain();
    }

    delete lp;
    numloops--;
  }
}

// Trigger the loop at index within the map
// The exact behavior varies depending on what is already happening with
// this loop and the settings passed- see .fweelin.rc
void LoopManager::Activate (int index, float vol, nframes_t ofs, 
			     char overdub, float od_feedback) {
  if (plist[index] != 0) {
    // We have a problem, we already have a processor on this index.
    // Queue the requested activate
    waitactivate[index] = 1;
    waitactivate_vol[index] = vol;
    waitactivate_od[index] = overdub;
    waitactivate_od_fb[index] = od_feedback;
    return;
  }
  
  Loop *lp = app->getTMAP()->GetMap(index);
  if (lp == 0) {
    // Record a new loop
    float *inputvol = &(app->getRP()->inputvol); // Where to get input vol from
    app->getRP()->AddChild(plist[index] =
			   new RecordProcessor(app,app->getISET(),inputvol,
					       GetCurPulse(),
					       app->getAUDIOMEM(),
					       app->getAUDIOMEMI(),
					       app->getCFG()->
					       loop_peaksavgs_chunksize));
    
    numrecordingloops++;
    status[index] = T_LS_Recording;
    // Keep track of this index in our record of last recorded indexes
    for (int i = LAST_REC_COUNT-1; i > 0; i--)
      lastrecidx[i] = lastrecidx[i-1];
    lastrecidx[0] = index;
  } else {
    // A loop exists at that index
    if (overdub) {
      // Overdub
      float *inputvol = &(app->getRP()->inputvol); // Get input vol from main
      app->getRP()->AddChild(plist[index] = 
			     new RecordProcessor(app,
						 app->getISET(),inputvol,
						 lp,vol,ofs,od_feedback));
      numrecordingloops++;
      status[index] = T_LS_Overdubbing;
    } else {
      // Play
      app->getRP()->AddChild(plist[index] = 
			     new PlayProcessor(app,lp,vol,ofs));
      status[index] = T_LS_Playing;
    }
  }
}

void LoopManager::Deactivate (int index) {
  if (plist[index] == 0) {
    // We have a problem, there is supposed to be a processor here!
    printf("Nothing happening on index %d to deactivate\n",index);
    return;
  }
  
  // If we recorded something new to this index, store it in the map
  if (status[index] == T_LS_Recording) {
    // *** Perhaps make a function in RecordProcessor called
    // ** 'createloop'.. which does the encapsulation from the blocks
    Pulse *curpulse = 0;
    if (curpulseindex != -1)
      curpulse = pulses[curpulseindex];

    // Adjust newloop volume so that it will match the volume
    // it was heard as-- since output volume does not scale
    // the initial monitor but does scale loops, we need to adjust
    float adjustednewloopvol = newloopvol / GetOutputVolume(); 
      
    Loop *newlp = new
      Loop(((RecordProcessor *) plist[index])->GetFirstRecordedBlock(),
	   curpulse,1.0,adjustednewloopvol,
	   ((RecordProcessor *) plist[index])->GetNBeats());
    app->getTMAP()->SetMap(index, newlp);
    numloops++;
    lastindex = index;

    // Record processor will broadcast when it is ready to end!
    ((RecordProcessor *) plist[index])->End();
  } else if (status[index] == T_LS_Overdubbing) {
    // Overdubbing record processor will end immediately and broadcast 
    // EndRecord event
    ((RecordProcessor *) plist[index])->End();
  } else if (status[index] == T_LS_Playing) {
    // Stop playing/overdubbing
    app->getRP()->DelChild(plist[index]);
    plist[index] = 0;
    status[index] = T_LS_Off;    
  }
}

void LoopManager::SaveLoop(int index) {
  Loop *l = app->getTMAP()->GetMap(index);
  if (l != 0)
    l->Save(app);
};

void LoopManager::SaveScene() {
  TriggerMap *tm = app->getTMAP();
  if (tm != 0)
    tm->Save(app);
};

// Load loop from disk into the given index
void LoopManager::LoadLoop(char *filename, int index, float vol) {
  DeleteLoop(index);
  AddLoopToLoadQueue(filename,index,vol);
};

// Load scene from disk
void LoopManager::LoadScene(char *filename) {
  // Load XML data for scene
  char tmp[FWEELIN_OUTNAME_LEN];
  snprintf(tmp,FWEELIN_OUTNAME_LEN,"%s%s",
	   filename,FWEELIN_OUTPUT_DATA_EXT);

  xmlDocPtr dat = xmlParseFile(tmp);
  if (dat == 0)
    printf("DISK: ERROR: Scene data '%s' invalid or missing!\n",tmp);
  else {
    xmlNode *root = xmlDocGetRootElement(dat);
    if (!root || !root->name ||
	xmlStrcmp(root->name,(const xmlChar *) FWEELIN_OUTPUT_SCENE_NAME))
      printf("DISK: ERROR: Scene data '%s' bad format!\n",tmp);
    else {
      for (xmlNode *cur_node = root->children; cur_node != NULL; 
	   cur_node = cur_node->next) {
	if ((!xmlStrcmp(cur_node->name, 
			(const xmlChar *) FWEELIN_OUTPUT_LOOP_NAME))) {
	  // Loop within scene-- read
	  int l_idx = loadloopid;
	  float vol = 1.0;

	  // Loopid
	  xmlChar *n = xmlGetProp(cur_node, (const xmlChar *) "loopid");
	  if (n != 0) {
	    l_idx = atoi((char *) n);
	    xmlFree(n);
	  }

	  // Volume
	  if ((n = xmlGetProp(cur_node, (const xmlChar *) "volume")) != 0) {
	    vol = atof((char *) n);
	    xmlFree(n);
	  }

	  // Hash
	  if ((n = xmlGetProp(cur_node, (const xmlChar *) "hash")) != 0) {
	    // Compose loop filename from hash
	    snprintf(tmp,FWEELIN_OUTNAME_LEN,"%s/%s-%s",
		     app->getCFG()->GetLibraryPath(),
		     FWEELIN_OUTPUT_LOOP_NAME,n);
	    xmlFree(n);

	    // Load the loop into the specified index
	    printf(" (loopid %d vol %.5f filename %s)\n",
	    	   l_idx,vol,tmp);
	    LoadLoop(tmp,l_idx,vol);
	    // sleep(2);
	  } else 
	    printf("DISK: Scene definition for loop (id %d) has missing "
		   "hash!\n",l_idx);
	}
      }
    }
  }

  xmlFreeDoc(dat);
  // Don't call cleanup because another thread may have xml open
  // xmlCleanupParser();
};

void LoopManager::ReceiveEvent(Event *ev, EventProducer *from) {
  switch (ev->GetType()) {
  case T_EV_EndRecord :
    // Recording has ended on one of the RecordProcessors- find it!
    for (int i = 0; i < app->getTMAP()->GetMapSize(); i++) 
      if (plist[i] == from) {
	// Should we keep this recording
	if (!((EndRecordEvent *) ev)->keeprecord) {
	  DeleteLoop(i); // Nope!
	}
	else {
	  nframes_t playofs = 0;
	  if (status[i] == T_LS_Recording) {
	    // Adjust number of beats in loop based on the recording
	    app->getTMAP()->GetMap(i)->nbeats = 
	      ((RecordProcessor *) plist[i])->GetNBeats();
	  }
	  else if (status[i] == T_LS_Overdubbing) {
	    // Start play at position where overdub left off
	    playofs = ((RecordProcessor *) plist[i])->GetRecordedLength();
	  }

	  // Remove recordprocessor from chain
	  app->getRP()->DelChild(plist[i]);
	  numrecordingloops--;
	  status[i] = T_LS_Off;
	  plist[i] = 0;

	  // Check if we need to activate a playprocessor
	  if (waitactivate[i]) {
	    waitactivate[i] = 0;
	    // Activate is not RT safe (new processor alloc)
	    // So event thread better be nonRT!
	    Activate(i,waitactivate_vol[i],playofs,waitactivate_od[i],
		     waitactivate_od_fb[i]);
	  }
	}
      }
    break;

  case T_EV_ToggleDiskOutput :
    {
      // OK!
      if (CRITTERS)
	printf("CORE: Received ToggleDiskOutputEvent\n");
      app->ToggleDiskOutput();
    }
    break;

  case T_EV_SetAutoLoopSaving :
    {
      SetAutoLoopSavingEvent *sev = (SetAutoLoopSavingEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received SetAutoLoopSavingEvent (%s)\n",
	       (sev->save ? "on" : "off"));
      SetAutoLoopSaving(sev->save);
    }
    break;

  case T_EV_SaveLoop :
    {
      SaveLoopEvent *sev = (SaveLoopEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received SaveLoopEvent (%d)\n",sev->index);
      SaveLoop(sev->index);
    }
    break;

  case T_EV_SaveScene :
    {
      // OK!
      if (CRITTERS)
	printf("CORE: Received SaveSceneEvent\n");
      SaveScene();
    }
    break;

  case T_EV_SetLoadLoopId :
    {
      SetLoadLoopIdEvent *sev = (SetLoadLoopIdEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received SetLoadLoopIdEvent (%d)\n",sev->index);
      loadloopid = sev->index;
    }
    break;

  case T_EV_SlideMasterInVolume :
    {
      SlideMasterInVolumeEvent *vev = (SlideMasterInVolumeEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received SlideMasterInVolumeEvent(%f)\n", vev->slide);
      AdjustInputVolume(vev->slide);
    }
    break;

  case T_EV_SlideMasterOutVolume :
    {
      SlideMasterOutVolumeEvent *vev = (SlideMasterOutVolumeEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received SlideMasterOutVolumeEvent(%f)\n", vev->slide);
      AdjustOutputVolume(vev->slide);
    }
    break;

  case T_EV_SlideInVolume :
    {
      SlideInVolumeEvent *vev = (SlideInVolumeEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received SlideInVolumeEvent(%d: %f)\n", vev->input, vev->slide);
      app->getISET()->AdjustInputVol(vev->input-1, vev->slide);
    }
    break;

  case T_EV_SetMasterInVolume :
    {
      SetMasterInVolumeEvent *vev = (SetMasterInVolumeEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received SetMasterInVolumeEvent(%f)\n", vev->vol);
      SetInputVolume(vev->vol);
    }
    break;

  case T_EV_SetMasterOutVolume :
    {
      SetMasterOutVolumeEvent *vev = (SetMasterOutVolumeEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received SetMasterOutVolumeEvent(%f)\n", vev->vol);
      SetOutputVolume(vev->vol);
    }
    break;

  case T_EV_SetInVolume :
    {
      SetInVolumeEvent *vev = (SetInVolumeEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received SetInVolumeEvent(%d: %f)\n", vev->input, vev->vol);
      app->getISET()->SetInputVol(vev->input-1, vev->vol);
    }
    break;

  case T_EV_ToggleInputRecord :
    {
      ToggleInputRecordEvent *vev = (ToggleInputRecordEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received ToggleInputRecordEvent(%d)\n", vev->input);
      app->getISET()->SelectInput(vev->input-1,(app->getISET()->InputSelected(vev->input-1) == 0 ? 1 : 0));
    }
    break;

  case T_EV_DeletePulse :
    {
      DeletePulseEvent *dev = (DeletePulseEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received DeletePulse(%d)\n", dev->pulse);
      DeletePulse(dev->pulse);
    }
    break;

  case T_EV_SelectPulse :
    {
      SelectPulseEvent *sev = (SelectPulseEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received SelectPulse(%d)\n", sev->pulse);
      SelectPulse(sev->pulse);
    }
    break;

  case T_EV_TapPulse :
    {
      TapPulseEvent *tev = (TapPulseEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received TapPulse(%d) %s\n", tev->pulse,
	       (tev->newlen ? "[new length]" : ""));
      TapPulse(tev->pulse,tev->newlen);
    }
    break;

  case T_EV_SwitchMetronome :
    {
      SwitchMetronomeEvent *swev = (SwitchMetronomeEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received SwitchMetronome(%d) %s\n", swev->pulse,
	       (swev->metronome ? "[on]" : "[off]"));
      SwitchMetronome(swev->pulse,swev->metronome);
    }
    break;

  case T_EV_SetTriggerVolume :
    {
      SetTriggerVolumeEvent *laev = (SetTriggerVolumeEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received SetTriggerVolume(%d,%f)\n", laev->index, 
	       laev->vol);
      SetTriggerVol(laev->index,laev->vol);
    }
    break;

  case T_EV_SlideLoopAmp :
    {
      SlideLoopAmpEvent *laev = (SlideLoopAmpEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received SlideLoopAmp(%d,%f)\n", laev->index, 
	       laev->slide);
      AdjustLoopVolume(laev->index,laev->slide);
    }
    break;

  case T_EV_SetLoopAmp :
    {
      SetLoopAmpEvent *laev = (SetLoopAmpEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received SetLoopAmp(%d,%f)\n", laev->index, laev->amp);
      SetLoopVolume(laev->index,laev->amp);
    }
    break;

  case T_EV_TriggerLoop :
    {
      TriggerLoopEvent *tev = (TriggerLoopEvent *) ev;
      int index = tev->index;
      float vol = tev->vol;
      char od = tev->od;
      float od_fb = tev->od_fb;

      // OK!
      if (CRITTERS) {
	printf("CORE: Received TriggerLoop(%d,%.2f)", index, vol);
	if (od) 
	  printf(" [overdub] (feedback %.2f)\n",od_fb);
	else
	  printf("\n");
      }

      if (GetStatus(index) == T_LS_Recording || 
	  GetStatus(index) == T_LS_Overdubbing ||
	  (GetStatus(index) == T_LS_Playing && od == 1)) {
	// Stop-start case
	nframes_t ofs = 0;
	if (GetStatus(index) == T_LS_Overdubbing && od == 1) {
	  // Don't allow retrigger from overdub to overdub-- override to play
	  od = 0;
	} else if (GetStatus(index) == T_LS_Playing) {
	  // Play->overdub case- start overdub where play left off
	  ofs = ((PlayProcessor *) plist[index])->GetPlayedLength();
	}

	Deactivate(index);                // Stop
	Activate(index,vol,ofs,od,od_fb); // Start
      } else if (IsActive(index)) {
	// Stop case (play and no overdub)
	Deactivate(index);
      }
      else {
	// Start case (record)
	Activate(index,vol,0,od,od_fb);
      }
    }
    break;

  case T_EV_MoveLoop :
    {
      MoveLoopEvent *mev = (MoveLoopEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received MoveLoop(%d->%d)\n", mev->oldloopid, 
	       mev->newloopid);
      MoveLoop(mev->oldloopid,mev->newloopid);
    }
    break;

  case T_EV_EraseLoop :
    {
      EraseLoopEvent *eev = (EraseLoopEvent *) ev;

      // OK!
      if (CRITTERS)
	printf("CORE: Received EraseLoop(%d)\n", eev->index);
      DeleteLoop(eev->index);
    }
    break;

  case T_EV_EraseAllLoops :
    {
      // OK!
      if (CRITTERS)
	printf("CORE: Received EraseAllLoops\n");

      // Erase all loops!
      for (int i = 0; i < app->getCFG()->GetNumTriggers(); i++) 
	DeleteLoop(i);
      // And all pulses!
      for (int i = 0; i < MAX_PULSES; i++)
	DeletePulse(i);
    }
    break;

  case T_EV_SlideLoopAmpStopAll :
    {
      // OK!
      if (CRITTERS)
	printf("CORE: Received SlideLoopAmpStopAll\n");

      for (int i = 0; i < app->getCFG()->GetNumTriggers(); i++) 
	SetLoopdVolume(i,1.0);
    }
    break;

  default:
    break;
  }
}

int Fweelin::go()
{
  running = 1;

  // Broadcast start session event!
  Event *proto = Event::GetEventByType(T_EV_StartSession);
  if (proto == 0) {
    printf("GO: Can't get start event prototype!\n");
  } else {
    Event *cpy = (Event *) proto->RTNew();
    if (cpy == 0)
      printf("CORE: WARNING: Can't send event- RTNew() failed\n");
    else 
      emg->BroadcastEventNow(cpy, this);
  }

  // Encourage the user!
  printf("\nOKIE DOKIE, KIDDO!\n");

  // Now just wait.. the threads will take care of everything
  while (keys->IsActive()) {
    usleep(100000);
  };
  
  // Cleanup
  vid->close();
  keys->close();
  midi->close();
  audio->close();
  delete vid;
  delete keys;
  delete midi;
  delete audio;
  delete iset;
  delete abufs;

#if USE_FLUIDSYNTH
  delete fluidp;
#endif

  printf("MAIN: end stage 1\n");

  //sleep(1);

  // Manually reset audio memory to its original state-
  // not preallocated!
  getAMPEAKS()->SetupPreallocated(0,Preallocated::PREALLOC_BASE_INSTANCE);
  getAMAVGS()->SetupPreallocated(0,Preallocated::PREALLOC_BASE_INSTANCE);
  audiomem->SetupPreallocated(0,Preallocated::PREALLOC_BASE_INSTANCE);

  // And main classes..
  delete tmap;
  printf(" 1\n");
  delete loopmgr;
  printf(" 2\n");
  delete rp; 
  printf(" 3\n");
  delete bmg;
  printf(" 4\n");

  printf("MAIN: end stage 2\n");

  //::delete audiomem;
  delete[] scope;

  printf("MAIN: end stage 3\n");

  // Delete preallocated type managers
  delete pre_audioblock;
  delete pre_extrachannel;
  delete pre_timemarker;

  printf("MAIN: end stage 4\n");
  //sleep(2);

  delete cfg;
  printf(" 1\n");
  //sleep(2);
  delete emg;
  //sleep(2);
  printf(" 2\n");
  delete mmg;

  SDL_Quit();
  printf("MAIN: end\n");

  return 0;
}

BED_MarkerPoints *Fweelin::getAMPEAKSPULSE() { 
  AudioBlock *peaks = getAMPEAKS();
  if (peaks != 0)
    return dynamic_cast<BED_MarkerPoints *>
      (getAMPEAKS()->GetExtendedData(T_BED_MarkerPoints));
  else
    return 0;
};

AudioBlock *Fweelin::getAMPEAKS() { 
  return 
    dynamic_cast<PeaksAvgsManager *>(bmg->GetBlockManager(audiomem,
							  T_MC_PeaksAvgs))->
    GetPeaks();
};

AudioBlock *Fweelin::getAMAVGS() { 
  return 
    dynamic_cast<PeaksAvgsManager *>(bmg->GetBlockManager(audiomem,
							  T_MC_PeaksAvgs))->
    GetAvgs();
  };

AudioBlockIterator *Fweelin::getAMPEAKSI() { 
  return
    dynamic_cast<PeaksAvgsManager *>(bmg->GetBlockManager(audiomem,
							  T_MC_PeaksAvgs))->
    GetPeaksI();
};

AudioBlockIterator *Fweelin::getAMAVGSI() { 
  return
    dynamic_cast<PeaksAvgsManager *>(bmg->GetBlockManager(audiomem,
							  T_MC_PeaksAvgs))->
    GetAvgsI();
};

AudioBlockIterator *Fweelin::getAUDIOMEMI() {
  return amrec->GetIterator();
};

Browser *Fweelin::GetBrowserFromConfig(BrowserItemType b) {
  FloDisplay *cur = cfg->displays;
  while (cur != 0) {
    if (cur->GetFloDisplayType() == FD_Browser &&
	((Browser *) cur)->GetType() == b)
      return (Browser *) cur;
    cur = cur->next;
  }
  return 0;
};

void Fweelin::ToggleDiskOutput()
{
  if (vs->GetStatus() == VorbisStreamer::STATUS_STOPPED) {
    // Create appropriate filename for output
    char tmp[FWEELIN_OUTNAME_LEN];
    char go = 1;
    do {
      snprintf(tmp,FWEELIN_OUTNAME_LEN,"%s/%s%d%s",
	       cfg->GetLibraryPath(),FWEELIN_OUTPUT_STREAM_NAME,
	       writenum,FWEELIN_OUTPUT_AUDIO_EXT);
      struct stat st;
      printf("DISK: Opening '%s' for streaming.\n",tmp);
      if (stat(tmp,&st) == 0) {
	printf("DISK: File exists, trying another.\n");
	writenum++;
      }
      else
	go = 0;
    } while (go);

    // Compose filename & start writing
    strcpy(writename,tmp);
    snprintf(timingname,FWEELIN_OUTNAME_LEN,"%s/%s%d%s",
	     cfg->GetLibraryPath(),FWEELIN_OUTPUT_STREAM_NAME,
	     writenum,FWEELIN_OUTPUT_TIMING_EXT);
    vs->StartWriting(writename,timingname);
  } else {
    // Stop disk output
    vs->StopWriting();
    strcpy(writename,"");
    strcpy(timingname,"");

    // Advance to next logical filename
    writenum++;
  }
};

int Fweelin::setup()
{
  char tmp[255];

  // Keep all memory inline
  mlockall(MCL_CURRENT | MCL_FUTURE);

  if (!XInitThreads()) {
    printf("MAIN: ERROR: FreeWheeling requires threaded Xlib support\n");
    return 0;
  }

  /* Initialize SDL- this happens here because it is common to video, keys &
     config */ 
  // SDL_INIT_NOPARACHUTE
  if ( SDL_Init(SDL_INIT_VIDEO | SDL_INIT_EVENTTHREAD) < 0) {
    printf("MAIN: ERROR: Can't initialize SDL: %s\n",SDL_GetError());
    return 0;
  }
  atexit(SDL_Quit);

  // Memory manager
  mmg = new MemoryManager();

  // Load configuration from .rc file
  cfg = new FloConfig(this);

  // Create system variables so that config will have them first!
  UserVariable *tmpv;
  tmpv = cfg->AddEmptyVariable("BROWSE_loop");
  tmpv->type = T_int;
  *tmpv = (int) B_Loop;
  tmpv = cfg->AddEmptyVariable("BROWSE_scene");
  tmpv->type = T_int;
  *tmpv = (int) B_Scene;
  cfg->AddEmptyVariable("SYSTEM_num_midi_outs");
  cfg->AddEmptyVariable("SYSTEM_midi_transpose");
  cfg->AddEmptyVariable("SYSTEM_master_in_volume");
  cfg->AddEmptyVariable("SYSTEM_master_out_volume");
  cfg->AddEmptyVariable("SYSTEM_cur_pitchbend");
  cfg->AddEmptyVariable("SYSTEM_bender_tune");
  cfg->AddEmptyVariable("SYSTEM_cur_limiter_gain");
  cfg->AddEmptyVariable("SYSTEM_audio_cpu_load");
#if USE_FLUIDSYNTH
  cfg->AddEmptyVariable("SYSTEM_fluidsynth_enabled");

  tmpv = cfg->AddEmptyVariable("BROWSE_fluidsynth");
  tmpv->type = T_int;
  *tmpv = (int) B_FluidPatch;
#endif
  cfg->AddEmptyVariable("SYSTEM_num_help_pages");
  cfg->AddEmptyVariable("SYSTEM_num_loops_in_map");
  cfg->AddEmptyVariable("SYSTEM_num_recording_loops_in_map");
  for (int i = 0; i < LAST_REC_COUNT; i++) {
    sprintf(tmp,"SYSTEM_loopid_lastrecord_%d",i);
    cfg->AddEmptyVariable(tmp);
  }

  // Now parse and setup config
  cfg->Parse();

  // Event manager
  emg = new EventManager();

  vid = new VideoIO(this);
  if (vid->activate()) {
    printf("MAIN: ERROR: Can't start video handler!\n");
    return 1;
  }
  while (!vid->IsActive())
    usleep(100000);

  abufs = new AudioBuffers(this);
  iset = new InputSettings(this,abufs->numins);
  audio = new AudioIO(this);
  if (audio->open()) {
    printf("MAIN: ERROR: Can't start system level audio!\n");
    return 1;
  }  
  fragmentsize = audio->getbufsz();
  printf("MAIN: Core block size: %d\n",fragmentsize);

  // Linkup to browsers
  browsers = new Browser *[(int) B_Last];
  memset(browsers,0,sizeof(Browser *) * (int) B_Last);

  // FluidSynth
#if USE_FLUIDSYNTH
  // Create synth
  printf("MAIN: Creating integrated FluidSynth.\n");
  fluidp = new FluidSynthProcessor(this,cfg->GetFluidStereo());

  // Setup patch browser, if defined in config
  {
    Browser *br = GetBrowserFromConfig(B_FluidPatch);
    if (br != 0) {
      browsers[B_FluidPatch] = br;
      br->Setup(this,fluidp);
    }
  }
  
  // Setup patch names
  fluidp->SetupPatches();
#endif

  // Setup sample buffer for visual scope
  scope = new sample_t[fragmentsize];
  scope_len = fragmentsize;

  // Block manager
  bmg = new BlockManager(this);

  // Preallocated type managers
  pre_audioblock = new PreallocatedType(mmg, ::new AudioBlock(),
					sizeof(AudioBlock),
					FloConfig::
					NUM_PREALLOCATED_AUDIO_BLOCKS);
  if (cfg->IsStereoMaster()) 
    // Only preallocate for stereo blocks if we are running in stereo
    pre_extrachannel = new PreallocatedType(mmg, ::new BED_ExtraChannel(),
					    sizeof(BED_ExtraChannel),
					    FloConfig::
					    NUM_PREALLOCATED_AUDIO_BLOCKS);
  else 
    pre_extrachannel = 0;
  pre_timemarker = new PreallocatedType(mmg,::new TimeMarker(),
					sizeof(TimeMarker));

  rp = new RootProcessor(this,iset);
  // Add monitor mix
  float *inputvol = &(rp->inputvol); // Where to get input vol from
  rp->AddChild(new PassthroughProcessor(this,iset,inputvol),
	       ProcessorItem::TYPE_GLOBAL);
  // Add disk writer
  rp->AddChild(vs = new VorbisStreamer(this),
    ProcessorItem::TYPE_FINAL);
  writenum = 1;
  strcpy(writename,"");
  strcpy(timingname,"");

  // Fixed audio memory
  nframes_t memlen = (nframes_t) (audio->get_srate() * 
				  cfg->AUDIO_MEMORY_LEN),
    scopelen = cfg->GetScopeSampleLen(),
    chunksize = memlen/scopelen;
  // Note here we bypass using Preallocated RTNew because we want
  // a single block of our own size, not many preallocated blocks
  // chained together..
  audiomem = ::new AudioBlock(memlen);
  if (audiomem == 0) {
    printf("CORE: ERROR: Can't create audio memory!\n");
    exit(1);
  }
  audiomem->Zero();
  // If we are running in stereo, create a custom right channel to match
  // our left channel audio memory
  if (cfg->IsStereoMaster()) {
    BED_ExtraChannel *audiomem_r = ::new BED_ExtraChannel(memlen);
    if (audiomem_r == 0) {
      printf("CORE: ERROR: Can't create audio memory (right channel)!\n");
      exit(1);
    }
    
    audiomem->AddExtendedData(audiomem_r);
  }

  // So we have to set a pointer manually to the manager..
  // Because some functions depend on using audiomem as a basis
  // to access RTNew
  audiomem->SetupPreallocated(pre_audioblock,
			      Preallocated::PREALLOC_BASE_INSTANCE);

  // Begin recording into audio memory (use mono/stereo memory as appropriate)
  amrec = new RecordProcessor(this,iset,inputvol,audiomem,
			      cfg->IsStereoMaster());
  if (amrec == 0) {
    printf("CORE: ERROR: Can't create core RecordProcessor!\n");
    exit(1);
  }
  rp->AddChild(amrec,ProcessorItem::TYPE_HIPRIORITY);
  // Compute running peaks and averages from audio mem (for scope)
  AudioBlock *peaks = ::new AudioBlock(scopelen),
    *avgs = ::new AudioBlock(scopelen);
  if (peaks == 0 || avgs == 0) {
    printf("CORE: ERROR: Can't create peaks/averages memory!\n");
    exit(1);
  }
  peaks->Zero();
  avgs->Zero();
  // **BUG-- small leak-- the above two are never deleted
  peaks->SetupPreallocated(pre_audioblock,
			   Preallocated::PREALLOC_BASE_INSTANCE);
  avgs->SetupPreallocated(pre_audioblock,
			  Preallocated::PREALLOC_BASE_INSTANCE);
  audiomem->AddExtendedData(new BED_PeaksAvgs(peaks,avgs,chunksize));
  bmg->PeakAvgOn(audiomem,amrec->GetIterator());

  int nt = cfg->GetNumTriggers();
  tmap = new TriggerMap(nt);
  loopmgr = new LoopManager(this);

  // Setup loop & scene browsers, if defined in config
  {
    Browser *br = GetBrowserFromConfig(B_Loop);
    if (br != 0) {
      browsers[B_Loop] = br;
      br->Setup(this,loopmgr);
    }
    loopmgr->SetupLoopBrowser();

    br = GetBrowserFromConfig(B_Scene);
    if (br != 0) {
      browsers[B_Scene] = br;
      br->Setup(this,loopmgr);
    }
    loopmgr->SetupSceneBrowser();
  }

  keys = new KeyIO(this);
  midi = new MidiIO(this);

  if (keys->activate()) {
    printf("(start) cant start keyboard handler\n");
    return 1;
  }
  if (midi->activate()) {
    printf("(start) cant start midi\n");
    return 1;
  }

  // Linkup system variables
  cfg->LinkSystemVariable("SYSTEM_num_midi_outs",T_int,
			  (char *) &(cfg->midiouts));
  cfg->LinkSystemVariable("SYSTEM_midi_transpose",T_int,
			  (char *) &(cfg->transpose));
  cfg->LinkSystemVariable("SYSTEM_master_in_volume",T_float,
			  (char *) &(rp->inputvol));
  cfg->LinkSystemVariable("SYSTEM_master_out_volume",T_float,
			  (char *) &(rp->outputvol));
  cfg->LinkSystemVariable("SYSTEM_cur_pitchbend",T_int,
			  (char *) &(midi->curbender));
  cfg->LinkSystemVariable("SYSTEM_bender_tune",T_int,
			  (char *) &(midi->bendertune));
  cfg->LinkSystemVariable("SYSTEM_cur_limiter_gain",T_float,
			  (char *) &(rp->curlimitvol));
  cfg->LinkSystemVariable("SYSTEM_audio_cpu_load",T_float,
			  (char *) &(audio->cpuload));
#if USE_FLUIDSYNTH
  cfg->LinkSystemVariable("SYSTEM_fluidsynth_enabled",T_char,
			  (char *) &(fluidp->enable));
#endif
  cfg->LinkSystemVariable("SYSTEM_num_help_pages",T_int,
			  (char *) &(vid->numhelppages));
  cfg->LinkSystemVariable("SYSTEM_num_loops_in_map",T_int,
			  (char *) &(loopmgr->numloops));
  cfg->LinkSystemVariable("SYSTEM_num_recording_loops_in_map",T_int,
			  (char *) &(loopmgr->numrecordingloops));
  for (int i = 0; i < LAST_REC_COUNT; i++) {
    sprintf(tmp,"SYSTEM_loopid_lastrecord_%d",i);
    cfg->LinkSystemVariable(tmp,T_int,
			    (char *) &(loopmgr->lastrecidx[i]));
  }
  for (int i = 0; i < iset->numins; i++) {
    snprintf(tmp,255,"SYSTEM_in_%d_volume",i+1);
    cfg->LinkSystemVariable(tmp,T_float,
			    (char *) &(iset->invols[i]));
    snprintf(tmp,255,"SYSTEM_in_%d_peak",i+1);
    cfg->LinkSystemVariable(tmp,T_float,
			    (char *) &(iset->inpeak[i]));
    snprintf(tmp,255,"SYSTEM_in_%d_record",i+1);
    cfg->LinkSystemVariable(tmp,T_char,
			    (char *) &(iset->selins[i]));
  }

  // Finally, final Config start
  cfg->Start();

  // Now start signal processing
  if (audio->activate(rp)) {
    printf("MAIN: Error with signal processing start!\n");
    return 1;
  }

  return 0;
}
