# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright 2013 Canonical
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.

"""friends_app autopilot tests."""

import os

from autopilot.testcase import AutopilotTestCase
from autopilot.input import Mouse, Touch, Pointer
from autopilot.display import Display
from autopilot.platform import model
from testtools.matchers import Equals
from autopilot.matchers import Eventually

from friends_app.emulators.timeline_view import TimelineView
from friends_app.emulators.post_view import PostView
from friends_app.emulators.friends_app_utils import FriendsAppUtils


class FriendsAppTestCase(AutopilotTestCase):
    """A common test case class that provides several useful methods for
    friends-app tests."""
    if model() == 'Desktop':
        scenarios = [
            ('with mouse', dict(input_device_class=Mouse)),
        ]
    else:
        scenarios = [
            ('with touch', dict(input_device_class=Touch)),
        ]

    local_location = "../../src/friends-app"

    def setUp(self):
        self.pointing_device = Pointer(self.input_device_class.create())
        super(FriendsAppTestCase, self).setUp()
        self.unlock_screen()
        if os.path.exists(self.local_location):
            self.launch_test_local()
        else:
            self.launch_test_installed()

        main_view = self.friends_app_utils.get_main_view()
        self.assertThat(main_view.visible, Eventually(Equals(True)))

    def unlock_screen(self):
        input_device = Touch.create()
        pointing_device = Pointer(input_device)
        x, y, w, h = Display.create().get_screen_geometry(0)
        tx = x + w
        ty = y + (h // 2)
        pointing_device.drag(tx, ty, tx // 2, ty)

    def launch_test_local(self):
        self.app = self.launch_test_application(
            self.local_location, app_type='qt')

    def launch_test_installed(self):
        if model() == 'Desktop':
            self.app = self.launch_test_application(
                "friends-app")
        else:
            self.app = self.launch_test_application(
                "friends-app", "--desktop_file_hint=/usr/share/applications/"
                "friends-app.desktop", app_type='qt')

    @property
    def friends_app_utils(self):
        return FriendsAppUtils(self.app)

    @property
    def timeline_view(self):
        return TimelineView(self.app)

    @property
    def post_view(self):
        return PostView(self.app)

    def reveal_toolbar(self):
        toolbar = self.friends_app_utils.get_toolbar()
        self.assertThat(toolbar.animating, Eventually(Equals(False)))

        if toolbar.opened:
            # Toolbar already open
            return

        qml_view = self.friends_app_utils.get_qml_view()
        x, y, w, h = toolbar.globalRect
        x_line = qml_view.x + qml_view.width * 0.5
        start_y = qml_view.y + qml_view.height - 1
        stop_y = start_y - 2 * h
        self.pointing_device.move_to_object(toolbar)
        self.pointing_device.drag(x_line, start_y, x_line, stop_y)
        self.assertThat(toolbar.state, Eventually(Equals('spread')))
