/* mt -- control magnetic tape drive operation
   Copyright (C) 1991, 1992, 1995 Free Software Foundation, Inc.

   Changes for use with ftape/zftape (C) 1997 Claus-Justus Heine

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
   */

 char ftmt_src[] = "$RCSfile: mt.c,v $";
 char ftmt_rev[] = "$Revision: 1.8 $";
 char ftmt_dat[] = "$Date: 2000/07/21 07:47:13 $";

/* If -f is not given, the environment variable TAPE is used;
   if that is not set, a default device defined in sys/mtio.h is used.
   The device must be either a character special file or a remote
   tape drive with the form "[user@]system:path".
   The default count is 1.  Some operations ignore it.

   Exit status:
   0	success
   1	invalid operation or device name
   2	operation failed

   Operations (unique abbreviations are accepted):
   reset        reset drive in case of problems
   fsf		Forward space COUNT files.
		Tape is positioned on the first block of the file.
   bsf		Backward space COUNT files.
		Tape is positioned on the first block of the file.
   fsr		Forward space COUNT records.
   bsr		Backward space COUNT records.
   eof, weof	Write COUNT EOF marks at current position on tape.
   rewind	Rewind the tape.
   offline, rewoffl
		Rewind the tape and, if applicable, unload the tape.
   nop          No operation, set status
   retension	Rewind the tape, then wind it to the end of the reel,
		then rewind it again.
   bsfm		Backward space COUNT file marks.
		Tape is positioned on the beginning-of-the-tape side of
		the file mark.
   fsfm		Forward space COUNT file marks.
		Tape is positioned on the beginning-of-the-tape side of
		the file mark.
   asf		Absolute space to file number COUNT.
		Equivalent to rewind followed by fsf COUNT.
   eom		Space to the end of the recorded media on the tape
		(for appending files onto tapes).
   erase	Erase the tape.
   ras1		run self test 1 (nondestructive)
   ras2		run self test 2 (destructive)
   ras3		reserved for self test 3
   setblk       set block length (SCSI and!! ftape one day)
   setdensity   set tape density (SCSI)
   seek         seek to block (Tandberg, ftape, etc.)
   tell         tell block (Tandberg, ftape, etc.)
   setdrvbuffer set the drive buffering according to SCSI-2
		ordinary buffered operation with code 1
   status	Print status information about the tape unit.

   David MacKenzie <djm@gnu.ai.mit.edu> */

/*
   new commands for Linux/ftape/zftape:

   rdftseg      Read an arbitrary tape segment, output to stdout.
   wrftseg      Oh, no, we don't. That is really too dangerous.
                Imagine erasing the header segments ...
   volinfo      Query information about the current volume
   getsize      Query tape capacity
   ftmode       Switch tape drive to raw-mode.

   Claus-Justus Heine <claus@instmath.rwth-aachen.de>
 */

/*
   some extensions, should probably check out st-mt.c
   
   lock
   unlock
   load
   unload
   setpart

   Claus-Justus Heine <heine@instmath.rwth-aachen.de>
 */

/*
   To report in real numbers a computer can understand for various values:
   --computer
   This is most useful in plain shell scripts to determine how much space
   remains on the tape. This is needed because the division for HUMAN
   readable forms truncates the numbers. Multiplication back shows a 
   different than real amount left, and sometimes it's larger than what is 
   actually there. This results in failing backup scripts. 
   Modified lines contain the string "--ajk".
   Default is HUMAN readable form as so not to break existing scripts.

   Andrew J. Kroll <ag784@freenet.buffalo.edu>

   Changed this to "-l" and "--long-numbers".

   Claus-Justus Heine <heine@instmath.rwth-aachen.de>
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <time.h>
#include <sys/file.h>
#include <fcntl.h>
#include <errno.h>
#include <getopt.h>
#if defined(HAVE_UNISTD_H)
#include <unistd.h>
#endif
#ifdef HAVE_SYS_IO_TRIOCTL_H
#include <sys/io/trioctl.h>
#endif

#include <locale.h>
#include <libintl.h>
#define _(String) gettext (String)

#include <asm/types.h>
#include <linux/mtio.h>

#ifndef DEFTAPE			/* From sys/mtio.h.  */
# define DEFTAPE "/dev/tape"
#endif

#include "rmt.h"
#include "version.h"

#if defined(HAVE_STRING_H) || defined(STDC_HEADERS)
#include <string.h>
#else
#include <strings.h>
#endif

#if defined(STDC_HEADERS)
#include <stdlib.h>
#else
extern int errno;
char *getenv ();
int atoi ();
void exit ();
#endif

int fstat ();

int argmatch ();
void check_type ();
void error ();
void invalid_arg ();
void perform_operation ();
void print_status ();
int  get_capacity();
int  volinfo();
int  get_sement();
void usage ();

struct mt_tape_info known_drives[]= MT_TAPE_INFO;
#include <linux/ftape-vendors.h>
vendor_struct ftape_vendors[]= QIC117_VENDORS;

char *opnames[] =
{
   "reset", "fsf", "bsf", "fsr", "bsr", "eof", "weof", "rewind",
   "offline", "rewoffl", "eject", /*   "nop", */ "retension",
   "bsfm", "fsfm", "asf", "eom", "seod", "erase", "ras1", 
   "ras2", "ras3", "setblk", "setdensity", "seek", "tell", 
   "setdrvbuffer",
#ifdef MTLOCK
   "lock",
#endif
#ifdef MTUNLOCK
   "unlock",
#endif
#ifdef MTLOAD
   "load",
#endif
#ifdef MTUNLOAD
   "unload",
#endif
#ifdef MTCOMPRESSION
   "compression",
#endif
#ifdef MTSETPART
   "setpart",
#endif
#ifdef MTIOCRDFTSEG
   "rdftseg",
#endif
#ifdef MTIOCVOLINFO
   "volinfo",
#endif
#ifdef MTIOCGETSIZE
   "getsize",
#endif
#ifdef MTIOCFTMODE
   "ftmode",
#endif
   "status",
   NULL
};
#define MTASF 600		/* Random unused number.  */
int operations[] =
{                   
   MTRESET, MTFSF, MTBSF, MTFSR, MTBSR, MTWEOF, MTWEOF, MTREW,
   MTOFFL, MTOFFL, MTOFFL, /*   MTNOP, */
   MTRETEN, MTBSFM, MTFSFM, MTASF, MTEOM, MTEOM, MTERASE,
   MTRAS1, MTRAS2, MTRAS3,
   MTSETBLK, MTSETDENSITY, MTSEEK, MTTELL, MTSETDRVBUFFER,
#ifdef MTLOCK
   MTLOCK,
#endif
#ifdef MTUNLOCK
   MTUNLOCK,
#endif
#ifdef MTLOAD
   MTLOAD,
#endif
#ifdef MTUNLOAD
   MTUNLOAD,
#endif
#ifdef MTCOMPRESSION
   MTCOMPRESSION,
#endif
#ifdef MTSETPART
   MTSETPART,
#endif
#ifdef MTIOCRDFTSEG
   MTIOCRDFTSEG,
#endif
#ifdef MTIOCVOLINFO
   MTIOCVOLINFO,
#endif
#ifdef MTIOCGETSIZE
   MTIOCGETSIZE,
#endif
#ifdef MTIOCFTMODE
   MTIOCFTMODE,
#endif
   MTNOP
 };

/* If nonzero, don't consider file names that contain a `:' to be
   on remote hosts; all files are local.  Always zero for mt;
   since when do local device names contain colons?  */
int f_force_local = 0;

struct option longopts[] =
{
  {"long-numbers",0,NULL,'l'},  /* --ajk */ 
  {"file", 1, NULL, 'f'},
  {"version", 0, NULL, 'V'},
  {"help", 0, NULL, 'H'},
  {NULL, 0, NULL, 0}
};

int long_numbers=0; /* global, because I'm lazy today --ajk */

/* The name this program was run with.  */
char *program_name;

int
main (argc, argv)
     int argc;
     char **argv;
{
  int operation;
  int count;
  char *tapedev;
  int tapedesc;
  int i;
  struct mtpos getblkno;

  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  program_name = argv[0];
  tapedev = NULL;
  count = 1;

  while ((i = getopt_long (argc, argv,
			   "cf:t:VH", longopts, (int *) 0)) != -1)
    {
      switch (i)
	{
	case 'l': /* computer readable format on numbers --ajk */
	  long_numbers=1;
	  break;
	case 'f':
	case 't':
	  tapedev = optarg;
	  break;

	case 'V':
	  printf (version_string);
	  exit (0);
	  break;

	case 'H':
	default:
	  usage (stdout, 0);
	}
    }

  if (optind == argc)
    usage (stderr, 1);

  i = argmatch (argv[optind], opnames);
  if (i < 0)
    {
      invalid_arg (_("tape operation"), argv[optind], i);
      exit (1);
    }
  operation = operations[i];

  if (++optind < argc)
    count = atoi (argv[optind]);
  if (++optind < argc)
    usage (stderr, 1);

  if (tapedev == NULL)
    {
      tapedev = getenv ("TAPE");
      if (tapedev == NULL)
#ifdef DEFTAPE			/* From sys/mtio.h.  */
        tapedev = DEFTAPE;
#else
	error (1, 0, _("no tape device specified"));
#endif
    }

  if ( (operation == MTWEOF)
#ifdef MTERASE
       || (operation == MTERASE)
#endif
	)
    tapedesc = rmtopen (tapedev, O_WRONLY, 0);
  else
    tapedesc = rmtopen (tapedev, O_RDONLY, 0);
  if (tapedesc == -1)
    error (1, errno, "%s", tapedev);
  check_type (tapedev, tapedesc);

  switch (operation) 
    {
#ifdef MTIOCRDFTSEG
    case MTIOCRDFTSEG:
      if (get_segment(tapedev, tapedesc, count)) 
	error (2, errno, "%s", tapedev);
      break;
#endif
#ifdef MTIOCVOLINFO
    case MTIOCVOLINFO: 
      if (volinfo(tapedev, tapedesc, 0)) 
	error (2, errno, "%s", tapedev);
      break;
#endif
#ifdef MTIOCGETSIZE
    case MTIOCGETSIZE:
      if (get_capacity(tapedev, tapedesc, 0)) 
	error (2, errno, "%s", tapedev);
      break;
#endif
#ifdef MTIOCFTMODE
    case MTIOCFTMODE:
      if (switch_ftmode(tapedev, tapedesc, count))
	error (2, errno, "%s", tapedev);
      break;
#endif
    default:
    {
      if (operation == MTASF)
      {
	perform_operation (tapedev, tapedesc, MTREW, 1);
	operation = MTFSF;
      }
      if ( operation == MTTELL )
      {
	struct mtop control;                       
	
	control.mt_op = operation;
	if(rmtioctl(tapedesc, MTIOCTOP, &control) == -1)
	{   
	  if (rmtioctl(tapedesc, MTIOCPOS , &getblkno) == -1)
	    error (2, errno, "%s", tapedev);
	  else
	    printf(_("At block %d.\n"),getblkno.mt_blkno);
	}   
	else
	  printf(_("At block %d.\n"),control.mt_count);
      }                         
      else
      {
	perform_operation (tapedev, tapedesc, operation, count);
	if (operation == MTNOP)
	  print_status (tapedev, tapedesc);
      }
      break;
    }
    } /* switch */

  if (rmtclose (tapedesc) == -1)
    error (2, errno, "%s", tapedev);

  exit (0);
}

void
check_type (dev, desc)
     char *dev;
     int desc;
{
  struct stat stats;

  if (_isrmt (desc))
    return;
  if (fstat (desc, &stats) == -1)
    error (1, errno, "%s", dev);
  if ((stats.st_mode & S_IFMT) != S_IFCHR)
    error (1, 0, _("%s is not a character special file"), dev);
}

void
perform_operation (dev, desc, op, count)
     char *dev;
     int desc;
     short op;
     int count;
{
  struct mtop control;

  control.mt_op = op;
  control.mt_count = count;
  if(rmtioctl (desc, MTIOCTOP, &control) == -1)
        error (2, errno, "%s", dev);
}

void
print_status (dev, desc)
     char *dev;
     int desc;
{
  struct mtget status;
  int i;

  if (rmtioctl(desc, MTIOCGET, &status) == -1)
    error (2, errno, "%s", dev);

  for( i=0; known_drives[i].t_type != 0; i++)
  {
    if( known_drives[i].t_type == status.mt_type )
    {
      printf(_("Drive is a %s, (drive type = %d)\n"),known_drives[i].t_name,status.mt_type);
      break;
    }
  }   
  if( known_drives[i].t_type == 0 )
  {
    for ( i = 0; i < sizeof(ftape_vendors)/sizeof(vendor_struct); i ++ ) 
    {
      if ( (ftape_vendors[i].vendor_id | MT_ISFTAPE_FLAG) == status.mt_type ) 
      {
	printf(_("This is a %s floppy tape drive, (drive type = 0x%04lx)\n"), 
	       ftape_vendors[i].name, status.mt_type & ~MT_ISFTAPE_FLAG );
	break;
      }
    }
    if ( i == sizeof(ftape_vendors)/sizeof(vendor_struct) )
      if ( (MT_ISFTAPE_FLAG & status.mt_type) != 0 )
	printf (_("This is a floppy tape drive, drive type = %d, drive name unknown.\n"), (int) status.mt_type & ~MT_ISFTAPE_FLAG);
      else
    printf (_("drive type = %d, drive name unknown.\n"), (int) status.mt_type);
  }
#if defined(hpux) || defined(__hpux)
  printf (_("drive status (high) = %d\n"), (int) status.mt_dsreg1);
  printf (_("drive status (low) = %d\n"), (int) status.mt_dsreg2);
#else
  printf (_("drive status         = 0x%08x\n"), (int) status.mt_dsreg);
  printf (_("generic drive status = 0x%08x\n"), (int) status.mt_gstat);
  printf(_("(In particular: "));
  if( GMT_EOF(status.mt_gstat) )printf("* %s ", _("at end of file"));
  if( GMT_BOT(status.mt_gstat) )printf("* %s ", _("at begin of tape"));
  if( GMT_EOT(status.mt_gstat) )printf("* %s ", _("at end of tape"));
  if( GMT_SM(status.mt_gstat) )printf("* %s ", _("at DDS setmark"));
  if( GMT_EOD(status.mt_gstat) )printf("* %s ", _("at end of data"));
  if( GMT_WR_PROT(status.mt_gstat) )printf("* %s ", _("tape write protected"));
  if( GMT_ONLINE(status.mt_gstat) )printf("* %s ", _("tape online"));
  if( GMT_D_6250(status.mt_gstat) )printf("* %s ", _("density 6250 (scsi)"));
  if( GMT_D_1600(status.mt_gstat) )printf("* %s ", _("density 1600 (scsi)"));
  if( GMT_D_800(status.mt_gstat) )printf("* %s ", _("density 800 (scsi)"));
  if( GMT_DR_OPEN(status.mt_gstat) )printf("* %s ", _("door open (no tape)"));
  if( GMT_IM_REP_EN(status.mt_gstat) )printf("* %s ", _("immediate report mode (scsi)"));
  printf("* )\n");


#endif
  printf (_("sense key error      = 0x%08x\n"), (int) status.mt_erreg);
  printf (_("residue count        = %d\n"), (int) status.mt_resid);
#if !defined(ultrix) && !defined(__ultrix__) && !defined(hpux) && !defined(__hpux) && !defined(__osf__)
  printf (_("file number          = %d\n"), (int) status.mt_fileno);
  printf (_("block number         = %d\n"), (int) status.mt_blkno);
#endif
#ifdef MTIOCVOLINFO
  (void)volinfo(dev, desc, 1);
#endif
#ifdef MTIOCGETSIZE
  (void)get_capacity(dev, desc, 1);
#endif
}

#ifdef MTIOCVOLINFO
int 
volinfo(dev, desc, noise)
     char *dev;
     int desc;
     int noise;
{
  struct mtvolinfo info_rec;
  int result;
  char *suffixes[] = {"kilo", "mega", "giga", "tera"};
  int suffix;
  double rawsize, size;

  if ((result = rmtioctl (desc, MTIOCVOLINFO, &info_rec)) >= 0) 
  {
    if (noise) printf(_("\nMTIOCVOLINFO result:\n"));
    printf(_("file number          = %d\n"), (int)info_rec.mt_volno);
    printf(_("block size           = %d\n"), (int)info_rec.mt_blksz);
    rawsize = (double)info_rec.mt_rawsize;
    size    = (double)info_rec.mt_size;

    if(!long_numbers) { /* --ajk */

      for (suffix = 0;
	   suffix < 3 && rawsize > 1024.0;
	   suffix++, rawsize /= 1024.0);

      printf(_("physical space used  = %6.1f %sbytes\n"),
	     rawsize, suffixes[suffix]);

      for (suffix = 0;
	   suffix < 3 && size > 1024.0;
	   suffix++, size /= 1024.0);

      printf(_("real size of volume  = %6.1f %sbytes\n"),
	     size, suffixes[suffix]);
    } else { /* --ajk 18446744073709551616 */
      printf(_("physical space used  = %10.0u kilobytes\n"),
	     info_rec.mt_rawsize); /* --ajk */
      printf(_("real size of volume  = %10.0u kilobytes\n"),
	     info_rec.mt_size); /* --ajk */
    } /* --ajk */
    if (info_rec.mt_cmpr) {
      printf(_("compression ratio    = %d %%\n"), 
	     (int)((double)info_rec.mt_rawsize*100.0 / 
		   (double)info_rec.mt_size));
    }
  }
  return result;
}
#endif

#ifdef MTIOCGETSIZE
int 
get_capacity(dev, desc, noise)
     char *dev;
     int desc;
     int noise;
{
  struct mttapesize capacity;
  int result;
  double space, used, free;
  char *suffixes[] = {"kilo", "mega", "giga", "tera"};
  int suffix;

  if ((result = rmtioctl (desc, MTIOCGETSIZE, &capacity)) >= 0) 
  {
    if (noise) printf(_("\nMTIOCGETSIZE result:\n"));

    if(!long_numbers) { /* --ajk */
      space = (double)capacity.mt_capacity;
      used  = (double)capacity.mt_used;
      free  = space-used;

      for (suffix = 0;
	   suffix < 3 && space > 1024.0;
	   suffix++, space /= 1024.0);
      printf(_("total bytes on tape  = %6.1f %sbytes\n"),
	   space, suffixes[suffix]);

      for (suffix = 0; suffix < 3 && used > 1024.0; suffix++, used /= 1024.0);
      printf(_("total bytes used     = %6.1f %sbytes\n"),
	     used, suffixes[suffix]);
      for (suffix = 0; suffix < 3 && free > 1024.0; suffix++, free /= 1024.0);
      printf(_("total bytes left     = %6.1f %sbytes\n"),
	     free, suffixes[suffix]);
    } else { /* --ajk */
      printf(_("total bytes on tape  = %10.0lu kilobytes\n"),
	     capacity.mt_capacity); /* --ajk */
      printf(_("total bytes used     = %10.0lu kilobytes\n"),
	     capacity.mt_used); /* --ajk */
      printf(_("total bytes left     = %10.0lu kilobytes\n"),
	     capacity.mt_capacity- capacity.mt_used); /* --ajk */
    } /* --ajk */

  }
  return result;
}
#endif

#ifdef MTIOCFTMODE
int
switch_ftmode(dev, desc, on)
     char *dev;
     int desc;
     int on;
{
  struct mtftmode mode;
  int result;

  mode.ft_rawmode = on ? 1 : 0;
  result = rmtioctl (desc, MTIOCFTMODE, &mode);
  return result;
}
#endif

#ifdef MTIOCRDFTSEG
int 
get_segment(dev, desc, seg_no)
     char *dev;
     int desc;
     int seg_no;
{
  struct mtftseg raw_seg;
  int result;
# ifdef MTIOCFTMODE
  struct mtftmode mode;

  mode.ft_rawmode = 1;
  rmtioctl (desc, MTIOCFTMODE, &mode); /* ignore possible error */
# endif

  raw_seg.mt_data = (void *)xmalloc(32*1024);
  raw_seg.mt_mode = 0;
  raw_seg.mt_result = 0;
  raw_seg.mt_segno = seg_no;

  result = rmtioctl (desc, MTIOCRDFTSEG, &raw_seg);

# ifdef MTIOCFTMODE
  mode.ft_rawmode = 0;
  rmtioctl (desc, MTIOCFTMODE, &mode); /* ignore possible error */
# endif

  if (result >= 0) 
  {

    if (raw_seg.mt_result >= 0 && raw_seg.mt_result <= 32*1024)
    {
      /*  print the raw data to stdout.
       */
      if (write(1, raw_seg.mt_data, raw_seg.mt_result) < 0) 
      {
	error (2, errno, "%s", dev);
      }
    }
    else
    {
      error (2, raw_seg.mt_result, 
	     _("Couldn't access segment %d on tape device %s"), 
	     seg_no, dev);
    }
  }
  free(raw_seg.mt_data);
  return result;
}
#endif

void
usage (fp, status)
  FILE *fp;
  int status;
{
  int i;
  char **opname;

  fprintf (fp, "\
Usage: %s [-V] [-f device] [--file=device] [-l] [--long-numbers] [--help] [--version] operation [count]\n",
	   program_name);
  fprintf (fp, _("\
Known tape operations:\n"));
  opname = opnames;
  while (*opname != NULL) {
    fprintf (fp, "%s ", *opname++);
  }
  fprintf(fp, "\n");

  exit (status);
}

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 2
 * End:
 */
