/*
   (c) Copyright 2000-2002  convergence integrated media GmbH.
   (c) Copyright 2002-2003  convergence GmbH.

   All rights reserved.

   Written by Denis Oliver Kropp <dok@directfb.org>,
              Andreas Hundt <andi@fischlustig.de> and
              Sven Neumann <sven@convergence.de>.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <unistd.h>

#include <fusionsound.h>

#include <direct/interface.h>
#include <direct/mem.h>

#include <core/core_sound.h>
#include <core/sound_buffer.h>

#include "ifusionsoundbuffer.h"
#include "ifusionsoundstream.h"
#include "media/ifusionsoundmusicprovider.h"


static DFBResult
Probe( void *arg );

static DFBResult
Construct( IFusionSound *thiz,
           void         *arg );

#include <direct/interface_implementation.h>

DIRECT_INTERFACE_IMPLEMENTATION( IFusionSound, default )

/*
 * private data struct of IFusionSound
 */
typedef struct {
     int              ref;       /* reference counter */

     CoreSound       *core;
} IFusionSound_data;


static void
IFusionSound_Destruct( IFusionSound *thiz )
{
     IFusionSound_data *data = (IFusionSound_data*)thiz->priv;

     fs_core_destroy( data->core );

     DIRECT_DEALLOCATE_INTERFACE( thiz );
}

static DFBResult
IFusionSound_AddRef( IFusionSound *thiz )
{
     DIRECT_INTERFACE_GET_DATA (IFusionSound);

     data->ref++;

     return DFB_OK;
}

static DFBResult
IFusionSound_Release( IFusionSound *thiz )
{
     DIRECT_INTERFACE_GET_DATA (IFusionSound)

     if (--data->ref == 0) {
          IFusionSound_Destruct( thiz );
     }

     return DFB_OK;
}

static DFBResult
IFusionSound_CreateBuffer( IFusionSound         *thiz,
                           FSBufferDescription  *desc,
                           IFusionSoundBuffer  **ret_interface )
{
     DFBResult                 ret;
     int                       length   = 0;
     int                       channels = 2;
     FSSampleFormat            format   = FSSF_S16;
     int                       rate     = 44100;
     FSBufferDescriptionFlags  flags;
     CoreSoundBuffer          *buffer;
     IFusionSoundBuffer       *interface;

     DIRECT_INTERFACE_GET_DATA (IFusionSound);

     if (!desc || !ret_interface)
          return DFB_INVARG;

     flags = desc->flags;

     if (flags & ~FSBDF_ALL)
          return DFB_INVARG;

     if (flags & FSBDF_LENGTH)
          length = desc->length;

     if (flags & FSBDF_CHANNELS)
          channels = desc->channels;

     if (flags & FSBDF_SAMPLEFORMAT)
          format = desc->sampleformat;

     if (flags & FSBDF_SAMPLERATE)
          rate = desc->samplerate;

     if (length < 1)
          return DFB_INVARG;

     switch (channels) {
          case 1:
          case 2:
               break;

          default:
               return DFB_INVARG;
     }

     switch (format) {
          case FSSF_S16:
          case FSSF_U8:
               break;

          default:
               return DFB_INVARG;
     }

     if (rate < 1)
          return DFB_INVARG;

     ret = fs_buffer_create( data->core,
                             length, channels, format, rate, &buffer );
     if (ret)
          return ret;

     DIRECT_ALLOCATE_INTERFACE( interface, IFusionSoundBuffer );

     ret = IFusionSoundBuffer_Construct( interface, data->core, buffer,
                                         length, channels, format, rate );
     if (ret)
          *ret_interface = NULL;
     else
          *ret_interface = interface;

     fs_buffer_unref( buffer );

     return ret;
}

static DFBResult
IFusionSound_CreateStream( IFusionSound         *thiz,
                           FSStreamDescription  *desc,
                           IFusionSoundStream  **ret_interface )
{
     DFBResult                 ret;
     int                       channels  = 2;
     FSSampleFormat            format    = FSSF_S16;
     int                       rate      = 44100;
     int                       size      = rate;   /* space for one second */
     int                       prebuffer = 0;      /* no prebuffer by default */
     FSStreamDescriptionFlags  flags     = FSSDF_NONE;
     CoreSoundBuffer          *buffer;
     IFusionSoundStream       *interface;

     DIRECT_INTERFACE_GET_DATA (IFusionSound);

     if (!ret_interface)
          return DFB_INVARG;

     if (desc) {
          flags = desc->flags;

          if (flags & ~FSSDF_ALL)
               return DFB_INVARG;

          if (flags & FSSDF_BUFFERSIZE)
               size = desc->buffersize;
          else if (flags & FSSDF_SAMPLERATE)
               size = desc->samplerate;

          if (flags & FSSDF_CHANNELS)
               channels = desc->channels;

          if (flags & FSSDF_SAMPLEFORMAT)
               format = desc->sampleformat;

          if (flags & FSSDF_SAMPLERATE)
               rate = desc->samplerate;

          if (flags & FSSDF_PREBUFFER)
               prebuffer = desc->prebuffer;
     }

     if (size < 1 || rate < 1 || prebuffer >= size)
          return DFB_INVARG;

     /* Limit ring buffer size to five seconds. */
     if (size > rate * 5)
          return DFB_LIMITEXCEEDED;

     switch (channels) {
          case 1:
          case 2:
               break;

          default:
               return DFB_INVARG;
     }

     switch (format) {
          case FSSF_S16:
          case FSSF_U8:
               break;

          default:
               return DFB_INVARG;
     }

     ret = fs_buffer_create( data->core,
                             size, channels, format, rate, &buffer );
     if (ret)
          return ret;

     DIRECT_ALLOCATE_INTERFACE( interface, IFusionSoundStream );

     ret = IFusionSoundStream_Construct( interface, data->core, buffer, size,
                                         channels, format, rate, prebuffer );
     if (ret)
          *ret_interface = NULL;
     else
          *ret_interface = interface;

     fs_buffer_unref( buffer );

     return ret;
}

static DFBResult
IFusionSound_CreateMusicProvider( IFusionSound               *thiz,
                                  const char                 *filename,
                                  IFusionSoundMusicProvider **interface )
{
     DFBResult                            ret;
     DirectInterfaceFuncs                *funcs = NULL;
     IFusionSoundMusicProvider           *musicprovider;
     IFusionSoundMusicProvider_ProbeContext ctx;

     DIRECT_INTERFACE_GET_DATA(IFusionSound)

     /* Check arguments */
     if (!interface || !filename)
          return DFB_INVARG;

     if (access( filename, R_OK ) != 0)
          return DFB_FILENOTFOUND;

     /* Fill out probe context */
     ctx.filename = filename;

     /* Find a suitable implemenation */
     ret = DirectGetInterface( &funcs,
                               "IFusionSoundMusicProvider", NULL, DirectProbeInterface, &ctx );

     if (ret)
          return ret;

     DIRECT_ALLOCATE_INTERFACE( musicprovider, IFusionSoundMusicProvider );

     /* Construct the interface */
     ret = funcs->Construct( musicprovider, filename );
     if (ret)
          return ret;

     *interface = musicprovider;

     return DFB_OK;
}

/* exported symbols */

static DFBResult
Probe( void *arg )
{
     return DFB_OK;
}

static DFBResult
Construct( IFusionSound *thiz,
           void         *arg )
{
     DFBResult ret;

     /* Allocate interface data. */
     DIRECT_ALLOCATE_INTERFACE_DATA( thiz, IFusionSound );

     /* Initialize interface data. */
     data->ref = 1;

     /* Create the core instance. */
     ret = fs_core_create( &data->core );
     if (ret) {
          DirectFBError( "FusionSound: fs_core_create() failed", ret );

          DIRECT_DEALLOCATE_INTERFACE( thiz );

          return ret;
     }

     /* Assign interface pointers. */
     thiz->AddRef        = IFusionSound_AddRef;
     thiz->Release       = IFusionSound_Release;
     thiz->CreateBuffer  = IFusionSound_CreateBuffer;
     thiz->CreateStream  = IFusionSound_CreateStream;
     thiz->CreateMusicProvider = IFusionSound_CreateMusicProvider;

     return DFB_OK;
}

