/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: NewFirewallDruid.cc,v 1.21 2003/10/22 07:17:20 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"

#include "fwbuilder/Firewall.hh"
#include "fwbuilder/Interface.hh"
#include "fwbuilder/IPv4.hh"
#include "fwbuilder/Management.hh"
#include "fwbuilder/snmp.hh"
#include "fwbuilder/dns.hh"

#include "main_window.hh"

#include "NewFirewallDruid.hh"
#include "MessageDialog.hh"

#include "InterfaceListWidget.hh"
#include "InterfaceTblWidget.hh"

#include "NewFirewallDruidP10Dialog.hh"
#include "NewFirewallDruidP12Dialog.hh"
#include "NewFirewallDruidP15Dialog.hh"
#include "NewFirewallDruidP17Dialog.hh"
#include "NewFirewallDruidP20Dialog.hh"
#include "NewFirewallDruidP30Dialog.hh"

#include "FWObjectBook.hh"
#include "FWObjectDatabaseGUI.hh"
#include "GenericBackgroundOpDialog.hh"

#include <gtk--.h>

#include "glademm_support.hh"

#include <iostream>


using namespace libfwbuilder;
using namespace std;

enum {
    PAGE_10,
    PAGE_12,
    PAGE_15,
    PAGE_17,
    PAGE_20,
    PAGE_30,
    PAGE_F,
} NewFirewallDruidPages;


NewFirewallDruid::NewFirewallDruid(main_window *mw,
                                   const string &t,
                                   const string& l) : Druid(t,l)
{
    main_w=mw;

    set_default_size(600,600);
    set_position(GTK_WIN_POS_CENTER);

/************************************************************************
 *    Page #10
 */
    p10=manage(new NewFirewallDruidP10Dialog());
    createStandardPage(PAGE_10,p10);

/************************************************************************
 *    Page #12
 */
    p12=manage(new NewFirewallDruidP12Dialog());
    createStandardPage(PAGE_12,p12);

/************************************************************************
 *    Page #15
 */
    p15=manage(new NewFirewallDruidP15Dialog());
    createStandardPage(PAGE_15,p15);

/************************************************************************
 *    Page #17
 */
    p17=manage(new NewFirewallDruidP17Dialog());
    createStandardPage(PAGE_17,p17);

/************************************************************************
 *    Page #20
 */
    p20=manage(new NewFirewallDruidP20Dialog());
    createStandardPage(PAGE_20,p20);


/************************************************************************
 *    Page #30
 */
    p30=manage(new NewFirewallDruidP30Dialog());
    createStandardPage(PAGE_30,p30);


/************************************************************************
 *    Final Page 
 */
    createFinishPage(PAGE_F,
 _("Click 'Finish' to create and file new object, click 'Cancel' to discard."));

    glademm_set_Widget("NewFirewallDruid", this);

    setPage(PAGE_10);
}


void NewFirewallDruid::on_prepare()
{
    int current_page=get_current_page_id();

    switch (current_page) {
    case -1:
    case PAGE_10:	p10->prepare();	break;
    case PAGE_12:	p12->prepare();	break;
    case PAGE_15:	p15->prepare();	break;
    case PAGE_17:	p17->prepare();	break;
    case PAGE_20:    
    {
        bool have_loopback=false;
        list<InterfaceData> il=p20->getInterfaces();
        for (list<InterfaceData>::iterator i=il.begin(); i!=il.end(); i++)
        {
            if (i->address=="127.0.0.1")  have_loopback=true;
        }
        if (!have_loopback)
        {
            InterfaceData idata;
            idata.id   =FWObjectDatabaseGUI::db->generateUniqueId();

            string lo_name=Resources::getTargetOptionStr( p12->host_os->get_value(),
                                                          "loopback_interface");
            if (lo_name!="")
            {
                idata.name=lo_name;
                idata.isUnnumbered=false;
                idata.isDyn=false;
                idata.address="127.0.0.1";
                idata.netmask="255.0.0.0";
                p20->addInterface(idata);
            }
        }
        p20->prepare();
        break;
    }
    case PAGE_30:	
    {
// we copy interfaces from p20 to ilist in on_next
        p30->setPlatform( p12->fw_platform->get_value() );
        p30->prepare();	
        InterfaceData::guessSecurityLevel( p12->fw_platform->get_value() , ilist );
        p30->setInterfaces( ilist );
        p30->show_all();
    }
    break;
    }
}

gboolean NewFirewallDruid::on_next()
{
    int current_page=get_current_page_id();

    switch (current_page) {

    case -1:
    case PAGE_10:
        if (p10->obj_name->get_text()=="") return true;
        setPage(PAGE_12);
	return true;

    case PAGE_12:
        setPage(PAGE_15);
	return true;

    case PAGE_15:
        if (p15->use_snmp->get_active())  setPage(PAGE_17);
        else setPage(PAGE_20);
	return true;

    case PAGE_17:
    {
        if (p17->snmp_community->get_text()=="") return true;

// run SNMP query

        list<InterfaceData> ifaces=getInterfacesDataViaSNMP();
        for (list<InterfaceData>::iterator i=ifaces.begin(); i!=ifaces.end(); i++)
            i->guessLabel( p12->fw_platform->get_value() );

        p20->setInterfaces( ifaces );
        setPage(PAGE_20);
    }
    return true;

    case PAGE_20:
    {
        ilist=p20->getInterfaces();

        for (list<InterfaceData>::iterator i=ilist.begin(); i!=ilist.end(); i++)
        {
            if ( ! i->isDyn && ! i->isUnnumbered)
            {
                try
                {
                    checkIPaddress( i->address );
                    checkNetmask(   i->netmask );
                } catch(FWException &ex)
                {
                    MessageDialog::Error(ex.toString(),this);
                    return true;
                }
            }
        }

        if (ilist.size()==1) 
        {
            if (ilist.front().address!="127.0.0.1") ilist.front().securityLevel=0;
            else                                    ilist.front().securityLevel=100;

            setPage(PAGE_F);
            return true;
        }
        if (ilist.size()==2)
        {
            if (ilist.front().address=="127.0.0.1") 
            {
                ilist.front().securityLevel=100;
                ilist.back().securityLevel=0;
            } else 
            {
                if (ilist.back().address=="127.0.0.1") 
                {
                    ilist.front().securityLevel=0;
                    ilist.back().securityLevel=100;
                } else
                {
                    setPage(PAGE_30);
                    return true;
                }
            }
            setPage(PAGE_F);
            return true;
        }
	setPage(PAGE_30);
        return true;
    }
    case PAGE_30:
        ilist=p30->getInterfaces();
	setPage(PAGE_F);
	return true;
    }

    return true;
}

gboolean NewFirewallDruid::on_back()
{
    int current_page=get_current_page_id();

    switch (current_page) {

    case PAGE_20:
    {
        list<InterfaceData> il=p20->getInterfaces();
        for (list<InterfaceData>::iterator i=il.begin(); i!=il.end(); i++)
        {
            if (i->address=="127.0.0.1")
            {
                il.erase(i);
                break;
            }
        }
        p20->setInterfaces(il);
	setPage(PAGE_12);
	return true;
    }
    case PAGE_17:	setPage(PAGE_15);	return true;
    case PAGE_15:	setPage(PAGE_12);	return true;
    case PAGE_12:	setPage(PAGE_10);	return true;
    case PAGE_30:       setPage(PAGE_20);	return true;
    case PAGE_F:        setPage(PAGE_30);	return true;
    }
    return  true;
}

gboolean NewFirewallDruid::on_cancel()
{
    destroy();
    return true;
}

std::list<InterfaceData> NewFirewallDruid::getInterfacesDataViaSNMP()
{
    list<InterfaceData> res;

#ifdef HAVE_LIBSNMP

    string rcomm=p17->snmp_community->get_text();

    if ( rcomm.empty() ) 
    {
	MessageDialog::Error(_("missing SNMP community string"));
	return res;
    }

    IPAddress addr;
    string name=p10->obj_name->get_text();
    try {
        vector<IPAddress> addrs=DNS::getHostByName( name );
        addr= addrs.front();
    } catch (FWException &ex)
    {
        char errstr[256];
        sprintf(errstr,_("Address of %s could not be obtained via DNS"), name.c_str());
        MessageDialog::Error(errstr, main_w );
        return res;
    }

    int t=Preferences::global_prefs->getOptInt("/FWBuilderPreferences/Network/SNMPTimeout", -1);

    SNMP_interface_query *q=new SNMP_interface_query();
    q->init(addr.toString(),  rcomm,
	    Preferences::global_prefs->getOptInt("/FWBuilderPreferences/Network/SNMPRetries",  SNMP_DEFAULT_RETRIES),
	    t==-1?SNMP_DEFAULT_TIMEOUT:(1000000L*t)
    );

    GenericBackgroundOpDialog bdisp(q);
    
    try
    {
        bdisp.execute();
        
        const map<int, Interface> &intf = q->getInterfaces();
        for(map<int, Interface>::const_iterator i=intf.begin();i!=intf.end(); ++i)
        {
            if ( i->second.isUp() ) 
            {
                InterfaceData idata( i->second );
                res.push_back(idata);
            }
        }

    } catch(const FWException &ex)
    {
        //do nothing
    }

    bdisp.disconnect();
    delete q;

#endif
    return res;
}

void NewFirewallDruid::on_finish()
{
    Firewall *fw=Firewall::cast( FWObjectDatabaseGUI::newFirewall() );

    fw->setName( p10->obj_name->get_text() );

    Management *mgmt=fw->getManagementObject();
    assert(mgmt!=NULL);

    mgmt->getSNMPManagement()->setReadCommunity(p17->snmp_community->get_text());        

    fw->setStr("host_OS",  p12->host_os->get_value());
    fw->setStr("platform", p12->fw_platform->get_value());

    try {
      Resources::setDefaultTargetOptions(fw->getStr("host_OS")  , fw );
      Resources::setDefaultTargetOptions(fw->getStr("platform") , fw );
    } catch (FWException &ex) { }

    for (list<InterfaceData>::iterator i=ilist.begin(); i!=ilist.end(); i++)
    {
        InterfaceData idata= *i;

        Interface *iface= Interface::cast(FWObjectDatabaseGUI::newInterface(fw->getId()));
        iface->setName( idata.name );
        iface->setLabel( idata.label );
        iface->setDyn(idata.isDyn);
        iface->setUnnumbered(idata.isUnnumbered);
        iface->setStr("network_zone", FWObjectDatabase::getAnyNetworkId() );

        IPv4        *addr=NULL;
        physAddress *pa=NULL;
        if ( ! idata.isDyn && ! idata.isUnnumbered)
        {
            addr=IPv4::cast( FWObjectDatabaseGUI::newIPv4(iface->getId()) );
            addr->setAddress( idata.address );
            addr->setNetmask( idata.netmask );

            if (idata.physicalAddress!="")
            {
                pa=physAddress::cast( FWObjectDatabaseGUI::newPhysAddress(iface->getId()) );
                pa->setPhysAddress( idata.physicalAddress );
            }
        }
        iface->setSecurityLevel( idata.securityLevel );

//        main_window::insertInAllObjectBooks(iface);
    }

    try
    {
        main_window::insertInAllObjectBooks(fw);

//        main_window::changeLabelInAllObjectBooks(fw->getId());
        FWObjectBook *w=main_w->getObjectBook();
        if (w!=NULL)  w->showObject( fw->getId() );
        main_window::conditionalRefreshAllDialogs(fw);

    } catch (FWException &ex)
    {
        cerr << ex.toString();
    } catch (std::string s) 
    {
        cerr << s;
    } catch (std::exception ex) 
    {
        cerr << ex.what();
    } catch (...) 
    {
        cerr << _("Unsupported exception");
    }


    destroy();
}

