/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: ObjectQuickView.cc,v 1.32 2003/12/05 16:19:45 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "ObjectQuickView.hh"
#include "fwbuilder/FWObject.hh"

#include "fwbuilder/Host.hh"
#include "fwbuilder/Network.hh"
#include "fwbuilder/Firewall.hh"
#include "fwbuilder/Group.hh"
#include "fwbuilder/Interface.hh"
#include "fwbuilder/IPv4.hh"
#include "fwbuilder/AddressRange.hh"

#include "fwbuilder/TCPService.hh"
#include "fwbuilder/UDPService.hh"
#include "fwbuilder/ICMPService.hh"
#include "fwbuilder/IPService.hh"

#include "FWObjectDatabaseGUI.hh"

#include "Preferences.hh"
#include "main_window.hh"

#include <sstream>

using namespace libfwbuilder;


string ObjectQuickView::getTextAsString()
{
    return onestring;
}


string ObjectQuickView::getSinglelineToolTip(libfwbuilder::FWObject *o)
{
    ostringstream str;

    if (Host::cast(o)!=NULL) 
    {
	str <<  Address::cast(o)->getAddress().toString() ;

	FWObject *co=o->getFirstByType("Interface");
	if (co!=NULL) 
        {
	    physAddress *paddr=(Interface::cast(co))->getPhysicalAddress();
	    if (paddr!=NULL) 
		str << "    " <<  paddr->getPhysAddress() ;
	}
    }

    if (Firewall::cast(o)!=NULL) 
    {
	str << "  ";

	list<FWObject*>::iterator m;
	Interface       *intf;

	for (m=o->begin(); m!=o->end(); ++m) 
        {
	    if(Interface::isA((*m))) 
            {
		intf=Interface::cast(*m);
                string s;

		if (intf->isDyn()) s=_("Dynamic address");
                else if (intf->isUnnumbered()) s=_("Unnumbered interface");
		else		   s =intf->getIPAddress().toString();

		if (intf->isExt()) s=s+" "+_("(ext)");

		str << " " << intf->getName()
                    << ":" << s << "  ";
            }
	}
    }

    if (Interface::cast(o)!=NULL) 
    {
        physAddress *paddr=(Interface::cast(o))->getPhysicalAddress();
        if (paddr!=NULL) 
        {
            str << "    ";
            str <<  paddr->getPhysAddress() ;
        }
    }

    if (physAddress::cast(o)!=NULL) 
    {
	str <<  physAddress::cast(o)->getPhysAddress();
    }

    if (IPv4::cast(o)!=NULL) 
    {
	str <<  Address::cast(o)->getAddress().toString();
        str << "/";
	str << Address::cast(o)->getNetmask().toString();
    }

    if (Network::cast(o)!=NULL) 
    {
	Network *n=Network::cast(o);
	str << n->getAddress().toString();
	str << "/";
	str << n->getNetmask().toString();
    }

    if (AddressRange::cast(o)!=NULL) 
    {
	AddressRange *ar=AddressRange::cast(o);
	str << ar->getRangeStart().toString();
        str << " - ";
	str << ar->getRangeEnd().toString();
    }

    if (Group::cast(o)!=NULL) 
    {
	string    val="";
	FWObject* m;
	list<FWObject*>::iterator j;
	bool  first=true;
	for(j=o->begin(); j!=o->end(); ++j)
        {
	    if ( (m=(*j))!=NULL )
            {
		if ( FWReference::cast(m)!=NULL ) m=(FWReference::cast(m))->getPointer();
		if (!first) val += ", ";
		val += m->getName();
		first=false;
	    }
	}
	str << val;
    }

    if (TCPService::cast(o)!=NULL || UDPService::cast(o)!=NULL) 
    {
	int sps,spe,dps,dpe;

	sps=o->getInt("src_range_start");
	spe=o->getInt("src_range_end");
	dps=o->getInt("dst_range_start");
	dpe=o->getInt("dst_range_end");

        str << _("src.ports:") << sps << "-" << spe << " , ";
        str << _("dst.ports:") << dps << "-" << dpe;
    }

    if (ICMPService::cast(o)!=NULL) 
    {
	str << _("type:") << o->getStr("type")
            << "  "
            << _("code:") << o->getStr("code");

    }

    if (IPService::cast(o)!=NULL) 
    {
	str << _("prot.:") << o->getStr("protocol_num");
    }

    string s=str.str();
    if (s.length()>120) s=s.substr(0,117)+"...";

    return s;
}

void ObjectQuickView::fillObjectQuickViewText( FWObject *o)
{
    string line;
    singleLine sl;

    onestring=getSinglelineToolTip(o);
    strings.clear();

    sl.name=_("Object:");

    sl.val=o->getTypeName();
    if (sl.val==IPv4::TYPENAME) sl.val="Address";
    strings.push_back(sl);

    sl.name=_("Name:");

    sl.val=o->getName();
    strings.push_back(sl);

    if (Host::cast(o)!=NULL) 
    {
	sl.name=_("Address:");
        sl.val=Address::cast(o)->getAddress().toString();

	strings.push_back(sl);

	FWObject *co=o->getFirstByType("Interface");
	if (co!=NULL) 
        {
	    physAddress *paddr=(Interface::cast(co))->getPhysicalAddress();
	    if (paddr!=NULL) 
            {
		sl.name=_("PhysAddress:");
		sl.val =paddr->getPhysAddress();
		strings.push_back(sl);
	    }
	}
    }

    if (Firewall::cast(o)!=NULL) 
    {
	sl.name=_("Platform:");
	sl.val=o->getStr("platform");

	strings.push_back(sl);

	list<FWObject*>::iterator m;
	Interface       *intf;

	for (m=o->begin(); m!=o->end(); ++m) 
        {
	    if(Interface::isA((*m))) 
            {
		intf=Interface::cast(*m);
		sl.name=" "+intf->getName();

		if (intf->isDyn()) sl.val=_("Dynamic address");
                else if (intf->isUnnumbered()) sl.val=_("Unnumbered interface");
		else		   sl.val =intf->getIPAddress().toString();

		if (intf->isExt()) sl.val=sl.val+_(" (ext)");

		strings.push_back(sl);
            }
	}
    }

    if (Interface::cast(o)!=NULL) 
    {
        physAddress *paddr=(Interface::cast(o))->getPhysicalAddress();
        if (paddr!=NULL) 
        {
            sl.name=_("PhysAddress:");
            sl.val =paddr->getPhysAddress();
            strings.push_back(sl);
        }

        sl.name=_("Belongs to:");
        sl.val = o->getParent()->getName();
        strings.push_back(sl);
    }

    if (physAddress::cast(o)!=NULL) 
    {
	sl.name=_("Physical Address:");
        sl.val=physAddress::cast(o)->getPhysAddress();

	strings.push_back(sl);

        sl.name=_("Belongs to:");
        sl.val = o->getParent()->getName();
        strings.push_back(sl);
    }

    if (IPv4::cast(o)!=NULL) 
    {
	sl.name=_("Address:");
        sl.val=Address::cast(o)->getAddress().toString();

	strings.push_back(sl);

	sl.name=_("Netmask:");
        sl.val=Address::cast(o)->getNetmask().toString();

	strings.push_back(sl);

        sl.name=_("Belongs to:");
        sl.val = o->getParent()->getName();
        strings.push_back(sl);
    }

    if (Network::cast(o)!=NULL) 
    {
	Network *n=Network::cast(o);

	sl.name=_("Address:");
	sl.val=n->getAddress().toString();
	strings.push_back(sl);

	sl.name=_("Netmask:");

	sl.val=n->getNetmask().toString();
	strings.push_back(sl);
    }

    if (AddressRange::cast(o)!=NULL) 
    {
	AddressRange *ar=AddressRange::cast(o);

	sl.name=_("Range start:");
	sl.val=ar->getRangeStart().toString();
	strings.push_back(sl);

	sl.name=_(" Range end:");
	sl.val=ar->getRangeEnd().toString();
	strings.push_back(sl);
    }

    if (Group::cast(o)!=NULL) 
    {
	sl.name=_("Members:");

        int       n=0;
	for(list<FWObject*>::iterator j=o->begin(); j!=o->end(); j++,n++)
        {
            FWObject* m;
            string    val="";
	    if ( (m=(*j))!=NULL )
            {
		if ( FWReference::cast(m)!=NULL ) m=(FWReference::cast(m))->getPointer();
		val += m->getName();
                val += "(";
                val += getSinglelineToolTip(m);
                val += ")";
	    }
            sl.val=val;
            strings.push_back(sl);
            sl.name=" ";
            if (n>10)
            {
                sl.val=_(" ( more objects ) ");
                strings.push_back(sl);
                break;
            }
	}
    }

    if (TCPService::cast(o)!=NULL || UDPService::cast(o)!=NULL) 
    {
	string sps,spe,dps,dpe;

	sps=o->getStr("src_range_start");
	spe=o->getStr("src_range_end");
	dps=o->getStr("dst_range_start");
	dpe=o->getStr("dst_range_end");

	if (sps==spe && dps==dpe) 
        {
	    sl.name = _("Source port:");
	    sl.val  = sps;
	    strings.push_back(sl);

	    sl.name = _("Destination port:");
	    sl.val  = dps;
	    strings.push_back(sl);
	} else 
        {
	    sl.name = _("Source port range start:");
	    sl.val  = sps;
	    strings.push_back(sl);

	    sl.name = _("Source port range end:");
	    sl.val  = spe;
	    strings.push_back(sl);

	    sl.name = _("Destination port range start:");
	    sl.val  = dps;
	    strings.push_back(sl);

	    sl.name = _("Destination port range end:");
	    sl.val  = dpe;
	    strings.push_back(sl);
	}
    }

    if (ICMPService::cast(o)!=NULL) 
    {
	sl.name=_("Type:");
	sl.val=o->getStr("type");
	strings.push_back(sl);

	sl.name=_("Code:");
	sl.val=o->getStr("code");
	strings.push_back(sl);
    }

    if (IPService::cast(o)!=NULL) 
    {
	sl.name=_("Protocol:");
	sl.val=o->getStr("protocol_num");
	strings.push_back(sl);
    }


    if (o->getComment()!="") 
    {
	sl.name=_("Comment:");
	sl.val=o->getComment();
	strings.push_back(sl);
    }
}

ObjectQuickView* ObjectQuickView::instance=NULL;

ObjectQuickView* ObjectQuickView::getInstance(main_window *mw)
{
    if (instance==NULL) 
	instance=new ObjectQuickView(mw);
    else
        instance->main_w=mw;

    return instance;
}

ObjectQuickView::ObjectQuickView(main_window *mw) :  Gtk::Window(GTK_WINDOW_POPUP)
{
    main_w=mw;

    border=8;
    spacing=6;
    active=false;
    widget=NULL;

    quick_view_timer_connected=false;
    quick_view_shown_in_status_bar=false;

    set_policy(true, true, true);

    set_name("ObjectQuickView");

    size_request.connect(SigC::slot(this,&ObjectQuickView::on_size_request));
    size_allocate.connect(SigC::slot(this,&ObjectQuickView::on_size_allocate));
}


ObjectQuickView::~ObjectQuickView()
{
    strings.clear();
}

void ObjectQuickView::setObject(FWObject *obj)
{
    fillObjectQuickViewText(obj);
}

void ObjectQuickView::setObjectById(const string &id)
{
    FWObject *object=FWObjectDatabaseGUI::db->getById(id,true);
    fillObjectQuickViewText(object);
}

void   ObjectQuickView::setText(const string &txt)
{
    onestring="";
    strings.clear();


    string line;
    singleLine sl;

    string::size_type b,n;
    b=0;
    while ( (n=txt.find('\n',b))!=string::npos ) {
	line= txt.substr(b,n-b);
	sl.name=line;
	sl.val ="";
	strings.push_back(sl);
	b+=n+1;
	

	onestring.append("   ");


	onestring.append(sl.name);
    }
    line= txt.substr(b);
    sl.name=line;
    sl.val ="";
    strings.push_back(sl);

    onestring.append("   ");
    onestring.append(sl.name);
}

void   ObjectQuickView::attachTo(Gtk::Widget *w)
{
    widget=w;
}


void   ObjectQuickView::activate()
{
    if (active) deactivate();

    if ( Preferences::global_prefs->getOptStr(
	     "/FWBuilderPreferences/UI/ObjectQuickView")=="window") {
/* 
 *  show on the status bar
 */
	string str=getTextAsString();

	main_w->clearStatusbar();
	main_w->showStatusbarMsg(str);
	quick_view_shown_in_status_bar=true;
    }

    if ( Preferences::global_prefs->getOptStr(
	     "/FWBuilderPreferences/UI/ObjectQuickView")=="popup") {

	int timeout= Preferences::global_prefs->getOptInt(
	    "/FWBuilderPreferences/UI/ObjectQuickViewTimeout");
	


	
	quick_view_timer_conn=Gtk::Main::timeout.connect(
	    SigC::slot(this,&ObjectQuickView::timer_callback),
	    timeout*1000 );
	    quick_view_timer_connected=true;	    
    }
    
    active=true;
}

void   ObjectQuickView::deactivate()
{
    if (quick_view_timer_connected) {
	quick_view_timer_conn.disconnect();
	quick_view_timer_connected=false;	    
    }

    if (is_visible()) hide();

    if (quick_view_shown_in_status_bar) {
	main_w->showStatusbarMsg("");
	quick_view_shown_in_status_bar=false;
    }

    active=false;
}

gint ObjectQuickView::timer_callback()
{
    int x,y,w,h,d;

    if (widget) {
	Gdk_Window   wnd=widget->get_window();

	wnd.get_geometry(x,y,w,h,d);
	wnd.get_origin(x,y);

	x-=10;
	y+=(h+10);

	set_uposition(x,y);
    }

    show();

    return false;
}



void ObjectQuickView::draw_impl (GdkRectangle* p0) { paint(p0); }

gint ObjectQuickView::expose_event_impl(GdkEventExpose* ex)
{
    paint( &ex->area );
    return true;
}

void ObjectQuickView::paint(GdkRectangle* p0)
{
    GtkWidget*   widget=GTK_WIDGET(gtkobj());
    Gdk_Window   wnd=get_window();

    Gtk::Style  *st=get_style();

    Gdk_Font     fn=st->get_font();

    Gdk_GC       gc=Gdk_GC(wnd);
    Gdk_GC       wgc;

    int          strx,stry;

    gdk_window_clear_area( wnd, p0->x,p0->y,p0->width,p0->height);

    gdk_gc_set_clip_rectangle( widget->style->white_gc , p0 );
    gdk_draw_rectangle( widget->window,
			widget->style->white_gc,
			true,
			p0->x,p0->y,p0->width,p0->height);

    gdk_gc_set_clip_rectangle( widget->style->black_gc , p0 );
    gdk_draw_rectangle( widget->window,
			widget->style->black_gc,
			false,
			p0->x,p0->y,p0->width-1,p0->height-1);

    strx=border;
    stry=border+sh;

    list<singleLine>::iterator m;
    singleLine                 sl;

    gdk_gc_set_clip_rectangle( widget->style->fg_gc[get_state()] , p0 );

    for (m=strings.begin(); m!=strings.end(); ++m) {
	sl=(*m);

	gdk_draw_text(  widget->window, 
			widget->style->font ,
			widget->style->fg_gc[get_state()],
			strx, stry,
			sl.name.c_str(),
			sl.name.length() );

	gdk_draw_text(  widget->window, 
			widget->style->font ,
			widget->style->fg_gc[get_state()],
			strx+sw1+spacing, stry,
			sl.val.c_str(),
			sl.val.length() );

	stry += (sh+spacing);
    }

}

void ObjectQuickView::on_size_request(GtkRequisition* req)
{
    Gtk::Style *st=get_style();
    Gdk_Font    fn=st->get_font();
    gint        resx, resy;

    list<singleLine>::iterator m;
    singleLine                 sl;
    int  i, n;

    n=sw1=sw2=sh=0;

    for (m=strings.begin(); m!=strings.end(); ++m) {
	sl=(*m);
	n++;
	i= fn.string_width(sl.name); if (sw1<i) sw1=i;
	i= fn.string_width(sl.val);  if (sw2<i) sw2=i;
	sh= fn.string_height(sl.name);
    }

    resx=sw1+sw2+spacing;
    resy=(sh+spacing)*n;

    resx+=(border*2);
    resy+=(border*2);

    req->width   = resx;
    req->height  = resy;
}

void ObjectQuickView::on_size_allocate(GtkAllocation* ga)
{
/*
 *  Remember position, width and height allocated for this widget. 
 *  This info is needed in paint
 */
    x=ga->x;
    y=ga->y;
    width=ga->width;
    height=ga->height;
}




