/* 

                          Firewall Builder

                 Copyright (C) 2002 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: ipfw.cc,v 1.4 2003/11/03 07:37:23 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#include <fstream>
#include <algorithm>
#include <functional>
#include <iostream>

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>

#include <assert.h>

#include "PolicyCompiler_ipfw.hh"
#include "NATCompiler_ipfw.hh"

#include "OSConfigurator_openbsd.hh"
#include "OSConfigurator_macosx.hh"
#include "OSConfigurator_freebsd.hh"
#include "OSConfigurator_solaris.hh"

#include "fwbuilder/Resources.hh"
#include "fwbuilder/FWObjectDatabase.hh"
#include "fwbuilder/XMLTools.hh"
#include "fwbuilder/FWException.hh"
#include "fwbuilder/Firewall.hh"
#include "fwbuilder/Interface.hh"

#include "fwcompiler/manifest.hh"

#ifdef HAVE_GETOPT_H
  #include <getopt.h>
#else
  #ifdef __MINGW32__
    #include <getopt.h>
  #else
    #include <stdlib.h>
  #endif
#endif



static const char      *filename       = NULL;
static const char      *wdir           = NULL;
static const char      *mfile          = NULL;
static const char      *fwobjectname   = NULL;
static int              dl             = 0;
static int              drp            = -1;
static int              drn            = -1;
static int              verbose        = 0;

using namespace libfwbuilder;
using namespace fwcompiler;

class UpgradePredicate: public XMLTools::UpgradePredicate
{
    public:
    virtual bool operator()(const string &msg) const 
    { 
	cout << _("Data file has been created in the old version of Firewall Builder. Use fwbuilder GUI to convert it.") << endl;
	return false;
    }
};
    
void usage(const char *name)
{
    cout << _("Firewall Builder:  policy compiler for ipfw") << endl;
    cout << _("Version ") << VERSION << RELEASE_NUM << endl;
    cout << _("Usage: ") << name << " [-x] [-v] [-V] [-f filename.xml] [-d destdir] [-m manifest] firewall_object_name" << endl;
}

int main(int argc, char * const *argv)
{   

#ifdef ENABLE_NLS
    setlocale (LC_ALL, "");

    bindtextdomain (PACKAGE, LOCALEDIR);
    textdomain (PACKAGE);
#else
#  ifdef HAVE_SETLOCALE
    setlocale (LC_ALL, "");
#  endif
#endif
    

    if (argc<=1)
    {
        usage(argv[0]);
        exit(1);
    }

    int   opt;

    while( (opt=getopt(argc,argv,"x:vVf:d:m:")) != EOF )
    {
        switch(opt)
        {
        case 'd':
            wdir = strdup(optarg);
            break;
        case 'm':
            mfile = strdup(optarg);
            break;
        case 'f':
            filename = strdup(optarg);
            break;
        case 'x':
            if (*optarg=='p') {
                ++optarg;
                drp  = atoi(optarg);
            } else {
                if (*optarg=='n') {
                    ++optarg;
                    drn  = atoi(optarg);
                } else {
                    if (isdigit(*optarg))  dl=atoi(optarg);  // increase debug level
                    else {
                        usage(argv[0]);
                        exit(1);
                    }
                }
            }
            break;
        case 'v':
            verbose++;
            break;
        case 'V':
            usage(argv[0]);
            exit(1);
        }
    }
    
    if((argc-1) != optind)
    {
        usage(argv[0]);
        exit(1);
    }

    fwobjectname = strdup( argv[optind++] );


    if (wdir==0) 	wdir="./";

    if(chdir(wdir))   {
	cerr << _("Can't change to: ") << wdir << endl;
	exit(1);
    }


    try   {

        libfwbuilder::init();

        new Resources(TEMPLATE_DIR "/resources.xml");

	/* create database */
	new FWObjectDatabase();

	/* load the data file */
	UpgradePredicate upgrade_predicate; 

	if (verbose) cout << _(" *** Loading data ...");
	FWObjectDatabase::db->load(filename,  &upgrade_predicate);
	if (verbose) cout << _(" done\n");


	string   fw_file_name=string(fwobjectname)+".fw";
	ofstream fw_file(fw_file_name.c_str());

	/* Review firewall and OS options and generate commands */
	Firewall*  fw=FWObjectDatabase::db->findFirewallByName(fwobjectname);

        /* some initial sanity checks */
        list<FWObject*> l2=fw->getByType(Interface::TYPENAME);
        for (list<FWObject*>::iterator i=l2.begin(); i!=l2.end(); ++i) 
        {
            Interface *iface=dynamic_cast<Interface*>(*i);
            assert(iface);

            string::size_type n;
            if ( iface->isDyn())  
            {
                list<FWObject*> l3=iface->getByType(IPv4::TYPENAME);
                if (l3.size()>0)
                {
                    char errstr[256];
                    for (list<FWObject*>::iterator j=l3.begin(); j!=l3.end(); ++j) 
                        if ( FWObjectDatabase::db->findAllReferences(*j).size()!=0 )
                        {
                            sprintf(errstr,
                                    _("Dynamic interface %s has an IP address that is used in the firewall policy rule.\n"),
                                    iface->getName().c_str() );
                            throw FWException(errstr);
                        }

                    sprintf(errstr,
                            _("Dynamic interface %s should not have an IP address object attached to it. This IP address object will be ignored.\n"),
                            iface->getName().c_str() );
                    cerr << errstr;
                    for (list<FWObject*>::iterator j=l3.begin(); j!=l3.end(); ++j) 
                        iface->remove(*j);
                }
            }
        }


	FWOptions* options=fw->getOptionsObject();
	string s;

	string firewall_dir=options->getStr("firewall_dir");
	if (firewall_dir=="") firewall_dir="/etc";

	bool debug=options->getBool("debug");
	string shell_dbg=(debug)?"-x":"" ;

	fw_file << "#!/bin/sh " << shell_dbg << endl << endl;
/*
 * Process firewall options, build OS network configuration script
 */
	OSConfigurator *oscnf=NULL;
	if (fw->getStr("host_OS")=="macosx")    oscnf=new OSConfigurator_macosx(FWObjectDatabase::db , fwobjectname);
	if (fw->getStr("host_OS")=="freebsd")   oscnf=new OSConfigurator_freebsd(FWObjectDatabase::db , fwobjectname);
	if (oscnf==NULL)
	    throw FWException(_("Unrecognized host OS ")+fw->getStr("host_OS"));

	oscnf->prolog();
/*
 * create compilers and run the whole thing 
 */
        Manifest manifest;
	Action a;

	PolicyCompiler_ipfw c( FWObjectDatabase::db , fwobjectname , oscnf );

	c.setDebugLevel( dl );
	c.setDebugRule(  drp );
	c.setVerbose( verbose );

	bool     have_ipfw=false;
	if ( c.prolog() > 0 ) 
        {
	    have_ipfw=true;

	    c.compile();
	    c.epilog();

	    a.action     = "COPY";
	    a.filename   = fw_file_name;
	    a.parameters = firewall_dir;
	    manifest.push_back(a);
	}

/*
 * now write generated scripts to files
 */


        char          *timestr;
        time_t         tm;
        struct tm     *stm;

        tm=time(NULL);
        stm=localtime(&tm);
        timestr=strdup(ctime(&tm));
        timestr[ strlen(timestr)-1 ]='\0';
    
        char* user_name=(char*)g_get_user_name();
        if (user_name==NULL) {
            cerr << _("Can't figure out your user name, aborting") << endl;
            exit(1);
        }

        fw_file << _("#\n\
#  This is automatically generated file. DO NOT MODIFY !\n\
#\n\
#  Firewall Builder  fwb_ipfw v") << VERSION << "-" << RELEASE_NUM << _(" \n\
#\n\
#  Generated ") << timestr << " " << tzname[stm->tm_isdst] << _(" by ") 
               << user_name << "\n#\n#\n";

        string fwcomment=fw->getComment();
        string::size_type n1,n2;
        n1=n2=0;
        while ( (n2=fwcomment.find("\n",n1))!=string::npos )
        {
            fw_file << "#  " << fwcomment.substr(n1,n2-n1) << endl;
            n1=n2+1;
        }
        fw_file << "#  " << fwcomment.substr(n1) << endl;
        fw_file << "#\n#\n#\n";

        fw_file << "cd " << firewall_dir << " || exit 1" << endl << endl;

	fw_file << oscnf->getCompiledScript();

        fw_file << endl;

        fw_file << "log \"";
        fw_file << _("Activating firewall script generated ")
               << timestr << " " << tzname[stm->tm_isdst] << _(" by ")
               << user_name;
        fw_file << "\"" << endl;

	fw_file << endl;




        fw_file << "$IPFW -f flush" << endl;
        fw_file << endl;

	if (have_ipfw)
        {
	    fw_file << c.getCompiledScript();
        }

	fw_file << endl;

	if(mfile)
            manifest.save(string(mfile), S_IRUSR|S_IWUSR);
        
        chmod(fw_file_name.c_str(),S_IXUSR|S_IRUSR|S_IWUSR|S_IRGRP);
        
        return 0;

    } catch(FWException &ex)  {
	cerr << ex.toString() << endl;
        return 1;
    } catch (std::string s) {
	cerr << s;
        return 1;
    } catch (std::exception ex) {
	cerr << ex.what();
        return 1;
    } catch (...) {
	cerr << _("Unsupported exception");
        return 1;
    }

}









