/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: InterfaceDialog.cpp,v 1.29 2007/04/14 00:18:43 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"
#include "global.h"
#include "utils.h"

#include "InterfaceDialog.h"
#include "ObjectManipulator.h"
#include "FWWindow.h"

#include "fwbuilder/Library.h"
#include "fwbuilder/Interface.h"
#include "fwbuilder/Interface.h"
#include "fwbuilder/Management.h"
#include "fwbuilder/FWException.h"
#include "fwbuilder/Host.h"
#include "fwbuilder/Firewall.h"
#include "fwbuilder/ObjectGroup.h"
#include "fwbuilder/Resources.h"

#include <qlineedit.h>
#include <qspinbox.h>
#include <qcheckbox.h>
#include <qtextedit.h>
#include <qcombobox.h>
#include <qmessagebox.h>
#include <qcheckbox.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <qpushbutton.h>
#include <qlabel.h>

using namespace std;
using namespace libfwbuilder;

InterfaceDialog::InterfaceDialog(QWidget *parent) :
    InterfaceDialog_q(parent)
{
/*
    seclevel->hide();    seclevelLabel->hide();
    netzone->hide();     netzoneLabel->hide();
*/
    obj=NULL;
}

void InterfaceDialog::loadFWObject(FWObject *o)
{
    obj=o;
    Interface *s = dynamic_cast<Interface*>(obj);
    assert(s!=NULL);

    init=true;

    fillLibraries(libs,obj);

    obj_name->setText( QString::fromUtf8(s->getName().c_str()) );
    label->setText( QString::fromUtf8(s->getLabel().c_str()) );

    regular->setChecked( ! s->isDyn() &&
                         ! s->isUnnumbered() &&
                         ! s->isBridgePort() );
    dynamic->setChecked( s->isDyn() );
    unnumbered->setChecked( s->isUnnumbered() );
    bridgeport->setChecked( s->isBridgePort() );

    management->setChecked( s->isManagement() );

    comment->setText( QString::fromUtf8(s->getComment().c_str()) );

/* interface should always belong to the host or firewall so we can't
 * move them from library to library */
    libs->setEnabled( false );

    FWObject *f=obj->getParent();

/* if parent is a host, hide checkbox 'external', security level and netzone */
    if (Host::isA( f ))
    {
        ext->setEnabled( false );
        ext->hide();
        seclevel->hide();
        seclevelLabel->hide();
        netzone->hide();
        netzoneLabel->hide();
    }

    bool supports_security_levels=false;
    bool supports_network_zones  =false;

    try  {
        supports_security_levels=
            Resources::getTargetCapabilityBool(f->getStr("platform"), "security_levels");
        supports_network_zones=
            Resources::getTargetCapabilityBool(f->getStr("platform"), "network_zones");
    } catch (FWException &ex)  { }

/* if parent is a firewall, it is more complex ... */
    if (Firewall::isA( f ))
    {
        if (supports_security_levels)
        {
            seclevel->show();
            seclevelLabel->show();
            ext->hide();
            seclevel->setValue( obj->getInt("security_level") );
        } else
        {
            seclevel->hide();
            seclevelLabel->hide();
            ext->show();
            ext->setChecked( obj->getInt("security_level")==0 );
        }

        if (supports_network_zones)
        {
            netzone->show();
            netzoneLabel->show();

            netzoneObjectIDs.clear();
            netzoneObjectNos.clear();

            QStringList netzoneObjectNames;

            int n=0;

            netzoneObjectIDs["sysid0"]=n;
            netzoneObjectNos[n]="sysid0";
            netzoneObjectNames.push_back(" Any ");

            ++n;

/* TODO: try to make this widget show object with appropriate icon */

            list<FWObject*> libs=mw->db()->getByType( Library::TYPENAME );
            for (list<FWObject*>::iterator l=libs.begin(); l!=libs.end(); ++l)
            {
                FWObject *library= *l;
                FWObject *o1,*o2;

                if ( library->getId()==DELETED_LIB ) continue;

                o1=library->findObjectByName(ObjectGroup::TYPENAME,"Objects");
                assert(o1!=NULL);
                o2=o1->findObjectByName(ObjectGroup::TYPENAME,"Groups");
                if (o2==NULL)
                {
                    if (fwbdebug)
                        qDebug("InterfaceDialog::loadFWObject  missing Groups group in %s",o1->getId().c_str());
                    continue;
                }
//                assert(o2!=NULL);

                for (FWObject::iterator i=o2->begin(); i!=o2->end(); ++i)
                {
                    netzoneObjectIDs[ (*i)->getId().c_str() ]=n;
                    netzoneObjectNos[n]= (*i)->getId().c_str();
                    netzoneObjectNames.push_back(
                        tr("Group: ")+ (*i)->getName().c_str() );
                    ++n;
                }

                o2=o1->findObjectByName(ObjectGroup::TYPENAME,"Networks");
                if (o2==NULL)
                {
                    if (fwbdebug)
                        qDebug("InterfaceDialog::loadFWObject  missing Networks group in %s",o1->getId().c_str());
                    continue;
                }
//                assert(o2!=NULL);

                for (FWObject::iterator i1=o2->begin(); i1!=o2->end(); ++i1)
                {
                    netzoneObjectIDs[ (*i1)->getId().c_str() ]=n;
                    netzoneObjectNos[n]= (*i1)->getId().c_str();
                    netzoneObjectNames.push_back(
                        tr("Network: ")+ (*i1)->getName().c_str() );
                    ++n;
                }
            }

            netzone->clear();            
            netzone->insertStringList( netzoneObjectNames );

            QString id=obj->getStr("network_zone").c_str();
            if (id=="") id="sysid0";  // any network
            netzone->setCurrentItem( netzoneObjectIDs[id] );
        }
        else
        {
            netzone->hide();
            netzoneLabel->hide();
        }
    }

    //apply->setEnabled( false );

    obj_name->setEnabled(!o->isReadOnly());
    setDisabledPalette(obj_name);

    libs->setEnabled(!o->isReadOnly());
    setDisabledPalette(libs);

    comment->setReadOnly(o->isReadOnly());
    setDisabledPalette(comment);

    label->setEnabled(!o->isReadOnly());
    setDisabledPalette(label);

    regular->setEnabled(!o->isReadOnly());
    setDisabledPalette(regular);

    dynamic->setEnabled(!o->isReadOnly());
    setDisabledPalette(dynamic);

    unnumbered->setEnabled(!o->isReadOnly());
    setDisabledPalette(unnumbered);

    bridgeport->setEnabled(!o->isReadOnly());
    setDisabledPalette(bridgeport);

    management->setEnabled(!o->isReadOnly());
    setDisabledPalette(management);

    ext->setEnabled(!o->isReadOnly());
    setDisabledPalette(ext);

    seclevel->setEnabled(!o->isReadOnly());
    setDisabledPalette(seclevel);

    netzone->setEnabled(!o->isReadOnly());
    setDisabledPalette(netzone);


    init=false;
}
    
void InterfaceDialog::changed()
{
    if (fwbdebug)
        qDebug("InterfaceDialog::changed()");

    //apply->setEnabled( true );
    emit changed_sign();
}

void InterfaceDialog::validate(bool *res)
{
    *res=true;
    if (!isTreeReadWrite(this,obj)) *res=false;
    if (!validateName(this,obj,obj_name->text())) *res=false;
}

void InterfaceDialog::isChanged(bool *res)
{
    //*res=(!init && apply->isEnabled());
}

void InterfaceDialog::libChanged()
{
    changed();
}

void InterfaceDialog::applyChanges()
{
    Interface *s = dynamic_cast<Interface*>(obj);
    assert(s!=NULL);

    string oldname=obj->getName();
    string oldlabel=s->getLabel();
    obj->setName( string(obj_name->text().utf8()) );
    obj->setComment( string(comment->text().utf8()) );

    s->setLabel( string(label->text().utf8()) );
    s->setDyn( dynamic->isChecked() );
    s->setUnnumbered( unnumbered->isChecked() );
    s->setBridgePort( bridgeport->isChecked() );

    FWObject *f = obj->getParent();
    bool supports_security_levels =false;
    bool supports_network_zones   =false;

    try  {
        supports_security_levels=
            Resources::getTargetCapabilityBool(f->getStr("platform"), "security_levels");
        supports_network_zones=
            Resources::getTargetCapabilityBool(f->getStr("platform"), "network_zones");
    } catch (FWException &ex)  { }


    if (Firewall::isA( f ))
    {
        if (supports_security_levels)
            obj->setInt("security_level", seclevel->value() );
        else
            obj->setInt("security_level", (ext->isChecked()) ? 0 : 100 );

        if (supports_network_zones)
            obj->setStr("network_zone",
                        netzoneObjectNos[ netzone->currentItem() ].latin1() );

        s->setManagement( management->isChecked() );
    }

    om->updateObjName(obj,
                      QString::fromUtf8(oldname.c_str()),
                      QString::fromUtf8(oldlabel.c_str()));

    //apply->setEnabled( false );
    om->updateLastModifiedTimestampForAllFirewalls(obj);
}

void InterfaceDialog::discardChanges()
{
    loadFWObject(obj);
}


/* ObjectEditor class connects its slot to this signal and does all
 * the verification for us, then accepts (or not) the event. So we do
 * nothing here and defer all the processing to ObjectEditor
 */
void InterfaceDialog::closeEvent(QCloseEvent *e)
{
    if (fwbdebug)
        qDebug("InterfaceDialog::coseEvent  got close event: %p",e);
    emit close_sign(e);
}

