/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: newFirewallDialog.cpp,v 1.32 2007/05/23 03:05:51 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "config.h"
#include "global.h"
#include "utils.h"
#include "utils_no_qt.h"
#include "platforms.h"

#include "newFirewallDialog.h"
#include "InterfaceData.h"
#include "ObjectManipulator.h"
#include "FWWindow.h"
#include "ObjConflictResolutionDialog.h"
#include "upgradePredicate.h"

#include "fwbuilder/Library.h"
#include "fwbuilder/Firewall.h"
#include "fwbuilder/Resources.h"
#include "fwbuilder/Policy.h"
#include "fwbuilder/InterfacePolicy.h"
#include "fwbuilder/BackgroundOp.h"

#include <qlineedit.h>
#include <qtextedit.h>
#include <qcombobox.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qcheckbox.h>
#include <qlistview.h>
#include <qtextbrowser.h>
#include <qmessagebox.h>
#include <qtimer.h>
#include <qlistbox.h>
#include <qapplication.h>
#include <qcursor.h>
#include <qpixmapcache.h>

#include <iostream>

// must be the last for win
#include "fwbuilder/snmp.h"

//#undef HAVE_LIBSNMP

using namespace libfwbuilder;
using namespace std;

newFirewallDialog::newFirewallDialog() : newFirewallDialog_q()
{
    nfw = NULL; 
    tmpldb = NULL;
    snmpPollCompleted = false;
    q = NULL;
    unloadTemplatesLib = false;
    getInterfacesBusy = false;

    timer = new QTimer(this);
    connect( timer, SIGNAL(timeout()), this, SLOT(monitor()) );

/* fill in platform */
    setPlatform(platform, "" );

/* fill in host OS  */
    setHostOS(hostOS, "" );

    setNextEnabled( QWizard::page(0), false );
    for (int i=0; i<pageCount(); ++i)
        setHelpEnabled( QWizard::page(i), false );

    iface_list->setItemMargin( 1 );
    iface_list->setAllColumnsShowFocus( true );

    iface_sl_list->setItemMargin( 1 );
    iface_sl_list->setAllColumnsShowFocus( true );
    QToolTip::add(iface_dyn,wordWrap(tr("Check option 'dynamic address' for the interface that gets its IP address dynamically via DHCP or PPP protocol.") ,80 ));
    QToolTip::add(iface_unnum,wordWrap(tr("Check option 'Unnumbered interface' for the interface that does not have an IP address. Examples of interfaces of this kind are those used to terminate PPPoE or VPN tunnels.") ,80 ));
    obj_name->setFocus();
}

newFirewallDialog::~newFirewallDialog()
{
    if (timer!=NULL) delete timer;
#ifdef HAVE_LIBSNMP
    if (q!=NULL) delete q;
#endif
}

void newFirewallDialog::changed()
{
    int p = indexOf( currentPage() );
    if (p==0)
    {
        setNextEnabled( QWizard::page(p), !obj_name->text().isEmpty() );
    }

    if (p==1)
    {

        bool f;

#ifdef HAVE_LIBSNMP
        f = use_snmp->isChecked();
#else
        f = false;
        use_snmp->setEnabled( f );
#endif

        snmp_community->setEnabled( f );
        snmpQuery->setEnabled( f );
        snmpProgress->setEnabled( f );
        if (f) snmp_community->setFocus();

        f = use_manual->isChecked() || snmpPollCompleted;
        setNextEnabled( QWizard::page(1), f );
    }

    if (p==2)
    {
        if (iface_dyn->isChecked() ||
            iface_unnum->isChecked() ||
            iface_bridgeport->isChecked())
        {
            iface_addr->clear();
            iface_addr->setEnabled(false);
            iface_netmask->clear();
            iface_netmask->setEnabled(false);
        } else
        {
            iface_addr->setEnabled(true);
            iface_netmask->setEnabled(true);
        }
    }
}

void  newFirewallDialog::monitor()
{
    if (logger==NULL || q==NULL) return;

#ifdef HAVE_LIBSNMP

    if( logger->ready() )
    {
        QString str = logger->getLine().c_str();
        snmpProgress->moveCursor( QTextEdit::MoveEnd , false );
        snmpProgress->insert( str );
        return;
    }

    if (q->isRunning()) return;

    timer->stop();

    const map<int, Interface> &intf = q->getInterfaces();
    for(map<int, Interface>::const_iterator i=intf.begin();i!=intf.end(); ++i)
    {
        if ( i->second.isUp() ) 
        {
            InterfaceData idata( i->second );

            idata.guessLabel( readPlatform(platform).latin1() );

            QString dn;
            if (idata.isDyn)        dn+="dyn";
            if (idata.isUnnumbered) dn+="unn";
            if (idata.isBridgePort) dn+="bridge";

            new QListViewItem(iface_list, 
                              idata.name.c_str(),
                              idata.label.c_str(),
                              idata.address.c_str(),
                              idata.netmask.c_str(),
                              dn,
                              idata.physicalAddress.c_str() );

//            cerr << "Added interface " << idata.name << endl;

        }
    }

    delete q;
    q=NULL;

#endif

    snmpPollCompleted=true;
    setNextEnabled( QWizard::page(1), true );
}

void newFirewallDialog::getInterfacesViaSNMP()
{
#ifdef HAVE_LIBSNMP

// need to protect from reentry because getAddrByName processes events
    if (q!=NULL || getInterfacesBusy) return;

    snmpPollCompleted=false;
    iface_list->clear();

    string rcomm=snmp_community->text().latin1();

    if ( rcomm.empty() ) 
    {
        QMessageBox::warning(
            this,"Firewall Builder", 
            tr("Missing SNMP community string."),
            "&Continue", QString::null, QString::null, 0, 1 );
        return ;
    }

    getInterfacesBusy = true;

    IPAddress addr;
    QString name=obj_name->text().latin1();
    try 
    {
        QApplication::setOverrideCursor( QCursor( Qt::WaitCursor) );
        QString a = getAddrByName(name);
        QApplication::restoreOverrideCursor();
        addr = a.ascii();
    } catch (FWException &ex)
    {
        QMessageBox::warning(
            this,"Firewall Builder", 
            tr("Address of %1 could not be obtained via DNS")
            .arg(obj_name->text()),
            "&Continue", QString::null, QString::null, 0, 1 );
        getInterfacesBusy = false;
        return ;
    }

    logger=NULL;
    snmpProgress->clear();

    if (q!=NULL) delete q;
    q=new SNMP_interface_query();
    q->init(addr.toString(),rcomm,SNMP_DEFAULT_RETRIES,SNMP_DEFAULT_TIMEOUT);

    timer->start(0, false);
    
    try
    {
        logger = q->start_operation();

    } catch(const FWException &ex)
    {
        //do nothing
    }

    getInterfacesBusy = false;

#endif
}

bool newFirewallDialog::appropriate(QWidget *page) const
{
    int p  = indexOf( page );

    if (fwbdebug)
    {
        qDebug("newFirewallDialog::appropriate  p=%d",p);
    }

    switch (p)
    {
    case 0:
    case 4:
        return true;

    case 1:
    case 2:
    case 3:
        return (!useTemplate->isChecked());
    }
    return true;
}

void newFirewallDialog::selected(const QString &title)
{
    int p = indexOf( currentPage() );

// p is a page number _after_ it changed
    switch (p)
    {
    case 1:
    {
        changed();  // to properly enable/disable widgets
        break;
    }

    case 2:
    {
        iface_name->setFocus();

        if (!Resources::getTargetCapabilityBool(readPlatform(platform).latin1(),
                                                "security_levels") )
        {
/* if chosen fw platform does not support security levels,
 * this is the last page
 */
            setNextEnabled( QWizard::page(2), false );
            setFinishEnabled( QWizard::page(2), true );
        }
        break;
    }

    case 3:
    {
        if (useTemplate->isChecked())
        {
            showPage( QWizard::page(0) );
            return;
        }

        fillInterfaceSLList();

        setFinishEnabled( QWizard::page(3), true );
        break;
    }

    case 4:
    {
        setFinishEnabled( QWizard::page(4), true );
/* load templates if not loaded */
        if (tmpldb==NULL)
        {

            MessageBoxUpgradePredicate upgrade_predicate(this);

            tmpldb = new FWObjectDatabase();
            tmpldb->setReadOnly( false );
            tmpldb->load( tempfname, &upgrade_predicate, librespath);
        }
        FWObject *tlib = tmpldb->getById(TEMPLATE_LIB);

#if 0
        FWObject *tlib = mw->db()->getById(TEMPLATE_LIB);
        if (tlib==NULL)
        {
            FWObject *cl = om->getCurrentLib();
            mw->loadLibrary(tempfname);
            unloadTemplatesLib = true;
            om->loadObjects();
            tlib = mw->db()->getById(TEMPLATE_LIB);
/* restore library that was opened prior loading templates */
            om->openLib(cl);
        }
#endif

        list<FWObject*> fl;
        findFirewalls(tlib, fl, false);

        QString icn = QString( Resources::global_res->getObjResourceStr(fl.front(), "icon-tree").c_str() );

        templateList->clear();

        int n=0;
        for (list<FWObject*>::iterator m=fl.begin(); m!=fl.end(); m++,n++)
        {
            FWObject *o=*m;

            QPixmap pm;
            if ( ! QPixmapCache::find( icn, pm) )
            {
                pm = QPixmap::fromMimeSource( icn );
                QPixmapCache::insert( icn, pm);
            }

            templateList->insertItem(pm, o->getName().c_str() );
            templates[ templateList->item( templateList->count()-1 ) ]=o;
        }
        templateList->setCurrentItem(0);
        templateList->setFocus();
        break;
    }
    }
}

void newFirewallDialog::fillInterfaceSLList()
{

    QListViewItem *itm = iface_list->firstChild();

    iface_sl_list->clear();

    while (itm!=NULL)
    {
        InterfaceData idata;

        idata.name         =  itm->text(0).latin1();
        idata.label        =  itm->text(1).latin1();
        idata.isDyn        =  itm->text(4).find("Dyn")!=-1;
        idata.isUnnumbered =  itm->text(4).find("Unn")!=-1;
        idata.isBridgePort =  itm->text(4).find("Bridge")!=-1;

        if (!idata.isDyn && !idata.isUnnumbered && !idata.isBridgePort)
            idata.address  =  itm->text(2).latin1();
        else
            idata.address  =  QObject::tr("dynamic").latin1();

        try 
        {
            idata.guessSecurityLevel( readPlatform(platform).latin1() );
        }
        catch (FWException &ex)
        {
            QMessageBox::warning(
                this,"Firewall Builder", ex.toString().c_str(),
                "&Continue", QString::null, QString::null, 0, 1 );
            
            showPage( QWizard::page(2) );
            return;
        }

        new QListViewItem(iface_sl_list, 
                          idata.name.c_str(),
                          idata.label.c_str(),
                          idata.address.c_str(),
                          QString::number(idata.securityLevel) );
        itm=itm->itemBelow();
    }
}

void newFirewallDialog::templateSelected(QListBoxItem *itm)
{
    FWObject *o=templates[itm];
    assert (o!=NULL);

    Firewall *fw = Firewall::cast(o);

    templateComment->clear();
    QString s=QString("<a name=\"top\">\n") + fw->getComment().c_str();
    templateComment->append( s );
    templateComment->scrollToAnchor("top");

    bool haveOutside = false;
    bool haveInside  = false;
    bool haveDMZ     = false;
    list<FWObject*> ll = fw->getByType(Interface::TYPENAME);
    for (FWObject::iterator i=ll.begin(); i!=ll.end(); i++)
    {
        Interface *intf = Interface::cast( *i );
        if (intf->getLabel()=="outside")
        {
            haveOutside=true;
            intfOutsideLine->show();
            intfOutsideText->show();
            fillInterfaceData(intf,intfOutsideText);
        }
        if (intf->getLabel()=="inside")
        {
            haveInside=true;
            intfInsideLine->show();
            intfInsideText->show();
            fillInterfaceData(intf,intfInsideText);
        }
        if (intf->getLabel()=="dmz")
        {
            haveDMZ=true;
            intfDMZLine->show();
            intfDMZText->show();
            fillInterfaceData(intf,intfDMZText);
        }
    }

    if (!haveOutside) { intfOutsideLine->hide(); intfOutsideText->hide(); }
    if (!haveInside)  { intfInsideLine->hide();  intfInsideText->hide();  }
    if (!haveDMZ)     { intfDMZLine->hide();     intfDMZText->hide();     }
}

void newFirewallDialog::fillInterfaceData(Interface *intf, QTextBrowser *qte)
{
    qte->clear();
    QString s;

    s += "<table border='0' cellspacing='0' cellpadding='0'>";

    s += "<tr>";
    s += "<td>";
    s +=  tr("Interface: %1 (%2)")
        .arg(intf->getName().c_str())
        .arg(intf->getLabel().c_str());
    s += "</td>";
    s += "</tr>";

    s += "<tr>";
    s += "<td>";
    if (intf->isDyn()) s +=  tr("Dynamic address");
    else
        if (intf->isUnnumbered()) s +=  tr("Unnumbered interface");
        else
            if (intf->isBridgePort()) s +=  tr("Bridge port");
            else
                s += QString("%1/%2")
                    .arg(intf->getAddress().toString().c_str())
                    .arg( intf->getNetmask().toString().c_str());
    s += "</td>";
    s += "</tr>";
    s += "</table>";
    qte->setText(s);
}

void newFirewallDialog::addInterface()
{
    QString dn = "";
    if (iface_dyn->isChecked())        dn+="Dyn";
    if (iface_unnum->isChecked())      dn+="Unn";
    if (iface_bridgeport->isChecked()) dn+="Bridge";

    QString addr;
    QString netm;

    if (!iface_dyn->isChecked() &&
        !iface_unnum->isChecked() &&
        !iface_bridgeport->isChecked())
    {
        addr = iface_addr->text();
        netm = iface_netmask->text();

        if (addr.isEmpty()) addr="0.0.0.0";
        if (netm.isEmpty()) netm="0.0.0.0";

        try
        {
            IPAddress(addr.latin1());
            Netmask(netm.latin1());
        }
        catch (FWException &ex)
        {
            QMessageBox::warning(
                this,"Firewall Builder", 
                tr("Illegal address '%1/%2'").arg(addr).arg(netm),
                "&Continue", QString::null, QString::null, 0, 1 );
            return;
        }
    }
    new QListViewItem(iface_list, 
                      iface_name->text(),
                      iface_label->text(),
                      addr,
                      netm,
                      dn,
                      iface_physaddr->text() );
}

void newFirewallDialog::selectedInterface(QListViewItem *itm)
{
    iface_name->setText( itm->text(0) );
    iface_label->setText( itm->text(1) );
    iface_addr->setText( itm->text(2) );
    iface_netmask->setText( itm->text(3) );
    iface_reg->setChecked( itm->text(4).isEmpty() );
    iface_dyn->setChecked( itm->text(4).find("Dyn")!=-1 );
    iface_unnum->setChecked( itm->text(4).find("Unn")!=-1 );
    iface_bridgeport->setChecked( itm->text(4).find("Bridge")!=-1 );
    iface_physaddr->setText( itm->text(5) );
}

void newFirewallDialog::updateInterface()
{
    QString dn = "";
    if (iface_dyn->isChecked())   dn+="Dyn";
    if (iface_unnum->isChecked()) dn+="Unn";
    if (iface_bridgeport->isChecked()) dn+="Bridge";

    QListViewItem *itm = iface_list->selectedItem();
    if (itm==NULL) return;

    itm->setText( 0 , iface_name->text() );
    itm->setText( 1 , iface_label->text() );
    itm->setText( 2 , iface_addr->text() );
    itm->setText( 3 , iface_netmask->text() );
    itm->setText( 4 , dn );
    itm->setText( 5 , iface_physaddr->text() );
}

void newFirewallDialog::deleteInterface()
{
    QListViewItem *itm = iface_list->selectedItem();
    if (itm==NULL) return;
    iface_list->takeItem( itm );
}

void newFirewallDialog::adjustSL(QListViewItem *itm1)
{
// interface 1 is above 2. Adjust their security levels accordingly
    int sl1 = itm1->text(3).toInt();
    QListViewItem *itm2 = itm1->itemBelow();        
    QListViewItem *itm3 = itm1->itemAbove();        

    if (itm2==NULL) sl1=100;
    else
    {
        if (itm3==NULL) sl1=0;
        else
        {
            int sl2 = itm2->text(3).toInt();
            int sl3 = itm3->text(3).toInt();
            sl1 = (sl2+sl3)/2;
        }
    }
    itm1->setText( 3 , QString("%1").arg(sl1) );
}

void newFirewallDialog::upInterface()
{
    QListViewItem *itm1 = iface_sl_list->selectedItem();
    if (itm1==NULL) return;
    QListViewItem *itm2 = itm1->itemAbove();
    if (itm2==NULL) return;
    itm2->moveItem(itm1);
    adjustSL(itm1);
}

void newFirewallDialog::downInterface()
{
    QListViewItem *itm1 = iface_sl_list->selectedItem();
    if (itm1==NULL) return;
    QListViewItem *itm2 = itm1->itemBelow();
    if (itm2==NULL) return;
    itm1->moveItem(itm2);
    adjustSL(itm1);
}


void newFirewallDialog::accept()
{
    int p = indexOf( currentPage() );
    
    if (p==2)  fillInterfaceSLList();

    if (p==4)
    {
        QListBoxItem *itm = templateList->item( templateList->currentItem() );
        FWObject *template_fw=templates[itm];
        assert (template_fw!=NULL);

        FWObject *no = om->duplicateObject(om->getCurrentLib(),
                                           template_fw,
                                           obj_name->text(),
                                           false );  // do not ask to autorename
        if (no==NULL)
        {
            QDialog::accept();
            return;
        }

        map<string,string> platforms = Resources::getPlatforms();
        map<string,string>::iterator i;
        for (i=platforms.begin(); i!=platforms.end(); i++)
            Resources::setDefaultTargetOptions( i->first , Firewall::cast(no) );

        map<string,string> OSs = Resources::getOS();
        for (i=OSs.begin(); i!=OSs.end(); i++)
            Resources::setDefaultTargetOptions( i->first  , Firewall::cast(no) );

        no->setStr("platform", readPlatform(platform).latin1() );
        no->setStr("host_OS",  readHostOS(hostOS).latin1() );

        nfw=Firewall::cast(no);
    } else
    {
        FWObject *o;
        o=om->createObject(Firewall::TYPENAME, obj_name->text() );

        if (o==NULL)
        {
            QDialog::accept();
            return;
        }

        map<string,string> platforms = Resources::getPlatforms();
        map<string,string>::iterator i;
        for (i=platforms.begin(); i!=platforms.end(); i++)
            Resources::setDefaultTargetOptions( i->first , Firewall::cast(o) );

        map<string,string> OSs = Resources::getOS();
        for (i=OSs.begin(); i!=OSs.end(); i++)
            Resources::setDefaultTargetOptions( i->first  , Firewall::cast(o) );

        o->setStr("platform", readPlatform(platform).latin1() );
        o->setStr("host_OS",  readHostOS(hostOS).latin1() );

        nfw=Firewall::cast(o);

/* create interfaces */

        QListViewItem *itm = iface_list->firstChild();

        while (itm!=NULL)
        {
            QString name    =  itm->text(0);
            QString label   =  itm->text(1);
            QString addr    =  itm->text(2);
            QString netmask =  itm->text(3);
            bool    dyn     =  itm->text(4).find("Dyn")!=-1;
            bool    unnum   =  itm->text(4).find("Unn")!=-1;
            bool    bridgeport =  itm->text(4).find("Bridge")!=-1;
            QString physaddr=  itm->text(5);

            QListViewItem *itm2 = iface_sl_list->findItem( name , 0 );
            assert(itm2!=NULL);

            int     sl = itm2->text(3).toInt();

            Interface *oi = Interface::cast(om->createObject(nfw,Interface::TYPENAME,
                                                             name));
#ifdef USE_INTERFACE_POLICY
            oi->add(new InterfacePolicy());
#endif
            oi->setLabel( label.latin1() );

            oi->setDyn(dyn);
            oi->setUnnumbered(unnum);
            oi->setBridgePort(bridgeport);
            oi->setSecurityLevel(sl);

            if (!dyn && !unnum && !bridgeport)
            {
                QString addrname=QString("%1:%2:ip").arg(obj_name->text()).arg(name);
                IPv4 *oa = IPv4::cast(om->createObject(oi, IPv4::TYPENAME,addrname));
                oa->setAddress( addr.latin1()    );
                oa->setNetmask( netmask.latin1() );
            }
            // updateObjName has a side effect: it causes redraw of the ruleset
            // views in the main window
            om->updateObjName(oi,"","",false);

            itm=itm->itemBelow();
        }
    }
    if (unloadTemplatesLib)
    {
#if 0
        FWObject *tlib = mw->db()->getById(TEMPLATE_LIB);
        assert (tlib!=NULL);
        string tlibID = tlib->getId();
        if (fwbdebug)
            qDebug("newFirewallDialog::accept  Delete template library");
        om->delObj(tlib,false);

/*
 * deleting an object places it in the "Deleted objects" library, so
 * we need to remove "templates" library from there.
 *
 * TODO: need to add flags to the API to be able to delete objects
 * without placing them in "Deleted objects" automatically
 */
        FWObject *delObjLib = mw->db()->getById( DELETED_LIB );
        if (delObjLib!=NULL && delObjLib->getById(tlibID)!=NULL)
        {
            if (fwbdebug) qDebug("newFirewallDialog::accept  Delete library of templates from 'Deleted objects'");
            om->delObj(tlib,false);  // this time from deleted objects lib
        }
#endif

        delete tmpldb;
        tmpldb = NULL;
        unloadTemplatesLib=false;
    }
    QDialog::accept();
}



