/* GADMIN-DHCPD - An easy to use GTK+ frontend for ISC DHCPD.
 * Copyright (C) 2004 - 2009 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/



#include "../config.h"
#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "allocate.h"
#include "gettext.h"
#include "widgets.h"
#include "functions.h"
#include "make_settings_entries.h"
#include "make_settings_combos.h"
#include "populate_scope_treeview.h"
#include "select_first_scope.h"
#include "set_num_ranges.h"
#include "populate_ranges.h"
#include "populate_scope_settings.h"
#include "populate_host_treeview.h"
#include "populate_host_settings.h"
#include "populate_leases.h"
#include "reread_conf.h"
#include "loadbalance_window.h"
#include "failover_functions.h"
#include "get_ranges_from_current_scope.h"
#include "show_info.h"
#include "handle_ssh.h"

extern char global_netmask[1024];
extern char global_subnet[1024];
extern char global_nic[1024];

extern char LEASE_FILE_BUF[1024];
extern char DHCPD_CONF_BUF[1024];

extern int num_ranges;



/* Adds or changes a loadbalance declaration including
   the scopes pool with ranges inside */
void apply_loadbalance_button_clicked(struct w *widgets)
{
    /* The apply button in the loadbalancing window */
    FILE *fp;
    long file_size;
    char *line, *new_conf, *sec_conf, *range_from, *range_to;
    int i, once = 0;
    int found_scope = 0, failover_pool_added = 0;
    int added_range_end = 0;
    gint active_index;
    gchar *nic, *subnet, *netmask, *declaration, *range_line;
    gchar *info, *current_decl_name, *temp_decl_name;
    gchar *sec_declaration, *settings_conf;

    G_CONST_RETURN gchar *decl_name;
    G_CONST_RETURN gchar *address;
    G_CONST_RETURN gchar *port;
    G_CONST_RETURN gchar *remote_address;
    G_CONST_RETURN gchar *remote_port;
    G_CONST_RETURN gchar *max_response_delay;
    G_CONST_RETURN gchar *max_unacked_updates;
    G_CONST_RETURN gchar *loadbalance_max_seconds;
    G_CONST_RETURN gchar *mclt;
    G_CONST_RETURN gchar *split;

    /* Get the entry settings, the combo is used later */
    decl_name = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_entry[0]));
    address = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_entry[1]));
    port = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_entry[2]));
    remote_address = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_entry[3]));
    remote_port = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_entry[4]));
    max_response_delay = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_entry[5]));
    max_unacked_updates = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_entry[6]));
    loadbalance_max_seconds = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_entry[7]));
    mclt = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_entry[8]));
    split = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_entry[9]));

    /* All values must be filled in */
    for(i=0; i<10; i++)
    if( 0 == strlen(gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_entry[i]))) )
    {
	info = g_strdup_printf(_("All loadbalancing values must be filled in.\n"));
	show_info(info);
	g_free(info);
	return;
    }

    if( ! num_ranges )
    {
	info = g_strdup_printf(_("This scope does not have any ranges, add a range first.\n"));
	show_info(info);
	g_free(info);
	return;
    }


    subnet = g_strdup_printf("%s", global_subnet);
    netmask = g_strdup_printf("%s", global_netmask);
    nic = g_strdup_printf("%s", global_nic);

    /* No failover exists for this scope, add a new one */
    if( ! failover_exists(nic, subnet, netmask) )
    {
	/* Avoid warning, dont pass decl_name */
	temp_decl_name = g_strdup_printf("%s", decl_name);

	/* If the failover name exists then return */
	if( failover_name_exists(temp_decl_name) )
	{
	    info = g_strdup_printf(_("Another loadbalance declaration with this name\nalready exists, choose another.\n"));
	    show_info(info);
	    g_free(info);

	    g_free(nic);
	    g_free(subnet);
	    g_free(netmask);
	    g_free(temp_decl_name);
    	    return;
	}

	g_free(temp_decl_name);



	if((fp=fopen(DHCPD_CONF_BUF, "r"))==NULL)
	{
	    info = g_strdup_printf(_("Could not open dhcpd.conf here: %s\n"), DHCPD_CONF_BUF);
	    show_info(info);
	    g_free(info);

	    g_free(nic);
	    g_free(subnet);
	    g_free(netmask);
    	    return;
	}
	fseek(fp, 0, SEEK_END);
	file_size = ftell(fp);
	rewind(fp);

	line = allocate(file_size+1);

	new_conf = allocate(file_size+(350*10)+1024+1);
	sec_conf = allocate(file_size+(350*10)+1024+1);


	/* Get the primary/secondary combo setting */
	gint active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->loadbalance_combo[0]));
	if( active_index <= 0 )
	{
	    /* Create a Primary failover peer declaration */
	    declaration = g_strconcat(
	    "failover peer \"", decl_name, "\" {\n",
	    "    primary;\n",
	    "    address ", address, ";\n",
	    "    port ", port, ";\n",
	    "    peer address ", remote_address, ";\n", 
	    "    peer port ", remote_port, ";\n", 
	    "    max-response-delay ", max_response_delay, ";\n", 
	    "    max-unacked-updates ", max_unacked_updates, ";\n", 
	    "    load balance max seconds ", loadbalance_max_seconds, ";\n", 
	    "    mclt ", mclt, ";\n",
	    "    split ", split, ";\n",
	    "}\n\n",
	    NULL);

	    /* Create a secondary declaration suitable
	       for upload to a remote secondary failover */
	    sec_declaration = g_strconcat(
	    "failover peer \"", decl_name, "\" {\n",
	    "    secondary;\n",
	    "    address ", remote_address, ";\n",
	    "    port ", remote_port, ";\n",
	    "    peer address ", address, ";\n", 
	    "    peer port ", port, ";\n", 
	    "    max-response-delay ", max_response_delay, ";\n", 
	    "    max-unacked-updates ", max_unacked_updates, ";\n", 
//	    "    load balance max seconds ", loadbalance_max_seconds, ";\n", 
	    "    mclt ", mclt, ";\n",
//	    "    split ", split, ";\n",
	    "}\n\n",
	    NULL);
	     

	}
	else
	{
	    /* Create a Secondary failover peer declaration */
	    declaration = g_strconcat(
	    "failover peer \"", decl_name, "\" {\n",
	    "    secondary;\n",
	    "    address ", address, ";\n",
	    "    port ", port, ";\n",
	    "    peer address ", remote_address, ";\n", 
	    "    peer port ", remote_port, ";\n", 
	    "    max-response-delay ", max_response_delay, ";\n", 
	    "    max-unacked-updates ", max_unacked_updates, ";\n", 
//	    "    load balance max seconds ", loadbalance_max_seconds, ";\n", 
//	    "    mclt ", mclt, ";\n",     // Not in a secondary server (Works though)
//	    "    split ", split, ";\n",   // Not in a secondary server 

	    "}\n\n",
	    NULL);
	    
	    sec_declaration = g_strdup_printf("%s", "Never uploaded\n");
	}
	

	/* Scroll to the insertion point */
	if( file_size > 1 )
	while(fgets(line, file_size, fp)!=NULL)
	{
	    /* Another failover or a subnet declaration is found */
	    if( (! once && strstr(line, "failover peer") && strstr(line, "{"))
	    ||  (! once && strstr(line, "subnet") && strstr(line, "{")) )
	    {
		once = 1;
		/* Insert our new failover declaration */
		strcat(new_conf, declaration);
		strcat(new_conf, line);
		
		/* Insert the secondary adapted declaration */
		strcat(sec_conf, sec_declaration);
		strcat(sec_conf, line);
	    }
	    else
	    {
	        strcat(new_conf, line);
		strcat(sec_conf, line);
	    }
	}

	g_free(declaration);
	g_free(sec_declaration);
	
	/* Gather the rest of the configuration */
	if( file_size > 1 )
	while(fgets(line, file_size, fp)!=NULL)
	{
	    strcat(new_conf, line);	
	    strcat(sec_conf, line);
	}
	fclose(fp);
	free(line);
	
	
	/* Write the new configuration */
	if((fp=fopen(DHCPD_CONF_BUF, "w+"))==NULL)
	{
	    info = g_strdup_printf(_("Could not write dhcpd.conf here: %s\n"), DHCPD_CONF_BUF);
	    show_info(info);
	    g_free(info);

	    g_free(nic);
	    g_free(subnet);
	    g_free(netmask);
	    
	    free(new_conf);
	    free(sec_conf);
    	    return;
	}
	fputs(new_conf, fp);
	fclose(fp);
	free(new_conf);


	/* Write the new secondary configuration in the settings directory */
	settings_conf = g_strdup_printf("%s/%s", SETTINGS_PATH, "dhcpd.conf");
	if((fp=fopen(settings_conf, "w+"))==NULL)
	{
	    info = g_strdup_printf(_("Could not write dhcpd.conf here: %s\n"), settings_conf);
	    show_info(info);
	    g_free(info);

	    g_free(nic);
	    g_free(subnet);
	    g_free(netmask);
	    
	    free(sec_conf);
	    g_free(settings_conf);
    	    return;
	}
	fputs(sec_conf, fp);
	fclose(fp);
	free(sec_conf);
	g_free(settings_conf);


	/* Now add the ranges in this subnet to a failover pool declaration 
	   in the same subnet declaration */
	gchar *scope_line1 = g_strdup_printf("subnet %s netmask %s\n", global_subnet, global_netmask);
	gchar *scope_line2 = g_strdup_printf("subnet %s netmask %s{\n", global_subnet, global_netmask);
	gchar *scope_line3 = g_strdup_printf("subnet %s netmask %s {\n", global_subnet, global_netmask);
	gchar *nic_line = g_strdup_printf("interface %s;\n", global_nic);

	if((fp=fopen(DHCPD_CONF_BUF, "r"))==NULL)
	{
	    info = g_strdup_printf(_("Could not open dhcpd.conf here: %s\n"), DHCPD_CONF_BUF);
	    show_info(info);
	    g_free(info);

	    g_free(nic);
	    g_free(subnet);
	    g_free(netmask);

	    g_free(scope_line1);
	    g_free(scope_line2);
	    g_free(scope_line3);
	    g_free(nic_line);
    	    return;
	}
	fseek(fp, 0, SEEK_END);
	file_size = ftell(fp);
	rewind(fp);

	line = allocate(file_size+1);

	/* +1024 for the static pool addition */
	new_conf = allocate(file_size+1024+1);
	sec_conf = allocate(file_size+1024+1);

	/* Find the correct scope and add the pool declaration */
	if( file_size > 1 )
	while(fgets(line, file_size, fp)!=NULL)
	{
	    strcat(new_conf, line);
	    strcat(sec_conf, line);

	    if( strstr(line, scope_line1) 
	    || strstr(line, scope_line2)
	    || strstr(line, scope_line3) )
	    {
		while(fgets(line, file_size, fp)!=NULL)
		{
		    /* Scope end is found */
		    if( strstr(line, "}") )
		      break;

		    strcat(new_conf, line);
		    strcat(sec_conf, line);
		
		    /* The interface line is found */
		    if( strstr(line, nic_line) )
		    {
			found_scope = 1;
			break;
		    }
		}
	    }

	    if( found_scope )
	      break;
	}

	g_free(scope_line1);
	g_free(scope_line2);
	g_free(scope_line3);
	g_free(nic_line);


	if( ! found_scope )
	{
	    info = g_strdup_printf(_("Could not add the loadbalancer, scope not found.\n"));
	    show_info(info);
	    g_free(info);

	    fclose(fp);

	    free(line);
	    free(new_conf);    
	    free(sec_conf);    

	    g_free(subnet);
	    g_free(netmask);
	    g_free(nic);
	    return;
	}


	/* Create the subnets failover peer declaration */
	declaration = g_strconcat(
	"    pool {\n",
	"        failover peer \"", decl_name, "\";\n",
	"        deny dynamic bootp clients;\n",
	NULL);

	/* Secondary suitable for upload */
	sec_declaration = g_strdup_printf("%s", declaration);


	/* We have found the new scope, add the new failover pool
	   before and after the scope declarations ranges */
	if( found_scope && file_size > 1 )
	while(fgets(line, file_size, fp)!=NULL)
	{
	    if( strlen(line) < 3 )
	      continue;

	    /* Break at end of scope */
	    if( strstr(line, "}") )
	    {
		strcat(new_conf, line);
		strcat(sec_conf, line);
	        break;
	    }

	    /* Add the failover pool at the first range then break */
	    if( strstr(line, "range ") )
	    {
		if( ! failover_pool_added )
		{
		    failover_pool_added = 1;

		    /* Added failover pool */
		    strcat(new_conf, declaration);
		    strcat(sec_conf, sec_declaration);
		}
				    
		range_from = allocate(file_size+1); // alloc too big
		range_to = allocate(file_size+1);
		sscanf(line, "%*s %s %s", range_from, range_to);

		if( ! strstr(range_to, ";") )
	          range_line = g_strdup_printf("        range %s %s;\n", range_from, range_to);
		else
	          range_line = g_strdup_printf("        range %s %s\n", range_from, range_to);

		/* Add the range */		
		strcat(new_conf, range_line);
		strcat(sec_conf, range_line);
	    
		g_free(range_line);
		free(range_from);
		free(range_to);

		break;
	    }
	    else
	      {
	          strcat(new_conf, line);
	          strcat(sec_conf, line);
		  continue;
	      }	
	}


	/* Failover pool added at first range. If not range, add end pool */
	if( found_scope && file_size > 1 )
	while(fgets(line, file_size, fp)!=NULL)
	{
	    /* Break at end of scope */
	    if( strstr(line, "}") && ! added_range_end )
	    {
		added_range_end = 1;
		strcat(new_conf, "    }\n");
		strcat(sec_conf, "    }\n");
		
		strcat(new_conf, line);
		strcat(sec_conf, line);
	        break;
	    }
	    /* Break at end of scope */

	    if( strstr(line, "}") )
	    {
		strcat(new_conf, line);
		strcat(sec_conf, line);
	        break;
	    }

	    /* Continue until theres no more ranges */
	    if( strstr(line, "range ") )
	    {
		range_from = allocate(file_size+1); // alloc too big
		range_to = allocate(file_size+1);
		sscanf(line, "%*s %s %s", range_from, range_to);

		if( ! strstr(range_to, ";") )
	          range_line = g_strdup_printf("        range %s %s;\n", range_from, range_to);
		else
	          range_line = g_strdup_printf("        range %s %s\n", range_from, range_to);

		/* Add the range */		
		strcat(new_conf, range_line);
		strcat(sec_conf, range_line);
	    
		g_free(range_line);
		free(range_from);
		free(range_to);

		continue;
	    }

	    if( ! strstr(line, "range ") )
	    {
		added_range_end = 1;
		strcat(new_conf, "    }\n");
		strcat(sec_conf, "    }\n");

		strcat(new_conf, line);
		strcat(sec_conf, line);
		break;
	    }
	}


	/* Gather everything else */
	if( found_scope && file_size > 1 )
	while(fgets(line, file_size, fp)!=NULL)
	{
            strcat(new_conf, line);
            strcat(sec_conf, line);
	}

	fclose(fp);
	free(line);


	if( ! failover_pool_added )
	{
	    info = g_strdup_printf(_("Could not add the loadbalancing pool, ranges not found.\n"));
	    show_info(info);
	    g_free(info);
	
	    free(new_conf);    
	    free(sec_conf);    
	    g_free(subnet);
	    g_free(netmask);
	    g_free(nic);
	    return;
	}

	/* Write the new conf */
	if((fp=fopen(DHCPD_CONF_BUF, "w+"))==NULL)
	{
	    info = g_strdup_printf(_("Could not write dhcpd.conf here:\n%s\n"), DHCPD_CONF_BUF);
	    show_info(info);
	    g_free(info);

	    free(new_conf);
	    free(sec_conf);
	    g_free(subnet);
	    g_free(netmask);
	    g_free(nic);
    	    return;
	}
	fputs(new_conf, fp);
	fclose(fp);
	free(new_conf);

	/* Write the secondary conf suitable for upload */
	settings_conf = g_strdup_printf("%s/%s", SETTINGS_PATH, "dhcpd.conf");
	if((fp=fopen(settings_conf, "w+"))==NULL)
	{
	    info = g_strdup_printf(_("Could not write dhcpd.conf here:\n%s\n"), settings_conf);
	    show_info(info);
	    g_free(info);

	    free(sec_conf);
	    g_free(settings_conf);
	    g_free(subnet);
	    g_free(netmask);
	    g_free(nic);
    	    return;
	}
	fputs(sec_conf, fp);
	fclose(fp);
	g_free(settings_conf);


	/* If this is a primary loadbalancer, ask to update the secondary */
	active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->loadbalance_combo[0]));
	if( active_index <= 0 )
	{
	    show_upload_menu(widgets);
	}

	info = g_strdup_printf(_("A new loadbalancing pool was added.\n"));
	show_info(info);
	g_free(info);
	
	g_free(subnet);
	g_free(netmask);
	g_free(nic);


	/* Reread entire gui */
	populate_scope_treeview(widgets);
	select_first_scope(widgets);

	set_num_ranges(global_nic, global_subnet, global_netmask);
	populate_ranges(widgets, global_nic, global_subnet, global_netmask);

	populate_scope_settings(widgets);

	populate_host_treeview(widgets);
	populate_host_settings(widgets);

	populate_leases(widgets);
    
	gtk_widget_destroy(widgets->loadbalance_window);

	reread_conf();
	return;
    }




    /* Change the current failover. The required checks have been made. */



    /* Delete the current failover declaration and insert the new one.
       Also change the declaration name in the pool. */
    if((fp=fopen(DHCPD_CONF_BUF, "r"))==NULL)
    {
	info = g_strdup_printf(_("Could not open dhcpd.conf here: %s\n"), DHCPD_CONF_BUF);
	show_info(info);
	g_free(info);

	g_free(nic);
	g_free(subnet);
	g_free(netmask);
        return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    line = allocate(file_size+1);

    new_conf = allocate(file_size+(350*10)+1024+1);
    sec_conf = allocate(file_size+(350*10)+1024+1);


    /* Get the primary/secondary combo setting */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->loadbalance_combo[0]));
    if( active_index <= 0 )
    {
	/* Create a Primary failover peer declaration */
	declaration = g_strconcat(
	"failover peer \"", decl_name, "\" {\n",
	"    primary;\n",
	"    address ", address, ";\n",
	"    port ", port, ";\n",
	"    peer address ", remote_address, ";\n", 
	"    peer port ", remote_port, ";\n", 
	"    max-response-delay ", max_response_delay, ";\n", 
	"    max-unacked-updates ", max_unacked_updates, ";\n", 
	"    load balance max seconds ", loadbalance_max_seconds, ";\n", 
	"    mclt ", mclt, ";\n",
	"    split ", split, ";\n",
	"}\n\n",
	NULL);

	/* Create a Secondary failover peer declaration suitable
	   for upload to a remote loadbalancer */
	sec_declaration = g_strconcat(
	"failover peer \"", decl_name, "\" {\n",
	"    secondary;\n",
	"    address ", remote_address, ";\n",
	"    port ", remote_port, ";\n",
	"    peer address ", address, ";\n", 
	"    peer port ", port, ";\n", 
	"    max-response-delay ", max_response_delay, ";\n", 
	"    max-unacked-updates ", max_unacked_updates, ";\n", 
//	"    load balance max seconds ", loadbalance_max_seconds, ";\n", 
//	"    mclt ", mclt, ";\n",
//	"    split ", split, ";\n",
	"}\n\n",
	NULL);
    }
    else
    {
	/* Create a Secondary failover peer declaration */
	declaration = g_strconcat(
	"failover peer \"", decl_name, "\" {\n",
	"    secondary;\n",
	"    address ", address, ";\n",
	"    port ", port, ";\n",
	"    peer address ", remote_address, ";\n", 
	"    peer port ", remote_port, ";\n", 
	"    max-response-delay ", max_response_delay, ";\n", 
	"    max-unacked-updates ", max_unacked_updates, ";\n", 
	"    load balance max seconds ", loadbalance_max_seconds, ";\n", 
/*	"    mclt ", mclt, ";\n",     // Not in a secondary server (Works though)
        "    split ", split, ";\n",   // Not in a secondary server 
*/
        "}\n\n",
        NULL);


	sec_declaration = g_strdup_printf("%s", "Will not be uploaded\n");
    }


    /* Get the current declarations name */
    current_decl_name = get_failover(nic, subnet, netmask, "return-failover-name");

    g_free(subnet);
    g_free(netmask);
    g_free(nic);

    temp_decl_name = g_strdup_printf("\"%s\"", current_decl_name);

    g_free(current_decl_name);

    /* Scroll to the insertion point */
    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
        /* The correct failover is found */
        if( strstr(line, temp_decl_name) && strstr(line, "failover")
        && strstr(line, "peer") && strstr(line, "{") && ! strstr(line, "#") )
        {
	    /* Insert our new failover declaration */
	    strcat(new_conf, declaration);
	    strcat(sec_conf, sec_declaration);

	    /* Scroll past the old pool declaration */
	    while(fgets(line, file_size, fp)!=NULL)
	      if( strstr(line, "}") && ! strstr(line, "#") )
	        break;
	}
	else /* Also change the pool's name */
	if( strstr(line, temp_decl_name) && strstr(line, "failover")
	&& strstr(line, "peer") && strstr(line, ";") && ! strstr(line, "#")  )
	{
	    strcat(new_conf, "        failover peer \"");
	    strcat(new_conf, decl_name);
	    strcat(new_conf, "\";\n");

	    strcat(sec_conf, "        failover peer \"");
	    strcat(sec_conf, decl_name);
	    strcat(sec_conf, "\";\n");
	}
	else
	{
	    strcat(new_conf, line);
	    strcat(sec_conf, line);
	}
    }
    g_free(declaration);
    g_free(sec_declaration);
    g_free(temp_decl_name);

    fclose(fp);
    free(line);


    /* Write the new configuration */
    if((fp=fopen(DHCPD_CONF_BUF, "w+"))==NULL)
    {
	info = g_strdup_printf(_("Could not write dhcpd.conf here: %s\n"), DHCPD_CONF_BUF);
	show_info(info);
	g_free(info);
	    
	free(new_conf);
	free(sec_conf);
    	return;
    }
    fputs(new_conf, fp);
    fclose(fp);
    free(new_conf);


    /* Write the secondary conf suitable for upload */
    settings_conf = g_strdup_printf("%s/%s", SETTINGS_PATH, "dhcpd.conf");
    if((fp=fopen(settings_conf, "w+"))==NULL)
    {
	info = g_strdup_printf(_("Could not write dhcpd.conf here:\n%s\n"), settings_conf);
	show_info(info);
	g_free(info);
	free(sec_conf);
	g_free(settings_conf);
	return;
    }
    fputs(sec_conf, fp);
    fclose(fp);
    free(sec_conf);
    g_free(settings_conf);


    /* If this is a primary loadbalancer then ask to update the secondary */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->loadbalance_combo[0]));
    if( active_index <= 0 )
    {
	show_upload_menu(widgets);
    }


    /* Close the loadbalance window */
    gtk_widget_destroy(widgets->loadbalance_window);
    

    /* Reread entire gui */
    populate_scope_treeview(widgets);
    select_first_scope(widgets);

    set_num_ranges(global_nic, global_subnet, global_netmask);
    populate_ranges(widgets, global_nic, global_subnet, global_netmask);

    populate_scope_settings(widgets);

    populate_host_treeview(widgets);
    populate_host_settings(widgets);

    populate_leases(widgets);
    
    reread_conf();
}



/* Delete the selected loadbalance declaration and pool */
void delete_loadbalance_button_clicked(struct w *widgets)
{
    /* The delete button in the loadbalancing window */
    FILE *fp;
    char *line, *new_conf;
    char *range_from, *range_to;
    long file_size;
    int found_scope = 0;
    int found_pool = 0;
    int found_failover_decl = 0;
    gchar *decl, *decl_name, *nic, *subnet, *netmask, *range_line;
    gchar *info;

    gchar *scope_line1 = g_strdup_printf("subnet %s netmask %s\n", global_subnet, global_netmask);
    gchar *scope_line2 = g_strdup_printf("subnet %s netmask %s{\n", global_subnet, global_netmask);
    gchar *scope_line3 = g_strdup_printf("subnet %s netmask %s {\n", global_subnet, global_netmask);
    gchar *nic_line = g_strdup_printf("interface %s;\n", global_nic);

    subnet = g_strdup_printf("%s", global_subnet);
    netmask = g_strdup_printf("%s", global_netmask);
    nic = g_strdup_printf("%s", global_nic);

    /* Get the name of the subnets failover declaration */
    decl = get_failover(nic, subnet, netmask, "return-failover-name");
    decl_name = g_strdup_printf("\"%s\"", decl);

    g_free(decl);


    if( ! failover_exists(nic, subnet, netmask) )
    {
	info = g_strdup_printf(_("Can not delete the loadbalancer, the selected scope does not seem to have one.\n"));
	show_info(info);
	g_free(info);

	g_free(scope_line1);
	g_free(scope_line2);
	g_free(scope_line3);
	g_free(nic_line);

	g_free(subnet);
	g_free(netmask);
	g_free(nic);
	g_free(decl_name);
	return;
    }

    g_free(subnet);
    g_free(netmask);
    g_free(nic);



    /* First delete the failover-peer declaration */
    if((fp=fopen(DHCPD_CONF_BUF, "r"))==NULL)
    {
	info = g_strdup_printf(_("Could not read dhcpd.conf here:\n%s\n\n"), DHCPD_CONF_BUF);
	show_info(info);
	g_free(info);

	g_free(scope_line1);
	g_free(scope_line2);
	g_free(scope_line3);
	g_free(nic_line);
	g_free(decl_name);
        return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    line = allocate(file_size+1);
    new_conf = allocate(file_size+1);

    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
	/* We have found the declaration, scroll past it */
	if( strstr(line, "failover peer") && strstr(line, decl_name) 
	&& strstr(line, "{") )
	{
	    while(fgets(line, file_size, fp)!=NULL)
	    if( strstr(line, "}") )
	    {
		found_failover_decl = 1;
	        break;
	    }
	}
	else
	  strcat(new_conf, line);
	  
	if( found_failover_decl )
	  break;
    }


    /* Find the correct scope and remove any corresponding pools */
    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
	strcat(new_conf, line);

	if( strstr(line, scope_line1) 
	|| strstr(line, scope_line2)
	|| strstr(line, scope_line3) )
	{
	    while(fgets(line, file_size, fp)!=NULL)
	    {
		strcat(new_conf, line);
		
		if( strstr(line, nic_line) )
		{
		    found_scope = 1;
		    break;
		}
	    
		if( strstr(line, "}") )
		  break;
	    }
	}

	if( found_scope )
	  break;
    }

    g_free(scope_line1);
    g_free(scope_line2);
    g_free(scope_line3);
    g_free(nic_line);
    g_free(decl_name);


    if( ! found_scope )
    {
	info = g_strdup_printf(_("Could not delete the loadbalancer, scope not found.\n"));
	show_info(info);
	g_free(info);

	fclose(fp);
	free(line);
	free(new_conf);    
	return;
    }


    /* Remove the pool { failover-peer.. } from the scope declaration */
    if( found_scope && file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
	/* Skip this pool declaration but keep any ranges */
	if( strstr(line, "pool") && strstr(line, "{") )
	{
	    while(fgets(line, file_size, fp)!=NULL)
	    {
		if( strstr(line, "}") )
		{
		    found_pool = 1;
	    	    break;
		}

		/* Keep the ranges but fix white spaces */
		if( strstr(line, "range ") )
		{
		    range_from = allocate(file_size+1); // alloc too big
		    range_to = allocate(file_size+1);

		    sscanf(line, "%*s %s %s", range_from, range_to);

		    if( ! strstr(range_to, ";") )
		      range_line = g_strdup_printf("    range %s %s;\n", range_from, range_to);
		    else
		      range_line = g_strdup_printf("    range %s %s\n", range_from, range_to);
		    
		    strcat(new_conf, range_line);

		    g_free(range_line);
		    free(range_from);
		    free(range_to);
		}

	    }
	}
	else
	  strcat(new_conf, line);
	    
	if( found_pool )
	  break;
    }


    /* Gather everything else */
    if( found_scope && file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
      strcat(new_conf, line);

    fclose(fp);
    free(line);


    if( ! found_pool )
    {
	info = g_strdup_printf(_("Could not delete the loadbalancer, pool not found.\n"));
	show_info(info);
	g_free(info);

	free(new_conf);    
	return;
    }

    /* Write the new conf */
    if((fp=fopen(DHCPD_CONF_BUF, "w+"))==NULL)
    {
	info = g_strdup_printf(_("Could not write dhcpd.conf here: %s\n"), DHCPD_CONF_BUF);
	show_info(info);
	g_free(info);

	free(new_conf);
        return;
    }
    fputs(new_conf, fp);
    fclose(fp);

    free(new_conf);

    /* Reread entire gui */
    populate_scope_treeview(widgets);
    select_first_scope(widgets);

    set_num_ranges(global_nic, global_subnet, global_netmask);
    populate_ranges(widgets, global_nic, global_subnet, global_netmask);

    populate_scope_settings(widgets);

    populate_host_treeview(widgets);
    populate_host_settings(widgets);

    populate_leases(widgets);
    
    gtk_widget_destroy(widgets->loadbalance_window);

    reread_conf();
}


void show_loadbalance_window(struct w *widgets)
{
    GtkWidget *frame;
    GtkWidget *table;
    GtkWidget *settings_vbox;
    GtkTooltips *tooltips;
    gchar *info;
    gchar *utf8=NULL;
    char *tmp;

    /* Get the failover values for this subnet */
    gchar *nic = g_strdup_printf("%s", global_nic);
    gchar *subnet = g_strdup_printf("%s", global_subnet);
    gchar *netmask = g_strdup_printf("%s", global_netmask);
    gchar *value;

    widgets->loadbalance_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_position(GTK_WINDOW (widgets->loadbalance_window), GTK_WIN_POS_CENTER);
    gtk_widget_set_size_request(widgets->loadbalance_window, -1, -1);

    /* Set window information */
    info = g_strdup_printf(_("GADMIN-DHCPD %s Loadbalancing server for the selected subnet"), VERSION);
    gtk_window_set_title(GTK_WINDOW(widgets->loadbalance_window), info);
    g_free(info);

    g_signal_connect(GTK_WINDOW(widgets->loadbalance_window), "delete_event", 
		     G_CALLBACK (gtk_widget_destroy), NULL);

    settings_vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add (GTK_CONTAINER (widgets->loadbalance_window), settings_vbox);


    tooltips = gtk_tooltips_new();
    frame = gtk_frame_new("Settings:");

    /* A table with 16 settings and 2 columns */
    table = gtk_table_new(16, 2, FALSE);

    gtk_box_pack_start(GTK_BOX(settings_vbox), frame, TRUE, TRUE, 1);
    gtk_container_add(GTK_CONTAINER(frame), table);




    /* Max lengths and input 350 chars */

    /* name of the failover peer */
    widgets->loadbalance_entry[0] = make_entry_with_label(GTK_TABLE(table), _(" Loadbalance name: "),   0,1,0,1,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_entry[0], _("A unique name, IE: dhcpd-failover-2"), NULL);

    /* Loadbalance Primary/Secondary combo. Text is moved left to be properly aligned w the others */
    widgets->loadbalance_combo[0] = make_combo_with_label(GTK_TABLE(table), _("Loadbalance type: "),    0,1,1,2,350);

    tmp = allocate(1024);
    snprintf(tmp, 1000, "%s", _("Primary"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->loadbalance_combo[0]), utf8);

    snprintf(tmp, 1000, "%s", _("Secondary"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->loadbalance_combo[0]), utf8);
    gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->loadbalance_combo[0]), 0);
    free(tmp);


    widgets->loadbalance_entry[1] = make_entry_with_label(GTK_TABLE(table), _(" This servers address: "), 0,1,2,3,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_entry[1], _("IE: 192.168.0.50"), NULL);

    widgets->loadbalance_entry[2] = make_entry_with_label(GTK_TABLE(table), _(" This servers port: "), 0,1,3,4,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_entry[2], _("This servers port IE: 520"), NULL);

    widgets->loadbalance_entry[3] = make_entry_with_label(GTK_TABLE(table), _(" Remote server address: "), 0,1,5,6,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_entry[3], _("Remote server address IE: 192.168.0.200"), NULL);

    widgets->loadbalance_entry[4] = make_entry_with_label(GTK_TABLE(table), _(" Remote server port: "), 0,1,6,7,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_entry[4], _("Remote server port IE: 520"), NULL);

    widgets->loadbalance_entry[5] = make_entry_with_label(GTK_TABLE(table), _(" Max response delay: "), 0,1,8,9,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_entry[5], _("30 is the standard value in seconds"), NULL);

    widgets->loadbalance_entry[6] = make_entry_with_label(GTK_TABLE(table), _(" Max unacked updates: "), 0,1,10,11,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_entry[6], _("10 is the standard value in seconds"), NULL);

    widgets->loadbalance_entry[7] = make_entry_with_label(GTK_TABLE(table), _(" Loadbalance max: "), 0,1,12,13,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_entry[7], _("3 is the standard value in seconds"), NULL);

    widgets->loadbalance_entry[8] = make_entry_with_label(GTK_TABLE(table), _(" MCLT: "), 0,1,14,15,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_entry[8], _("1800 is the standard MCLT value"), NULL);

    widgets->loadbalance_entry[9] = make_entry_with_label(GTK_TABLE(table), _(" Split ratio: "), 0,1,16,17,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_entry[9], _("128 is the standard split ratio"), NULL);




    /* Populate the widgets... */

    /* The name of this failover declaration */
    value = get_failover(nic, subnet, netmask, "return-failover-name");
    if( value!=NULL && strlen(value) > 0 )
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[0]), value);
    else
      {
          value = g_strdup_printf(_("Loadbalancer %s"), subnet);
	  utf8 = g_locale_to_utf8(value, strlen(value), NULL, NULL, NULL);
          gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[0]), utf8);
          g_free(utf8);
      }
    g_free(value);


    /* The primary/secondary combo */
    value = get_failover(nic, subnet, netmask, "primary;");
    if( strstr(value, "primary") )
      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->loadbalance_combo[0]), 0); /* Primary */
    g_free(value);

    value = get_failover(nic, subnet, netmask, "secondary;");
    if( strstr(value, "secondary") )
      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->loadbalance_combo[0]), 1); /* Secondary */
    g_free(value);

     
    /* This servers address */
    value = get_failover(nic, subnet, netmask, "address");
    gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[1]), value);
    g_free(value);
        
    /* This servers port */
    value = get_failover(nic, subnet, netmask, "port");
    if( value!=NULL && strlen(value) > 0 )
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[2]), value);
    else
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[2]), "520");
    g_free(value);

    /* Remote server address */
    value = get_failover(nic, subnet, netmask, "peer address");
    gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[3]), value);
    g_free(value);

    /* Remote server port */
    value = get_failover(nic, subnet, netmask, "peer port");
    if( value!=NULL && strlen(value) > 0 )
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[4]), value);
    else
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[4]), "520");
    g_free(value);

    /* Max response delay */
    value = get_failover(nic, subnet, netmask, "max-response-delay");
    if( value!=NULL && strlen(value) > 0 )
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[5]), value);
    else
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[5]), "30");
    g_free(value);

    /* Max unacked updates */
    value = get_failover(nic, subnet, netmask, "max-unacked-updates");
    if( value!=NULL && strlen(value) > 0 )
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[6]), value);
    else
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[6]), "10");
    g_free(value);

    /* Loadbalance max seconds */
    value = get_failover(nic, subnet, netmask, "load balance max seconds");
    if( value!=NULL && strlen(value) > 0 )
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[7]), value);
    else
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[7]), "3");
    g_free(value);

    /* MCLT */
    value = get_failover(nic, subnet, netmask, "mclt");
    if( value!=NULL && strlen(value) > 0 )
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[8]), value);
    else
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[8]), "1800");
    g_free(value);

    /* Split ratio */
    value = get_failover(nic, subnet, netmask, "split");
    if( value!=NULL && strlen(value) > 0 )
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[9]), value);
    else
      gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_entry[9]), "128");
    g_free(value);

    g_free(nic);
    g_free(subnet);
    g_free(netmask);


    /* The buttons... */
    GtkWidget *settings_hbuttonbox = gtk_hbutton_box_new();
    gtk_box_pack_start(GTK_BOX(settings_vbox), settings_hbuttonbox, FALSE, FALSE, 0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(settings_hbuttonbox), GTK_BUTTONBOX_SPREAD);

    GtkWidget *delete_loadbalance_settings_button = gtk_button_new_from_stock(GTK_STOCK_DELETE);
    gtk_container_add(GTK_CONTAINER(settings_hbuttonbox), delete_loadbalance_settings_button);
    g_signal_connect_swapped(G_OBJECT(delete_loadbalance_settings_button), "clicked", 
                             G_CALLBACK(delete_loadbalance_button_clicked), widgets);

    GtkWidget *apply_loadbalance_settings_button = gtk_button_new_from_stock(GTK_STOCK_APPLY);
    gtk_container_add(GTK_CONTAINER(settings_hbuttonbox), apply_loadbalance_settings_button);
    g_signal_connect_swapped(G_OBJECT(apply_loadbalance_settings_button), "clicked", 
                             G_CALLBACK(apply_loadbalance_button_clicked), widgets);
    
    GtkWidget *cancel_loadbalance_settings_button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
    gtk_container_add(GTK_CONTAINER(settings_hbuttonbox), cancel_loadbalance_settings_button);
    g_signal_connect_swapped(G_OBJECT(cancel_loadbalance_settings_button), "clicked", 
    		G_CALLBACK(gtk_widget_destroy), GTK_OBJECT(widgets->loadbalance_window));


    gtk_widget_show_all(widgets->loadbalance_window);
}


/* Upload the secondary configuration to the remote server */
void apply_loadbalance_update_clicked(struct w *widgets)
{
    char **new_argv;
    int i = 0;
    gchar *settings_conf, *user_host_dir, *info;
    G_CONST_RETURN gchar *remote_path;
    G_CONST_RETURN gchar *remote_address;
    G_CONST_RETURN gchar *remote_port;
    G_CONST_RETURN gchar *remote_user;
    G_CONST_RETURN gchar *remote_password;
    G_CONST_RETURN gchar *remote_restart_cmd;

    remote_path = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_upload_entry[0]));
    remote_address = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_upload_entry[1]));
    remote_port = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_upload_entry[2]));
    remote_user = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_upload_entry[3]));
    remote_password = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_upload_entry[4]));
    remote_restart_cmd = gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_upload_entry[5]));

    /* Return if any field is empty */
    for(i=0; i<6; i++)
    {
	info = g_strdup_printf("%s", gtk_entry_get_text(GTK_ENTRY(widgets->loadbalance_upload_entry[i])));
	if( info != NULL )
	{
	    g_free(info);
	    continue;
	}
	else
	{
	    info = g_strdup_printf(_("All fields must be filled in.\n"));
	    show_info(info);
	    g_free(info);
	    return;
	}
    }

    settings_conf = g_strdup_printf("%s/%s", SETTINGS_PATH, "dhcpd.conf");

    user_host_dir = g_strdup_printf("%s@%s:%s", remote_user, remote_address, remote_path);

    /* SCP Upload the secondary conf to the remote loadbalancer */
    new_argv = create_scp_upload_cmd((char *)remote_user, (char *)remote_port,
                    		     (char *)settings_conf, (char *)user_host_dir);

    if( ! run_ssh_cmd(new_argv, (char *)remote_password) )
    {
	info = g_strdup_printf(_("Failed to upload secondary configuration to the remote loadbalancer.\n"));
	show_info(info);
	g_free(info);

	g_free(user_host_dir);
	free(new_argv);
	return;
    }
    g_free(user_host_dir);
    free(new_argv);


    /* SSH in to the remote server and run the dhcpd restart command on it */
    new_argv = create_ssh_cmd((char *)remote_user, (char *)remote_address,
			      (char *)remote_port, (char *)remote_restart_cmd);

    if( ! run_ssh_cmd(new_argv, (char *)remote_password) )
    {
	info = g_strdup_printf(_("Failed to restart the remote dhcp server.\n"));
	show_info(info);
	g_free(info);
    }
    free(new_argv);

    gtk_widget_destroy(widgets->loadbalance_upload_window);
}


/* We get here if a primary loadbalancer has been applied.
   Show a question to upload the secondary-adjusted conf to the remote server. */
void show_upload_menu(struct w *widgets)
{
    GtkWidget *frame;
    GtkWidget *table;
    GtkWidget *settings_vbox;
    GtkTooltips *tooltips;
    gchar *info;

    /* Get the failover values for this subnet */
    gchar *nic = g_strdup_printf("%s", global_nic);
    gchar *subnet = g_strdup_printf("%s", global_subnet);
    gchar *netmask = g_strdup_printf("%s", global_netmask);
    gchar *value;

    widgets->loadbalance_upload_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_position(GTK_WINDOW(widgets->loadbalance_upload_window), GTK_WIN_POS_CENTER);
    gtk_widget_set_size_request(widgets->loadbalance_upload_window, -1, -1);

    /* Set window information */
    info = g_strdup_printf(_("GADMIN-DHCPD %s Update secondary loadbalancer ?"), VERSION);
    gtk_window_set_title(GTK_WINDOW(widgets->loadbalance_upload_window), info);
    g_free(info);

    g_signal_connect(GTK_WINDOW(widgets->loadbalance_upload_window), "delete_event", 
		     G_CALLBACK(gtk_widget_destroy), NULL);

    settings_vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(widgets->loadbalance_upload_window), settings_vbox);


    tooltips = gtk_tooltips_new();
    frame = gtk_frame_new("Upload settings:");

    /* A table with 16 settings and 2 columns */
    table = gtk_table_new(16, 2, FALSE);

    gtk_box_pack_start(GTK_BOX(settings_vbox), frame, TRUE, TRUE, 1);
    gtk_container_add(GTK_CONTAINER(frame), table);


    /* Remote configuration path entry */
    widgets->loadbalance_upload_entry[0] = make_entry_with_label(GTK_TABLE(table), _(" Remote configuration: "),   0,1,0,1,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_upload_entry[0], _("Configuration path on the remote server"), NULL);

    /* Remote server address entry */
    widgets->loadbalance_upload_entry[1] = make_entry_with_label(GTK_TABLE(table), _(" Remote server address: "),   0,1,1,2,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_upload_entry[1], _("The remote servers address"), NULL);

    /* Remote servers ssh port entry */
    widgets->loadbalance_upload_entry[2] = make_entry_with_label(GTK_TABLE(table), _(" Remote ssh port: "),   0,1,2,3,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_upload_entry[2], _("The remote servers ssh port"), NULL);

    /* Remote servers ssh user entry */
    widgets->loadbalance_upload_entry[3] = make_entry_with_label(GTK_TABLE(table), _(" Remote ssh user: "),   0,1,3,4,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_upload_entry[3], _("The remote servers ssh admin user"), NULL);

    /* Remote servers ssh password entry */
    widgets->loadbalance_upload_entry[4] = make_entry_with_label(GTK_TABLE(table), _(" Remote ssh password: "),   0,1,4,5,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_upload_entry[4], _("The remote servers ssh admin password"), NULL);

    /* Remote dhcpd restart command entry */
    widgets->loadbalance_upload_entry[5] = make_entry_with_label(GTK_TABLE(table), _(" Remote restart command: "),   0,1,5,6,350);
    gtk_tooltips_set_tip(tooltips, widgets->loadbalance_upload_entry[5], _("The remote dhcp servers restart command"), NULL);



    /* Set remote configuration path as the local one */
    gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_upload_entry[0]), DHCPD_CONF_BUF);

    /* Set remote server address (the conf addresses have been reversed) */
    value = get_failover(nic, subnet, netmask, "peer address");
    gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_upload_entry[1]), value);
    g_free(value);
    // FIX: Make it insensitive.

    /* Set remote ssh port */
    gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_upload_entry[2]), "22");

    /* Set remote servers ssh user */
    gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_upload_entry[3]), "root");

    /* The password is empty */

    /* Set remote servers dhcpd restart command */
    gtk_entry_set_text(GTK_ENTRY(widgets->loadbalance_upload_entry[5]), "/etc/init.d/dhcp* restart");



    /* The buttons... */
    GtkWidget *settings_hbuttonbox = gtk_hbutton_box_new();
    gtk_box_pack_start(GTK_BOX(settings_vbox), settings_hbuttonbox, FALSE, FALSE, 0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(settings_hbuttonbox), GTK_BUTTONBOX_SPREAD);

    GtkWidget *apply_button = gtk_button_new_from_stock(GTK_STOCK_APPLY);
    gtk_container_add(GTK_CONTAINER(settings_hbuttonbox), apply_button);
    g_signal_connect_swapped(G_OBJECT(apply_button), "clicked", 
                             G_CALLBACK(apply_loadbalance_update_clicked), widgets);
    
    GtkWidget *cancel_button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
    gtk_container_add(GTK_CONTAINER(settings_hbuttonbox), cancel_button);

    g_signal_connect_swapped(G_OBJECT(cancel_button), "clicked", 
         G_CALLBACK(gtk_widget_destroy), GTK_OBJECT(widgets->loadbalance_upload_window));


    gtk_widget_show_all(widgets->loadbalance_upload_window);
}
