/* GADMIN-SQUID - An easy to use GTK+ frontend for the Squid proxy server.
 * Copyright (C) 2006 - 2009 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/




#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "allocate.h"
#include <gtk/gtk.h>
#include "gettext.h"
#include "widgets.h"
#include "functions.h"
#include "commands.h"
#include "commented.h"
#include "get_option_pos.h"
#include "standard_conf.h"
#include "create_swap_dirs.h"
#include "system_defines.h"
#include "populate_gadmin_squid.h"
#include "show_info.h"



int conf_ok(gchar *file_path)
{
    FILE *fp;
    long file_size;
    char *line;
    int retval = 0;
    int good   = 0;

    if((fp=fopen(file_path, "r"))==NULL)
      return retval;

    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    /* Conf too small to be ok */
    if( file_size < 100 )
    {
	fclose(fp);
	return retval;
    }

    line = allocate(file_size+1);

    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
	if( commented(line) )
	  continue;

	/* Some options must exist */
	if( strstr(line, "http_port") )
	  good++;
	if( strstr(line, "client_netmask") )
	  good++;
	if( strstr(line, "cache_effective_user") )
	  good++;
    }
    fclose(fp);
    free(line);

    if( good >= 3 )
      retval = 1;

    return retval;
}


/* Automated configuration backups for new versions */
void backup_configuration(struct w *widgets)
{
    gchar *cmd, *info;

    if( ! file_exists(SQUID_CONF_BACKUP) && file_exists(SQUID_CONF) )
    {
	cmd = g_strdup_printf("cp %s %s", SQUID_CONF, SQUID_CONF_BACKUP);
        if( ! run_command(cmd) )
	{
	    info = g_strdup_printf(_("Error: Could not backup the current configuration:\n%s\n"), SQUID_CONF);
	    show_info(info);
	    g_free(info);
	}
	else
	{
	    info = g_strdup_printf(_("The configuration was backed up here for safety:\n%s\n"), SQUID_CONF_BACKUP);
	    show_info(info);
	    g_free(info);
	}
	g_free(cmd);
    }
}


void add_standard_conf(struct w *widgets)
{
    /* Adds default squid.conf configurations */
    FILE *fp;
    int i = 0;
    gchar *info, *cmd, *path, *conf;

    /* Create the squid configuration directory if it doesnt exist */
    path = g_strdup_printf("%s", SQUID_CONF);
    for(i=strlen(path)-1; path[i]!='\0'; i--)
    {
	if( path[i]=='/' )
	{
	    path[i]='\0';
	    break;
	}
    }

    if( ! file_exists(path) )
    {
	cmd = g_strdup_printf("mkdir -p %s", path);
	if( ! run_command(cmd) )
	{
	    info = g_strdup_printf(_("Error creating squid configuration directory using command: %s\n"), cmd);
	    show_info(info);
	    g_free(info);
	    g_free(cmd);
	    g_free(path);
	    return;
	}
	g_free(cmd);
    }
    g_free(path);

    if((fp=fopen(SQUID_CONF, "w+"))==NULL)
    {
	info = g_strdup_printf(_("Cant write a new squid.conf here:\n%s\n"), SQUID_CONF);
	show_info(info);
	g_free(info);
	return;
    }

    /* Make the standard squid configuration */
    conf = g_strconcat("",
    "http_port 0.0.0.0:3128\n",
    "#https_port 0.0.0.0:3128 cert=/etc/gadmin-squid/certificate.pem\n",

    "visible_hostname host.example.org\n",
    "httpd_suppress_version_string off\n",
    "client_netmask 255.255.255.255\n",
    "udp_incoming_address 0.0.0.0\n",
    "udp_outgoing_address 255.255.255.255\n",
    "icp_port 3130 proxy-only\n",
    "icp_query_timeout 0\n",
    "maximum_icp_query_timeout 2000\n",
    "mcast_icp_query_timeout 2000\n",
    "dead_peer_timeout 10 seconds\n",
    "hierarchy_stoplist cgi-bin ?\n",
    "cache_mem 8 MB\n",
    "cache_swap_low 90\n",
    "cache_swap_high 95\n",
    "maximum_object_size 4096 KB\n",
    "minimum_object_size 0 KB\n",
    "maximum_object_size_in_memory 8 KB\n",
    "ipcache_size 1024\n",
    "ipcache_low 90\n",
    "ipcache_high 95\n",
    "fqdncache_size 1024\n",
    "cache_replacement_policy lru\n",
    "memory_replacement_policy lru\n",
    "cache_dir ufs ", TOPLEVEL_SWAP_PATH, " 100 16 256\n",
    "access_log /var/log/squid/access.log squid\n",
    "cache_log /var/log/squid/cache.log\n",
    "cache_store_log /var/log/squid/store.log\n",
    "pid_filename /var/run/squid.pid\n",
    "hosts_file /etc/hosts\n",
    "diskd_program /usr/lib/squid/diskd-daemon\n",
    "unlinkd_program /usr/lib/squid/unlinkd\n",
    "emulate_httpd_log off\n",
    "log_ip_on_direct on\n",
    "log_mime_hdrs off\n",
    "debug_options ALL,1\n",
    "log_fqdn off\n",
    "ftp_user Squid@\n",
    "ftp_list_width 32\n",
    "ftp_passive on\n",
    "ftp_sanitycheck on\n",
    "ftp_telnet_protocol on\n",
    "check_hostnames on\n",
    "allow_underscore on\n",
    "dns_retransmit_interval 5 seconds\n",
    "dns_timeout 2 minutes\n",
    "dns_defnames off\n",
    "authenticate_cache_garbage_interval 1 hour\n",
    "authenticate_ttl 1 hour\n",
    "authenticate_ip_ttl 20 seconds\n",
    "request_header_max_size 20 KB\n",
    "request_body_max_size 0 KB\n",
    "quick_abort_min 16 KB\n",
    "quick_abort_max 16 KB\n",
    "quick_abort_pct 95\n",
    "read_ahead_gap 16 KB\n",
    "negative_ttl 5 minutes\n",
    "positive_dns_ttl 6 hours\n",
    "negative_dns_ttl 1 minute\n",
    "range_offset_limit 0 KB\n",
    "forward_timeout 4 minutes\n",
    "connect_timeout 1 minute\n",
    "peer_connect_timeout 30 seconds\n",
    "read_timeout 15 minutes\n",
    "request_timeout 5 minutes\n",
    "persistent_request_timeout 1 minute\n",
    "client_lifetime 1 day\n",
    "half_closed_clients on\n",
    "pconn_timeout 120 seconds\n",
    "ident_timeout 10 seconds\n",
    "shutdown_lifetime 30 seconds\n",
    "reply_header_max_size 20 KB\n",
    "cache_mgr root\n",
    "mail_from ", SERVER_USER, "@example.org\n",
    "mail_program mail\n",
    "cache_effective_user ",  SERVER_USER, "\n",
    "cache_effective_group ", SERVER_GROUP, "\n",
    "umask 027\n",
    "announce_period 50 seconds\n",
    "announce_host tracker.ircache.net\n",
    "announce_port 3131\n",
    "logfile_rotate 0\n",
    "tcp_recv_bufsize 0 bytes\n",
    "memory_pools on\n",
    "memory_pools_limit 5 MB\n",
    "via on\n",
    "forwarded_for on\n",
    "log_icp_queries on\n",
    "icp_hit_stale off\n",
    "minimum_direct_hops 4\n",
    "minimum_direct_rtt 400\n",
    "store_avg_object_size 13 KB\n",
    "store_objects_per_bucket 20\n",
    "client_db on\n",
    "netdb_low 900\n",
    "netdb_high 1000\n",
    "netdb_ping_period 5 minutes\n",
    "query_icmp off\n",
    "test_reachability off\n",
    "buffered_logs off\n",
    "reload_into_ims off\n",
    "icon_directory /usr/share/squid/icons\n",
    "global_internal_static on\n",
    "short_icon_urls off\n",
    "error_directory /usr/share/squid/errors/English\n",
    "maximum_single_addr_tries 1\n",
    "retry_on_error off\n",
    "snmp_port 0\n",
    "snmp_incoming_address 0.0.0.0\n",
    "snmp_outgoing_address 255.255.255.255\n",
    "as_whois_server whois.ra.net\n",
    "wccp_router 0.0.0.0\n",
    "wccp_version 4\n",
    "wccp2_rebuild_wait on\n",
    "wccp2_forwarding_method 1\n",
    "wccp2_return_method 1\n",
    "wccp2_assignment_method 1\n",
    "wccp2_service standard 0\n",
    "wccp2_weight 10000\n",
    "wccp_address 0.0.0.0\n",
    "wccp2_address 0.0.0.0\n",
    "max_open_disk_fds 0\n",
    "offline_mode off\n",
    "uri_whitespace strip\n",
    "nonhierarchical_direct on\n",
    "prefer_direct off\n",
    "strip_query_terms on\n",
    "coredump_dir none\n",
    "coredump_dir ", TOPLEVEL_SWAP_PATH, "\n",
    "redirector_bypass off\n",
    "ignore_unknown_nameservers on\n",
    "digest_generation on\n",
    "digest_bits_per_entry 5\n",
    "digest_rebuild_period 1 hour\n",
    "digest_rewrite_period 1 hour\n",
    "digest_swapout_chunk_size 4096 bytes\n",
    "digest_rebuild_chunk_percentage 10\n",
    "client_persistent_connections on\n",
    "server_persistent_connections on\n",
    "persistent_connection_after_error off\n",
    "detect_broken_pconn off\n",
    "balance_on_multiple_ip on\n",
    "pipeline_prefetch off\n",
    "request_entities off\n",
    "high_response_time_warning 0\n",
    "high_page_fault_warning 0\n",
    "high_memory_warning 30 bytes\n",
    "store_dir_select_algorithm least-load\n",
    "ie_refresh off\n",
    "vary_ignore_expire off\n",
    "sleep_after_fork 0\n",
    "minimum_expiry_time 60 seconds\n",
    "relaxed_header_parser on\n",
    "\n\n",
    "acl our_networks src 192.168.0.0/24 192.168.1.0/24\n",
    "acl localhost src 127.0.0.1/255.255.255.255\n",
    "acl to_localhost dst 127.0.0.0/8\n",
    "acl manager proto cache_object\n",
    "acl QUERY urlpath_regex cgi-bin \\?\n",
    "acl apache rep_header Server ^Apache\n",
    "acl SSL_ports port 443\n",
    "acl Safe_ports port 80		# http\n",
    "acl Safe_ports port 21		# ftp\n",
    "acl Safe_ports port 443		# https\n",
    "acl Safe_ports port 70		# gopher\n",
    "acl Safe_ports port 210		# wais\n",
    "acl Safe_ports port 1025-65535	# unregistered ports\n",
    "acl Safe_ports port 280		# http-mgmt\n",
    "acl Safe_ports port 488		# gss-http\n",
    "acl Safe_ports port 591		# filemaker\n",
    "acl Safe_ports port 777		# multiling http\n",
    "acl CONNECT method CONNECT\n",
    "\n",
    "http_access allow localhost\n",
    "http_access allow our_networks\n",
    "http_reply_access allow all\n",
    "icp_access allow all\n",
    "miss_access allow all\n",
    "http_access deny !Safe_ports\n",
    "http_access deny CONNECT !SSL_ports\n",
    "http_access deny to_localhost\n",
    "ident_lookup_access deny all\n",
    "cache deny QUERY\n",
    "http_access deny all\n",
    "snmp_access deny all\n",
    NULL);


/* Has commented gui settings... */
/*
    "incoming_icp_average 6\n",
    "incoming_http_average 4\n",
    "incoming_dns_average 4\n",
    "min_icp_poll_cnt 8\n",
    "min_dns_poll_cnt 8\n",
    "min_http_poll_cnt 8\n",
*/


/* Unhandeled settings follow... */
/*
    "logformat squid  %ts.%03tu %6tr %>a %Ss/%03Hs %<st %rm %ru %un %Sh/%<A %mt\n",
    "logformat squidmime  %ts.%03tu %6tr %>a %Ss/%03Hs %<st %rm %ru %un %Sh/%<A %mt [%>h] [%<h]\n",
    "logformat common %>a %ui %un [%tl] \"%rm %ru HTTP/%rv\" %Hs %<st %Ss:%Sh\n",
    "logformat combined %>a %ui %un [%tl] \"%rm %ru HTTP/%rv\" %Hs %<st \"%{Referer}>h\" \"%{User-Agent}>h\" %Ss:%Sh\n",

    "refresh_pattern ^ftp:		1440	20%	10080\n",
    "refresh_pattern ^gopher:	1440	0%	1440\n",
    "refresh_pattern .		0	20%	4320\n",

    "#dns_nameservers 10.0.0.1 10.0.0.2\n",
    "#mcast_groups 239.128.16.128 224.0.1.20\n",
    "#mime_table /etc/squid/mime.conf\n",

    "#ssl_unclean_shutdown off\n",
    "#ssl_engine none\n",
    "#sslproxy_client_certificate none\n",
    "#sslproxy_client_key none\n",
    "#sslproxy_version 1\n",
    "#sslproxy_options none\n",
    "#sslproxy_cipher none\n",
    "#sslproxy_cafile none\n",
    "#sslproxy_capath none\n",
    "#sslproxy_flags none\n",
    "#sslpassword_program none\n",

    "#auth_param negotiate program <uncomment and complete this line to activate>\n",
    "#auth_param negotiate children 5\n",
    "#auth_param negotiate keep_alive on\n",
    "#auth_param ntlm program <uncomment and complete this line to activate>\n",
    "#auth_param ntlm children 5\n",
    "#auth_param ntlm keep_alive on\n",
    "#auth_param digest program <uncomment and complete this line>\n",
    "#auth_param digest children 5\n",
    "#auth_param digest realm Squid proxy-caching web server\n",
    "#auth_param digest nonce_garbage_interval 5 minutes\n",
    "#auth_param digest nonce_max_duration 30 minutes\n",
    "#auth_param digest nonce_max_count 50\n",
    "#auth_param basic program <uncomment and complete this line>\n",
    "#auth_param basic children 5\n",
    "#auth_param basic realm Squid proxy-caching web server\n",
    "#auth_param basic credentialsttl 2 hours\n",
    "#auth_param basic casesensitive off\n",
*/
    /* Delay pools must be placed after the ACL's */
/*
    "delay_pools 0\n",
    "#delay_pools 2      # 2 delay pools\n",
    "#delay_class 1 2    # pool 1 is a class 2 pool\n",
    "#delay_class 2 3    # pool 2 is a class 3 pool\n",

    "#delay_access 1 allow some_big_clients\n",
    "#delay_access 2 allow lotsa_little_clients\n",
    "#delay_access 1 deny all\n",
    "#delay_access 2 deny all\n",

    "#delay_parameters pool aggregate\n",
    "#delay_parameters pool aggregate individual\n",
    "#delay_parameters pool aggregate network individual\n",
    "#delay_parameters 1 -1/-1 8000/8000\n",
    "#delay_parameters 2 32000/32000 8000/8000 600/8000\n",
    "#delay_initial_bucket_level 50\n",

    "#cache_peer parent.foo.net       parent    3128  3130  proxy-only\n",
    "#cache_peer sib1.foo.net         sibling   3128  3130  proxy-only\n",
    "#cache_peer sib2.foo.net         multicast 3128  3130  proxy-only\n",
    "#cache_peer_domain parent.foo.net .net\n",
    "#cache_peer_domain parent.bar.net !hello.org\n",
    "#cache_peer  parent cache.foo.org 3128 3130\n",
    "#neighbor_type_domain cache.foo.org sibling .com .net\n",
    "#neighbor_type_domain cache.foo.org sibling .au .de\n",
    NULL);
*/
    /* End of default configuration for squid */

    fputs(conf, fp);
    fclose(fp);

    g_free(conf);

    if( GTK_IS_WINDOW(widgets->default_conf_question_window) )
      gtk_widget_destroy(widgets->default_conf_question_window);

    populate_gadmin_squid(widgets);

    /* Create squid swap directories and
       server user/group if they dont exist */
    create_swap_dirs(widgets);
}


/* This should probably also write a ~"dont_ask_again=1" to the settings file */
void dont_add_standard_conf(struct w *widgets)
{
    gtk_widget_destroy(widgets->default_conf_question_window);
    populate_gadmin_squid(widgets);
}


void create_standard_conf_question(struct w *widgets)
{
  GtkWidget *vbox18, *label182;
  GtkWidget *scrolledwindow18;
  GtkWidget *default_question_textview;
  GtkWidget *hbuttonbox11;
  GtkWidget *yes_default_question_button;
  GtkWidget *alignment44, *hbox98;
  GtkWidget *image44, *label184;
  GtkWidget *no_default_question_button;
  GtkWidget *alignment45, *hbox99;
  GtkWidget *image45, *label185;

  widgets->default_conf_question_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_size_request(widgets->default_conf_question_window, 500, 340);
  gtk_window_set_title(GTK_WINDOW(widgets->default_conf_question_window), _("GADMIN SQUID question"));
  gtk_window_set_position(GTK_WINDOW(widgets->default_conf_question_window), GTK_WIN_POS_CENTER);

  vbox18 = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(widgets->default_conf_question_window), vbox18);

  label182 = gtk_label_new ("");
  gtk_box_pack_start(GTK_BOX(vbox18), label182, FALSE, FALSE, 0);
  gtk_widget_set_size_request(label182, -1, 20);
  gtk_label_set_justify(GTK_LABEL(label182), GTK_JUSTIFY_LEFT);

  scrolledwindow18 = gtk_scrolled_window_new(NULL, NULL);
  gtk_box_pack_start(GTK_BOX(vbox18), scrolledwindow18, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow18), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  default_question_textview = gtk_text_view_new();
  gtk_container_add(GTK_CONTAINER(scrolledwindow18), default_question_textview);
  gtk_widget_set_size_request(default_question_textview, 156, 260);
  gtk_text_view_set_editable(GTK_TEXT_VIEW(default_question_textview), FALSE);
  gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(default_question_textview), FALSE);
  gtk_text_view_set_pixels_above_lines(GTK_TEXT_VIEW(default_question_textview), 3);
  gtk_text_view_set_left_margin(GTK_TEXT_VIEW(default_question_textview), 30);

  gtk_text_buffer_set_text(gtk_text_view_get_buffer(GTK_TEXT_VIEW(default_question_textview)),
     _("GADMIN SQUID has detected that your current configuration\nfile is lacking required options or is missing.\n\nDo you want to write a new configuration ?\n\n\t\t\t(If you dont know then press yes)\n"), -1);

  hbuttonbox11 = gtk_hbutton_box_new();
  gtk_box_pack_start(GTK_BOX(vbox18), hbuttonbox11, FALSE, FALSE, 0);
  gtk_widget_set_size_request(hbuttonbox11, -1, 40);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox11), GTK_BUTTONBOX_SPREAD);

  yes_default_question_button = gtk_button_new();
  gtk_container_add(GTK_CONTAINER(hbuttonbox11), yes_default_question_button);
  GTK_WIDGET_SET_FLAGS(yes_default_question_button, GTK_CAN_DEFAULT);

  alignment44 = gtk_alignment_new(0.5, 0.5, 0, 0);
  gtk_container_add(GTK_CONTAINER(yes_default_question_button), alignment44);

  hbox98 = gtk_hbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(alignment44), hbox98);

  image44 = gtk_image_new_from_stock("gtk-yes", GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start(GTK_BOX(hbox98), image44, FALSE, FALSE, 0);

  label184 = gtk_label_new_with_mnemonic(_("Yes"));
  gtk_box_pack_start(GTK_BOX(hbox98), label184, FALSE, FALSE, 0);
  gtk_label_set_justify(GTK_LABEL(label184), GTK_JUSTIFY_LEFT);

  no_default_question_button = gtk_button_new();
  gtk_container_add(GTK_CONTAINER(hbuttonbox11), no_default_question_button);
  GTK_WIDGET_SET_FLAGS(no_default_question_button, GTK_CAN_DEFAULT);

  alignment45 = gtk_alignment_new(0.5, 0.5, 0, 0);
  gtk_container_add(GTK_CONTAINER(no_default_question_button), alignment45);

  hbox99 = gtk_hbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(alignment45), hbox99);

  image45 = gtk_image_new_from_stock("gtk-no", GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start(GTK_BOX(hbox99), image45, FALSE, FALSE, 0);

  label185 = gtk_label_new_with_mnemonic(_("No"));
  gtk_box_pack_start(GTK_BOX(hbox99), label185, FALSE, FALSE, 0);
  gtk_label_set_justify(GTK_LABEL(label185), GTK_JUSTIFY_LEFT);

  g_signal_connect_swapped(G_OBJECT(yes_default_question_button), "clicked",
                           G_CALLBACK(add_standard_conf), widgets);

  g_signal_connect_swapped(G_OBJECT(no_default_question_button), "clicked",
                           G_CALLBACK(dont_add_standard_conf), widgets);

  gtk_widget_show_all(widgets->default_conf_question_window);
}
