/**
 * @file meta-person.c MetaPerson API
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include "meta-person.h"
#include "filter.h"
#include "meta-account.h"
#include "person-list.h"

#include <string.h>
#include <time.h>

/* Provided by libgalago */
void _galago_account_set_person(GalagoAccount *account, GalagoPerson *person);
void _galago_person_set_id(GalagoPerson *person, const char *id);
void _galago_person_set_session_id(GalagoPerson *person,
								   const char *session_id);


/* person-list.c */
extern GHashTable *meta_people_table;

GalagodMetaPerson *
galagod_meta_person_new(void)
{
	GalagodMetaPerson *meta_person;
	char *obj_path;

	meta_person = g_new0(GalagodMetaPerson, 1);

	galago_context_push(galagod_get_context());
	meta_person->person = galago_create_person(NULL);
	g_object_set_data(G_OBJECT(meta_person->person), "valid_uid_set",
					  GINT_TO_POINTER(FALSE));

	obj_path = galagod_person_generate_path(meta_person->person);
	galago_object_set_dbus_path(GALAGO_OBJECT(meta_person->person), obj_path);
	g_free(obj_path);
	galago_context_pop();

	g_object_set_data(G_OBJECT(meta_person->person),
					  "meta-person", meta_person);

	galagod_person_add_filter(meta_person->person);
	galagod_people_add_meta_person(meta_person);

	return meta_person;
}

void
galagod_meta_person_destroy(GalagodMetaPerson *meta_person)
{
	const GList *l, *l_next;

	g_return_if_fail(meta_person != NULL);

	for (l = galago_person_get_accounts(meta_person->person, FALSE);
		 l != NULL;
		 l = l_next)
	{
		GalagoAccount *account = (GalagoAccount *)l->data;
		GalagodMetaAccount *meta_account = GALAGOD_META_ACCOUNT(account);

		l_next = l->next;

		galagod_meta_account_destroy(meta_account);
	}

	while (meta_person->people != NULL)
	{
		galagod_meta_person_remove_person(meta_person,
			(GalagoPerson *)meta_person->people->data);
	}

	galagod_people_remove_meta_person(meta_person);
	galagod_person_remove_filter(meta_person->person);
	g_object_unref(meta_person->person);

	g_free(meta_person);
}

void
galagod_meta_person_merge(GalagodMetaPerson *dest, GalagodMetaPerson *src)
{
	const GList *l, *l_next;
	GalagodMetaAccount *meta_account;
	GalagoAccount *account, *main_account;

	g_return_if_fail(dest != NULL);
	g_return_if_fail(src  != NULL);

	while ((l = galagod_meta_person_get_people(src)) != NULL)
	{
		GalagoPerson *person = (GalagoPerson *)l->data;

		galagod_meta_person_remove_person(src, person);
		galagod_meta_person_add_person(dest, person);
	}

	for (l = galago_person_get_accounts(src->person, FALSE);
		 l != NULL;
		 l = l_next)
	{
		l_next = l->next;

		account      = (GalagoAccount *)l->data;
		meta_account = GALAGOD_META_ACCOUNT(account);
		main_account = galagod_meta_account_get_account(meta_account);

		_galago_account_set_person(main_account, dest->person);
	}

	galagod_meta_person_destroy(src);

	/* XXX This is a hack to get the new MetaPerson's ID in the table. */
	if (galago_person_get_id(dest->person) != NULL)
	{
		g_hash_table_replace(meta_people_table,
							 g_strdup(galago_person_get_id(dest->person)),
							 dest);
	}

}

GalagoPerson *
galagod_meta_person_get_person(const GalagodMetaPerson *meta_person)
{
	g_return_val_if_fail(meta_person != NULL, NULL);

	return meta_person->person;
}

GList *
galagod_meta_person_get_people(const GalagodMetaPerson *meta_person)
{
	g_return_val_if_fail(meta_person != NULL, NULL);

	return meta_person->people;
}

void
galagod_meta_person_add_person(GalagodMetaPerson *meta_person,
							   GalagoPerson *person)
{
	const char *person_id;

	g_return_if_fail(meta_person != NULL);
	g_return_if_fail(person      != NULL);

	meta_person->people = g_list_append(meta_person->people, person);
	g_object_set_data(G_OBJECT(person), "meta-person", meta_person);

	person_id = galago_person_get_id(person);

	if (galago_person_get_id(meta_person->person) == NULL &&
		person_id != NULL)
	{
		_galago_person_set_id(meta_person->person, person_id);
		g_object_set_data(G_OBJECT(meta_person->person), "valid_uid_set",
						  GINT_TO_POINTER(TRUE));
		g_hash_table_replace(meta_people_table, g_strdup(person_id),
							 meta_person);
	}

	common_meta_object_merge_attributes(GALAGO_OBJECT(meta_person->person),
										GALAGO_OBJECT(person),
										meta_person->people);

	galagod_person_add_filter(person);
}

void
galagod_meta_person_remove_person(GalagodMetaPerson *meta_person,
								  GalagoPerson *person)
{
	g_return_if_fail(meta_person != NULL);
	g_return_if_fail(person      != NULL);

	galagod_person_remove_filter(person);

	g_object_set_data(G_OBJECT(person), "meta-person", NULL);

	meta_person->people = g_list_remove(meta_person->people, person);
}
