namespace Galago
{
	using System;
	using System.Runtime.InteropServices;

	public class Presence
		: Object
	{
		~Presence()
		{
			Dispose();
		}

		public Presence(IntPtr raw) : base(raw) {}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_presence_new(IntPtr account);

		public Presence(Account account)
		{
			Raw = galago_presence_new(account.Handle);
		}

		[DllImport("libgalago.so")]
		static extern void galago_presence_set_idle(IntPtr presence,
													bool idle, uint idle_time);

		public void SetIdle(bool idle, uint idle_time)
		{
			galago_presence_set_idle(Handle, idle, idle_time);
		}

		[DllImport("libgalago.so")]
		static extern bool galago_presence_is_idle(IntPtr presence);

		public bool Idle
		{
			get { return galago_presence_is_idle(Handle); }
			set { SetIdle(value, 0);                      }
		}

		[DllImport("libgalago.so")]
		static extern uint galago_presence_get_idle_time(IntPtr presence);

		public uint IdleTime
		{
			get { return galago_presence_get_idle_time(Handle); }
			set { SetIdle(true, value);                         }
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_presence_get_statuses(IntPtr presence);

		[DllImport("libgalago.so")]
		static extern void galago_presence_clear_statuses(IntPtr presence);

		[DllImport("libgalago.so")]
		static extern void galago_presence_set_statuses(IntPtr presence,
														IntPtr statuses);

		public List Statuses
		{
			get
			{
				IntPtr raw_ret = galago_presence_get_statuses(Handle);

				return new List(raw_ret);
			}

			set
			{
				if (value == null)
					galago_presence_clear_statuses(Handle);
				else
					galago_presence_set_statuses(Handle, value.Handle);
			}
		}

		[DllImport("libgalago.so")]
		static extern void galago_presence_add_status(IntPtr presence,
													  IntPtr status);

		public void AddStatus(Status status)
		{
			galago_presence_add_status(Handle, status.Handle);
		}

		[DllImport("libgalago.so")]
		static extern void galago_presence_remove_status(IntPtr presence,
														 IntPtr status);

		public void RemoveStatus(Status status)
		{
			galago_presence_remove_status(Handle, status.Handle);
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_presence_get_account(IntPtr presence);

		public Account Account
		{
			get
			{
				IntPtr raw_ret = galago_presence_get_account(Handle);
				return Object.Wrap(raw_ret) as Account;
			}
		}

		[DllImport("libgalago.so")]
		static extern bool galago_presence_is_discarded(IntPtr presence);

		public bool Discarded
		{
			get { return galago_presence_is_discarded(Handle); }
		}

		[DllImport("libgalago.so")]
		static extern bool galago_presence_is_available(IntPtr presence);

		public bool Available
		{
			get { return galago_presence_is_available(Handle); }
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_presence_get_active_status(IntPtr presence);

		public Status ActiveStatus
		{
			get
			{
				IntPtr raw_ret = galago_presence_get_active_status(Handle);

				return Object.Wrap(raw_ret) as Status;
			}
		}

		[DllImport("libgalago.so")]
		static extern bool galago_presence_is_status_exclusive(IntPtr presence,
															   string status_id);

		public bool IsStatusExclusive(string status_id)
		{
			return galago_presence_is_status_exclusive(Handle, status_id);
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_presence_get_status(IntPtr presence,
														string status_id);

		public Status GetStatus(string status_id)
		{
			IntPtr raw_ret = galago_presence_get_status(Handle, status_id);

			return Object.Wrap(raw_ret) as Status;
		}

		[DllImport("libgalago.so")]
		static extern bool galago_presence_has_status(IntPtr presence,
													  string status_id);

		public bool HasStatus(string status_id)
		{
			return galago_presence_has_status(Handle, status_id);
		}

		[DllImport("libgalago.so")]
		static extern bool galago_presence_has_status_type(IntPtr presence,
														   StatusType type);

		public bool HasStatusType(StatusType type)
		{
			return galago_presence_has_status_type(Handle, type);
		}

		[DllImport("libgalago.so")]
		static extern int galago_presence_compare(IntPtr presence1,
												  IntPtr presence2);

		public static int Compare(Presence p1, Presence p2)
		{
			return galago_presence_compare(p1.Handle, p2.Handle);
		}

		/*
		 * Signals
		 */
		[Galago.Signal("idle-changed")]
		public event IdleChangedHandler OnIdleChanged
		{
			add
			{
				AddSignalHandler("idle-changed",
								 typeof(SignalMarshals.VoidBoolULongSignal),
								 typeof(IdleChangedArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("idle-changed", value);
			}
		}

		[Galago.Signal("status-added")]
		public event AddedHandler OnStatusAdded
		{
			add
			{
				AddSignalHandler("status-added",
								 typeof(SignalMarshals.VoidPointerSignal),
								 typeof(AddedArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("status-added", value);
			}
		}

		[Galago.Signal("status-removed")]
		public event RemovedHandler OnStatusRemoved
		{
			add
			{
				AddSignalHandler("status-removed",
								 typeof(SignalMarshals.VoidPointerSignal),
								 typeof(RemovedArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("status-removed", value);
			}
		}

		[Galago.Signal("status-updated")]
		public event StatusUpdatedHandler OnStatusUpdated
		{
			add
			{
				AddSignalHandler("status-updated",
								 typeof(SignalMarshals.VoidPointerStringSignal),
								 typeof(StatusUpdatedArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("status-updated", value);
			}
		}

		[Galago.Signal("updated")]
		public event EventHandler OnUpdated
		{
			add
			{
				AddSignalHandler("updated",
								 typeof(SignalMarshals.VoidVoidSignal),
								 typeof(EventArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("updated", value);
			}
		}
	}
}
