/*
 *  Copyright (C) 2003 Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "galeon-action-go.h"
#include "galeon-shell.h"
#include "eel-gconf-extensions.h"
#include "prefs-strings.h"
#include "gul-gobject-misc.h"
#include "gul-toolbutton.h"
#include "gul-string.h"
#include "gul-gui.h"

#include <gtk/gtkstock.h>
#include <glib/gi18n.h>
#include <string.h>

static void		galeon_action_go_init			(GaleonActionGo *action);
static void		galeon_action_go_class_init		(GaleonActionGoClass *class);
static void		galeon_action_go_finalize		(GObject *object);
static void		galeon_action_go_connect_proxy		(GtkAction *action, GtkWidget *proxy);
static void		galeon_action_go_activate		(GtkAction *action);

static GtkActionClass *parent_class = NULL;


MAKE_GET_TYPE (galeon_action_go, "GaleonActionGo", GaleonActionGo,
	       galeon_action_go_class_init, 
	       galeon_action_go_init, GALEON_TYPE_ACTION);


static void
galeon_action_go_class_init (GaleonActionGoClass *class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (class);
	GtkActionClass *action_class = GTK_ACTION_CLASS (class);

	object_class->finalize = galeon_action_go_finalize;

	action_class->connect_proxy = galeon_action_go_connect_proxy;
	action_class->menu_item_type    = GTK_TYPE_IMAGE_MENU_ITEM;
	action_class->toolbar_item_type = GUL_TYPE_TOOLBUTTON;
	action_class->activate          = galeon_action_go_activate;

	parent_class = g_type_class_peek_parent (class);
}

static void
galeon_action_go_init (GaleonActionGo *action)
{
	g_object_set (G_OBJECT (action), "stock_id", GTK_STOCK_JUMP_TO, NULL);
	g_object_set (G_OBJECT (action), "tooltip", _("Go to the specified location"), NULL);
}

static void
galeon_action_go_finalize (GObject *object)
{
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static gchar *
galeon_action_go_get_location (GaleonActionGo *b)
{
	GaleonWindow *window;
	char *ret;
	
	window = galeon_action_get_window (GALEON_ACTION (b));
	g_return_val_if_fail (window != NULL, NULL);
	
	ret = galeon_window_get_location_entry_location (window);

	return ret;
}

static void
galeon_action_go_activate (GtkAction *a)
{
	GaleonEmbed *embed = galeon_action_get_embed (GALEON_ACTION (a));
	char *location;
	
	g_return_if_fail (embed != NULL);

	location = galeon_action_go_get_location (GALEON_ACTION_GO (a));
	g_return_if_fail (location != NULL);
	
	galeon_embed_load_url (embed, location);

	g_free (location);
}


static gboolean
galeon_action_go_button_press_event_cb (GtkWidget *widget, GdkEventButton *event, 
					GaleonActionGo *b)
{
	GaleonWindow *window = galeon_action_get_window (GALEON_ACTION (b));

	if (event->button == 2 && IS_GALEON_WINDOW (window))
	{
		char *location;

		location = galeon_action_go_get_location (b);
		g_return_val_if_fail (location != NULL, FALSE);
	
		if (location)
		{
			GaleonTab *tab = galeon_window_get_active_tab (window);
			galeon_shell_new_tab (galeon_shell, window, tab, location, 
					      GALEON_NEW_TAB_JUMP);
			g_free (location);
		}
		return TRUE;
	}
	
	return FALSE;
}

static void
galeon_action_go_activate_cb (GtkMenuItem *mi, GaleonActionGo *b)
{
	GaleonEmbed *embed = galeon_action_get_embed (GALEON_ACTION (b));
	char *location;
	
	g_return_if_fail (embed != NULL);

	location = galeon_action_go_get_location (b);
	g_return_if_fail (location != NULL);
	
	galeon_embed_load_url (embed, location);

	g_free (location);
}

static void 
galeon_action_go_new_tab_activate_cb (GtkMenuItem *mi, GaleonActionGo *b)
{
	GaleonWindow *window = galeon_action_get_window (GALEON_ACTION (b));
	char *location;

	location = galeon_action_go_get_location (b);
	g_return_if_fail (location != NULL);
	
	if (location)
	{
		GaleonTab *tab = galeon_window_get_active_tab (window);
		galeon_shell_new_tab (galeon_shell, window, tab, location, 
				      GALEON_NEW_TAB_JUMP | GALEON_NEW_TAB_IN_EXISTING_WINDOW);
		g_free (location);
	}
}

static void 
galeon_action_go_new_window_activate_cb (GtkMenuItem *mi, GaleonActionGo *b)
{
	GaleonWindow *window = galeon_action_get_window (GALEON_ACTION (b));
	char *location;

	location = galeon_action_go_get_location (b);
	g_return_if_fail (location != NULL);
	
	if (location)
	{
		GaleonTab *tab = galeon_window_get_active_tab (window);
		galeon_shell_new_tab (galeon_shell, window, tab, location, 
				      GALEON_NEW_TAB_JUMP | GALEON_NEW_TAB_IN_NEW_WINDOW);
		g_free (location);
	}
}

static void
galeon_action_go_connect_proxy (GtkAction *a, GtkWidget *proxy)
{
	GtkWidget *widget = proxy;

	if (GUL_IS_TOOLBUTTON (proxy))
	{
		GtkMenuShell *ms;

		widget = gul_toolbutton_get_button (GUL_TOOLBUTTON (proxy));
		ms = gul_toolbutton_get_menu (GUL_TOOLBUTTON (proxy));

		gul_gui_append_new_menuitem (GTK_WIDGET (ms), _("Go To The Specified _Location"), 
				     G_CALLBACK (galeon_action_go_activate_cb), a);
		gul_gui_append_new_menuitem (GTK_WIDGET (ms), _("Go To The Specified Location in Another _Tab"), 
				     G_CALLBACK (galeon_action_go_new_tab_activate_cb), a);
		gul_gui_append_new_menuitem (GTK_WIDGET (ms), _("Go To The Specified Location in Another _Window"), 
				     G_CALLBACK (galeon_action_go_new_window_activate_cb), a);
	}

	g_signal_connect (widget, "button-press-event",
			  G_CALLBACK (galeon_action_go_button_press_event_cb), a);

	(* GTK_ACTION_CLASS (parent_class)->connect_proxy) (a, proxy);
}

