/***************************************************************************

  stream_direct.c

  The direct stream management routines

  (c) 2000-2004 Benot Minisini <gambas@users.sourceforge.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/

#define __STREAM_IMPL_C

#include "gb_common.h"
#include "gb_error.h"
#include "gbx_value.h"
#include "gb_limit.h"

#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <errno.h>
#include <sys/ioctl.h>

#include "gbx_stream.h"


#define FD (stream->direct.fd)


static int stream_open(STREAM *stream, const char *path, int mode)
{
  int fd;
  struct stat info;
  int fmode;

  if (mode & ST_CREATE)
    fmode = O_CREAT | O_TRUNC;
  else if (mode & ST_APPEND)
    fmode = O_APPEND | O_CREAT;
  else
    fmode = 0;

  switch (mode & ST_MODE)
  {
    case ST_READ: fmode |= O_RDONLY; break;
    case ST_WRITE: fmode |= O_WRONLY; break;
    case ST_READ_WRITE: fmode |= O_RDWR; break;
    default: fmode |= O_RDONLY;
  }

  fd = open(path, fmode, 0666);
  if (fd < 0)
    return TRUE;

  if (fstat(fd, &info) < 0)
    return TRUE;

  if (S_ISDIR(info.st_mode))
  {
    errno = EISDIR;
    return TRUE;
  }

  stream->direct.size = info.st_size;

  FD = fd;
  return FALSE;
}


static int stream_close(STREAM *stream)
{
  if (close(FD) < 0)
    return TRUE;

  FD = -1;
  return FALSE;
}


static int stream_read(STREAM *stream, char *buffer, long len)
{
  ssize_t eff_read;
  ssize_t len_read;

  /*
  if (stream->direct.has_car)
  {
    *buffer++ = stream->direct.car;
    len--;
    STREAM_eff_read++;
    stream->direct.has_car = FALSE;
  }
  */

  while (len > 0)
  {
    len_read = Min(len, MAX_IO);
    eff_read = read(FD, buffer, len_read);

    if (eff_read > 0)
    {
      STREAM_eff_read += eff_read;
      len -= eff_read;
      buffer += eff_read;
    }

    if (eff_read < len_read)
    {
      if (eff_read == 0)
        errno = 0;
      if (eff_read <= 0 && errno != EINTR)
        return TRUE;
    }

  }

  return FALSE;
}


static int stream_write(STREAM *stream, char *buffer, long len)
{
  ssize_t eff_write;
  ssize_t len_write;

  while (len > 0)
  {
    len_write = Min(len, MAX_IO);
    eff_write = write(FD, buffer, len_write);

    if (eff_write < len_write)
    {
      if (eff_write <= 0 && errno != EINTR)
        return TRUE;
    }

    len -= eff_write;
    buffer += eff_write;
  }

  return FALSE;
}


static int stream_seek(STREAM *stream, long pos, int whence)
{
  return (lseek(FD, pos, whence) < 0);
  /*
  if (!err)
    stream->direct.has_car = FALSE;
  return err;
  */
}


static int stream_tell(STREAM *stream, long *pos)
{
  *pos = lseek(FD, 0, SEEK_CUR);
  return (*pos < 0);
}


static int stream_flush(STREAM *stream)
{
  return FALSE;
}


static int stream_eof(STREAM *stream)
{
  int ilen;

  if (ioctl(FD, FIONREAD, &ilen))
    return TRUE;
    
  return (ilen == 0);
}


static int stream_lof(STREAM *stream, long *len)
{
  *len = stream->direct.size;
  return FALSE;
}


static int stream_handle(STREAM *stream)
{
  return FD;
}


DECLARE_STREAM(STREAM_direct);

