/***************************************************************************

  gba.c

  Welcome to the GAMBAS archiver !

  (c) 2000-2004 Benot Minisini <gambas@users.sourceforge.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/

#define __GBA_C

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <dirent.h>

#ifdef __GNU_LIBRARY__
#include <getopt.h>
#endif

#include "gb_common.h"
#include "gb_error.h"
#include "gb_str.h"
#include "gb_file.h"
#include "gb_array.h"

#include "gbc_archive.h"

#ifdef __GNU_LIBRARY__
PRIVATE struct option Long_options[] =
{
  { "version", 0, NULL, 'V' },
  { "help", 0, NULL, 'h' },
  { "verbose", 0, NULL, 'v' },
  { 0 }
};
#endif

PRIVATE char **path_list;
PRIVATE int path_current;

PRIVATE void get_arguments(int argc, char **argv)
{
  int opt;
  #ifdef __GNU_LIBRARY__
  int index = 0;
  #endif

  for(;;)
  {
    #ifdef __GNU_LIBRARY__
      opt = getopt_long(argc, argv, "vVh", Long_options, &index);
    #else
      opt = getopt(argc, argv, "vVh");
    #endif

    if (opt < 0) break;

    switch (opt)
    {
      case 'V':
        printf("gba-" VERSION "\n");
        exit(0);

      case 'v':
        ARCH_verbose = TRUE;
        break;

      case 'h': case '?':
        printf(
          "\n"
          "GAMBAS Archiver version " VERSION " " __DATE__ " " __TIME__ "\n"
          COPYRIGHT
          "Usage: gba [options] [<project directory>]\n\n"
          "Options:"
          #ifdef __GNU_LIBRARY__
          "\n"
          "  -v  --verbose              verbose output\n"
          "  -V  --version              display version\n"
          "  -h  --help                 display this help\n"
          #else
          " (no long options on this system)\n"
          "  -v                         verbose output\n"
          "  -V                         display version\n"
          "  -h                         display this help\n"
          #endif
          "\n"
          );

        exit(0);

      default:
        exit(1);

    }
  }

  if (optind < (argc - 1))
  {
    fprintf(stderr, "gba: too many arguments.\n");
    exit(1);
  }

  if (optind == argc)
    ARCH_define_project(NULL);
  else
    ARCH_define_project(argv[optind]);

  if (!FILE_exist(ARCH_project))
  {
    fprintf(stderr, "gba: project file not found: %s\n", ARCH_project);
    exit(1);
  }
}


PRIVATE void path_add(const char *path)
{
  *((char **)ARRAY_add(&path_list)) = STR_copy(path);
}


PRIVATE void path_init(const char *first)
{
  ARRAY_create(&path_list);

  if (*first)
    chdir(first);

  path_add(FILE_get_current_dir());

  path_current = 0;
}


PRIVATE void path_exit(void)
{
  ARRAY_delete(&path_list);
}


PRIVATE long path_count(void)
{
  return ARRAY_count(path_list);
}


int main(int argc, char **argv)
{
  DIR *dir;
  char *path;
  struct dirent *dirent;
  char *file_name;
  const char *file;
  struct stat info;
  const char *ext;
  int len;

  get_arguments(argc, argv);

  TRY
  {
    ARCH_init();

    path_init(FILE_get_dir(ARCH_project));

    /* .project file always first ! */
    ARCH_add_file(ARCH_project);

    for(;;)
    {
      if (path_current >= path_count())
        break;

      path = path_list[path_current++];
      dir = opendir(path);

      if (dir == NULL)
      {
        fprintf(stderr, "gba: Warning: Cannot open dir: %s\n", path);
        goto _NEXT_PATH;
      }

      if (chdir(path) != 0)
      {
        fprintf(stderr, "gba: Warning: Cannot change dir: %s\n", path);
        goto _NEXT_PATH;
      }

      while ((dirent = readdir(dir)) != NULL)
      {
        file_name = dirent->d_name;
        len = strlen(file_name);

        if ((*file_name == '.')
            && (strcmp(file_name, ".gambas") != 0)
            && (strcmp(file_name, ".lang") != 0))
          continue;

        if (file_name[len - 1] == '~')
          continue;

        if (strcmp(file_name, ARCH_project_name) == 0)
          continue;

        if ((len >= 8) && (strncmp(file_name, "Makefile", 8) == 0))
          continue;

        file = FILE_cat(path, file_name, NULL);

        if (stat(file_name, &info))
        {
          fprintf(stderr, "gba: Warning: Cannot stat file: %s\n", file);
          continue;
        }

        if (S_ISDIR(info.st_mode))
        {
          path_add(file);
        }
        else
        {
          ext = FILE_get_ext(file);

          if ((strcasecmp(ext, "module") == 0)
              || (strcasecmp(ext, "class") == 0)
              || (strcasecmp(ext, "form") == 0)
              || (strcasecmp(ext, "pot") == 0)
              || (strcasecmp(ext, "po") == 0))
            continue;

          ARCH_add_file(file);
        }
      }

_NEXT_PATH:
      if (dir != NULL) closedir(dir);
      FREE(&path, "main");
    }

    path_exit();

    ARCH_exit();
    /*MEM_check();*/
  }
  CATCH
  {
    fflush(NULL);
    fprintf(stderr, "gba: ERROR: ");
    ERROR_print();
    abort();
  }
  END_TRY

  return 0;
}

