/***************************************************************************

  CArray.c

  The Array native classes.

  (c) 2000-2004 Benot Minisini <gambas@users.sourceforge.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/

#define __GBX_C_ARRAY_C

#include "gb_common.h"
#include "gb_error.h"
#include "gb_array.h"
#include "gbx_class.h"

#include "gbx_exec.h"
#include "gbx_date.h"
#include "gbx_variant.h"
#include "gbx_compare.h"
#include "gbx_class.h"
#include "gbx_object.h"
#include "gbx_api.h"

#include "gbx_c_file.h"
#include "gbx_c_array.h"



PRIVATE void *get_data(CARRAY *_object, long i)
{
  if ((i < 0) || (i >= ARRAY_count(THIS->data)))
  {
    GB_Error((char *)E_BOUND);
    return NULL;
  }

  return (void *)((char *)(THIS->data) + i * TYPE_sizeof_memory(THIS->type));
}


PRIVATE void release_one(CARRAY *_object, long i)
{
  switch(THIS->type)
  {
    case T_STRING:
      STRING_unref(&(((char **)(THIS->data))[i]));
      break;

    case T_OBJECT:
      OBJECT_unref(&(((void **)(THIS->data))[i]));
      break;

    case T_VARIANT:
      VARIANT_free(&(((VARIANT *)(THIS->data))[i]));
      break;

    default:
      break;
  }
}


PRIVATE void release(CARRAY *_object, int start, int end)
{
  int i;
  
  if (end < 0)
    end = ARRAY_count(THIS->data);

  switch(THIS->type)
  {
    case T_STRING:
      for (i = start; i < end; i++)
        STRING_unref(&((char **)(THIS->data))[i]);
      break;

    case T_OBJECT:
      for (i = start; i < end; i++)
        OBJECT_unref(&((void **)(THIS->data))[i]);
      break;

    case T_VARIANT:
      for (i = start; i < end; i++)
        VARIANT_free(&((VARIANT *)(THIS->data))[i]);
      break;

    default:
      break;
  }
}


PRIVATE void borrow(CARRAY *_object, int start, int end)
{
  int i;

  if (end < 0)
    end = ARRAY_count(THIS->data);

  switch(THIS->type)
  {
    case T_STRING:
      for (i = start; i < end; i++)
        STRING_ref(((char **)(THIS->data))[i]);
      break;

    case T_OBJECT:
      for (i = start; i < end; i++)
        OBJECT_ref(((void **)(THIS->data))[i]);
      break;

    case T_VARIANT:
      for (i = start; i < end; i++)
        VARIANT_keep(&((VARIANT *)(THIS->data))[i]);
      break;

    default:
      break;
  }
}


PRIVATE void clear(CARRAY *_object)
{
  release(THIS, 0, -1);
  ARRAY_delete(&THIS->data);
  ARRAY_create_with_size(&THIS->data, TYPE_sizeof_memory(THIS->type), 8);
}


PRIVATE void *insert(CARRAY *_object, long index)
{
  return ARRAY_insert(&THIS->data, index);
}


/*
PUBLIC CARRAY *CARRAY_create(CLASS *klass)
{
  void *object;

  OBJECT_new(&object, CLASS_Array, NULL, NULL);
  OBJECT_UNREF_KEEP(&object, "CARRAY_create");
  return (CARRAY *)object;
}
*/


PUBLIC void CARRAY_split(CARRAY *_object, const char *str, const char *sep, const char *esc)
{
  char c;
  const char *p;
  long len;
  const char *elt;
  bool escape;

  if (sep == NULL || *sep == 0)
    sep = ",";

  if (esc == NULL)
    esc = "";

  clear(THIS);

  p = str;
  escape = FALSE;

  for(;;)
  {
    c = *str;

    if (p == str && c != 0 && index(esc, c))
    {
      escape = TRUE;
      goto _NEXT;
    }

    if (escape)
    {
      if (c != 0 && index(esc, c) == NULL)
        goto _NEXT;
      else
        c = *(++str);
    }

    if (c != 0 && index(sep, c) == NULL)
      goto _NEXT;


    if (escape)
    {
      len = str - p - 2;
      if (len > 0)
      {
        /*str[-1] = 0;*/
        elt = p + 1;
      }
    }
    else
    {
      len = str - p;
      if (len > 0)
      {
        /**str = 0;*/
        elt = p;
      }
    }

    if (len <= 0)
    {
      ARRAY_add_void(&THIS->data);
    }
    else
    {
      STRING_new((char **)ARRAY_add(&THIS->data), elt, len);
    }

    if (c == 0)
      break;

    p = str + 1;
    escape = FALSE;

  _NEXT:

    str++;
  }
}



BEGIN_METHOD(CARRAY_new, GB_INTEGER size)

  TYPE type;
  CLASS *klass;
  long size;
  long inc;

  klass = OBJECT_class(THIS);

  if (klass == CLASS_IntegerArray)
    type = T_INTEGER;
  else if (klass == CLASS_ShortArray)
    type = T_SHORT;
  else if (klass == CLASS_ByteArray)
    type = T_BYTE;
  else if (klass == CLASS_FloatArray)
    type = T_FLOAT;
  else if (klass == CLASS_DateArray)
    type = T_DATE;
  else if (klass == CLASS_StringArray)
    type = T_STRING;
  else if (klass == CLASS_ObjectArray)
    type = T_OBJECT;
  else
    type = T_VARIANT;

  THIS->type = type;

  size = VARGOPT(size, 0);
  if (size < 0)
    size = 0;

  inc = (size / 8) & ~7;
  if (inc < 8)
    inc = 8;

  ARRAY_create_with_size(&THIS->data, TYPE_sizeof_memory(type), inc);
  if (size > 0)
    ARRAY_add_many_void(&THIS->data, size);

END_METHOD


BEGIN_METHOD_VOID(CARRAY_free)

  release(THIS, 0, -1);
  ARRAY_delete(&THIS->data);

END_METHOD


BEGIN_METHOD_VOID(CARRAY_clear)

  clear(THIS);

END_METHOD


BEGIN_PROPERTY(CARRAY_count)

  GB_ReturnInt(ARRAY_count(THIS->data));

END_PROPERTY


BEGIN_PROPERTY(CARRAY_max)

  GB_ReturnInt(ARRAY_count(THIS->data) - 1);

END_PROPERTY


BEGIN_METHOD(CARRAY_remove, GB_INTEGER index; GB_INTEGER count)

  long index = VARG(index);
  long count = VARGOPT(count, 1);

  /* do it after 1.0
    if (index < 0)
    index = ARRAY_count(THIS->data) - index;*/
  
  if ((index < 0) || (index >= ARRAY_count(THIS->data)))
  {
    GB_Error((char *)E_ARG);
    return;
  }

  if (count < 0)
    count = ARRAY_count(THIS->data) - index;

  release(THIS, index, index + count);
  ARRAY_remove_many(&THIS->data, index, count);

END_METHOD


BEGIN_METHOD(CARRAY_resize, GB_INTEGER size)

  long size = VARG(size);
  long count = ARRAY_count(THIS->data);

  if (size > count)
  {
    ARRAY_add_many_void(&THIS->data, size - count);
  }
  else if ((size < count) && (size >= 0))
  {
    release(THIS, size, -1);
    ARRAY_remove_many(&THIS->data, size, count - size);
  }

END_METHOD


PRIVATE void add(CARRAY *_object, GB_VALUE *value, long index)
{
  void *data = insert(THIS, index);
  /*GB_Conv(value, THIS->type);*/
  GB_Store(THIS->type, value, data);
}


#define IMPLEMENT_add(_type, _gtype) \
BEGIN_METHOD(CARRAY_##_type##_add, GB_##_gtype value; GB_INTEGER index) \
\
  add(THIS, (GB_VALUE *)ARG(value), VARGOPT(index, -1)); \
  \
END_METHOD \
\
BEGIN_METHOD(CARRAY_##_type##_push, GB_##_gtype value) \
\
  add(THIS, (GB_VALUE *)ARG(value), -1); \
  \
END_METHOD

IMPLEMENT_add(integer, INTEGER)
IMPLEMENT_add(short, INTEGER)
IMPLEMENT_add(byte, INTEGER)
IMPLEMENT_add(float, FLOAT)
IMPLEMENT_add(date, DATE)
IMPLEMENT_add(string, STRING)
IMPLEMENT_add(object, OBJECT)
IMPLEMENT_add(variant, VARIANT)


#define IMPLEMENT_put(_type, _gtype) \
BEGIN_METHOD(CARRAY_##_type##_put, GB_##_gtype value; GB_INTEGER index) \
\
  void *data = get_data(THIS, VARG(index)); \
  if (!data) return; \
  GB_Store(GB_T_##_gtype, (GB_VALUE *)ARG(value), data); \
  \
END_METHOD

IMPLEMENT_put(integer, INTEGER)
IMPLEMENT_put(short, INTEGER)
IMPLEMENT_put(byte, INTEGER)
IMPLEMENT_put(float, FLOAT)
IMPLEMENT_put(date, DATE)
IMPLEMENT_put(string, STRING)
IMPLEMENT_put(object, OBJECT)
IMPLEMENT_put(variant, VARIANT)


BEGIN_METHOD(CARRAY_insert, GB_OBJECT array; GB_INTEGER pos)

  long pos = VARGOPT(pos, -1);
  CARRAY *array = (CARRAY *)VARG(array);
  int count = ARRAY_count(array->data);
  void *data;

  if (count <= 0)
    return;
    
   if (pos < 0)
     pos = ARRAY_count(THIS->data);

  data = ARRAY_insert_many(&THIS->data, pos, count);
  borrow(array, 0, -1);
  memmove(data, array->data, count * TYPE_sizeof_memory(THIS->type));

END_METHOD


BEGIN_METHOD(CARRAY_copy, GB_INTEGER start; GB_INTEGER length)

  CARRAY *array;
  int count = ARRAY_count(THIS->data);
  void *data;
  int start = VARGOPT(start, 0);
  int length = VARGOPT(length, -1);

  if (start < 0 || start >= count)
  {  
    GB_Error((char *)E_BOUND);
    return;
  }
  
  if (length == -1)
    length = count - start;
     
  if (length < 0 || (start + length) > count)
  {  
    GB_Error((char *)E_BOUND);
    return;
  }
  
  GB_ArrayNew((GB_ARRAY)&array, THIS->type, 0);  
  
  if (length > 0)
  {
    data = ARRAY_insert_many(&array->data, 0, length);
    memmove(data, get_data(THIS, start), length * TYPE_sizeof_memory(THIS->type));
    borrow(array, 0, -1);
  }
  
  GB_ReturnObject(array);

END_METHOD


BEGIN_METHOD_VOID(CARRAY_pop)

  long index = ARRAY_count(THIS->data) - 1;

  if (index < 0)
  {
    GB_Error((char *)E_ARG);
    return;
  }

  GB_ReturnPtr(THIS->type, get_data(THIS, index));

  BORROW(&TEMP);
  release_one(THIS, index);
  ARRAY_remove(&THIS->data, index);
  UNBORROW(&TEMP);

END_METHOD


BEGIN_METHOD(CARRAY_get, GB_INTEGER index)

  void *data = get_data(THIS, VARG(index));

  if (data)
    GB_ReturnPtr(THIS->type, data);

END_METHOD


BEGIN_METHOD_VOID(CARRAY_next)

  long *index = (long *)GB_GetEnum();

  if (*index >= ARRAY_count(THIS->data))
    GB_StopEnum();
  else
  {
    GB_ReturnPtr(THIS->type, get_data(THIS, *index));
    (*index)++;
  }

END_METHOD


BEGIN_METHOD(CARRAY_sort, GB_INTEGER mode)

  COMPARE_FUNC compare;
  long mode = VARGOPT(mode, 0);
  void *data = THIS->data;

  if (ARRAY_count(data) <= 1)
    return;

  compare = COMPARE_get(THIS->type, mode);

  qsort(data, ARRAY_count(data), TYPE_sizeof_memory(THIS->type), compare);

END_METHOD


PRIVATE void find(CARRAY *_object, int mode, void *value)
{
  COMPARE_FUNC compare = COMPARE_get(THIS->type, mode);
  int i;

  for (i = 0; i < ARRAY_count(THIS->data); i++)
  {
    if ((*compare)(value, get_data(THIS, i)) == 0)
    {
      GB_ReturnInt(i);
      return;
    }
  }

  GB_ReturnInt(-1);
}

#define IMPLEMENT_find(_type, _gtype) \
BEGIN_METHOD(CARRAY_##_type##_find, _gtype value; GB_INTEGER mode) \
\
  find(THIS, VARGOPT(mode, 0), &VARG(value)); \
\
END_METHOD

IMPLEMENT_find(integer, GB_INTEGER)
IMPLEMENT_find(short, GB_INTEGER)
IMPLEMENT_find(byte, GB_INTEGER)
IMPLEMENT_find(float, GB_FLOAT)
IMPLEMENT_find(date, GB_DATE)
IMPLEMENT_find(object, GB_OBJECT)


BEGIN_METHOD(CARRAY_string_find, GB_STRING value; GB_INTEGER mode)

  char *str = GB_ToZeroString(ARG(value));

  find(THIS, VARGOPT(mode, 0), &str);

END_METHOD


BEGIN_METHOD(CARRAY_string_join, GB_STRING sep)

  char *sep = ",";
  char *res = NULL;
  int i;
  char **data = (char **)THIS->data;

  if (!MISSING(sep))
    sep = GB_ToZeroString(ARG(sep));

  for (i = 0; i < ARRAY_count(data); i++)
  {
    if (i)
      STRING_add(&res, sep, 0);
    STRING_add(&res, data[i], STRING_length(data[i]));
  }

  if (res)
  {
    STRING_extend_end(&res);
    GB_ReturnString(res);
  }
  else
    GB_ReturnNull();

END_METHOD


BEGIN_METHOD_VOID(CARRAY_reverse)

  size_t size;
  long count;
  char *buffer[16];
  char *pi, *pj;

  count = ARRAY_count(THIS->data);
  if (count <= 1)
    return;

  size = TYPE_sizeof_memory(THIS->type);
  pi = get_data(THIS, 0);
  pj = get_data(THIS, count - 1);

  do
  {
    memcpy(buffer, pi, size);
    memcpy(pi, pj, size);
    memcpy(pj, buffer, size);
    pi += size;
    pj -= size;
  }
  while (pi < pj);

END_METHOD


BEGIN_METHOD(CARRAY_read, GB_OBJECT file; GB_INTEGER start; GB_INTEGER length)

  long count = ARRAY_count(THIS->data);
  long start = VARGOPT(start, 0);
  long length = VARGOPT(length, count);

  if ((start + length) > count)
    length = count - start;

  STREAM_read(CSTREAM_stream(VARG(file)), get_data(THIS, start), length * TYPE_sizeof_memory(THIS->type));

END_METHOD


BEGIN_METHOD(CARRAY_write, GB_OBJECT file; GB_INTEGER start; GB_INTEGER length)

  long count = ARRAY_count(THIS->data);
  long start = VARGOPT(start, 0);
  long length = VARGOPT(length, count);

  if ((start + length) > count)
    length = count - start;

  STREAM_write(CSTREAM_stream(VARG(file)), get_data(THIS, start), length * TYPE_sizeof_memory(THIS->type));

END_METHOD


PUBLIC GB_DESC NATIVE_Array[] =
{
  GB_DECLARE(".Array", sizeof(CARRAY)), GB_NOT_CREATABLE(),

  GB_METHOD("_free", NULL, CARRAY_free, NULL),

  GB_PROPERTY_READ("Count", "i", CARRAY_count),
  GB_PROPERTY_READ("Max", "i", CARRAY_max),
  GB_PROPERTY_READ("Length", "i", CARRAY_count),

  GB_METHOD("Remove", NULL, CARRAY_remove, "(Index)i[(Count)i]"),
  GB_METHOD("Clear", NULL, CARRAY_clear, NULL),
  GB_METHOD("Resize", NULL, CARRAY_resize, "(Size)i"),
  GB_METHOD("Sort", NULL, CARRAY_sort, "[(Mode)i]"),
  GB_METHOD("Reverse", NULL, CARRAY_reverse, NULL),

  //GB_METHOD("FromString", "i", CARRAY_from_string, "(String)s[(Separator)s(Escape)s]"),
  //GB_METHOD("Sort", NULL, CARRAY_sort, "[(Compare)i]"),
  //GB_METHOD("Find", "i", CARRAY_find, "(Value)s[(Compare)i]"),

  GB_END_DECLARE
};


PUBLIC GB_DESC NATIVE_IntegerArray[] =
{
  GB_DECLARE("Integer[]", sizeof(CARRAY)), GB_INHERITS(".Array"),

  GB_METHOD("_new", NULL, CARRAY_new, "[(Size)i]"),
  
  GB_METHOD("Add", NULL, CARRAY_integer_add, "(Value)i[(Index)i]"),
  GB_METHOD("Push", NULL, CARRAY_integer_push, "(Value)i"),
  GB_METHOD("_put", NULL, CARRAY_integer_put, "(Value)i(Index)i"),
  GB_METHOD("Find", "i", CARRAY_integer_find, "(Value)i[(Mode)i]"),

  GB_METHOD("Pop", "i", CARRAY_pop, NULL),
  GB_METHOD("_get", "i", CARRAY_get, "(Index)i"),
  GB_METHOD("_next", "i", CARRAY_next, NULL),

  GB_METHOD("Read", NULL, CARRAY_read, "(File)File;[(Start)i(Length)i]"),
  GB_METHOD("Write", NULL, CARRAY_write, "(File)File;[(Start)i(Length)i]"),

  GB_METHOD("Insert", NULL, CARRAY_insert, "(Array)Integer[];[(Pos)i]"),
  GB_METHOD("Copy", "Integer[]", CARRAY_copy, "[(Start)i(Length)i]"),
  //GB_METHOD("Extract", NULL, CARRAY_extract, "(Start)i[(Length)i]"),

  GB_END_DECLARE
};


PUBLIC GB_DESC NATIVE_ShortArray[] =
{
  GB_DECLARE("Short[]", sizeof(CARRAY)), GB_INHERITS(".Array"),

  GB_METHOD("_new", NULL, CARRAY_new, "[(Size)i]"),
  
  GB_METHOD("Add", NULL, CARRAY_integer_add, "(Value)i[(Index)i]"),
  GB_METHOD("Push", NULL, CARRAY_integer_push, "(Value)i"),
  GB_METHOD("_put", NULL, CARRAY_integer_put, "(Value)i(Index)i"),
  GB_METHOD("Find", "i", CARRAY_integer_find, "(Value)i[(Mode)i]"),

  GB_METHOD("Pop", "i", CARRAY_pop, NULL),
  GB_METHOD("_get", "i", CARRAY_get, "(Index)i"),
  GB_METHOD("_next", "i", CARRAY_next, NULL),

  GB_METHOD("Read", NULL, CARRAY_read, "(File)File;[(Start)i(Length)i]"),
  GB_METHOD("Write", NULL, CARRAY_write, "(File)File;[(Start)i(Length)i]"),

  GB_METHOD("Insert", NULL, CARRAY_insert, "(Array)Short[];[(Pos)i]"),
  GB_METHOD("Copy", "Short[]", CARRAY_copy, "[(Start)i(Length)i]"),

  GB_END_DECLARE
};


PUBLIC GB_DESC NATIVE_ByteArray[] =
{
  GB_DECLARE("Byte[]", sizeof(CARRAY)), GB_INHERITS(".Array"),

  GB_METHOD("_new", NULL, CARRAY_new, "[(Size)i]"),
  
  GB_METHOD("Add", NULL, CARRAY_integer_add, "(Value)i[(Index)i]"),
  GB_METHOD("Push", NULL, CARRAY_integer_push, "(Value)i"),
  GB_METHOD("_put", NULL, CARRAY_integer_put, "(Value)i(Index)i"),
  GB_METHOD("Find", "i", CARRAY_integer_find, "(Value)i[(Mode)i]"),

  GB_METHOD("Pop", "i", CARRAY_pop, NULL),
  GB_METHOD("_get", "i", CARRAY_get, "(Index)i"),
  GB_METHOD("_next", "i", CARRAY_next, NULL),

  GB_METHOD("Read", NULL, CARRAY_read, "(File)File;[(Start)i(Length)i]"),
  GB_METHOD("Write", NULL, CARRAY_write, "(File)File;[(Start)i(Length)i]"),

  GB_METHOD("Insert", NULL, CARRAY_insert, "(Array)Byte[];[(Pos)i]"),
  GB_METHOD("Copy", "Byte[]", CARRAY_copy, "[(Start)i(Length)i]"),

  GB_END_DECLARE
};


PUBLIC GB_DESC NATIVE_StringArray[] =
{
  GB_DECLARE("String[]", sizeof(CARRAY)), GB_INHERITS(".Array"),

  GB_METHOD("_new", NULL, CARRAY_new, "[(Size)i]"),
  
  GB_METHOD("Add", NULL, CARRAY_string_add, "(Value)s[(Index)i]"),
  GB_METHOD("Push", NULL, CARRAY_string_push, "(Value)s"),
  GB_METHOD("_put", NULL, CARRAY_string_put, "(Value)s(Index)i"),
  GB_METHOD("Find", "i", CARRAY_string_find, "(Value)s[(Mode)i]"),

  GB_METHOD("Pop", "s", CARRAY_pop, NULL),
  GB_METHOD("_get", "s", CARRAY_get, "(Index)i"),
  GB_METHOD("_next", "s", CARRAY_next, NULL),

  GB_METHOD("Insert", NULL, CARRAY_insert, "(Array)String[];[(Pos)i]"),

  GB_METHOD("Join", "s", CARRAY_string_join, "[(Separator)s]"),

  /* Inutile ! NE sert qu' faire apparatre la mthode dans
     l'explorateur de composant ! */

  GB_METHOD("Sort", NULL, CARRAY_sort, "[(Mode)i]"),

  GB_METHOD("Copy", "String[]", CARRAY_copy, "[(Start)i(Length)i]"),
  
  GB_END_DECLARE
};


PUBLIC GB_DESC NATIVE_FloatArray[] =
{
  GB_DECLARE("Float[]", sizeof(CARRAY)), GB_INHERITS(".Array"),

  GB_METHOD("_new", NULL, CARRAY_new, "[(Size)i]"),
  
  GB_METHOD("Add", NULL, CARRAY_float_add, "(Value)f[(Index)i]"),
  GB_METHOD("Push", NULL, CARRAY_float_push, "(Value)f"),
  GB_METHOD("_put", NULL, CARRAY_float_put, "(Value)f(Index)i"),
  GB_METHOD("Find", "i", CARRAY_float_find, "(Value)f[(Mode)i]"),

  GB_METHOD("Pop", "f", CARRAY_pop, NULL),
  GB_METHOD("_get", "f", CARRAY_get, "(Index)i"),
  GB_METHOD("_next", "f", CARRAY_next, NULL),

  GB_METHOD("Read", NULL, CARRAY_read, "(File)File;[(Start)i(Length)i]"),
  GB_METHOD("Write", NULL, CARRAY_write, "(File)File;[(Start)i(Length)i]"),

  GB_METHOD("Insert", NULL, CARRAY_insert, "(Array)Float[];[(Pos)i]"),

  GB_METHOD("Copy", "Float[]", CARRAY_copy, "[(Start)i(Length)i]"),
  
  GB_END_DECLARE
};


PUBLIC GB_DESC NATIVE_DateArray[] =
{
  GB_DECLARE("Date[]", sizeof(CARRAY)), GB_INHERITS(".Array"),

  GB_METHOD("_new", NULL, CARRAY_new, "[(Size)i]"),
  
  GB_METHOD("Add", NULL, CARRAY_date_add, "(Value)d[(Index)i]"),
  GB_METHOD("Push", NULL, CARRAY_date_push, "(Value)d"),
  GB_METHOD("_put", NULL, CARRAY_date_put, "(Value)d(Index)i"),
  GB_METHOD("Find", "i", CARRAY_date_find, "(Value)d[(Mode)i]"),

  GB_METHOD("Pop", "d", CARRAY_pop, NULL),
  GB_METHOD("_get", "d", CARRAY_get, "(Index)i"),
  GB_METHOD("_next", "d", CARRAY_next, NULL),

  GB_METHOD("Read", NULL, CARRAY_read, "(File)File;[(Start)i(Length)i]"),
  GB_METHOD("Write", NULL, CARRAY_write, "(File)File;[(Start)i(Length)i]"),

  GB_METHOD("Insert", NULL, CARRAY_insert, "(Array)Date[];[(Pos)i]"),

  GB_METHOD("Copy", "Date[]", CARRAY_copy, "[(Start)i(Length)i]"),
  
  GB_END_DECLARE
};


PUBLIC GB_DESC NATIVE_ObjectArray[] =
{
  GB_DECLARE("Object[]", sizeof(CARRAY)), GB_INHERITS(".Array"),

  GB_METHOD("_new", NULL, CARRAY_new, "[(Size)i]"),
  
  GB_METHOD("Add", NULL, CARRAY_object_add, "(Value)o[(Index)i]"),
  GB_METHOD("Push", NULL, CARRAY_object_push, "(Value)o"),
  GB_METHOD("_put", NULL, CARRAY_object_put, "(Value)o(Index)i"),
  GB_METHOD("Find", "i", CARRAY_object_find, "(Value)o[(Mode)i]"),

  GB_METHOD("Pop", "o", CARRAY_pop, NULL),
  GB_METHOD("_get", "o", CARRAY_get, "(Index)i"),
  GB_METHOD("_next", "o", CARRAY_next, NULL),

  GB_METHOD("Insert", NULL, CARRAY_insert, "(Array)Object[];[(Pos)i]"),

  GB_METHOD("Copy", "Object[]", CARRAY_copy, "[(Start)i(Length)i]"),
  
  GB_END_DECLARE
};


PUBLIC GB_DESC NATIVE_VariantArray[] =
{
  GB_DECLARE("Variant[]", sizeof(CARRAY)), GB_INHERITS(".Array"),

  GB_METHOD("_new", NULL, CARRAY_new, "[(Size)i]"),
  
  GB_METHOD("Add", NULL, CARRAY_variant_add, "(Value)v[(Index)i]"),
  GB_METHOD("Push", NULL, CARRAY_variant_push, "(Value)v"),
  GB_METHOD("_put", NULL, CARRAY_variant_put, "(Value)v(Index)i"),

  GB_METHOD("Pop", "v", CARRAY_pop, NULL),
  GB_METHOD("_get", "v", CARRAY_get, "(Index)i"),
  GB_METHOD("_next", "v", CARRAY_next, NULL),

  GB_METHOD("Insert", NULL, CARRAY_insert, "(Array)Variant[];[(Pos)i]"),

  GB_METHOD("Copy", "Variant[]", CARRAY_copy, "[(Start)i(Length)i]"),
  
  GB_END_DECLARE
};


/* Gambas API */

PUBLIC void GB_ArrayNew(GB_ARRAY *array, long type, long size)
{
  CLASS *class;
  GB_INTEGER param;

  if (size > 0)
  {
    param.type = GB_T_INTEGER;
    param.value = size;
  }
  else
    param.type = T_VOID;

  switch(type)
  {
    case GB_T_INTEGER: class = CLASS_IntegerArray; break;
    case GB_T_SHORT: class = CLASS_ShortArray; break;
    case GB_T_BYTE: class = CLASS_ByteArray; break;
    case GB_T_STRING: class = CLASS_StringArray; break;
    case GB_T_FLOAT: class = CLASS_FloatArray; break;
    case GB_T_DATE: class = CLASS_DateArray; break;
    case GB_T_OBJECT: class = CLASS_ObjectArray; break;
    default: class = CLASS_VariantArray; break;
  }

  OBJECT_create_native((void **)array, class, (VALUE *)&param);
}

PUBLIC long GB_ArrayCount(GB_ARRAY array)
{
  return ARRAY_count(((CARRAY *)array)->data);
}

PUBLIC void *GB_ArrayAdd(GB_ARRAY array)
{
  return insert((CARRAY *)array, -1);
}

PUBLIC void *GB_ArrayGet(GB_ARRAY array, long index)
{
  return get_data((CARRAY *)array, index);
}



