# -*- coding: utf-8 -*-
# The GTK graphical user interface

import pygtk
pygtk.require('2.0')
import pango
import gtk
import gobject
import time
from gameclock.game import Game
import gameclock.clock

class GameclockUI:
    """this class handles most of the UI and turned-based logic
    
    It is not designed for UI-abstraction yet, but could be, if the
    turn-based logic is ripped in a seperate class
    """

    verbose = 0
    first_clock = None
    cur_clock = None
    clock_widget_cnt = 0
    fullscreen = False
    # the game this UI references
    game = None
    sec_loop_timeout = 200 # in ms, clocks are incremented every time this timeout ends
    ms_loop_timeout = 50 # in ms, the same as above, in milisecond mode
    menubar = None # used in the fullscreen toggle

    class ClockUI:
        """this class is used to encapsulate the various controls related with a clock"""

        # like the game Clock, it is a linked list
        next = None
        ui = None

        def __init__(self, ui, clock = None):
            self.ui = ui
            self.next = clock
            self.label = gtk.Label()
            self.label.modify_font(pango.FontDescription("sans 72"))
            self.label.show()

            # event boxes to be able to color the labels
            evbox = gtk.EventBox()
            evbox.modify_bg(gtk.STATE_NORMAL, evbox.get_colormap().alloc_color("green"))
            evbox.show()
            evbox.add(self.label)
            ui.add_clock_widget(evbox)

        def __del__(self):
            self.ui.remove_clock_widget(self.label.parent)
            self.label.destroy()
            del self.label

    def quit(self, action = None):
        gtk.main_quit()

    def add_clock_widget(self, widget):
        """add a new clock to the UI"""
        self.clock_widget_cnt += 1
        cols = 2 # all the time
        rows = (self.clock_widget_cnt - 1)/ cols + 1
        self.clock_table.resize(rows, cols)
        l = ( self.clock_widget_cnt % cols + 1 ) % cols
        r = l + 1
        t = rows - 1
        b = rows
        self.clock_table.attach(widget, l, r, t, b)

    def remove_clock_widget(self, widget):
        self.clock_table.remove(widget)
        self.clock_widget_cnt -= 1
        cols = 2 # all the time
        rows = (self.clock_widget_cnt - 1)/ cols + 1
        self.clock_table.resize(rows, cols)

    def addmenubar(self, widget):
        self.vlayout.pack_start(widget, False, True, 0)
        self.menubar = widget

    class MenuItemFactory:
        """
        This class takes care of creating the menus and keyboard shortcuts.
        """

        ui_desc = """
<ui>
  <menubar name="menubar">
    <menu action="game">
      <menuitem name="New" action="new" />
      <menuitem name="Pause" action="pause" />
      <menuitem name="Full screen" action="fullscreen" />
      <menuitem name="Quit" action="quit" />
    </menu>
    <menu action="help">
      <menuitem name="About" action="about" />
      <menuitem name="Keyboard shortcuts" action="keyboard" />
    </menu>
  </menubar>
</ui>
"""

        def __init__(self, ui):
            self.ui = ui

            # Create a UIManager instance
            uimanager = gtk.UIManager()

            # Add the accelerator group to the toplevel window
            accelgroup = uimanager.get_accel_group()
            ui.window.add_accel_group(accelgroup)

            # Create an ActionGroup
            actiongroup = gtk.ActionGroup('UIManagerExample')
            self.actiongroup = actiongroup

            # Create actions
            actiongroup.add_actions([('game', None, '_Game'),
                                     ('new', None, '_New game', '<control>n',
                                      'Create a new game', self.new_dialog),
                                     ('pause', None, '_Pause', '<control>p',
                                      'Pause game', ui.handle_pause),
                                     ('fullscreen', None, '_Full screen', '<control>f',
                                      'Full screen mode', ui.handle_fullscreen),
                                     ('quit', gtk.STOCK_QUIT, '_Quit', '<control>q',
                                      'Quit the Program', ui.quit),
                                     ('help', None, '_Help'),
                                     ('about', None, '_About', None,
                                      'More information about this software', self.about_dialog),
                                     ('keyboard', None, 'Keyboard shortcuts', None,
                                      'Display the available keyboard shortcuts', self.shortcuts_dialog),
                                     ])
            actiongroup.get_action('quit').set_property('short-label', '_Quit')


            # Add the actiongroup to the uimanager
            uimanager.insert_action_group(actiongroup, 0)

            # Add a UI description
            uimanager.add_ui_from_string(self.ui_desc)

            # Create a MenuBar
            menubar = uimanager.get_widget('/menubar')
            uimanager.get_widget('/menubar/help').set_right_justified(True)

            ui.addmenubar(menubar)
            menubar.show()

        def handle_new_dialog(self, dialog, response_id):
            dialog.destroy()
            if response_id == gtk.RESPONSE_ACCEPT:
                if self.ms_ctrl.get_active():
                    self.ui.maybe_print("running in miliseconds")
                    self.ui.loop_timeout = self.ui.ms_loop_timeout
                else:
                    self.ui.maybe_print("running in seconds")
                    self.ui.loop_timeout = self.ui.sec_loop_timeout

                hours = self.hours_val.get_value()
                minutes = self.minutes_val.get_value()
                seconds = self.seconds_val.get_value()
                default_time = ( ( hours * 60 * 60 ) + ( minutes * 60 ) + seconds ) * 1000

                for type, button in self.type_buttons.items():
                    if button.get_active():
                        game = Game(eval("gameclock.clock.%sClock" % type), self.count_players(), default_time, self.ms_ctrl.get_active(), self.get_fisher())
                        self.ui.maybe_print("created %s game" % type)

                self.ui.game = game
                self.ui.setup_clocks(int(self.players_val.get_value()))
                if not self.left_starts.get_active():
                    self.ui.switch_clock()

                self.ui.refresh()
                return game
            else:
                return False

        def handle_fisher(self, widget):
            if widget.get_active():
                self.fisher_val_btn.show()
                self.fisher_label.show()
            else:
                self.fisher_val_btn.hide()
                self.fisher_label.hide()
            return

        def count_players(self):
            """return the number of players configured"""
            try:
                return int(self.players_val.get_value())
            except:
                return Game.default_players

        def get_fisher(self):
            """return the current fisher delay selected in the UI or the default if there is no UI yet"""
            try:
                return int(self.fisher_val.get_value() * 1000)
            except:
                return FisherClock.delay

        def new_dialog(self, action = None):
            window = gtk.Dialog("Create new game", self.ui.window, gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                     (gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT,
                      gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
            window.set_default_response(gtk.RESPONSE_ACCEPT)
            window.connect("delete_event", lambda a, b: False)
            window.connect("destroy", lambda a: window.destroy)
            window.connect('response', self.handle_new_dialog)
            window.show()
            controls = gtk.HBox(False, 0)
            controls.show()
            window.vbox.pack_start(controls)

            # this may be a reset, init from old values
            try:
                miliseconds = self.game.miliseconds
                default_clock = self.game.first_clock.__class__.__name__
            except AttributeError:
                self.ui.maybe_print("exception")
                miliseconds = Game.default_time
                default_clock = Game.clock_type

            left_controls = gtk.VBox(False, 0)
            left_controls.show()

            # the widgets to change the starting time
            (hours, minutes, seconds) = map(int, time.strftime("%H:%M:%S", time.gmtime(miliseconds/1000)).split(":"))
            # XXX: spinbuttons limited to 24h-1s because of limitations of strftime
            self.hours_val = gtk.Adjustment(hours, 0, 23, 1, 10, 0)
            self.minutes_val = gtk.Adjustment(minutes, 0, 59, 1, 10, 0)
            self.seconds_val = gtk.Adjustment(seconds, 0, 59, 1, 10, 0)
            hours_val_btn = gtk.SpinButton(self.hours_val, 0.0, 0)
            minutes_val_btn = gtk.SpinButton(self.minutes_val, 0.0, 0)
            seconds_val_btn = gtk.SpinButton(self.seconds_val, 0.0, 0)
            hours_val_btn.show()
            minutes_val_btn.show()
            seconds_val_btn.show()
            
            clock_controls = gtk.HBox(False, 0)
            clock_controls.pack_start(hours_val_btn, False, False, 10)
            clock_controls.pack_start(minutes_val_btn, False, False, 10)
            clock_controls.pack_start(seconds_val_btn, False, False, 10)
            clock_controls.show()
            
            clock_controls_box = gtk.VBox(False, 0)
            label = gtk.Label("Time limit")
            label.show()
            
            clock_controls_box.pack_start(label, False, False, 10)
            clock_controls_box.pack_start(clock_controls, False, False, 10)
            clock_controls_box.show()

            left_controls.pack_start(clock_controls_box, False, False, 10)

            self.ms_ctrl = gtk.CheckButton("Display _miliseconds")
            self.ms_ctrl.show()
            left_controls.pack_start(self.ms_ctrl)

            label = gtk.Label('Number of players: ')
            label.show()
            left_controls.pack_start(label, False, False, 10)
            
            self.players_val = gtk.Adjustment(Game.default_players, 1, 10000, 1, 10, 0)
            players_val_btn = gtk.SpinButton(self.players_val, 0.0, 0)
            players_val_btn.show()
            left_controls.pack_start(players_val_btn, False, False, 0)

            controls.pack_start(left_controls)
            
            # The game mode radio buttons
            type_layout = gtk.VBox(False, 10)
            type_layout.set_border_width(10)
            type_layout.show()

            label = gtk.Label("Game type")
            label.show()
            type_layout.pack_start(label, False, False, 0)

            self.type_buttons = {}
            button = None
            for type in [ "Board", "Hourglass", "Chess" ]:
                button = gtk.RadioButton(button, type)
                clock_type = eval("gameclock.clock.%sClock" % type)
                button.set_tooltip_text(clock_type.__doc__)
                if default_clock == clock_type:
                    button.set_active(True)
                type_layout.pack_start(button, False, False, 0)
                button.show()
                self.type_buttons[type] = button

            # fisher is special: we need to pack more stuff on its line
            fisher_layout = gtk.HBox(False, 10)
            type_layout.pack_start(fisher_layout, False, False, 0)

            button = gtk.RadioButton(button, "Fisher")
            button.connect("toggled", self.handle_fisher)
            button.set_tooltip_text(gameclock.clock.FisherClock.__doc__)
            fisher_layout.pack_start(button, False, False, 0)
            self.type_buttons["Fisher"] = button
            button.show()

            self.fisher_label = gtk.Label('Delay: ')
            fisher_layout.pack_start(self.fisher_label, True, False, 0)

            self.fisher_val = gtk.Adjustment(gameclock.clock.FisherClock.delay, 1, 10000, 1, 10, 0)
            self.fisher_val_btn = gtk.SpinButton(self.fisher_val, 0.0, 0)
            fisher_layout.pack_start(self.fisher_val_btn, False, False, 0)
            fisher_layout.show()

            controls.pack_start(type_layout)

            right_controls = gtk.VBox(False, 10)
            right_controls.set_border_width(10)
            right_controls.show()

            # the main toggle button, used by various signal handlers
            label = gtk.Label('Starting player')
            label.show()
            right_controls.pack_start(label, False, False, 0)
            button = gtk.RadioButton(None, "Left")
            button.set_active(True)
            button.show()
            right_controls.pack_start(button, False, False, 0)
            self.left_starts = button
            button = gtk.RadioButton(button, "Right")
            button.show()
            right_controls.pack_start(button, False, False, 0)
            controls.pack_start(right_controls)
            return window

        def shortcuts_dialog(self, action = None):
            window = gtk.Dialog("Shortcuts", self.ui.window, gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                     (gtk.STOCK_OK, gtk.RESPONSE_OK))
            window.set_default_response(gtk.RESPONSE_OK)
            # little help
            label = gtk.Label("""<b>shift</b> and <b>space</b> keys end turns
<b>control-q</b> quits
<b>control-n</b> starts a new fgame
<b>control-f</b> enables fullscreen mode
<b>control-p, F5-F8</b> pause the game

Left side of the keyboard ends left side's turn and vice-versa for right turn.

We do not currently handle the numpad and arrow keys as we can't tell
if they are present or not (e.g. laptops) and that would favor too
much right right side if it *is* present.
""")
            label.set_use_markup(True)
            label.show()
            window.vbox.pack_start(label)
            window.connect("response", lambda a, b: window.destroy())
            window.show()

        def about_dialog(self, action = None):
            dialog = gtk.AboutDialog()
            dialog.set_version(version)
            dialog.set_copyright(copyright)
            #dialog.set_license("license?")
            dialog.set_comments("A simple game clock to be used for Chess or any board game")
            dialog.set_website("https://redmine.koumbit.net/projects/gameclock")
            dialog.set_authors(["Antoine Beaupré <anarcat@anarcat.ath.cx>"])
            dialog.connect("response", lambda a, b: dialog.destroy())
            dialog.show()

    def __init__(self):
        self.loop_timeout = self.sec_loop_timeout

    def main(self):
        """create the main user interface with GTK"""

        # create a new window
        self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)

        # handle window close events
        self.window.connect("delete_event", lambda a, b: False)    
        self.window.connect("destroy", self.quit)
    
        self.window.connect('key_press_event', self.handle_key_press)

        event_box = gtk.EventBox()
        self.window.add(event_box)
        event_box.show()
        # catch clicks as end turn
        event_box.set_extension_events(gtk.gdk.EXTENSION_EVENTS_ALL)
        event_box.set_events(gtk.gdk.BUTTON_PRESS_MASK)
        event_box.connect("button_press_event", self.end_turn)
        event_box.modify_bg(gtk.STATE_NORMAL, gtk.gdk.Color('black'))

        # main window consists of a vbox containing two hbox
        self.vlayout = vlayout = gtk.VBox(False, 0)
        event_box.add(vlayout)

        # the menu on top, which also acts as a dispatcher for the "New" dialog
        menu = GameclockUI.MenuItemFactory(self)

        # turn counter
        self.turns = gtk.Label()
        self.turns.modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color('white'))
        vlayout.pack_start(self.turns, False, False, 0)

        # the clocks
        self.clock_table = gtk.Table(1, 2, True)
        self.clock_table.show()
        vlayout.pack_start(self.clock_table, True, True, 0)

        vlayout.show()

        if self.fullscreen:
            self.window.fullscreen()

        menu.new_dialog().run()

        if self.game:
            self.window.show()
            gtk.main()

    def setup_clocks(self, players):
        if self.first_clock:
            # count the clocks
            p = self.first_clock
            count = 1
            while p.next:
                p = p.next
                count += 1
            # two special cases
            # add missing clocks
            while count < players:
                p.next = GameclockUI.ClockUI(self)
                p = p.next
                count += 1
            if count > players:
                count = 1
                p = self.first_clock
                while count < players:
                    count += 1
                    p = p.next
                while p.next:
                    q = p.next
                    del p.next
                    p = q
            # go back to the first clock anyways
            self.cur_clock = self.first_clock
        else:
            p = self.cur_clock = self.first_clock = GameclockUI.ClockUI(self)
            for i in range(players-1):
                p.next = GameclockUI.ClockUI(self)
                p = p.next
        cols = 2
        rows = (players - 1)/ cols + 1
        self.clock_table.resize(rows, cols)
        self.refresh()
        self.maybe_print("now at %d clocks, table size is %dX%d" % (players, rows, cols))

    def switch_clock(self):
        """change the current clock

        simply switch the clock in the game engine and rehilight
        """
        self.game.switch_clock()
        self.cur_clock = self.cur_clock.next
        if not self.cur_clock:
            self.cur_clock = self.first_clock

    def maybe_print(self, text):
        if self.verbose:
            t = ""
            state = ""
            if self.verbose > 1:
                t = "[%f] " % time.time()
                if self.verbose > 2:
                    state = "\n  game engine state: %s" % self.game
            print t + text + state

    def refresh(self):
        if self.game.miliseconds:
            font = "sans 48"
        else:
            font = "sans 72"

        p = self.first_clock
        q = self.game.first_clock
        while p:
            p.label.modify_font(pango.FontDescription(font))
            p.label.set_label(q.text)
            p = p.next
            q = q.next
        
        self.turns.set_label("Turn %d" % self.game.get_turns())
        self.hilight()
        
    def refresh_current(self):
        """refresh the active clock
        
        this handler is ran periodically through a timeout signal to make sure that the current clock is updated
        """

        if isinstance(self.game.first_clock, gameclock.clock.HourglassClock):
            p = self.game.first_clock
            while p:
                p.update()
                p = p.next
            self.refresh()
            return self.game.running()

        self.cur_clock.label.set_label(self.game.cur_clock.update())
        active = self.cur_clock.label.get_parent()
        if self.game.cur_clock.dead:
            active.modify_bg(gtk.STATE_NORMAL, active.get_colormap().alloc_color("red"))
        return self.game.running()

    def hilight(self):
        """hilight the proper clocks with proper colors
        
        this is 'transparent' for the inactive clock and colored for the
        active clock. the color depends on wether the clock is 'dead' or
        not
        """
        p = self.first_clock
        q = self.game.first_clock
        while p:
            widget = p.label.get_parent()
            if p == self.cur_clock:
                if q.dead:
                    color = gtk.gdk.Color("red")
                else:
                    color = gtk.gdk.Color("green")
                p.label.modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color('black'))
            else:
                color = gtk.gdk.Color('black')
                p.label.modify_fg(gtk.STATE_NORMAL, gtk.gdk.Color('white'))
            widget.modify_bg(gtk.STATE_NORMAL, color)
            p = p.next
            q = q.next


    def handle_key_press(self, widgets, event):
        keyname = gtk.gdk.keyval_name(event.keyval)
        # notice how we do not handle the arrow keys/home/end/scrlock and num pad
        if ( keyname == 'Shift_L' or keyname == 'Caps_Lock'
          or keyname == 'Alt_L' or keyname == 'Super_L'
          or event.hardware_keycode == 49 # ~
          or event.hardware_keycode in range(52, 56) # z-b
          or event.hardware_keycode in range(38, 42) # a-g
          or event.hardware_keycode in range(23, 28) # tab-q-t
          or event.hardware_keycode in range(10, 15) # 1-6
          or event.hardware_keycode in range(67, 70) # F1-F4
          or keyname == 'Escape'):
            if self.game.cur_clock == self.game.first_clock:
                self.end_turn()
        elif ( keyname == 'Shift_R' or keyname == 'Return'
          or keyname == 'Alt_R' or keyname == 'Super_R'
          or event.hardware_keycode in range(57, 61) # n-/
          or event.hardware_keycode in range(43, 48) # h-'
          or event.hardware_keycode in range(29, 35) # y-]
          or event.hardware_keycode in range(16, 22) # 7-backspace
          or event.hardware_keycode in range(75, 76) # F9-F10
          or event.hardware_keycode in range(95, 96) # F10-F11 (wtf?)
          or keyname == 'Menu' or event.hardware_keycode == 51): # \
            if self.game.cur_clock != self.game.first_clock:
                self.end_turn()
        elif event.hardware_keycode in range(71, 74): # F5-F8 is pause
            self.handle_pause()
        elif keyname == 'space':
            self.end_turn()
        self.maybe_print("key %s (%d/%d) was pressed" % (keyname, event.keyval, event.hardware_keycode))

    def handle_fullscreen(self, action = None):
        if self.fullscreen:
            self.window.unfullscreen()
            self.menubar.show()
        else:
            self.window.fullscreen()
            self.menubar.hide()
        self.fullscreen = not self.fullscreen

    def end_turn(self, widget = None, event = None):
        """handle end turn events
        
        this passes the message to the gaming engine as quickly as
        possible then goes around updating the UI
        """
        # it may be that we need to start the display
        if not self.game.running():
            self.game.end_turn()
            self.start_game()
        elif not self.first_clock.next:
            self.game.pause()
        else:
            self.game.end_turn()

        # update the current clock pointer
        self.cur_clock = self.cur_clock.next
        if not self.cur_clock:
            self.cur_clock = self.first_clock

        # some reason it doesn't work to just update the old clock label, we need to update both
        self.refresh()
        self.hilight()
        self.maybe_print("ended turn")
        # mouse pointer debugging code, see #4511
        if event and event.device:
            self.maybe_print("widget: %s" % widget)
            self.maybe_print("device name: %s" % event.device.name)
            self.maybe_print("device idt: %s" % event.deviceid)

    def handle_pause(self, action = None):
        """pause handler
        
        just a stub for the game engine for now
        """
        self.maybe_print("pausing game")
        self.game.pause()

    def start_game(self):
        self.game.start()
        self.timeout_source = gobject.timeout_add(self.loop_timeout, self.refresh_current)
        self.turns.set_label("Turn %d" % self.game.get_turns())
        self.turns.show()
        self.maybe_print("starting game, refresh rate %dms" % self.loop_timeout)
