# -*- coding: utf-8 -*-
#
# The Gameclock clock engines
#
# This is where the clock engines reside.
# 
# (C) Anarcat 2011

import time
import math

class Clock:
    """The main clock engine

    Each clock is an instance of this class. This could be considered
    like a merge between a controler and a model in an MVC
    model. However, this software isn't based on the MVC model in any
    remote way.
    
    This should be pretty precise and doesn't lag from empirical
    observations (watching the clocks compared to the Gnome clock
    applet)
    """
    last_start = 0 # 0 if stopped
    text = "" # rendered version of the clock
    dead = False
    game = None
    # a clock is part of a chained list
    next = None
    # usually, clocks go backwards, but some games might require it to go other ways
    factor = -1

    def __init__(self, game, delay = None):
        """Setup the clock engine
        
        The clock is stopped by default and the display is refreshed
        """
        self.time = game.default_time
        self.dead = self.time <= 0 # calculated only after update() and should be used only this way, it is a cache
        self.game = game
        self.update()

    def start(self):
        """Start the timer
        
        This marks the new timestamp and sets the background color
        """
        self.last_start = time.time()
        # no need to update, time shouldn't have changed since last stop

    def stop(self):
        """Stop the timer
        
        This computes the new cumulatative time based on the start
        timestamp. It resets the timestamp to zero to mark the timer as
        stopped.
        
        This also resets the event box's color and triggers an update of
        the label.
        
        XXX: Note that this function takes *some* time to process. This
        time is lost and not given to any participant. Maybe that should
        be fixed for the clock to be really precised, by compensation
        for the duration of the function.
        
        Another solution would be to create a thread for the Game engine
        """
        if self.last_start:
            self.time = self.get_time()
            self.last_start = 0
        self.update()

    def pause(self):
        """pause/unpause the timer
        
        this will start the timeer if stopped and stop it if started
        """
        if self.last_start:
            self.time = self.get_time()
            self.last_start = 0          
            self.update()
        else:
            self.start()

    def running(self):
        return self.last_start

    def get_time(self):
        """return the current time of the clock in ms"""
        if self.last_start:
            diff = time.time() - self.last_start
        else:
            diff = 0
        return self.time + (self.factor * diff*1000)

    def update(self):
        """Refresh the display of the clock's widget"""
        # convert to seconds because that's what gmtime wants
        miliseconds = self.get_time()
        secs = miliseconds/1000
        self.text = time.strftime("%M:%S", time.gmtime(abs(secs)))
        days = abs(secs) / ( 24 * 60 * 60 )
        if days >= 1:
            self.text = "%dd " % days + self.text
        self.dead = secs <= 0
        if self.dead:
            self.text = "-" + self.text
        if self.game.miliseconds:
            mantissa = ".%03d" % (abs(miliseconds) - (math.floor(abs(secs))*1000))
            self.text += mantissa[:2]
        return self.text

    def __str__(self):
        """make a better string representation of the objects
        
        we basically dump all variables and some functions
        """
        return "  clock engine %s time: %d last: %d diff: %f dead: %d text: %s\n  next %s" % ( object.__str__(self), self.time, self.last_start, time.time() - self.last_start, self.dead, self.text, self.next)


class ChessClock(Clock):
    """A typical Chess clock
    
This clock will stop at the end of your turn, and should represent fairly faithfully tabletop chess clocks.

A typical setting is 5 minutes each, which is considered to be a "blitz". 2 minutes is often called "lightning chess".
    """
    # this is just an alias to the base Game class which implements everything
    pass


class FisherClock(Clock):
    """A chess clock, as modified by Garry Fisher

This is a regular chess clock with one little twist: every time a player finishes its turn, he gets extra time. This allows for timed game that are still fairly interactive, as the player is forced to move within a certain timeframe.

A typical setting is 2 minutes + 10 seconds delay, which leads to games of around 10 to 20 minutes."""
    delay = 10

    def __init__(self, game, delay):
        Clock.__init__(self, game, delay)
        self.delay = delay
    
    def stop(self):
        """end the turn, fisher style

        this increments the current clock before switching turns as normal
        """
        self.time += self.delay
        Clock.stop(self)

class BoardClock(Clock):
    """A simple clock for general board games.

A player gets a specific amount of time to play his turn, but the leftover time isn't carried over to the next turn."""

    # we need to remember the original time
    default_time = None

    def __init__(self, game, delay = None):
        Clock.__init__(self, game, delay)
        self.default_time = self.time

    def stop(self):
        """override the end_turn function to reset the timers at the end of turns"""
        Clock.stop(self)
        self.time = self.default_time
        self.update()

class HourglassClock(Clock):
    """Hourglass emulation.

This clock is similar to having an hourglass on a table that is flipped at the end of each turn.

We do allow each player to start with a certain amonut of time (or "sand", if you will) on his side, in other words, this is as if the hourglass was half-empty/half-full when the game starts.

Note that this doesn't make much sense with more than two players..."""

    def __init__(self, game, delay = None):
        """this overrides the base constructor to make only one clock have an initial time

        basically, this is to represent that when you start it, the hour glass is empty on one side
        """
        Clock.__init__(self, game, delay)

    def start(self):
        """reimplement the start() function altogether

        make sure all the clocks are started and they are in the right direction
        """
        Clock.stop(self)
        self.factor = -1
        Clock.start(self)

    def stop(self):
        """reimplement the end_turn function altogether

        we go to the next clock, stop it, reverse its direction and
        start it again
        """
        Clock.stop(self)
        self.factor = 1
        Clock.start(self)

