/**
 * File:          $RCSfile: endian_io.c,v $
 * Module:        Little and big-endian I/O for architecture independent I/O
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.4 $
 * Last edited:   $Date: 2002/05/16 08:43:10 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <gandalf/common/endian_io.h>
#include <gandalf/common/misc_error.h>
#include <gandalf/config.h>

/**
 * \addtogroup Common
 * \{
 */

/**
 * \defgroup CommonEndianIO I/O Dependent on Endian-ness of Architecture
 * \{
 */

#ifdef GAN_INT16

/**
 * \brief Reads an array of signed 16-bit integers from a file.
 * \param fp Pointer to a binary file opened for reading
 * \param ai16 Array of integers
 * \param nitems The number of integers to be read into the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Reads an array of signed 16-bit integers from a file in little-endian byte
 * order.
 */
Gan_Bool
 gan_fread_lendian_i16 ( FILE *fp, gan_i16 *ai16, size_t nitems )
{
   /* reverse bytes if on big-endian architecture */
#ifdef WORDS_BIGENDIAN
   {
      char ca[2];
      int i;

      for ( i = (int)nitems-1; i >= 0; i--, ai16++ )
      {
         /* read bytes */
         if ( fread ( ca, 2, 1, fp ) != 1 )
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_fread_lendian_i16",
                               GAN_ERROR_READING_FROM_FILE, "" );
            return GAN_FALSE;
         }

         /* reverse bytes */
         ((char *)ai16)[1] = ca[0];
         ((char *)ai16)[0] = ca[1];
      }
   }
#else
   /* little-endian architecture: read bytes directly */
   if ( fread ( ai16, 2, nitems, fp ) != nitems )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_fread_lendian_i16",
                         GAN_ERROR_READING_FROM_FILE, "" );
      return GAN_FALSE;
   }
#endif

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Writes an array of signed 16-bit integers to a file.
 * \param fp Pointer to a binary file opened for reading
 * \param ai16 Array of integers
 * \param nitems The number of integers to be written from the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Writes an array of signed 16-bit integers to a file in little-endian byte
 * order.
 */
Gan_Bool
 gan_fwrite_lendian_i16 ( FILE *fp, gan_i16 *ai16, size_t nitems )
{
   /* reverse bytes if on big-endian architecture */
#ifdef WORDS_BIGENDIAN
   {
      char ca[2];
      int i;

      for ( i = (int)nitems-1; i >= 0; i--, ai16++ )
      {
         /* build reversed bytes */
         ca[0] = ((char *)ai16)[1];
         ca[1] = ((char *)ai16)[0];

         /* write reversed bytes */
         if ( fwrite ( ca, 2, 1, fp ) != 1 )
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_fwrite_lendian_i16",
                               GAN_ERROR_WRITING_TO_FILE, "" );
            return GAN_FALSE;
         }
      }
   }
#else
   /* little-endian architecture: write bytes directly */
   if ( fwrite ( ai16, 2, nitems, fp ) != nitems )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_fwrite_lendian_i16",
                         GAN_ERROR_WRITING_TO_FILE, "" );
      return GAN_FALSE;
   }
#endif

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Reads an array of unsigned 16-bit integers from a file.
 * \param fp Pointer to a binary file opened for reading
 * \param aui16 Array of integers
 * \param nitems The number of integers to be read into the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Reads an array of unsigned 16-bit integers from a file in little-endian byte
 * order.
 */
Gan_Bool
 gan_fread_lendian_ui16 ( FILE *fp, gan_ui16 *aui16, size_t nitems )
{
   /* reverse bytes if on big-endian architecture */
#ifdef WORDS_BIGENDIAN
   {
      char ca[2];
      int i;

      for ( i = (int)nitems-1; i >= 0; i--, aui16++ )
      {
         /* read bytes */
         if ( fread ( ca, 2, 1, fp ) != 1 )
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_fread_lendian_ui16",
                               GAN_ERROR_READING_FROM_FILE, "" );
            return GAN_FALSE;
         }

         /* reverse bytes */
         ((char *)aui16)[1] = ca[0];
         ((char *)aui16)[0] = ca[1];
      }
   }
#else
   /* little-endian architecture: read bytes directly */
   if ( fread ( aui16, 2, nitems, fp ) != nitems )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_fread_lendian_ui16",
                         GAN_ERROR_READING_FROM_FILE, "" );
      return GAN_FALSE;
   }
#endif

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Writes an array of unsigned 16-bit integers to a file
 * \param fp Pointer to a binary file opened for reading
 * \param aui16 Array of integers
 * \param nitems The number of integers to be written from the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Writes an array of unsigned 16-bit integers to a file in little-endian byte
 * order.
 */
Gan_Bool
 gan_fwrite_lendian_ui16 ( FILE *fp, gan_ui16 *aui16, size_t nitems )
{
   /* reverse bytes if on big-endian architecture */
#ifdef WORDS_BIGENDIAN
   {
      char ca[2];
      int i;

      for ( i = (int)nitems-1; i >= 0; i--, aui16++ )
      {
         /* build reversed bytes */
         ca[0] = ((char *)aui16)[1];
         ca[1] = ((char *)aui16)[0];

         /* write reversed bytes */
         if ( fwrite ( ca, 2, 1, fp ) != 1 )
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_fwrite_lendian_ui16",
                               GAN_ERROR_WRITING_TO_FILE, "" );
            return GAN_FALSE;
         }
      }
   }
#else
   /* little-endian architecture: write bytes directly */
   if ( fwrite ( aui16, 2, nitems, fp ) != nitems )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_fwrite_lendian_ui16",
                         GAN_ERROR_WRITING_TO_FILE, "" );
      return GAN_FALSE;
   }
#endif

   /* success */
   return GAN_TRUE;
}

#endif /* #ifdef GAN_INT16 */

#ifdef GAN_INT32

/**
 * \brief Reads an array of signed 32-bit integers from a file
 * \param fp Pointer to a binary file opened for reading
 * \param ai32 Array of integers
 * \param nitems The number of integers to be read into the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Reads an array of signed 32-bit integers from a file in little-endian byte
 * order.
 */
Gan_Bool
 gan_fread_lendian_i32 ( FILE *fp, gan_i32 *ai32, size_t nitems )
{
   /* reverse bytes if on big-endian architecture */
#ifdef WORDS_BIGENDIAN
   {
      char ca[4];
      int i;

      for ( i = (int)nitems-1; i >= 0; i--, ai32++ )
      {
         /* read bytes */
         if ( fread ( ca, 4, 1, fp ) != 1 )
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_fread_lendian_i32",
                               GAN_ERROR_READING_FROM_FILE, "" );
            return GAN_FALSE;
         }

         /* reverse bytes */
         ((char *)ai32)[3] = ca[0];
         ((char *)ai32)[2] = ca[1];
         ((char *)ai32)[1] = ca[2];
         ((char *)ai32)[0] = ca[3];
      }
   }
#else
   /* little-endian architecture: read bytes directly */
   if ( fread ( ai32, 4, nitems, fp ) != nitems )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_fread_lendian_i32",
                         GAN_ERROR_READING_FROM_FILE, "" );
      return GAN_FALSE;
   }
#endif

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Writes an array of signed 32-bit integers to a file
 * \param fp Pointer to a binary file opened for reading
 * \param ai32 Array of integers
 * \param nitems The number of integers to be written from the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Writes an array of signed 32-bit integers to a file in little-endian byte
 * order.
 */
Gan_Bool
 gan_fwrite_lendian_i32 ( FILE *fp, gan_i32 *ai32, size_t nitems )
{
   /* reverse bytes if on big-endian architecture */
#ifdef WORDS_BIGENDIAN
   {
      char ca[4];
      int i;

      for ( i = (int)nitems-1; i >= 0; i--, ai32++ )
      {
         /* build reversed bytes */
         ca[0] = ((char *)ai32)[3];
         ca[1] = ((char *)ai32)[2];
         ca[2] = ((char *)ai32)[1];
         ca[3] = ((char *)ai32)[0];

         /* write reversed bytes */
         if ( fwrite ( ca, 4, 1, fp ) != 1 )
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_fwrite_lendian_i32",
                               GAN_ERROR_WRITING_TO_FILE, "" );
            return GAN_FALSE;
         }
      }
   }
#else
   /* little-endian architecture: write bytes directly */
   if ( fwrite ( ai32, 4, nitems, fp ) != nitems )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_fwrite_lendian_i32",
                         GAN_ERROR_WRITING_TO_FILE, "" );
      return GAN_FALSE;
   }
#endif

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Reads an array of unsigned 32-bit integers from a file
 * \param fp Pointer to a binary file opened for reading
 * \param aui32 Array of integers
 * \param nitems The number of integers to be read into the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Reads an array of unsigned 32-bit integers from a file in little-endian byte
 * order.
 */
Gan_Bool
 gan_fread_lendian_ui32 ( FILE *fp, gan_ui32 *aui32, size_t nitems )
{
   /* reverse bytes if on big-endian architecture */
#ifdef WORDS_BIGENDIAN
   {
      char ca[4];
      int i;

      for ( i = (int)nitems-1; i >= 0; i--, aui32++ )
      {
         /* read bytes */
         if ( fread ( ca, 4, 1, fp ) != 1 )
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_fread_lendian_ui32",
                               GAN_ERROR_READING_FROM_FILE, "" );
            return GAN_FALSE;
         }

         /* reverse bytes */
         ((char *)aui32)[3] = ca[0];
         ((char *)aui32)[2] = ca[1];
         ((char *)aui32)[1] = ca[2];
         ((char *)aui32)[0] = ca[3];
      }
   }
#else
   /* little-endian architecture: read bytes directly */
   if ( fread ( aui32, 4, nitems, fp ) != nitems )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_fread_lendian_ui32",
                         GAN_ERROR_READING_FROM_FILE, "" );
      return GAN_FALSE;
   }
#endif

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Writes an array of unsigned 32-bit integers to a file
 * \param fp Pointer to a binary file opened for reading
 * \param aui32 Array of integers
 * \param nitems The number of integers to be written from the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Writes an array of unsigned 32-bit integers to a file in little-endian byte
 * order.
 */
Gan_Bool
 gan_fwrite_lendian_ui32 ( FILE *fp, gan_ui32 *aui32, size_t nitems )
{
   /* reverse bytes if on big-endian architecture */
#ifdef WORDS_BIGENDIAN
   {
      char ca[4];
      int i;

      for ( i = (int)nitems-1; i >= 0; i--, aui32++ )
      {
         /* build reversed bytes */
         ca[0] = ((char *)aui32)[3];
         ca[1] = ((char *)aui32)[2];
         ca[2] = ((char *)aui32)[1];
         ca[3] = ((char *)aui32)[0];

         /* write reversed bytes */
         if ( fwrite ( ca, 4, 1, fp ) != 1 )
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_fwrite_lendian_ui32",
                               GAN_ERROR_WRITING_TO_FILE, "" );
            return GAN_FALSE;
         }
      }
   }
#else
   /* little-endian architecture: write bytes directly */
   if ( fwrite ( aui32, 4, nitems, fp ) != nitems )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_fwrite_lendian_ui32",
                         GAN_ERROR_WRITING_TO_FILE, "" );
      return GAN_FALSE;
   }
#endif

   /* success */
   return GAN_TRUE;
}

#endif /* #ifdef GAN_INT32 */

/**
 * \brief Reads an array of floats from a file
 * \param fp Pointer to a binary file opened for reading
 * \param af Array of floats
 * \param nitems The number of integers to be read into the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Reads an array of floats from a file in little-endian byte order.
 */
Gan_Bool
 gan_fread_lendian_f ( FILE *fp, float *af, size_t nitems )
{
   
   /* reverse bytes if on big-endian architecture */
#ifdef WORDS_BIGENDIAN
   {
      char ca[SIZEOF_FLOAT];
      int i;

      for ( i = (int)nitems-1; i >= 0; i--, af++ )
      {
         /* read bytes */
         if ( fread ( ca, sizeof(float), 1, fp ) != 1 )
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_fread_lendian_f",
                               GAN_ERROR_READING_FROM_FILE, "" );
            return GAN_FALSE;
         }

         /* reverse bytes */
#if (SIZEOF_FLOAT == 4)
         ((char *)af)[3] = ca[0];
         ((char *)af)[2] = ca[1];
         ((char *)af)[1] = ca[2];
         ((char *)af)[0] = ca[3];
#elif (SIZEOF_FLOAT == 8)
         ((char *)af)[7] = ca[0];
         ((char *)af)[6] = ca[1];
         ((char *)af)[5] = ca[2];
         ((char *)af)[4] = ca[3];
         ((char *)af)[3] = ca[4];
         ((char *)af)[2] = ca[5];
         ((char *)af)[1] = ca[6];
         ((char *)af)[0] = ca[7];
#else
#error "unsupported float size"      
#endif
      }
   }
#else
   /* little-endian architecture: read bytes directly */
   if ( fread ( af, sizeof(float), nitems, fp ) != nitems )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_fread_lendian_f",
                         GAN_ERROR_READING_FROM_FILE, "" );
      return GAN_FALSE;
   }
#endif

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Writes an array of floats to a file
 * \param fp Pointer to a binary file opened for writing
 * \param af Array of floats
 * \param nitems The number of integers to be written from the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Writes an array of floats to a file in little-endian byte order.
 */
Gan_Bool
 gan_fwrite_lendian_f ( FILE *fp, float *af, size_t nitems )
{
   
   /* reverse bytes if on big-endian architecture */
#ifdef WORDS_BIGENDIAN
   {
      char ca[SIZEOF_FLOAT];
      int i;

      for ( i = (int)nitems-1; i >= 0; i--, af++ )
      {
         /* build reversed bytes */
#if (SIZEOF_FLOAT == 4)
         ca[0] = ((char *)af)[3];
         ca[1] = ((char *)af)[2];
         ca[2] = ((char *)af)[1];
         ca[3] = ((char *)af)[0];
#elif (SIZEOF_FLOAT == 8)
         ca[0] = ((char *)af)[7];
         ca[1] = ((char *)af)[6];
         ca[2] = ((char *)af)[5];
         ca[3] = ((char *)af)[4];
         ca[4] = ((char *)af)[3];
         ca[5] = ((char *)af)[2];
         ca[6] = ((char *)af)[1];
         ca[7] = ((char *)af)[0];
#else
#error "unsupported float size"      
#endif

         /* write reversed bytes */
         if ( fwrite ( ca, sizeof(float), 1, fp ) != 1 )
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_fwrite_lendian_f",
                               GAN_ERROR_WRITING_TO_FILE, "" );
            return GAN_FALSE;
         }
      }
   }
#else
   /* little-endian architecture: write bytes directly */
   if ( fwrite ( af, sizeof(float), nitems, fp ) != nitems )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_fwrite_lendian_f",
                         GAN_ERROR_WRITING_TO_FILE, "" );
      return GAN_FALSE;
   }
#endif

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Reads an array of doubles from a file
 * \param fp Pointer to a binary file opened for reading
 * \param ad Array of doubles
 * \param nitems The number of values to be read into the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Reads an array of doubles from a file in little-endian byte order.
 */
Gan_Bool
 gan_fread_lendian_d ( FILE *fp, double *ad, size_t nitems )
{
   
   /* reverse bytes if on big-endian architecture */
#ifdef WORDS_BIGENDIAN
   {
      char ca[SIZEOF_DOUBLE];
      int i;

      for ( i = (int)nitems-1; i >= 0; i--, ad++ )
      {
         /* read bytes */
         if ( fread ( ca, sizeof(double), 1, fp ) != 1 )
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_fread_lendian_d",
                               GAN_ERROR_READING_FROM_FILE, "" );
            return GAN_FALSE;
         }

         /* reverse bytes */
#if (SIZEOF_DOUBLE == 4)
         ((char *)ad)[3] = ca[0];
         ((char *)ad)[2] = ca[1];
         ((char *)ad)[1] = ca[2];
         ((char *)ad)[0] = ca[3];
#elif (SIZEOF_DOUBLE == 8)
         ((char *)ad)[7] = ca[0];
         ((char *)ad)[6] = ca[1];
         ((char *)ad)[5] = ca[2];
         ((char *)ad)[4] = ca[3];
         ((char *)ad)[3] = ca[4];
         ((char *)ad)[2] = ca[5];
         ((char *)ad)[1] = ca[6];
         ((char *)ad)[0] = ca[7];
      }
#else
#error "unsupported double size"      
#endif
   }
#else
   /* little-endian architecture: read bytes directly */
   if ( fread ( ad, sizeof(double), nitems, fp ) != nitems )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_fread_lendian_d",
                         GAN_ERROR_READING_FROM_FILE, "" );
      return GAN_FALSE;
   }
#endif

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Writes an array of doubles to a file.
 * \param fp Pointer to a binary file opened for writing
 * \param ad Array of doubles
 * \param nitems The number of values to be written from the array
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Writes an array of doubles to a file in little-endian byte order.
 */
Gan_Bool
 gan_fwrite_lendian_d ( FILE *fp, double *ad, size_t nitems )
{
   
   /* reverse bytes if on big-endian architecture */
#ifdef WORDS_BIGENDIAN
   {
      char ca[SIZEOF_DOUBLE];
      int i;

      for ( i = (int)nitems-1; i >= 0; i--, ad++ )
      {
         /* build reversed bytes */
#if (SIZEOF_DOUBLE == 4)
         ca[0] = ((char *)ad)[3];
         ca[1] = ((char *)ad)[2];
         ca[2] = ((char *)ad)[1];
         ca[3] = ((char *)ad)[0];
#elif (SIZEOF_DOUBLE == 8)
         ca[0] = ((char *)ad)[7];
         ca[1] = ((char *)ad)[6];
         ca[2] = ((char *)ad)[5];
         ca[3] = ((char *)ad)[4];
         ca[4] = ((char *)ad)[3];
         ca[5] = ((char *)ad)[2];
         ca[6] = ((char *)ad)[1];
         ca[7] = ((char *)ad)[0];
#else
#error "unsupported double size"      
#endif

         /* write reversed bytes */
         if ( fwrite ( ca, sizeof(double), 1, fp ) != 1 )
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_fwrite_lendian_d",
                               GAN_ERROR_WRITING_TO_FILE, "" );
            return GAN_FALSE;
         }
      }
   }
#else
   /* little-endian architecture: write bytes directly */
   if ( fwrite ( ad, sizeof(double), nitems, fp ) != nitems )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_fwrite_lendian_d",
                         GAN_ERROR_WRITING_TO_FILE, "" );
      return GAN_FALSE;
   }
#endif

   /* success */
   return GAN_TRUE;
}

/**
 * \}
 */

/**
 * \}
 */
