/**
 * File:          $RCSfile: 3x3matrix_eigsym.c,v $
 * Module:        3x3 symmetric matrix eigenvalues/vectors (double precision)
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.20 $
 * Last edited:   $Date: 2003/01/31 18:57:10 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <gandalf/linalg/3x3matrix_eigsym.h>
#include <gandalf/linalg/mat_symmetric.h>
#include <gandalf/linalg/mat_diagonal.h>
#include <gandalf/common/numerics.h>
#include <gandalf/common/misc_error.h>

/**
 * \addtogroup LinearAlgebra
 * \{
 */

/**
 * \addtogroup FixedSizeMatVec
 * \{
 */

/**
 * \addtogroup FixedSizeMatrix
 * \{
 */

/**
 * \addtogroup FixedSizeMatrixEigen
 * \{
 */

#if 0

/* my attempt at speeding up 3x3 symmetric matrix calculation, needs a bit
 * of work...
 */

/* compute eigenvector given matrix and eigenvalue */
static Gan_Bool
 compute_eigenvector ( Gan_SquMatrix33 *A, double l,
                       double *ex, double *ey, double *ez )
{
   double a = A->xx-l, d = A->yy-l, f = A->zz-l, n1, n2, n3;
   double e1[3], e2[3], e3[3];

   /* generate solution three ways to improve precision */
   e1[0] = A->zx*d - A->yx*A->zy;
   e1[1] = a*A->zy - A->yx*A->zx;
   e1[2] = A->yx*A->yx - a*d;
   n1 = e1[0]*e1[0] + e1[1]*e1[1] + e1[2]*e1[2];
   e2[0] = d*f - A->zy*A->zy;
   e2[1] = A->zx*A->zy - A->yx*f;
   e2[2] = A->yx*A->zy - A->zx*d;
   n2 = e2[0]*e2[0] + e2[1]*e2[1] + e2[2]*e2[2];
   e3[0] = A->yx*f - A->zx*A->zy;
   e3[1] = A->zx*A->zx - a*f;
   e3[2] = a*A->zy - A->yx*A->zx;
   n3 = e3[0]*e3[0] + e3[1]*e3[1] + e3[2]*e3[2];

   /* choose solution with largest norm */
   if ( n1 > n2 && n1 > n3 )
   {
      n1 = sqrt(n1);
      *ex = e1[0]/n1;
      *ey = e1[1]/n1;
      *ez = e1[2]/n1;
   }
   else if ( n2 > n3 )
   {
      n2 = sqrt(n2);
      *ex = e2[0]/n2;
      *ey = e2[1]/n2;
      *ez = e2[2]/n2;
   }
   else
   {
      if ( n3 == 0.0 )
      {
         gan_err_flush_trace();
         gan_err_register ( "compute_eigenvector", GAN_ERROR_DIVISION_BY_ZERO,
                            "" );
         return GAN_FALSE;
      }

      n3 = sqrt(n3);
      *ex = e3[0]/n3;
      *ey = e3[1]/n3;
      *ez = e3[2]/n3;
   }

   /* success */
   return GAN_TRUE;
}

/* compute pair of orthogonal eigenvectors given matrix and repeated
   eigenvalue */
static Gan_Bool
 compute_orthogonal_pair ( double gx, double gy, double gz,
                           double *ex, double *ey, double *ez,
                           double *fx, double *fy, double *fz )
{
   double n;
   Gan_Vector3 v3e, v3f, v3g = {gx,gy,gz};

   /* now compute vector guaranteed not parallel to g */
   if ( fabs(v3g.x) < fabs(v3g.y) && fabs(v3g.x) < fabs(v3g.z) )
      (void)gan_vec3_fill_q ( &v3f, 1.0, 0.0, 0.0 );
   else if ( fabs(v3g.y) < fabs(v3g.z) )
      (void)gan_vec3_fill_q ( &v3f, 0.0, 1.0, 0.0 );
   else
      (void)gan_vec3_fill_q ( &v3f, 0.0, 0.0, 1.0 );

   /* compute eigenvectors orthogonal to g */
   (void)gan_vec3_cross_q ( &v3f, &v3g, &v3e );
   (void)gan_vec3_cross_q ( &v3g, &v3e, &v3f );

   /* normalise to unit vectors */
   n = sqrt(gan_vec3_sqrlen_q(&v3e));
   if ( n == 0.0 )
   {
      gan_err_flush_trace();
      gan_err_register ( "compute_eigenvector_pair",
                         GAN_ERROR_DIVISION_BY_ZERO, "" );
      return GAN_FALSE;
   }

   *ex = v3e.x/n; *ey = v3e.y/n; *ez = v3e.z/n;

   n = sqrt(gan_vec3_sqrlen_q(&v3f));
   if ( n == 0.0 )
   {
      gan_err_flush_trace();
      gan_err_register ( "compute_eigenvector_pair",
                         GAN_ERROR_DIVISION_BY_ZERO, "" );
      return GAN_FALSE;
   }

   *fx = v3f.x/n; *fy = v3f.y/n; *fz = v3f.z/n;

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Eigenvalues and maybe eigenvectors of symmetric 3x3 matrix.
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Computes the eigenvalues and, optionally, the eigenvectors of a symmetric
 * 3x3 input matrix \a A, i.e. computing diagonal matrix \a W as the
 * eigenvalues \a W, and orthogonal matrix of eigenvectors \a Z such that
 * \f[
 *   A\:Z = Z\:W, \;\; Z^{\top}\:A = W\:`Z^{\top}
 * \f]
 * If \a Z is passed as \c NULL then the eigenvectors are not computed.
 *
 * NOTE: the contents of \a A are NOT destroyed.
 *
 * The eigenvalues \a W are provided in ascending order.
 */
Gan_Bool
 gan_symmat33_eigen ( Gan_SquMatrix33 *A, Gan_Vector3 *W, Gan_Matrix33 *Z )
{
   double b, c, d;
   Gan_Complex x[3];

   assert ( A->type == GAN_SYMMETRIC_MATRIX33 );

   /* build and solve cubic equation for eigenvalues */
   b = -(A->xx + A->yy + A->zz);
   c = A->xx*A->yy + A->yy*A->zz + A->zz*A->xx
       - A->zy*A->zy - A->zx*A->zx - A->yx*A->yx;
   d = A->xx*A->zy*A->zy + A->yy*A->zx*A->zx + A->zz*A->yx*A->yx
       - A->xx*A->yy*A->zz - 2.0*A->zy*A->zx*A->yx;

   if ( gan_solve_cubic ( 1.0, b, c, d, x ) != 3 )
   {
      gan_err_register ( "sym_eigen_solve_3x3", GAN_ERROR_FAILURE, "" );
      return GAN_FALSE;
   }

   /* sort eigenvalues */
   if ( x[0].r <= x[1].r && x[0].r <= x[2].r )
   {
      /* first value is the smallest */
      W->x = x[0].r;
      if ( x[1].r < x[2].r ) { W->y = x[1].r; W->z = x[2].r; }
      else                   { W->y = x[2].r; W->z = x[1].r; }
   }
   else if ( x[1].r <= x[2].r )
   {
      /* second value is the smallest */
      W->x = x[1].r;
      if ( x[0].r < x[2].r ) { W->y = x[0].r; W->z = x[2].r; }
      else                   { W->y = x[2].r; W->z = x[0].r; }
   }
   else
   {
      /* third value is the smallest */
      W->x = x[2].r;
      if ( x[0].r < x[1].r ) { W->y = x[0].r; W->z = x[1].r; }
      else                   { W->y = x[1].r; W->z = x[0].r; }
   }

   /* we have the eigenvalues. Now solve for the eigenvectors */

   /* check for eigenvalue duplication */
   if ( W->x == W->y )
   {
      if ( W->x == W->z )
         /* all eigenvalues identical. Eigenvector matrix can be set to
            identity */
         (void)gan_mat33_ident_q ( Z );
      else if ( !compute_eigenvector ( A, W->z, &Z->xz, &Z->yz, &Z->zz ) ||
                !compute_orthogonal_pair ( Z->xz, Z->yz, Z->zz,
                                           &Z->xx, &Z->yx, &Z->zx,
                                           &Z->xy, &Z->yy, &Z->zy ) )
      {
         gan_err_register ( "sym_eigen_solve_3x3", GAN_ERROR_FAILURE, "" );
         return GAN_FALSE;
      }
   }
   else if ( W->y == W->z )
   {
      if ( !compute_eigenvector ( A, W->x, &Z->xx, &Z->yx, &Z->zx ) ||
           !compute_orthogonal_pair ( Z->xx, Z->yx, Z->zx,
                                      &Z->xy, &Z->yy, &Z->zy,
                                      &Z->xz, &Z->yz, &Z->zz ) )
      {
         gan_err_register ( "sym_eigen_solve_3x3", GAN_ERROR_FAILURE, "" );
         return GAN_FALSE;
      }
   }
   else if ( !compute_eigenvector ( A, W->x, &Z->xx, &Z->yx, &Z->zx ) ||
             !compute_eigenvector ( A, W->y, &Z->xy, &Z->yy, &Z->zy ) ||
             !compute_eigenvector ( A, W->z, &Z->xz, &Z->yz, &Z->zz ) )
   {
      gan_err_register ( "sym_eigen_solve_3x3", GAN_ERROR_FAILURE, "" );
      return GAN_FALSE;
   }

   /* success */
   return GAN_TRUE;
}

#else

#define gel(M,i,j)   gan_mat_get_el(M,i,j)
#define sel(M,i,j,v) gan_mat_set_el(M,i,j,v)
#define sgel(M,i,j)   gan_squmat_get_el(M,i,j)
#define ssel(M,i,j,v) gan_squmat_set_el(M,i,j,v)

#ifdef HAVE_LAPACK
#define WORKSPACE_SIZE 200
#else
#define WORKSPACE_SIZE 0
#endif

/**
 * \brief Eigenvalues and maybe eigenvectors of symmetric 3x3 matrix.
 * \return #GAN_TRUE on success, #GAN_FALSE on failure.
 *
 * Computes the eigenvalues and, optionally, the eigenvectors of a symmetric
 * 3x3 input matrix \a A, i.e. computing diagonal matrix \a W as the
 * eigenvalues \a W, and orthogonal matrix of eigenvectors \a Z such that
 * \f[
 *   A\:Z = Z\:W, \;\; Z^{\top}\:A = W\:`Z^{\top}
 * \f]
 * If \a Z is passed as \c NULL then the eigenvectors are not computed.
 *
 * NOTE: the contents of \a A are NOT destroyed.
 *
 * The eigenvalues \a W are provided in ascending order.
 */
Gan_Bool
 gan_symmat33_eigen ( Gan_SquMatrix33 *A, Gan_Vector3 *W, Gan_Matrix33 *Z )
{
   Gan_SquMatrix Am, Wm;
   Gan_Matrix    Zm;
   double        Adata[3*(3+1)/2], Wdata[3], Zdata[3*3],
   #ifdef HAVE_LAPACK
                 wkdata[WORKSPACE_SIZE];
   #else
                 wkdata[1]; /* Can't declare an array with 0 elements */
   #endif

   /* convert to general matrices */
   if ( gan_symmat_form_data ( &Am, 3, Adata, 3*(3+1)/2 ) == NULL ||
        gan_diagmat_form_data ( &Wm, 3, Wdata, 3 ) == NULL ||
        ( Z != NULL && gan_mat_form_data ( &Zm, 3, 3, Zdata, 3*3 ) == NULL) )
   {
      gan_err_register ( "gan_symmat33_eigen", GAN_ERROR_FAILURE, "" );
      return GAN_FALSE;
   }

   /* copy data from matrix A into general matrix Am */
   if ( gan_symmat_fill_va ( &Am, 3, A->xx,
                                     A->yx, A->yy,
                                     A->zx, A->zy, A->zz ) == NULL )
   {
      gan_err_register ( "gan_symmat33_eigen", GAN_ERROR_FAILURE, "" );
      return GAN_FALSE;
   }

   /* perform eigendecomposition */
   if ( !gan_symmat_eigen ( &Am, &Wm, (Z == NULL) ? NULL : &Zm, GAN_TRUE,
                            wkdata, WORKSPACE_SIZE ) )
   {
      gan_err_register ( "gan_symmat33_eigen", GAN_ERROR_FAILURE, "" );
      return GAN_FALSE;
   }

   /* copy eigenvalues and eigenvectors */
   gan_vec3_fill_q ( W, sgel(&Wm,0,0), sgel(&Wm,1,1), sgel(&Wm,2,2) );
   if ( Z != NULL )
      gan_mat33_fill_q ( Z,
                         gel(&Zm,0,0), gel(&Zm,0,1), gel(&Zm,0,2),
                         gel(&Zm,1,0), gel(&Zm,1,1), gel(&Zm,1,2),
                         gel(&Zm,2,0), gel(&Zm,2,1), gel(&Zm,2,2) );

   /* success */
   return GAN_TRUE;
}
 
#endif

/**
 * \}
 */

/**
 * \}
 */

/**
 * \}
 */

/**
 * \}
 */
