/**
 * File:          $RCSfile: cineon_io.c,v $
 * Module:        DPX format image file I/O functions
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.14 $
 * Last edited:   $Date: 2005/05/21 22:49:00 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>
#include <gandalf/image/io/cineon_io.h>
#include <gandalf/image/io/dpx_io.h>
#include <gandalf/image/image_gl_uint8.h>
#include <gandalf/image/image_rgb_uint8.h>
#include <gandalf/image/image_rgba_uint8.h>
#include <gandalf/image/image_rgb_uint16.h>
#include <gandalf/common/misc_error.h>
#include <gandalf/common/allocate.h>
#include <gandalf/common/compare.h>

/**
 * \addtogroup ImagePackage
 * \{
 */

/**
 * \defgroup ImageIO Image I/O
 * \{
 */

/* file information offsets */
#define OFFSET_MAGIC          0
#define OFFSET_IMAGEOFFSET    4
#define OFFSET_GENERICSIZE    8
#define OFFSET_INDUSTRYSIZE  12
#define OFFSET_VARIABLESIZE  16
#define OFFSET_FILESIZE      20
#define OFFSET_VERSION       24
#define OFFSET_FILENAME      32
#define OFFSET_CREATIONDATE 132
#define OFFSET_CREATIONTIME 144
#define OFFSET_RESERVED     156

/* image information offsets */
#define OFFSET_ORIENTATION                192
#define OFFSET_NUMBEROFCHANNELS           193
#define OFFSET_DESIGNATORBYTE0            196
#define OFFSET_DESIGNATORBYTE1            197
#define OFFSET_BITSPERPIXEL               198
#define OFFSET_PIXELSPERLINE              200
#define OFFSET_LINESPERIMAGE              204
#define OFFSET_MINIMUMDATAVALUE           208
#define OFFSET_MINIMUMQUANTITYREPRESENTED 212
#define OFFSET_MAXIMUMDATAVALUE           216
#define OFFSET_MAXIMUMQUANTITYREPRESENTED 220
#define OFFSET_CHROMATICITY               420
#define OFFSET_DATAINTERLEAVE             680
#define OFFSET_PACKING                    681
#define OFFSET_SIGNED                     682
#define OFFSET_EOLPADDING                 684
#define OFFSET_EOIMAGEPADDING             688
#define OFFSET_XINPUTDEVICEPITCH          972
#define OFFSET_YINPUTDEVICEPITCH          976
#define OFFSET_IMAGEGAMMA                 980

#define BIG_BUFFER_SIZE 2048

/**
 * \brief Reads a RGB colour image file in CINEON format from a stream.
 * \param infile The file stream to be read
 * \param image The image structure to read the image data into or \c NULL
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads the CINEON image from the given file stream \a infile into the given
 * \a image. If \a image is \c NULL a new image is dynamically allocated,
 * otherwise the already allocated \a image structure is reused.
 *
 * \sa gan_write_cineon_image_stream().
 */
Gan_Image *
 gan_read_cineon_image_stream ( FILE *infile, Gan_Image *image )
{
   char acHeader[BIG_BUFFER_SIZE], *acAlignedHeader;
   Gan_ImageFormat eFormat;
   Gan_Type eType;
   Gan_Bool bReversedEndianness=GAN_FALSE;

   /* CINEON file header info */
   gan_uint32 ui32Magic;          /* Magic number */
   gan_uint32 ui32ImageOffset;    /* Offset to start of image data in bytes */

   /* CINEON image information variables */
   gan_uint8  ui8Orientation;           /* image orientation */
   gan_uint8  ui8NumberOfChannels;      /* number of image channels */
   gan_uint32 ui32PixelsPerLine;        /* or x value */
   gan_uint32 ui32LinesPerImageEle;     /* or y value, per element */
   gan_uint8  ui8BitsPerPixel;          /* bit size for channel */
   gan_uint8  ui8DataInterleave;        /* data interleave */
   gan_uint8  ui8Packing;               /* packing type*/
   gan_uint32 ui32eolPadding;           /* end of line padding used in element */
   gan_uint32 ui32eoImagePadding;       /* end of image padding used in element */

   /* align the header array */
   acAlignedHeader = (char*)((gan_uint32)acHeader + 7 - (((gan_uint32)acHeader + 7) % 8));

   /* read the generic file header */
   if(fread(acAlignedHeader, 1, 1024, infile) != 1024)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_cineon_image_stream", GAN_ERROR_CORRUPTED_FILE, "corrupted CINEON file header (truncated file?)" );
      return NULL;
   }

   /* determine endianness from magic number */
   ui32Magic = *((gan_uint32*)(acAlignedHeader + OFFSET_MAGIC));
   if(ui32Magic == 0x802a5fd7)
      bReversedEndianness = GAN_FALSE;
   else if(ui32Magic == 0xd75f2a80)
      bReversedEndianness = GAN_TRUE;
   else
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_cineon_image_stream", GAN_ERROR_CORRUPTED_FILE, "corrupted CINEON file header (illegal magic number)" );
      return NULL;
   }

   /* get offset of image data */
   ui32ImageOffset = *((gan_uint32*)(acAlignedHeader + OFFSET_IMAGEOFFSET));
   if(bReversedEndianness)
      vReverseEndiannessUI32(&ui32ImageOffset);

   /* decode image information header stuff */
   ui8Orientation = *((gan_uint8*)(acAlignedHeader + OFFSET_ORIENTATION));

   /* we only support top-to-bottom orientation */
   if(ui8Orientation != 0)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_cineon_image_stream", GAN_ERROR_NOT_IMPLEMENTED, "unsupported CINEON image orientation" );
      return NULL;
   }

   ui8NumberOfChannels = *((gan_uint8*)(acAlignedHeader + OFFSET_NUMBEROFCHANNELS));
   ui8BitsPerPixel     = *((gan_uint8*)(acAlignedHeader + OFFSET_BITSPERPIXEL));
   ui32PixelsPerLine = *((gan_uint32*)(acAlignedHeader + OFFSET_PIXELSPERLINE));
   if(bReversedEndianness)
      vReverseEndiannessUI32(&ui32PixelsPerLine);

   ui32LinesPerImageEle = *((gan_uint32*)(acAlignedHeader + OFFSET_LINESPERIMAGE));
   if(bReversedEndianness)
      vReverseEndiannessUI32(&ui32LinesPerImageEle);

   /* determine image format */
   switch(ui8NumberOfChannels)
   {
      case 1:
        eFormat = GAN_GREY_LEVEL_IMAGE;
        break;

      case 3:
        eFormat = GAN_RGB_COLOUR_IMAGE;
        break;

      case 4:
        eFormat = GAN_RGB_COLOUR_ALPHA_IMAGE;
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_read_cineon_image_stream", GAN_ERROR_NOT_IMPLEMENTED, "unsupported CINEON descriptor" );
        return NULL;
   }

   /* Determine interleave and packing type */
   ui8DataInterleave = *((gan_uint8*)(acAlignedHeader + OFFSET_DATAINTERLEAVE));
   if(ui8DataInterleave != 0)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_cineon_image_stream", GAN_ERROR_NOT_IMPLEMENTED, "unsupported CINEON data interleave" );
      return NULL;
   }

   ui8Packing = *((gan_uint8*)(acAlignedHeader + OFFSET_PACKING));
   if(ui8Packing != 5)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_cineon_image_stream", GAN_ERROR_NOT_IMPLEMENTED, "unsupported CINEON packing type" );
      return NULL;
   }

   switch(ui8BitsPerPixel)
   {
      case 1:
        eType = GAN_BOOL;
        break;

      case 8:
        eType = GAN_UINT8;
        break;

      case 10:
      case 12:
      case 16:
        eType = GAN_UINT16;
        eType = GAN_UINT16;
        eType = GAN_UINT16;
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_read_cineon_image_stream", GAN_ERROR_NOT_IMPLEMENTED, "unsupported CINEON bit size" );
        return NULL;
        break;
   }
   
   /* Determine end-of-line padding */
   ui32eolPadding = *((gan_uint32*)(acAlignedHeader + OFFSET_EOLPADDING));
   if(bReversedEndianness)
      vReverseEndiannessUI32(&ui32eolPadding);

   /* Determine end-of-image padding */
   ui32eoImagePadding = *((gan_uint32*)(acAlignedHeader + OFFSET_EOIMAGEPADDING));
   if(bReversedEndianness)
      vReverseEndiannessUI32(&ui32eoImagePadding);

   /* reset file point to start of image data */
   if(fseek(infile, ui32ImageOffset, SEEK_SET) != 0)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_cineon_image_stream", GAN_ERROR_CORRUPTED_FILE, "corrupted CINEON image orientation header (truncated file?)" );
      return NULL;
   }

   /* now read the image data */
   switch(ui8BitsPerPixel)
   {
      case 10:
        image = pgiRead10BitDPXImageData(infile, bReversedEndianness, (gan_uint16)ui8Packing, ui32eolPadding, ui32eoImagePadding,
                                         eFormat, eType, ui32PixelsPerLine, ui32LinesPerImageEle, image);
        if(image == NULL)
        {
           gan_err_register ( "gan_read_cineon_image_stream", GAN_ERROR_FAILURE, "" );
           return NULL;
        }

        break;

      case 8:
        image = pgiRead8BitDPXImageData(infile, bReversedEndianness, (gan_uint16)ui8Packing, ui32eolPadding, ui32eoImagePadding,
                                        eFormat, eType, ui32PixelsPerLine, ui32LinesPerImageEle, image);
        if(image == NULL)
        {
           gan_err_register ( "gan_read_cineon_image_stream", GAN_ERROR_FAILURE, "" );
           return NULL;
        }

        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_read_cineon_image_stream", GAN_ERROR_NOT_IMPLEMENTED, "unsupported CINEON bit depth" );
        return NULL;
   }
        
   /* success */
   return image;
}

/**
 * \brief Reads a RGB colour image file in CINEON format.
 * \param filename The name of the image file
 * \param image The image structure to read the image data into or \c NULL
 * \return Pointer to image structure, or \c NULL on failure.
 *
 * Reads the CINEON image with the in the file \a filename into the given
 * \a image. If \a image is \c NULL a new image is dynamically allocated;
 * otherwise the already allocated \a image structure is reused.
 *
 * \sa gan_write_cineon_image().
 */
Gan_Image *
 gan_read_cineon_image ( const char *filename, Gan_Image *image )
{
   FILE *infile;
   Gan_Image *result;

   /* attempt to open file */
   infile = fopen ( filename, "rb" );
   if ( infile == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_read_cineon_image", GAN_ERROR_OPENING_FILE,
                         filename );
      return NULL;
   }

   result = gan_read_cineon_image_stream ( infile, image );
   fclose(infile);
   return result;
}

/**
 * \brief Initialises the write control structure for Cineon files
 *
 * This function should be called on the structure to set the parameters to default values.
 * Then set any non-default values yourself before calling gan_write_cineon_image() or
 * gan_write_cineon_image_stream().
 */
void gan_initialise_cineon_write_control_struct(Gan_CineonWriteControlStruct *controlstr)
{
   controlstr->bit_size = 0;
}

#define IMAGE_DATA_OFFSET 0x7e00

/* file size function assumes unpacked I/O */
static gan_uint32
 ui32CineonFileSize(Gan_ImageFormat eFormat, gan_uint8 ui8BitSize, unsigned int uiHeight, unsigned int uiWidth)
{
   switch(ui8BitSize)
   {
      case 1:
        switch(eFormat)
        {
           case GAN_GREY_LEVEL_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*((uiWidth+31)/32)*4);
           default: return UINT_MAX;
        }
        break;
        
      case 8:
        switch(eFormat)
        {
           case GAN_GREY_LEVEL_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*((uiWidth+3)/4)*4);
           case GAN_RGB_COLOUR_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*uiWidth*4);
           case GAN_RGB_COLOUR_ALPHA_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*uiWidth*4);
           default: return UINT_MAX;
        }
        break;

      case 10:
        switch(eFormat)
        {
           case GAN_GREY_LEVEL_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*((uiWidth+2)/3)*4);
           case GAN_RGB_COLOUR_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*uiWidth*4);
           default: return UINT_MAX;
        }
        break;
        
      case 12:
        switch(eFormat)
        {
           case GAN_GREY_LEVEL_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*((uiWidth+1)/2)*4);
           case GAN_RGB_COLOUR_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*(((uiWidth*3)+1)/2)*4);
           default: return UINT_MAX;
        }
        break;

      case 16:
        switch(eFormat)
        {
           case GAN_GREY_LEVEL_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*((uiWidth+1)/2)*4);
           case GAN_RGB_COLOUR_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*(((uiWidth*3)+1)/2)*4);
           default: return UINT_MAX;
        }
        break;

      case 32:
        switch(eFormat)
        {
           case GAN_GREY_LEVEL_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*uiWidth*4);
           case GAN_RGB_COLOUR_IMAGE: return (IMAGE_DATA_OFFSET + uiHeight*3*uiWidth*4);
           default: return UINT_MAX;
        }
        break;

      default: return UINT_MAX;
   }

   return UINT_MAX;
}

#define REVERSE_BYTES_IN_WRITE

/**
 * \brief Writes a RGB colour image to a file stream in CINEON format.
 * \param outfile The file stream
 * \param image The image structure to write to the file
 * \param controlstr Pointer to structure controlling CINEON output or \c NULL
 * \return #GAN_TRUE on successful write operation, or #GAN_FALSE on failure.
 *
 * Writes the \a image into the file stream \a outfile in CINEON format.
 *
 * \sa gan_read_cineon_image_stream().
 */
Gan_Bool
 gan_write_cineon_image_stream ( FILE *outfile, Gan_Image *image, Gan_CineonWriteControlStruct *controlstr )
{
   char acHeader[BIG_BUFFER_SIZE], *acAlignedHeader;
   gan_uint8 ui8BitSize;
   gan_uint32 ui32eolPadding = 0;
   gan_uint8 ui8NumberOfChannels = 0, ui8Channel;
   unsigned int uiVal;

   /* determine bit size to use */
   if(controlstr != NULL && controlstr->bit_size != 0)
      ui8BitSize = (gan_uint8)controlstr->bit_size;
   else
      switch(image->type)
      {
         case GAN_BOOL:    ui8BitSize =  1; break;
         case GAN_UINT8:   ui8BitSize =  8; break;
         case GAN_UINT16:  ui8BitSize = 16; break;
         case GAN_FLOAT32: ui8BitSize = 32; break;

         default:
           gan_err_flush_trace();
           gan_err_register ( "gan_write_cineon_image", GAN_ERROR_NOT_IMPLEMENTED, "unsupported image type" );
           return GAN_FALSE;
           break;
      }

   /* align the header array */
   acAlignedHeader = (char*)((gan_uint32)acHeader + 7 - (((gan_uint32)acHeader + 7) % 8));

   /* build header */
   memset((void*)acAlignedHeader, 0, 1024);
   
   *((gan_uint32*)(acAlignedHeader + OFFSET_MAGIC)) = 0x802a5fd7; /* Magic number */
   *((gan_uint32*)(acAlignedHeader + OFFSET_IMAGEOFFSET)) = IMAGE_DATA_OFFSET;
   *((gan_uint32*)(acAlignedHeader + OFFSET_GENERICSIZE)) = 0x400;
   *((gan_uint32*)(acAlignedHeader + OFFSET_INDUSTRYSIZE)) = 0x400;
   *((gan_uint32*)(acAlignedHeader + OFFSET_VARIABLESIZE)) = 0x7600;
   *((gan_uint32*)(acAlignedHeader + OFFSET_FILESIZE)) = ui32CineonFileSize(image->format, ui8BitSize, image->height, image->width);
#ifdef REVERSE_BYTES_IN_WRITE
   vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_MAGIC));
   vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_IMAGEOFFSET));
   vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_GENERICSIZE));
   vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_INDUSTRYSIZE));
   vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_VARIABLESIZE));
   vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_FILESIZE));
#endif /* #ifdef REVERSE_BYTES_IN_WRITE */

   strcpy((char *)(acAlignedHeader + OFFSET_VERSION), "V4.5");

   switch(image->format)
   {
      case GAN_GREY_LEVEL_IMAGE:
        ui8NumberOfChannels = 1;
        *((gan_uint8*)(acAlignedHeader + OFFSET_DESIGNATORBYTE1)) = 0; /* B/W */
        break;

      case GAN_RGB_COLOUR_IMAGE:
        ui8NumberOfChannels = 3;
        *((gan_uint8*)(acAlignedHeader + OFFSET_DESIGNATORBYTE1)) = 1; /* red */
        *((gan_uint8*)(acAlignedHeader + OFFSET_DESIGNATORBYTE1 + 28)) = 2; /* green */
        *((gan_uint8*)(acAlignedHeader + OFFSET_DESIGNATORBYTE1 + 56)) = 3; /* blue */
        break;

      case GAN_RGB_COLOUR_ALPHA_IMAGE:
        ui8NumberOfChannels = 4;
        *((gan_uint8*)(acAlignedHeader + OFFSET_DESIGNATORBYTE1)) = 1; /* red */
        *((gan_uint8*)(acAlignedHeader + OFFSET_DESIGNATORBYTE1 + 28)) = 2; /* green */
        *((gan_uint8*)(acAlignedHeader + OFFSET_DESIGNATORBYTE1 + 56)) = 3; /* blue */
        *((gan_uint8*)(acAlignedHeader + OFFSET_DESIGNATORBYTE1 + 84)) = 0; /* B/W */
        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_write_cineon_image", GAN_ERROR_NOT_IMPLEMENTED, "unsupported image format" );
        return GAN_FALSE;
   }

   *((gan_uint8*)(acAlignedHeader + OFFSET_ORIENTATION)) = 0;
   *((gan_uint8*)(acAlignedHeader + OFFSET_NUMBEROFCHANNELS)) = ui8NumberOfChannels;
   for(ui8Channel=0; ui8Channel<ui8NumberOfChannels; ui8Channel++)
   {
      *((gan_uint8*)(acAlignedHeader + OFFSET_DESIGNATORBYTE0             + 28*ui8Channel)) = 0;
      *((gan_uint8*)(acAlignedHeader + OFFSET_BITSPERPIXEL                + 28*ui8Channel)) = ui8BitSize; /* bit size, e.g. 10-bit */
      *((gan_uint32*)(acAlignedHeader + OFFSET_PIXELSPERLINE              + 28*ui8Channel)) = image->width;
      *((gan_uint32*)(acAlignedHeader + OFFSET_LINESPERIMAGE              + 28*ui8Channel)) = image->height;
      *((gan_uint32*)(acAlignedHeader + OFFSET_MINIMUMDATAVALUE           + 28*ui8Channel)) = 0;
      *((gan_uint32*)(acAlignedHeader + OFFSET_MINIMUMQUANTITYREPRESENTED + 28*ui8Channel)) = 0;
      *((gan_uint32*)(acAlignedHeader + OFFSET_MAXIMUMDATAVALUE           + 28*ui8Channel)) = 0x447fc000;
      *((gan_uint32*)(acAlignedHeader + OFFSET_MAXIMUMQUANTITYREPRESENTED + 28*ui8Channel)) = 0x4002f1aa;

#ifdef REVERSE_BYTES_IN_WRITE
      vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_PIXELSPERLINE              + 28*ui8Channel));
      vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_LINESPERIMAGE              + 28*ui8Channel));
      vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_MINIMUMDATAVALUE           + 28*ui8Channel));
      vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_MINIMUMQUANTITYREPRESENTED + 28*ui8Channel));
      vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_MAXIMUMDATAVALUE           + 28*ui8Channel));
      vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_MAXIMUMQUANTITYREPRESENTED + 28*ui8Channel));
#endif /* #ifdef REVERSE_BYTES_IN_WRITE */
   }

   for(uiVal=0; uiVal<8; uiVal++)
   {
      *((gan_uint32*)(acAlignedHeader + OFFSET_CHROMATICITY + uiVal*4)) = 0x7f800000;
#ifdef REVERSE_BYTES_IN_WRITE
      vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_CHROMATICITY + uiVal*4));
#endif /* #ifdef REVERSE_BYTES_IN_WRITE */
   }

   *((gan_uint8*)(acAlignedHeader + OFFSET_DATAINTERLEAVE)) = 0;
   *((gan_uint8*)(acAlignedHeader + OFFSET_PACKING)) = 5;
   *((gan_uint32*)(acAlignedHeader + OFFSET_EOLPADDING)) = ui32eolPadding; /* end-of-line padding */
   *((gan_uint32*)(acAlignedHeader + OFFSET_EOIMAGEPADDING)) = 0; /* no end-of-image padding */

   *((gan_uint32*)(acAlignedHeader + OFFSET_XINPUTDEVICEPITCH)) = 0x7f800000;
   *((gan_uint32*)(acAlignedHeader + OFFSET_YINPUTDEVICEPITCH)) = 0x7f800000;
   *((gan_uint32*)(acAlignedHeader + OFFSET_IMAGEGAMMA)) = 0x7f800000;

#ifdef REVERSE_BYTES_IN_WRITE
   vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_EOLPADDING));
   vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_EOIMAGEPADDING));
   vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_XINPUTDEVICEPITCH));
   vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_YINPUTDEVICEPITCH));
   vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + OFFSET_IMAGEGAMMA));
#endif /* #ifdef REVERSE_BYTES_IN_WRITE */
   
   /* write image information header */
   if(fwrite((const void *)acAlignedHeader, 1, 1024, outfile) != 1024)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_cineon_image", GAN_ERROR_WRITING_TO_FILE, "");
      return GAN_FALSE;
   }

   /* motion picture header info */
   memset((void*)acAlignedHeader, 0, 1024);
   for(uiVal=0; uiVal<3; uiVal++)
   {
      *((gan_uint32*)(acAlignedHeader + uiVal*4)) = 0xffffffff;
#ifdef REVERSE_BYTES_IN_WRITE
      vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + uiVal*4));
#endif /* #ifdef REVERSE_BYTES_IN_WRITE */
   }
   
   *((gan_uint32*)(acAlignedHeader + 44)) = 0xffffffff;
   *((gan_uint32*)(acAlignedHeader + 48)) = 0x7f800000;
#ifdef REVERSE_BYTES_IN_WRITE
   vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + 44));
   vReverseEndiannessUI32((gan_uint32*)(acAlignedHeader + 48));
#endif /* #ifdef REVERSE_BYTES_IN_WRITE */
   
   /* write motion picture info header */
   if(fwrite((const void *)acAlignedHeader, 1, 1024, outfile) != 1024)
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_cineon_image", GAN_ERROR_WRITING_TO_FILE, "");
      return GAN_FALSE;
   }

   /* write zeros for rest of header */
   {
      unsigned int uiNumBytesLeftToWrite = IMAGE_DATA_OFFSET - 1024 - 1024;

      memset((void*)acHeader, 0, BIG_BUFFER_SIZE);
      while(uiNumBytesLeftToWrite > 0)
      {
         unsigned int uiNumBytes = gan_min2(uiNumBytesLeftToWrite, BIG_BUFFER_SIZE);

         if(fwrite((const void*)acHeader, 1, uiNumBytes, outfile) != uiNumBytes)
         {
            gan_err_flush_trace();
            gan_err_register ( "gan_write_cineon_image", GAN_ERROR_WRITING_TO_FILE, "");
            return GAN_FALSE;
         }

         uiNumBytesLeftToWrite -= uiNumBytes;
      }
   }

   /* now write the image data */
   switch(ui8BitSize)
   {
      case 10:
        if(!bWrite10BitDPXImageData(outfile, image,
#ifdef REVERSE_BYTES_IN_WRITE
                                    GAN_TRUE
#else
                                    GAN_FALSE
#endif /* #ifdef REVERSE_BYTES_IN_WRITE */
                                    ))
        {
           gan_err_register ( "gan_write_cineon_image", GAN_ERROR_FAILURE, "" );
           return GAN_FALSE;
        }

        break;

      case 8:
        if(!bWrite8BitDPXImageData(outfile, image,
#ifdef REVERSE_BYTES_IN_WRITE
                                    GAN_TRUE
#else
                                    GAN_FALSE
#endif /* #ifdef REVERSE_BYTES_IN_WRITE */
                                    ))
        {
           gan_err_register ( "gan_write_cineon_image", GAN_ERROR_FAILURE, "" );
           return GAN_FALSE;
        }

        break;

      default:
        gan_err_flush_trace();
        gan_err_register ( "gan_write_cineon_image", GAN_ERROR_NOT_IMPLEMENTED, "unsupported CINEON bit depth" );
        return GAN_FALSE;
   }

   /* success */
   return GAN_TRUE;
}

/**
 * \brief Writes a RGB colour image file in CINEON format.
 * \param filename The name of the image file
 * \param image The image structure to write to the file
 * \param controlstr Pointer to structure controlling CINEON output or \c NULL
 * \return #GAN_TRUE on successful write operation, #GAN_FALSE on failure.
 *
 * Writes the \a image into CINEON file \a filename.
 *
 * \sa gan_read_cineon_image().
 */
Gan_Bool
 gan_write_cineon_image ( const char *filename, Gan_Image *image, Gan_CineonWriteControlStruct *controlstr )
{
   FILE *outfile;
   Gan_Bool result;

   /* attempt to open file */
   outfile = fopen ( filename, "wb" );
   if ( outfile == NULL )
   {
      gan_err_flush_trace();
      gan_err_register ( "gan_write_cineon_image", GAN_ERROR_OPENING_FILE, filename );
      return GAN_FALSE;
   }

   result = gan_write_cineon_image_stream ( outfile, image, controlstr );
   fclose(outfile);
   return result;
}

/**
 * \}
 */

/**
 * \}
 */
