#if HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

/* $Id: gexec_process.c,v 1.2 2002/06/20 03:23:24 massie Exp $ */

#ifdef LINUX

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/time.h>
#include <ganglia/error.h>
#include <ganglia/gmond_config.h>
#include <ganglia/gexec_process.h>
#include <ganglia/file.h>
#include <sys/param.h>     /* for HZ */
#include <asm/page.h>      /* for PAGE_SHIFT */
#include <pthread.h>

/* useful macros */
#define bytetok(x)   (((x) + 512) >> 10)
#define pagetok(x)   ((x) << (PAGE_SHIFT - 10))

typedef struct
   {
   char *gpid;
   int pid;
   gmond_config_t *config;
   }
gexec_process_func_arg_t;

typedef struct proc_s 
{
   char name[64];  /* Name of the person who owns this process */

/* I might want to autoconf this later */
#define SIGNAL_STRING 1
#ifdef SIGNAL_STRING
    char
	/* Linux 2.1.7x and up have more signals. This handles 88. */
	signal[24],	/* mask of pending signals */
	blocked[24],	/* mask of blocked signals */
	sigignore[24],	/* mask of ignored signals */
	sigcatch[24];	/* mask of caught  signals */
#else
    long long
	/* Linux 2.1.7x and up have more signals. This handles 64. */
	signal,		/* mask of pending signals */
	blocked,	/* mask of blocked signals */
	sigignore,	/* mask of ignored signals */
	sigcatch;	/* mask of caught  signals */
#endif
    long
	cutime,		/* cumulative utime of process and reaped children */
	cstime,		/* cumulative stime of process and reaped children */
	priority,	/* kernel scheduling priority */
	timeout,	/* ? */
	nice,		/* standard unix nice level of process */
	rss,		/* resident set size from /proc/#/stat (pages) */
	it_real_value,	/* ? */
    /* the next 7 members come from /proc/#/statm */
	size,		/* total # of pages of memory */
	resident,	/* number of resident set (non-swapped) pages (4k) */
	share,		/* number of pages of shared (mmap'd) memory */
	trs,		/* text resident set size */
	lrs,		/* shared-lib resident set size */
	drs,		/* data resident set size */
	dt;		/* dirty pages */
    unsigned long
	/* FIXME: are these longs? Maybe when the alpha does PCI bounce buffers */
	vm_size,        /* same as vsize in kb */
	vm_lock,        /* locked pages in kb */
	vm_rss,         /* same as rss in kb */
	vm_data,        /* data size */
	vm_stack,       /* stack size */
	vm_exe,         /* executable size */
	vm_lib,         /* library size (all pages, not just used ones) */
	vsize,		/* number of pages of virtual memory ... */
	rss_rlim,	/* resident set size limit? */
	flags,		/* kernel flags for the process */
	min_flt,	/* number of minor page faults since process start */
	maj_flt,	/* number of major page faults since process start */
	cmin_flt,	/* cumulative min_flt of process and child processes */
	cmaj_flt,	/* cumulative maj_flt of process and child processes */
	nswap,		/* ? */
	cnswap,		/* cumulative nswap ? */
	utime,		/* user-mode CPU time accumulated by process */
	stime,		/* kernel-mode CPU time accumulated by process */
	start_code,	/* address of beginning of code segment */
	end_code,	/* address of end of code segment */
	start_stack,	/* address of the bottom of stack for the process */
	kstk_esp,	/* kernel stack pointer */
	kstk_eip,	/* kernel instruction pointer */
	start_time,	/* start time of process -- seconds since 1-1-70 */
   runtime,    /* total time in second the process has been running - matt */
	wchan;		/* address of kernel wait channel proc is sleeping in */
    struct proc_s *l,	/* ptrs for building arbitrary linked structs */
                  *r;	/* (i.e. singly/doubly-linked lists and trees */
    char
	**environ,	/* environment string vector (/proc/#/environ) */
	**cmdline;	/* command line string vector (/proc/#/cmdline) */
    char
	/* Be compatible: Digital allows 16 and NT allows 14 ??? */
    	ruser[16],	/* real user name */
    	euser[16],	/* effective user name */
    	suser[16],	/* saved user name */
    	fuser[16],	/* filesystem user name */
    	rgroup[16],	/* real group name */
    	egroup[16],	/* effective group name */
    	sgroup[16],	/* saved group name */
    	fgroup[16],	/* filesystem group name */
    	cmd[16];	/* basename of executable file in call to exec(2) */
    int
        ruid, rgid,     /* real      */
        euid, egid,     /* effective */
        suid, sgid,     /* saved     */
        fuid, fgid,     /* fs (used for file access only) */
    	pid,		/* process id */
    	ppid,		/* pid of parent process */
	pgrp,		/* process group id */
	session,	/* session id */
	tty,		/* full device number of controlling terminal */
	tpgid,		/* terminal process group id */
	exit_signal,	/* might not be SIGCHLD */
	processor;      /* current (or most recent?) CPU */
    double
        pcpu,           /* %CPU usage (is not filled in by readproc!!!) */
        pmem;           /* %MEM usage */
    char
    	state;		/* single-char code for process state (S=sleeping) */
} proc_t;

/*
 * The function stat2proc(), and get_elapsed_time() 
 * functions below were taken from procps-2.0.7
 * Copyright (C) 1996 Charles L. Blake.
 * Copyright (C) 1998 Michael K. Johnson
 * May be distributed under the conditions of the
 * GNU Library General Public License
 */
static float 
get_elapsed_time(void)
{
    struct timeval t;
    static struct timeval oldtime;
    struct timezone timez;
    float elapsed_time;

    gettimeofday(&t, &timez);
    elapsed_time = (t.tv_sec - oldtime.tv_sec)
   + (float) (t.tv_usec - oldtime.tv_usec) / 1000000.0;
    oldtime.tv_sec  = t.tv_sec;
    oldtime.tv_usec = t.tv_usec;
    return (elapsed_time);
}

static void 
stat2proc(char* S, proc_t* P) 
{
    int num;
    char* tmp;
  
    tmp  = strrchr(S, ')'); /* split into "PID (cmd" and "<rest>" */
    *tmp = '\0';			/* replace trailing ')' with NUL */
    /* fill in default values for older kernels */
    P->exit_signal = SIGCHLD;
    P->processor = 0;
    /* parse these two strings separately, skipping the leading "(". */
    memset(P->cmd, 0, sizeof P->cmd);	/* clear even though *P xcalloc'd ?! */
    sscanf(S, "%d (%15c", &P->pid, P->cmd);   /* comm[16] in kernel */
    num = sscanf(tmp + 2,			/* skip space after ')' too */
       "%c "
       "%d %d %d %d %d "
       "%lu %lu %lu %lu %lu %lu %lu "
       "%ld %ld %ld %ld %ld %ld "
       "%lu %lu "
       "%ld "
       "%lu %lu %lu %lu %lu %lu "
       "%*s %*s %*s %*s " /* discard, no RT signals & Linux 2.1 used hex */
       "%lu %lu %lu "
       "%d %d",
       &P->state,
       &P->ppid, &P->pgrp, &P->session, &P->tty, &P->tpgid,
       &P->flags, &P->min_flt, &P->cmin_flt, &P->maj_flt, &P->cmaj_flt, &P->utime, &P->stime,
       &P->cutime, &P->cstime, &P->priority, &P->nice, &P->timeout, &P->it_real_value,
       &P->start_time, &P->vsize,
       &P->rss,
       &P->rss_rlim, &P->start_code, &P->end_code, &P->start_stack, &P->kstk_esp, &P->kstk_eip,
/*     P->signal, P->blocked, P->sigignore, P->sigcatch,   */ /* can't use */
       &P->wchan, &P->nswap, &P->cnswap,
/* -- Linux 2.0.35 ends here -- */
       &P->exit_signal, &P->processor  /* 2.2.1 ends with "exit_signal" */
/* -- Linux 2.2.8 and 2.3.47 end here -- */
    );
    
    /* fprintf(stderr, "stat2proc converted %d fields.\n",num); */
    if (P->tty == 0)
	    P->tty = -1;  /* the old notty val, update elsewhere bef. moving to 0 */
}

static uid_t
proc_owner(pid_t pid)
{
   struct stat sb;
   char buffer[32];
   sprintf(buffer, "/proc/%d", pid);

   if (stat(buffer, &sb) < 0)
      return -1;
   else
      return sb.st_uid;
}

static unsigned long int
get_kb_main_total(void)
{
   int fd, len;
   char buffer[2048], *p;
  
   fd = open("/proc/meminfo", O_RDONLY);
   if(fd <0)
      return 0;

   len = read(fd, buffer, sizeof(buffer)-1);
   close(fd);

   buffer[len] = '\0'; 

   p = strstr( buffer, "MemTotal:" );
   if (!p)
      return 0;

   p = skip_token(p);
   p = skip_whitespace(p);

   return atol(p); 
}

static unsigned long int
get_boottime(void)
{
   int fd, len;
   char buffer[4096], *p;
 
   fd = open("/proc/stat",  O_RDONLY);
   if (fd <0)
      return 0;

   len = read(fd, buffer, sizeof(buffer)-1);
   close(fd);
  
   buffer[len] = '\0';

   p = strstr( buffer, "btime" );
   if(!p)
      return 0;

   p = skip_token(p);
   p = skip_whitespace(p);

   return atol(p); 
}

static int
mcast_data( gmond_config_t *config, proc_t *ps )
{

   return 0;
}

void *
gexec_process_func( void *arg )
{
/* autoconf these later */
#define POLLING_INTERVAL  15 
#define TIME_THRESHOLD    60 
#define PCPU_THRESHOLD    5
#define PMEM_THRESHOLD    5

   int rval;
   char statfile[512];
   char buffer[4096];
   long ticks, last_ticks = 0;
   float elapsed_time;
   struct passwd *pw;
   proc_t ps;
   uid_t uid, last_uid =0;
   int uid_set= 0;
   unsigned long kb_main_total;
   struct timeval t;
   unsigned long boottime;
   unsigned long last_mcast  = 0;
   double last_pmem =0, last_pcpu =0;
   gexec_process_func_arg_t *a = (gexec_process_func_arg_t *)arg;
    
   /* We need to get the total memory in order
      to calculate percentage memory */
   kb_main_total = get_kb_main_total();
   if (! kb_main_total )
      goto end;

   /* We need to know the bootime to get the
      time this process is running */
   boottime = get_boottime();
   if (! boottime )
      goto end;

   sprintf(statfile, "/proc/%d/stat", a->pid);

   err_quiet();

   for(;;)
      {
         rval = slurpfile(statfile, buffer, sizeof(buffer)-1 );
         if(rval <0)
            goto end;
                
         stat2proc(buffer, &ps);

         /* Convert to kilobytes */
         ps.vsize = bytetok(ps.vsize);  /* vsize */
         ps.rss   = pagetok(ps.rss);    /* rss */

         /* Calculate percentage CPU */
         ticks = ps.utime + ps.stime - last_ticks; 
         last_ticks = ps.utime + ps.stime;
         elapsed_time = get_elapsed_time();

         ps.pcpu = (ticks * 100/HZ) / elapsed_time;
         if (ps.pcpu > 99.9)
             ps.pcpu = 99.9;

         /* Calculate percentage MEM */
         ps.pmem = 100.00 * ( (float)(ps.rss) / (float)(kb_main_total));
         if (ps.pmem > 99.9) ps.pmem = 99.9;

         /* Get the UID and username */
         uid = proc_owner(a->pid);

         if ( !uid_set || uid != last_uid )
            {
               pw = getpwuid(uid);
               if (pw == NULL)
                  sprintf(ps.name, "%d", uid);
               else
                  strcpy( ps.name, pw->pw_name);
               uid_set = 1;
            }
         last_uid = uid;

         /* Calculate the time in seconds we've been running */
         gettimeofday(&t, NULL);
         ps.runtime = (t.tv_sec - boottime) - (ps.start_time/HZ);

         /* Decide if we should multicast the process data */
         if ( ((t.tv_sec - last_mcast)   > TIME_THRESHOLD )
            ||( abs(ps.pcpu - last_pcpu) > PCPU_THRESHOLD )
            ||( abs(ps.pmem - last_pmem) > PMEM_THRESHOLD ))
            {
               mcast_data(a->config, &ps);
               last_mcast = t.tv_sec;
            }
         last_pcpu = ps.pcpu;
         last_pmem = ps.pmem;

         printf("ps.name = %s\n", ps.name);
         printf("ps.pid = %d\n", ps.pid);
         printf("ps.rss = %ld\n", ps.rss);
         printf("ps.vsize = %ld\n", ps.vsize);
         printf("ps.priority = %ld\n", ps.priority);
         printf("ps.nice = %ld\n", ps.nice);
         printf("ps.utime = %ld\n", ps.utime);
         printf("ps.stime = %ld\n", ps.stime);
         printf("ps.cutime = %ld\n", ps.cutime);
         printf("ps.cstime = %ld\n", ps.cstime);
         printf("ps.pcpu = %f\n", ps.pcpu);
         printf("ps.pmem = %f\n", ps.pmem);
         printf("ps.runtime = %ld\n", ps.runtime);

         sleep(POLLING_INTERVAL);
      }

  end:
   free(a->gpid);
   free(a);
   return NULL;
}

int
gexec_process( const char *gpid, int pid, gmond_config_t *config, int block )
{
   pthread_t tid;
   pthread_attr_t attr; 
   gexec_process_func_arg_t *a;

   /* gexec_process_func will free the memory allocated here */
   a = (gexec_process_func_arg_t *)malloc(sizeof(gexec_process_func_arg_t));
   if(!a)
      return -1;

   /* gexec_process_func will free this too */
   a->gpid = strdup(gpid);
   if(!a->gpid)
      return -1;
   a->pid = pid;
   /* I want config changes to propagate .. no dup*/
   a->config = config;

   if( block )
      {
         /* Block until process, pid, no longer exists */
         gexec_process_func((void *)a);
      }
   else
      {
         pthread_attr_init( &attr );
         pthread_attr_setdetachstate( &attr, PTHREAD_CREATE_DETACHED );
         /* Spin off a thread */
         pthread_create(&tid, &attr, gexec_process_func, (void *)a);
      }
   return 0;
}

#endif
