/*
 * gauche-gtk.h - Gauche+Gtk extension
 *
 *  Copyright(C) 2002 by Shiro Kawai (shiro@acm.org)
 *
 *  Permission to use, copy, modify, distribute this software and
 *  accompanying documentation for any purpose is hereby granted,
 *  provided that existing copyright notices are retained in all
 *  copies and that this notice is included verbatim in all
 *  distributions.
 *  This software is provided as is, without express or implied
 *  warranty.  In no circumstances the author(s) shall be liable
 *  for any damages arising out of the use of this software.
 *
 *  $Id: gauche-gtk.c,v 1.46 2003/12/09 20:16:03 shirok Exp $
 */

#include "gauche-gtk.h"

/*===============================================================
 * GObject <-> ScmObj mapping
 */

/*
   NB: it is not trivial to keep reference-counting Gtk memory allocation
   and mark-sweep GC happily together.   The naive method (increment Gtk
   refcount when Scheme obtains the pointer to it, and decrement it when
   Scheme objects are garbage-collected) has a problem:

   We need to protect Scheme pointers passed to Gtk object.  Gtk
   has a callback when it releases the passed pointer, so we can
   register the Scheme pointer to a global table to protect from
   being GC'ed, and remove it in the Gtk's callback.  However,
   if the reference consists a cycle, i.e. the Scheme object passed
   to Gtk has a reference to other Scheme objects which eventually
   points back to the Gtk object, then the Gtk object's refcount
   never go down to zero, and whole structure will never be reclaimed.

   This issue has been discussed in gtk mailing list.
   http://mail.gnome.org/archives/gtk-list/1998-April/msg00525.html
   Vollmer Marius told how Guile-gtk handled it, which seems the
   best approximation strategy so far I've seen:
   http://mail.gnome.org/archives/gtk-list/1998-April/msg00596.html

   It may be possible that I hook Boehm GC (by providing user-defined
   mark_proc) to implement Vollmer's strategy, but I haven't understood
   innards of Boehm GC enough yet.

 */

static struct {
    ScmHashTable *protected;    /* Table of Scheme objects that is passed to
                                   GTk, to protect them from GC'ed.
                                   The key is the object itself, and the value
                                   is a # of times the object is protected. */
    ScmInternalMutex protected_mutex;
    ScmHashTable *typemap;      /* Map ScmClass to GType.  It is rarely
                                   used, but needed in some API that handles
                                   meta information (e.g. liststore) */
    ScmInternalMutex typemap_mutex;
    
    GQuark scmclass_key;        /* A Quark used in the property list of
                                   GType to keep its associated ScmClass. */
    GQuark scmobj_key;          /* A Quark used in the property list of
                                   GObject to point back Scheme object. */
} gtkdata = {
    /* Initialize the first item so that the structure is placed
       in the data area. */
    NULL
};

/*
 * Type mapping
 */

/* In order to 'box' given GObject in a Scheme object, we need to know
 * the Scheme class corresponding to the given GObject type.  GObject
 * type system has a sort of property list (called qdata), so we use
 * it to keep the type's corresponding ScmClass.
 * Scm_GtkRegisterClass establishes the bidirectional link between
 * GType and ScmClass.  It is called from initialization routine.
 */
typedef struct ScmGTypeRec {
    SCM_HEADER;
    GType gtype;
} ScmGType;

static SCM_DEFINE_BUILTIN_CLASS_SIMPLE(Scm_GTypeClass, NULL);

void Scm_GtkRegisterClass(GType type, ScmClass *klass)
{
    ScmGType *gtype = SCM_NEW(ScmGType);
    SCM_SET_CLASS(gtype, &Scm_GTypeClass);
    gtype->gtype = type;
    g_type_set_qdata(type, gtkdata.scmclass_key, (gpointer)klass);
    (void)SCM_INTERNAL_MUTEX_LOCK(gtkdata.typemap_mutex);
    Scm_HashTablePut(gtkdata.typemap, SCM_OBJ(klass), SCM_OBJ(gtype));
    (void)SCM_INTERNAL_MUTEX_UNLOCK(gtkdata.typemap_mutex);
}

ScmClass *Scm_GtkTypeToScmClass(GType type)
{
    ScmClass *c = NULL;
    GType t = type;
    /* Gtk API may return an object of private subtype of the published
       type, so if we don't find the corresponding ScmClass, need to
       look for its ancestors. */
    for (;;) {
        c = (ScmClass*)g_type_get_qdata(t, gtkdata.scmclass_key);
        if (c) return c;
        t = g_type_parent(t);
        if (t == 0) {
            const char *name = g_type_name(type);
            Scm_Warn("Unknown GType %x(%s); GObject assumed", type,
                     name? name : "noname");
            return SCM_CLASS_GOBJECT;
        }
    }
    /*NOTREACHED*/
}

GType Scm_ClassToGtkType(ScmClass *klass)
{
    ScmHashEntry *e;
    (void)SCM_INTERNAL_MUTEX_LOCK(gtkdata.typemap_mutex);
    e = Scm_HashTableGet(gtkdata.typemap, SCM_OBJ(klass));
    (void)SCM_INTERNAL_MUTEX_UNLOCK(gtkdata.typemap_mutex);
    if (!e) return G_TYPE_INVALID;
    else return ((ScmGType*)(e->value))->gtype;
}

int Scm_ClassListToGtkTypeList(ScmObj klasses, GType *types)
{
    int len, i = 0;
    ScmObj k = SCM_NIL;
    GType gt;
    
    if ((len = Scm_Length(klasses)) > 0) {
        ScmObj sp;
        SCM_FOR_EACH(sp, klasses) {
            k = SCM_CAR(sp);
            if (!Scm_TypeP(k, SCM_CLASS_CLASS)) goto noklass;
            gt = Scm_ClassToGtkType(SCM_CLASS(k));
            if (gt == G_TYPE_INVALID) goto notype;
            types[i++] = gt;
        }
        return i;
    } else if (SCM_VECTORP(klasses)) {
        ScmObj *sp = SCM_VECTOR_ELEMENTS(klasses);
        len = SCM_VECTOR_SIZE(klasses);
        for (i=0; i<len; i++) {
            k = *sp++;
            if (!Scm_TypeP(k, SCM_CLASS_CLASS)) goto noklass;
            gt = Scm_ClassToGtkType(SCM_CLASS(k));
            if (gt == G_TYPE_INVALID) goto notype;
            types[i] = gt;
        }
        return i;
    } else {
        Scm_Error("list or vector of <class> expected, but got %S", klasses);
    }
  noklass:
    Scm_Error("<class> required, but got %S", k);
  notype:
    Scm_Error("Class %S doesn't have corresponding Gtk type", k);
    return -1;                  /* dummy */
}

/* pre-registered primitive types */
static struct predef_type {
    ScmClass *scmklass;
    GType gtype;
} predef_types[] = {
    { SCM_CLASS_BOOL,       G_TYPE_BOOLEAN },
    { SCM_CLASS_CHAR,       G_TYPE_CHAR },
    { SCM_CLASS_INTEGER,    G_TYPE_INT },
    { SCM_CLASS_REAL,       G_TYPE_DOUBLE },
    { SCM_CLASS_STRING,     G_TYPE_STRING },
    { NULL }
};

static void typemap_initialize(ScmHashTable *table)
{
    struct predef_type *ptype = predef_types;
    for (; ptype->scmklass; ptype++) {
        ScmGType *g = SCM_NEW(ScmGType);
        SCM_SET_CLASS(g, &Scm_GTypeClass);
        g->gtype = ptype->gtype;
        Scm_HashTablePut(table, SCM_OBJ(ptype->scmklass), SCM_OBJ(g));
    }
}

/*
 * GObject
 */

/*static void gobject_print(ScmObj obj, ScmPort *port, ScmWriteContext *ctx)*/

static int gobject_compare(ScmObj x, ScmObj y, int equalp)
{
    if (equalp) {
        return (SCM_GOBJECT_OBJECT(x) == SCM_GOBJECT_OBJECT(y))? 0 : -1;
    } else {
        Scm_Error("can't order GObject %S and %S", x, y);
        return 0;
    }
}

static void gobject_destroy(GtkObject *gobj, void *data)
{
    ScmGObject *g = (ScmGObject*)data;
    Scm_GObjectUnref(g);
}

ScmClass *Scm_GObjectCPL[] = {
    SCM_CLASS_STATIC_PTR(Scm_GObjectClass),
    SCM_CLASS_STATIC_PTR(Scm_TopClass),
    NULL
};

SCM_DEFINE_BASE_CLASS(Scm_GObjectClass, ScmGObject,
                      NULL, gobject_compare, NULL, Scm_GtkObjectAllocate,
                      Scm_GObjectCPL+1);

#if 0 /* for now, we rely on explicit deallocation */
static void gobject_finalize(ScmObj obj, void *data)
{
    ScmGObject *g = SCM_GOBJECT(obj);
    g_object_set_qdata(SCM_GOBJECT_OBJECT(g), gtkdata.scmobj_key, NULL);
    g_object_unref(SCM_GOBJECT_OBJECT(g));
}
#endif

/* Internal routine to create a Scheme wrapper for gobject. */
static ScmGObject *make_gobject(ScmClass *klass, GObject *gobj)
{
    ScmGObject *g;
    SCM_ASSERT(Scm_SubtypeP(klass, SCM_CLASS_GOBJECT));
    g = SCM_ALLOCATE(ScmGObject, klass);
    SCM_SET_CLASS(g, klass);
    g->gobject = gobj;
    g->data = SCM_NIL;
    Scm_GtkProtect(SCM_OBJ(g));
    g_object_set_qdata_full(gobj, gtkdata.scmobj_key, (gpointer)g,
                            (GDestroyNotify)Scm_GtkUnprotect);
#if 0 /* for now, we rely on explicit deallocation */
    Scm_RegisterFinalizer(SCM_OBJ(g), gobject_finalize, NULL);
#endif
    g_object_ref(gobj);
    if (g_type_is_a(G_OBJECT_TYPE(gobj), GTK_TYPE_OBJECT)) {
        /* Take floating reference */
        gtk_object_sink(GTK_OBJECT(gobj));
        /* Drop the reference upon destruction. */
        g_signal_connect(GTK_OBJECT(gobj), "destroy",
                         (GCallback)gobject_destroy, (void*)g);
    }
    return g;
}

/* 'Box' a pointer to GObject. */
ScmObj Scm_MakeGObject(void *obj)
{
    ScmClass *klass;
    ScmGObject *g;
    GObject *gobj;
    
    /* Allow obj == NULL */
    if (obj == NULL) return SCM_FALSE;
    gobj = G_OBJECT(obj);

    /* First, see if this GObject already has corresponding ScmObj */
    g = (ScmGObject*)g_object_get_qdata(gobj, gtkdata.scmobj_key);
    if (g == NULL) {
        /* Creates ScmGObject */
        klass = Scm_GtkTypeToScmClass(G_OBJECT_TYPE(gobj));
        g = make_gobject(klass, gobj);
    }
    return SCM_OBJ(g);
}

/* Common allocator of GtkObject.  Should be used only from gtk*.stub. */
ScmObj Scm_GtkObjectAllocate(ScmClass *klass, ScmObj initargs)
{
    ScmClass **k = klass->cpa;
    GType gbase = G_TYPE_INVALID, t;

    /* Find out which GtkObject should be instantiated, and also
       there's no conflicting GtkType in CPL. */
    t = Scm_ClassToGtkType(klass);
    if (t != G_TYPE_INVALID) gbase = t;
    for (; *k; k++) {
        t = Scm_ClassToGtkType(*k);
        if (t != G_TYPE_INVALID) {
            if (gbase == G_TYPE_INVALID) {
                gbase = t;
            } else {
                if (!g_type_is_a(gbase, t)) {
                    const gchar *gn = g_type_name(gbase);
                    const gchar *tn = g_type_name(t);
                    Scm_Error("class precedence list of %S contains conflicting GtkObject types: %s and %s",
                              klass, (gn? gn : "?"), (tn? tn : "?"));
                }
            }
        }
    }
    if (gbase == G_TYPE_INVALID) {
        Scm_Error("can't instantiate object of class %S", klass);
    }
    return SCM_OBJ(make_gobject(klass, g_object_new(gbase, NULL)));
}

/* Protect and unprotect the Scheme object passed to Gtk */
void Scm_GtkProtect(ScmObj data)
{
    ScmHashEntry *e;
    int count;
    (void)SCM_INTERNAL_MUTEX_LOCK(gtkdata.protected_mutex);
    e = Scm_HashTableAdd(gtkdata.protected, data, SCM_MAKE_INT(0));
    count = SCM_INT_VALUE(e->value) + 1;
    e->value = SCM_MAKE_INT(count);
    (void)SCM_INTERNAL_MUTEX_UNLOCK(gtkdata.protected_mutex);
}

void Scm_GtkUnprotect(gpointer data)
{
    ScmHashEntry *e;
    int count;
    
    if (!data) return;
    (void)SCM_INTERNAL_MUTEX_LOCK(gtkdata.protected_mutex);
    e = Scm_HashTableGet(gtkdata.protected, SCM_OBJ(data));
    if (e) {
        count = SCM_INT_VALUE(e->value) - 1;
        if (count == 0) {
            Scm_HashTableDelete(gtkdata.protected, SCM_OBJ(data));
        } else {
            e->value = SCM_MAKE_INT(count);
        }
    }
    (void)SCM_INTERNAL_MUTEX_UNLOCK(gtkdata.protected_mutex);
}

/* Explicitly unreference GObject.
   This is necessary to break the cyclic reference until the customized
   mark procedure is implemented (see the discussion above).
   Once unreferenced, gobject field becomes NULL, even if the pointed
   GObject has more than one reference.  The idea is that the Scheme
   object here will be garbage-collected soon. */

void Scm_GObjectUnref(ScmGObject *gobj)
{
    if (gobj->gobject) {
        GObject *g = gobj->gobject;
        g_object_set_qdata(g, gtkdata.scmobj_key, NULL);
        gobj->gobject = NULL;
        g_object_unref(g);
    }
}

/* Checks if GObject is not unreferenced */
GObject *Scm_GObjectCheck(ScmGObject *gobj)
{
    if (!gobj->gobject) {
        Scm_Error("GObject has been unreferenced from %S", gobj);
    }
    return gobj->gobject;
}

/* Scheme-world GObject data & properties */
ScmObj Scm_GObjectGetData(ScmGObject *gobj, ScmObj key, ScmObj fallback)
{
    ScmObj p = Scm_Assq(key, gobj->data);
    if (SCM_PAIRP(p)) return SCM_CDR(p);
    if (SCM_UNBOUNDP(fallback)) Scm_Error("GObject %S doesn't have a property for the key %S", gobj, key);
    return fallback;
}

ScmObj Scm_GObjectSetData(ScmGObject *gobj, ScmObj key, ScmObj data)
{
    if (SCM_UNBOUNDP(data)) {
        gobj->data = Scm_AssocDeleteX(key, gobj->data, SCM_CMP_EQ);
    } else {
        ScmObj p = Scm_Assq(key, gobj->data);
        if (SCM_PAIRP(p)) SCM_SET_CDR(p, data);
        else {
            gobj->data = Scm_Acons(key, data, gobj->data);
        }
    }
    return SCM_UNDEFINED;
}

#if 0
static const char *get_key(ScmObj key)
{
    if (SCM_STRINGP(key)) return Scm_GetStringConst(SCM_STRING(key));
    if (SCM_SYMBOLP(key)) return Scm_GetStringConst(SCM_SYMBOL_NAME(key));
    if (SCM_IDENTIFIERP(key)) return Scm_GetStringConst(SCM_SYMBOL_NAME(SCM_IDENTIFIER(key)->name));
    Scm_Error("property key must be a string or a symbol, but got %S", key);
    return "";                  /* dummy */
}
#endif

/*===============================================================
 * Callbacks
 */

/* Issues:
 *
 *  * Error handling and non-local exit in callbacked Scheme program:
 *    The Scheme closure is effectively invoked inside with-error-handler
 *    so that the errors are captured and handled (right now by
 *    Scm_ReportError) before returning to C.
 *    Outbound continuation invocation, that is, the invocation of a
 *    continuation captured below the Gtk loop, is more problematic.
 *    It can happen, for example, if the user builds his/her own error
 *    handling mechanism using call/cc.
 *    If the program 'restarts', that is, re-enters gtk-main again,
 *    it would be a problem.  It is OK if the program just exits.
 *    We won't know which is the case here, so we assume the user
 *    knows what he/she is doing. 
 *
 *  * Argument marshalling.   Gtk/Glib callback mechanism provide
 *    complete type information of arguments & return value, so what
 *    I need to do is just map those arguments to Scheme values.
 *    The old Gtk used GtkArg structure for this purpose, but the new
 *    Gtk uses Glib's GValue mechanism.  There are still some callbacks
 *    in Gtk that uses GtkArg, so we need to support both.
 */

/* Call callback.  Errors are blocked here and handled accordingly. */

static ScmObj callcallback_error(ScmObj *args, int nargs, void *data)
{
    ScmObj exc = SCM_OBJ(args[0]);
    /* TODO: better error handling? */
    Scm_ReportError(exc);
    return SCM_FALSE;
}

static SCM_DEFINE_STRING_CONST(callcallback_error__NAME,
                               "%gtk-call-callback-error", 24, 24);
static SCM_DEFINE_SUBR(callcallback_error__STUB, 1, 0,
                       SCM_OBJ(&callcallback_error__NAME),
                       callcallback_error, NULL, NULL);

static ScmObj callcallback_thunk(ScmObj *args, int nargs, void *data)
{
    ScmObj sargs = SCM_OBJ(data);
    SCM_ASSERT(SCM_PAIRP(sargs));
    return Scm_Apply(SCM_CAR(sargs), SCM_CDR(sargs));
}
static SCM_DEFINE_STRING_CONST(callcallback_thunk__NAME,
                               "%gtk-call-callback-thunk", 24, 24);

static ScmObj callcallback_proc(ScmObj *args, int nargs, void *data)
{
    ScmObj closure = SCM_OBJ(args[0]);
    ScmObj sargs = SCM_OBJ(args[1]);
    ScmObj thunk;
    
    if (SCM_NULLP(sargs)) {
        /* closure is really a thunk, so we can avoid creating
           extra closure */
        thunk = closure;
    } else {
        thunk = Scm_MakeSubr(callcallback_thunk,
                             (void*)Scm_Cons(closure, sargs),
                             0, 0,
                             SCM_OBJ(&callcallback_thunk__NAME));
    }
    return Scm_VMWithErrorHandler(SCM_OBJ(&callcallback_error__STUB),
                                  thunk);
}

static SCM_DEFINE_STRING_CONST(callcallback_proc__NAME,
                               "%gtk-call-callback", 18, 18);
static SCM_DEFINE_SUBR(callcallback_proc__STUB, 2, 0,
                       SCM_OBJ(&callcallback_proc__NAME),
                       callcallback_proc, NULL, NULL);

#define SCM_CALL_CALLBACK   SCM_OBJ(&callcallback_proc__STUB)

/* Argument & return value marshalling - GValue version */

ScmObj Scm_UnboxGValue(const GValue *gv)
{
    GType gt = G_VALUE_TYPE(gv);
    switch (G_TYPE_FUNDAMENTAL(gt)) {
    case G_TYPE_CHAR:  return SCM_MAKE_INT((int)g_value_get_char(gv));
    case G_TYPE_UCHAR: return SCM_MAKE_INT((int)g_value_get_uchar(gv));
    case G_TYPE_BOOLEAN: return SCM_MAKE_BOOL(g_value_get_boolean(gv));
    case G_TYPE_INT:   return Scm_MakeInteger(g_value_get_int(gv));
    case G_TYPE_UINT:  return Scm_MakeIntegerFromUI(g_value_get_uint(gv));
    case G_TYPE_LONG:  return Scm_MakeInteger(g_value_get_long(gv));
    case G_TYPE_ULONG: return Scm_MakeInteger(g_value_get_ulong(gv));
    case G_TYPE_FLOAT: return Scm_MakeFlonum((double)g_value_get_float(gv));
    case G_TYPE_DOUBLE:return Scm_MakeFlonum(g_value_get_double(gv));
    case G_TYPE_STRING: 
        return SCM_MAKE_STR_COPYING(g_value_get_string(gv));
    case G_TYPE_OBJECT:
        return Scm_MakeGObject(G_OBJECT(g_value_get_object(gv)));
    case G_TYPE_POINTER: {
        Scm_Warn("got G_TYPE_POINTER (really a %s)", g_type_name(gt));
        return SCM_UNDEFINED;
    }
    default:
        /* I'm not sure this is a right thing, but for now...*/
        if (gt == GDK_TYPE_EVENT) {
            return Scm_MakeGdkEvent((GdkEvent*)g_value_get_boxed(gv));
        }
        if (g_type_is_a (gt, G_TYPE_FLAGS)) {
            return Scm_MakeInteger(g_value_get_flags(gv));
        }
        if (gt == GDK_TYPE_EVENT) {
            return Scm_MakeGdkEvent((GdkEvent*)g_value_get_boxed(gv));
        }
        if (gt == gtk_tree_path_get_type()) {
            return SCM_MAKE_GTK_TREE_PATH((GtkTreePath*)g_value_get_boxed(gv));
        }
        Scm_Warn("cannot convert a GValue of type %s to a Scheme object (%d)",
                 g_type_name(gt), G_TYPE_FUNDAMENTAL(gt));
        return SCM_UNDEFINED;
    }
}

void Scm_BoxGValue(GValue *gv, ScmObj sv)
{
    GType gt = G_VALUE_TYPE(gv);
    switch (G_TYPE_FUNDAMENTAL(gt)) {
    case G_TYPE_INVALID:
        /* this happens in some callbacks.   I assume the receiver doesn't
           need the return value. */
        return;
    case G_TYPE_CHAR: {
        int v;
        if (SCM_INTP(sv)) v = SCM_INT_VALUE(sv);
        else if (SCM_CHARP(sv)) v = SCM_CHAR_VALUE(sv);
        else goto err;
        if (v < -128 || v > 127) goto err;
        g_value_set_char(gv, (gchar)v);
        return;
    }
    case G_TYPE_UCHAR: {
        int v;
        if (SCM_INTP(sv)) v = SCM_INT_VALUE(sv);
        else if (SCM_CHARP(sv)) v = SCM_CHAR_VALUE(sv);
        else goto err;
        if (v < 0 || v > 255) goto err;
        g_value_set_uchar(gv, (guchar)v);
        return;
    }
    case G_TYPE_BOOLEAN: {
        g_value_set_boolean(gv, SCM_BOOL_VALUE(sv));
        return;
    }
    case G_TYPE_INT: {
        if (!SCM_EXACTP(sv)) goto err;
        g_value_set_int(gv, Scm_GetInteger(sv));
        return;
    }
    case G_TYPE_UINT: {
        if (!SCM_EXACTP(sv)) goto err;
        g_value_set_uint(gv, Scm_GetUInteger(sv));
        return;
    }
    case G_TYPE_LONG: {
        if (!SCM_EXACTP(sv)) goto err;
        g_value_set_long(gv, Scm_GetInteger(sv));
        return;
    }
    case G_TYPE_ULONG: {
        if (!SCM_EXACTP(sv)) goto err;
        g_value_set_ulong(gv, Scm_GetUInteger(sv));
        return;
    }
    case G_TYPE_FLOAT: {
        if (!SCM_REALP(sv)) goto err;
        g_value_set_float(gv, (gfloat)Scm_GetDouble(sv));
        return;
    }
    case G_TYPE_DOUBLE: {
        if (!SCM_REALP(sv)) goto err;
        g_value_set_double(gv, Scm_GetDouble(sv));
        return;
    }
    case G_TYPE_STRING: {
        if (!SCM_STRINGP(sv)) goto err;
        g_value_set_string(gv, Scm_GetStringConst(SCM_STRING(sv)));
        return;
    }
    case G_TYPE_OBJECT: {
        if (!Scm_TypeP(sv, SCM_CLASS_GOBJECT)) goto err;
        g_value_set_object(gv, SCM_GOBJECT_OBJECT(sv));
        return;
    }
    default:
      err:
        Scm_Error("cannot convert a Scheme object %S to a GValue of type %s",
                  sv, g_type_name(gt));
    }
}

/* Like BoxGValue, except the type is determined by Scheme value.
   GValue structure is initialized by this. */
GValue *Scm_ObjToGValue(ScmObj obj, GValue *gv)
{
    gv->g_type = 0;
    if (SCM_INTP(obj)) {
        g_value_init(gv, G_TYPE_INT);
        g_value_set_int(gv, Scm_GetInteger(obj));
        return gv;
    }
    if (SCM_BIGNUMP(obj)) {
        /*NB: check the value range */
        g_value_init(gv, G_TYPE_INT);
        g_value_set_int(gv, Scm_GetInteger(obj));
        return gv;
    }
    if (SCM_STRINGP(obj)) {
        g_value_init(gv, G_TYPE_STRING);
        g_value_set_string(gv, Scm_GetStringConst(SCM_STRING(obj)));
        return gv;
    }
    if (SCM_SYMBOLP(obj)) {
        g_value_init(gv, G_TYPE_STRING);
        g_value_set_string(gv, Scm_GetStringConst(SCM_SYMBOL_NAME(obj)));
        return gv;
    }
    if (SCM_BOOLP(obj)) {
        g_value_init(gv, G_TYPE_BOOLEAN);
        g_value_set_boolean(gv, SCM_BOOL_VALUE(obj));
        return gv;
    }
    if (SCM_FLONUMP(obj)) {
        g_value_init(gv, G_TYPE_DOUBLE);
        g_value_set_double(gv, Scm_GetDouble(obj));
        return gv;
    }
    if (SCM_GOBJECT_P(obj)) {
        GType gt = Scm_ClassToGtkType(SCM_CLASS_OF(obj));
        if (gt != G_TYPE_INVALID) {
            g_value_init(gv, gt);
            g_value_set_object(gv, SCM_GOBJECT_OBJECT(obj));
            return gv;
        }
    }
    Scm_Error("can't convert Scheme value %S to GValue", obj);
    return NULL;
}

/*
 * GClosure interface
 */
typedef struct {
    GClosure closure;
    ScmProcedure *proc;
} SClosure;

void Scm_GClosureMarshal(GClosure *closure, GValue *retval,
                         guint nparams, const GValue *params,
                         gpointer ihint, gpointer data)
{
    ScmObj argh = SCM_NIL, argt = SCM_NIL, ret;
    ScmProcedure *proc = ((SClosure*)closure)->proc;
    int i;

    SCM_ASSERT(proc && SCM_PROCEDUREP(proc));
    for (i=0; i<nparams; i++) {
        SCM_APPEND1(argh, argt, Scm_UnboxGValue(params+i));
    }
    ret = Scm_Apply(SCM_CALL_CALLBACK,
                    SCM_LIST2(SCM_OBJ(proc), argh));
    if (retval) Scm_BoxGValue(retval, ret);
}

void Scm_GClosureDestroy(gpointer data, GClosure *closure)
{
    Scm_GtkUnprotect(data);
}

GClosure *Scm_MakeGClosure(ScmProcedure *procedure)
{
    GClosure *c = g_closure_new_simple(sizeof(SClosure), NULL);
    ((SClosure*)c)->proc = procedure;
    Scm_GtkProtect(SCM_OBJ(procedure));
    g_closure_add_finalize_notifier(c, (gpointer)procedure,
                                    Scm_GClosureDestroy);
    g_closure_set_marshal(c, Scm_GClosureMarshal);
    return c;
}

/* This can be passed to gtk_idle_add etc. */
gboolean Scm_GtkCallThunk(gpointer closure)
{
    ScmObj r;
    SCM_ASSERT(closure != NULL && SCM_PROCEDUREP(closure));
    r = Scm_Apply(SCM_OBJ(&callcallback_proc__STUB),
                  SCM_LIST2(SCM_OBJ(closure), SCM_NIL));
    return SCM_BOOL_VALUE(r);
}

/* More general version.  Returns a list of values. */
ScmObj Scm_GtkApply(ScmObj proc, ScmObj args)
{
    Scm_Apply(SCM_OBJ(&callcallback_proc__STUB),
              SCM_LIST2(proc, args));
    return Scm_VMGetResult(Scm_VM());
}

/*===============================================================
 * Unix signal handling
 */

/* After gtk-main-loop, Gtk takes over the control of the
 * application.  When an unix signal arrives, it is queued
 * in the Gauche VM signal queue and also it terminates the
 * poll() function inside Gtk main loop.  However, Gtk knows
 * nothing about Gauche VM, so it re-invokes poll() if no other
 * event occurs---thus Gauche's signal handler will never be
 * called.
 *
 * Gtk doesn't provide a direct way to address this (idle
 * handler can't be used, for it would be a busy wait for the
 * unix signals).  However, the underlying g_main_loop mechanism
 * that Gtk main loop uses has very flexible way to hook our
 * function inside the main loop.
 */

static gboolean scm_signal_prepare(GSource *source, gint *timeout)
{
    *timeout = -1;
    return FALSE;
}

static gboolean scm_signal_check(GSource *source)
{
    ScmVM *vm = Scm_VM();
    return SCM_SIGPENDING(vm);
}

static gboolean scm_signal_dispatch(GSource *source,
                                    GSourceFunc callback,
                                    gpointer user_data)
{
    Scm_SigCheck(Scm_VM());
    return TRUE;
}

GSourceFuncs scm_signal_watch_funcs = {
    scm_signal_prepare,
    scm_signal_check,
    scm_signal_dispatch,
    NULL
};

static gboolean scm_signal_watcher_add(gpointer data)
{
    GSource *source = g_source_new(&scm_signal_watch_funcs,
                                   sizeof(GSource));
    /* attach to the default context, which Gtk seems to use. */
    g_source_attach(source, NULL);
    return TRUE;
}

void Scm_GtkInitUnixSignalHook(void)
{
    gtk_init_add(scm_signal_watcher_add, NULL);
}

/*===============================================================
 * GTimer
 */

static void g_timer_finalize(ScmObj obj, void *data)
{
    ScmGTimer *g = (ScmGTimer*)obj;
    g_timer_destroy(g->data);
    g->data = NULL;
}

ScmObj Scm_MakeGTimer(GTimer *r)
{
    ScmGTimer *g = SCM_NEW(ScmGTimer);
    SCM_SET_CLASS(g, SCM_CLASS_GTIMER);
    g->data = r;
    Scm_RegisterFinalizer(SCM_OBJ(g), g_timer_finalize, NULL);
    return SCM_OBJ(g);
}

/*===============================================================
 * GdkAtom <-> ScmObj mapping
 */

SCM_DEFINE_BUILTIN_CLASS(Scm_GdkAtomClass,
                         NULL, NULL, NULL, NULL,
                         Scm_GObjectCPL+1);

ScmObj Scm_MakeGdkAtom(GdkAtom atom)
{
    ScmGdkAtom *z = SCM_NEW(ScmGdkAtom);
    SCM_SET_CLASS(z, SCM_CLASS_GDK_ATOM);
    z->atom = atom;             /* no refcounting needed */
    return SCM_OBJ(z);
}

/*===============================================================
 * gchar** <-> string list mapping
 */

gint Scm_GtkStringsToGcharArrays(ScmObj list, gchar ***chars)
{
    int len = Scm_Length(list), i = 0;
    ScmObj cp;
    gchar **s = SCM_NEW2(gchar**, sizeof(gchar*)*len);
    SCM_FOR_EACH(cp, list) {
        if (!SCM_STRINGP(SCM_CAR(cp))) {
            Scm_Error("string requried, but got %S", SCM_CAR(cp));
        }
        s[i++] = (gchar*)Scm_GetString(SCM_STRING(SCM_CAR(cp)));
    }
    *chars = s;
    return len;
}

ScmObj Scm_GtkGcharArraysToStrings(gint count, gchar **chars)
{
    ScmObj h = SCM_NIL, t = SCM_NIL;
    int i;
    for (i=0; i<count; i++) {
        SCM_APPEND1(h, t, SCM_MAKE_STR_COPYING(chars[i]));
    }
    return h;
}

/* Box returned allocated gchar*.  We need to copy it to own the memory. */
ScmObj Scm_GtkGcharPtrBox(gchar *string)
{
    ScmObj s = SCM_MAKE_STR_COPYING(string);
    g_free(string);
    return s;
}

/*===============================================================
 * GdkEvent
 */

extern ScmClass Scm_GdkEventAnyClass;
extern ScmClass Scm_GdkEventExposeClass;
extern ScmClass Scm_GdkEventMotionClass;
extern ScmClass Scm_GdkEventButtonClass;
extern ScmClass Scm_GdkEventKeyClass;
extern ScmClass Scm_GdkEventCrossingClass;
extern ScmClass Scm_GdkEventFocusClass;
extern ScmClass Scm_GdkEventConfigureClass;
extern ScmClass Scm_GdkEventPropertyClass;
extern ScmClass Scm_GdkEventSelectionClass;
extern ScmClass Scm_GdkEventProximityClass;
extern ScmClass Scm_GdkEventDNDClass;
extern ScmClass Scm_GdkEventClientClass;
extern ScmClass Scm_GdkEventVisibilityClass;
extern ScmClass Scm_GdkEventNoExposeClass;
extern ScmClass Scm_GdkEventScrollClass;
extern ScmClass Scm_GdkEventWindowStateClass;
extern ScmClass Scm_GdkEventSettingClass;

/* maps event->type to the class of the event */
static struct EvClassTableRec {
    GdkEventType type;
    ScmClass *klass;
} evClassTable[] = {
    { GDK_DELETE,           &Scm_GdkEventAnyClass },
    { GDK_DESTROY,          &Scm_GdkEventAnyClass },
    { GDK_EXPOSE,           &Scm_GdkEventExposeClass },
    { GDK_MOTION_NOTIFY,    &Scm_GdkEventMotionClass },
    { GDK_BUTTON_PRESS,     &Scm_GdkEventButtonClass },
    { GDK_2BUTTON_PRESS,    &Scm_GdkEventButtonClass },
    { GDK_3BUTTON_PRESS,    &Scm_GdkEventButtonClass },
    { GDK_BUTTON_RELEASE,   &Scm_GdkEventButtonClass },
    { GDK_KEY_PRESS,        &Scm_GdkEventKeyClass },
    { GDK_KEY_RELEASE,      &Scm_GdkEventKeyClass },
    { GDK_ENTER_NOTIFY,     &Scm_GdkEventCrossingClass },
    { GDK_LEAVE_NOTIFY,     &Scm_GdkEventCrossingClass },
    { GDK_FOCUS_CHANGE,     &Scm_GdkEventFocusClass },
    { GDK_CONFIGURE,        &Scm_GdkEventConfigureClass },
    { GDK_MAP,              &Scm_GdkEventAnyClass },
    { GDK_UNMAP,            &Scm_GdkEventAnyClass },
    { GDK_PROPERTY_NOTIFY,  &Scm_GdkEventPropertyClass },
    { GDK_SELECTION_CLEAR,  &Scm_GdkEventSelectionClass },
    { GDK_SELECTION_REQUEST,&Scm_GdkEventSelectionClass },
    { GDK_SELECTION_NOTIFY, &Scm_GdkEventSelectionClass },
    { GDK_PROXIMITY_IN,     &Scm_GdkEventProximityClass },
    { GDK_PROXIMITY_OUT,    &Scm_GdkEventProximityClass },
    { GDK_DRAG_ENTER,       &Scm_GdkEventDNDClass },
    { GDK_DRAG_LEAVE,       &Scm_GdkEventDNDClass },
    { GDK_DRAG_MOTION,      &Scm_GdkEventDNDClass },
    { GDK_DRAG_STATUS,      &Scm_GdkEventDNDClass },
    { GDK_DROP_START,       &Scm_GdkEventDNDClass },
    { GDK_DROP_FINISHED,    &Scm_GdkEventDNDClass },
    { GDK_CLIENT_EVENT,     &Scm_GdkEventClientClass },
    { GDK_VISIBILITY_NOTIFY,&Scm_GdkEventVisibilityClass },
    { GDK_NO_EXPOSE,        &Scm_GdkEventNoExposeClass },
    { GDK_SCROLL,           &Scm_GdkEventScrollClass },
    { GDK_WINDOW_STATE,     &Scm_GdkEventWindowStateClass },
    { GDK_SETTING,          &Scm_GdkEventSettingClass },
    { -1,                   &Scm_GdkEventAnyClass }
};

SCM_DEFINE_BUILTIN_CLASS(Scm_GdkEventClass,
                         NULL, NULL, NULL, NULL,
                         Scm_GObjectCPL+1);

static void gdk_event_finalize(ScmObj obj, void *data)
{
    ScmGdkEvent *e = (ScmGdkEvent*)obj;
    gdk_event_free(e->data);
    e->data = NULL;
}

ScmObj Scm_MakeGdkEvent(GdkEvent *r)
{
    ScmClass *klass = &Scm_GdkEventAnyClass;
    ScmGdkEvent *g;
    struct EvClassTableRec *ctab;

    for (ctab = evClassTable; ctab->type >= 0; ctab++) {
        if (((GdkEventAny*)r)->type == ctab->type) {
            klass = ctab->klass;
            break;
        }
    }
    g = SCM_NEW(ScmGdkEvent);
    SCM_SET_CLASS(g, klass);
    g->data = gdk_event_copy(r);
    Scm_RegisterFinalizer(SCM_OBJ(g), gdk_event_finalize, NULL);
    return SCM_OBJ(g);
}

/*===============================================================
 * GList & GSList
 */

ScmObj Scm_GoListToList(GList *list)
{
    ScmObj h = SCM_NIL, t = SCM_NIL;
    while (list) {
        if (!list->data) Scm_Error("GList->List: list contains NULL");
        if (!G_IS_OBJECT(list->data)) {
            Scm_Error("GList->List: list contains non-GObject");
        }
        SCM_APPEND1(h, t, SCM_GOBJECT_BOX(list->data));
        list = g_list_next(list);
    }
    return h;
}

ScmObj Scm_GoSListToList(GSList *list)
{
    ScmObj h = SCM_NIL, t = SCM_NIL;
    while (list) {
        if (!list->data) Scm_Error("GSList->List: list contains NULL");
        if (!G_IS_OBJECT(list->data)) {
            Scm_Error("GSList->List: list contains non-GObject");
        }
        SCM_APPEND1(h, t, SCM_GOBJECT_BOX(list->data));
        list = g_slist_next(list);
    }
    return h;
}

GList *Scm_ListToGList(ScmObj list)
{
    GList *glist = NULL;
    ScmObj lp;
    SCM_FOR_EACH(lp, list) {
        ScmObj elt = SCM_CAR(lp);
        if (!SCM_GOBJECT_P(elt)) {
            if (glist) g_list_free(glist);
            Scm_Error("List of <g-object> required, but the list contains %S", elt);
        }
        glist = g_list_append(glist, SCM_GOBJECT_OBJECT(elt));
    }
    return glist;
}

GSList *Scm_ListToGSList(ScmObj list)
{
    GSList *glist = NULL;
    ScmObj lp;
    SCM_FOR_EACH(lp, list) {
        ScmObj elt = SCM_CAR(lp);
        if (!SCM_GOBJECT_P(elt)) {
            if (glist) g_slist_free(glist);
            Scm_Error("List of <g-object> required, but the list contains %S", elt);
        }
        glist = g_slist_append(glist, SCM_GOBJECT_OBJECT(elt));
    }
    return glist;
}

/*===============================================================
 * String list and array
 */

const char **Scm_StringListToStringArray(ScmObj list)
{
    int len = Scm_Length(list), i = 0;
    ScmObj lp;
    const char **a;
    if (len < 0) Scm_Error("proper list required, but got %S", list);
    a = SCM_NEW2(const char **, (len+1)*sizeof(char *));
    SCM_FOR_EACH(lp, list) {
        if (!SCM_STRINGP(SCM_CAR(lp)))
            Scm_Error("string required, but got %S", SCM_CAR(lp));
        a[i] = Scm_GetStringConst(SCM_STRING(SCM_CAR(lp)));
        i++;
    }
    a[i] = NULL;
    return a;
}

/*===============================================================
 * Arrays of primitive types
 */

SCM_DEFINE_BUILTIN_CLASS(Scm_GdkPointVectorClass,
                         NULL, NULL, NULL, NULL,
                         Scm_GObjectCPL+1);

ScmObj Scm_MakeGdkPointVector(GdkPoint *pts, int npts)
{
    ScmGdkPointVector *r = SCM_NEW(ScmGdkPointVector);
    SCM_SET_CLASS(r, SCM_CLASS_GDK_POINT_VECTOR);
    r->size = npts;
    r->elements = SCM_NEW_ATOMIC2(GdkPoint*, sizeof(GdkPoint[1])*npts);
    if (pts) memcpy(r->elements, pts, sizeof(GdkPoint[1])*npts);
    return SCM_OBJ(r);
}

SCM_DEFINE_BUILTIN_CLASS(Scm_GdkSegmentVectorClass,
                         NULL, NULL, NULL, NULL,
                         Scm_GObjectCPL+1);

ScmObj Scm_MakeGdkSegmentVector(GdkSegment *segs, int nsegs)
{
    ScmGdkSegmentVector *r = SCM_NEW(ScmGdkSegmentVector);
    SCM_SET_CLASS(r, SCM_CLASS_GDK_SEGMENT_VECTOR);
    r->size = nsegs;
    r->elements = SCM_NEW_ATOMIC2(GdkSegment*, sizeof(GdkSegment[1])*nsegs);
    if (segs) memcpy(r->elements, segs, sizeof(GdkSegment[1])*nsegs);
    return SCM_OBJ(r);
}

SCM_DEFINE_BUILTIN_CLASS(Scm_GdkRectangleVectorClass,
                         NULL, NULL, NULL, NULL,
                         Scm_GObjectCPL+1);

ScmObj Scm_MakeGdkRectangleVector(GdkRectangle *rects, int nrects)
{
    ScmGdkRectangleVector *r = SCM_NEW(ScmGdkRectangleVector);
    SCM_SET_CLASS(r, SCM_CLASS_GDK_RECTANGLE_VECTOR);
    r->size = nrects;
    r->elements = SCM_NEW_ATOMIC2(GdkRectangle*, sizeof(GdkRectangle[1])*nrects);
    if (rects) memcpy(r->elements, rects, sizeof(GdkRectangle[1])*nrects);
    return SCM_OBJ(r);
}

SCM_DEFINE_BUILTIN_CLASS(Scm_GdkColorVectorClass,
                         NULL, NULL, NULL, NULL,
                         Scm_GObjectCPL+1);

ScmObj Scm_MakeGdkColorVector(GdkColor *colors, int ncolors)
{
    ScmGdkColorVector *r = SCM_NEW(ScmGdkColorVector);
    SCM_SET_CLASS(r, SCM_CLASS_GDK_COLOR_VECTOR);
    r->size = ncolors;
    r->elements = SCM_NEW_ATOMIC2(GdkColor*, sizeof(GdkColor[1])*ncolors);
    if (colors) memcpy(r->elements, colors, sizeof(GdkColor[1])*ncolors);
    return SCM_OBJ(r);
}

/*===============================================================
 * RadioGroup
 */

/* See the comment of header file about ScmGtkRadioGroup */

static int radio_group_compare(ScmObj x, ScmObj y, int equalp)
{
    ScmObj rx, ry;
    GObject *gx, *gy;

    if (!equalp) Scm_Error("can't order %S and %S", x, y);
    rx = SCM_GTK_RADIO_GROUP(x)->radio;
    ry = SCM_GTK_RADIO_GROUP(x)->radio;
    if (SCM_FALSEP(rx)) {
        return SCM_FALSEP(ry)? 0 : -1;
    }
    SCM_ASSERT(SCM_GOBJECT_P(rx) && SCM_GOBJECT_P(ry));
    gx = SCM_GOBJECT_OBJECT(rx);
    gy = SCM_GOBJECT_OBJECT(ry);
    if (GTK_IS_RADIO_BUTTON(gx)) {
        if (GTK_IS_RADIO_BUTTON(gy)) {
            GtkRadioButton *bx = GTK_RADIO_BUTTON(gx);
            GtkRadioButton *by = GTK_RADIO_BUTTON(gy);
            return (gtk_radio_button_get_group(bx) == gtk_radio_button_get_group(by))? 0 : -1;
        }
        return -1;
    }
    if (GTK_IS_RADIO_MENU_ITEM(gx)) {
        if (GTK_IS_RADIO_MENU_ITEM(gy)) {
            GtkRadioMenuItem *bx = GTK_RADIO_MENU_ITEM(gx);
            GtkRadioMenuItem *by = GTK_RADIO_MENU_ITEM(gy);
            return (gtk_radio_menu_item_get_group(bx) == gtk_radio_menu_item_get_group(by))? 0 : -1;
        }
        return -1;
    }
    return -1;
}

SCM_DEFINE_BUILTIN_CLASS(Scm_GtkRadioGroupClass,
                         NULL, radio_group_compare, NULL, NULL,
                         NULL);

/* 'radio' must be either a GtkRadioButton or a GtkRadioMenuItem */
ScmObj Scm_MakeGtkRadioGroup(GObject *radio)
{
    GSList *glist = NULL;
    ScmGtkRadioGroup *group;
    if (GTK_IS_RADIO_BUTTON(radio)) {
        glist = gtk_radio_button_get_group(GTK_RADIO_BUTTON(radio));
    } else if (GTK_IS_RADIO_MENU_ITEM(radio)) {
        glist = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(radio));
    } else {
        Scm_Error("<gtk-radio-group> can be created only for <gtk-radio-button> or <gtk-radio-menu-item> object, but got an instance of %s",
                  g_type_name(G_OBJECT_TYPE(radio)));
    }
    group = SCM_NEW(ScmGtkRadioGroup);
    SCM_SET_CLASS(group, SCM_CLASS_GTK_RADIO_GROUP);
    if (glist != NULL) {
        group->radio = Scm_MakeGObject(radio);
    } else {
        group->radio = SCM_FALSE;
    }
    return SCM_OBJ(group);
}

GSList *Scm_GtkRadioGroupGetGroup(ScmObj g)
{
    GObject *gradio;
    ScmGtkRadioGroup *group;
    if (SCM_FALSEP(g)) return NULL;
    if (!SCM_GTK_RADIO_GROUP_P(g)) {
        Scm_Error("<gtk-radio-group> or #f required, but got %S", g);
    }
    group = SCM_GTK_RADIO_GROUP(g);
    if (SCM_FALSEP(group->radio)) return NULL;
    SCM_ASSERT(SCM_GOBJECT_P(group->radio));
    gradio = SCM_GOBJECT_OBJECT(group->radio);
    if (GTK_IS_RADIO_BUTTON(gradio)) {
        return gtk_radio_button_get_group(GTK_RADIO_BUTTON(gradio));
    }
    if (GTK_IS_RADIO_MENU_ITEM(gradio)) {
        return gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(gradio));
    }
    Scm_Error("internal inconsistency in %S", group);
    return NULL;                /* dummy */
}

ScmObj Scm_GtkRadioGroupToList(ScmGtkRadioGroup *group)
{
    GSList *glist = Scm_GtkRadioGroupGetGroup(SCM_OBJ(group));
    if (glist == NULL) return SCM_NIL;
    else return Scm_GoSListToList(glist);
}

/*===============================================================
 * Initialization 
 */

#include "gtk-lib.inits"
extern void Scm_Init_gauche_glib(ScmModule*);

void Scm_Init_gauche_gtk(void)
{
    ScmModule *mod;
    SCM_INIT_EXTENSION(gauche_gtk);
    mod = SCM_MODULE(SCM_FIND_MODULE("gtk", TRUE));

    g_type_init();
    gtkdata.scmclass_key = g_quark_from_static_string("ScmClass");
    gtkdata.scmobj_key = g_quark_from_static_string("ScmObj");

    (void)SCM_INTERNAL_MUTEX_INIT(gtkdata.protected_mutex);
    gtkdata.protected = SCM_HASHTABLE(Scm_MakeHashTable(SCM_HASH_ADDRESS, NULL, 0));
    (void)SCM_INTERNAL_MUTEX_INIT(gtkdata.typemap_mutex);
    gtkdata.typemap = SCM_HASHTABLE(Scm_MakeHashTable(SCM_HASH_ADDRESS, NULL, 0));
    typemap_initialize(gtkdata.typemap);

    Scm_InitBuiltinClass(&Scm_GObjectClass, "<g-object>",
                         NULL, sizeof(ScmGObject),
                         mod);
    Scm_InitBuiltinClass(&Scm_GTypeClass, "<g-type>",
                         NULL, sizeof(ScmGType),
                         mod);
    Scm_InitBuiltinClass(&Scm_GdkAtomClass, "<gdk-atom>",
                         NULL, sizeof(ScmGdkAtom),
                         mod);
    Scm_InitBuiltinClass(&Scm_GdkEventClass, "<gdk-event>",
                         NULL, sizeof(ScmGdkEvent),
                         mod);
    Scm_InitBuiltinClass(&Scm_GdkRegionClass, "<gdk-region>",
                         NULL, sizeof(ScmGdkRegion),
                         mod);
    Scm_InitBuiltinClass(&Scm_GdkPointVectorClass, "<gdk-point-vector>",
                         NULL, sizeof(ScmGdkPointVector),
                         mod);
    Scm_InitBuiltinClass(&Scm_GdkSegmentVectorClass, "<gdk-segment-vector>",
                         NULL, sizeof(ScmGdkSegmentVector),
                         mod);
    Scm_InitBuiltinClass(&Scm_GdkRectangleVectorClass, "<gdk-rectangle-vector>",
                         NULL, sizeof(ScmGdkRectangleVector),
                         mod);
    Scm_InitBuiltinClass(&Scm_GdkColorVectorClass, "<gdk-color-vector>",
                         NULL, sizeof(ScmGdkColorVector),
                         mod);
    Scm_InitBuiltinClass(&Scm_GtkRadioGroupClass, "<gtk-radio-group>",
                         NULL, sizeof(ScmGtkRadioGroup),
                         mod);
    Scm_Init_gauche_glib(mod);
    Scm_Init_gauche_gdklib(mod);
    Scm_Init_gtk_lib(mod);
    Scm_GtkInitUnixSignalHook();
}
