package GCExtract::GCExtractMusics;

###################################################
#
#  Copyright 2005-2006 Tian
#
#  This file is part of GCstar.
#
#  GCstar is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCstar is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCstar; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################

use strict;
use GCExtract;

{
    package GCExtract::GCmusicsExtracter;
    use base 'GCItemExtracter';
    
    sub new
    {
        my $proto = shift;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new(@_);
        bless ($self, $class);

        $self->{hasMP3Info} = $self->checkModule('MP3::Info');
        $self->{hasMP3Tag} = $self->checkModule('MP3::Tag');
        $self->{hasOggVorbisHeader} = $self->checkModule('Ogg::Vorbis::Header::PurePerl');
        # Even previous check fails, we want to use it for tracks feature
        $self->{errors} = 0;

        $self->{fields} = ['title', 'artist', 'release', 'genre', 'running', 'tracks'];
        return $self;
    }

    sub resetTracks
    {
        my $self = shift;
        $self->{tracks} = {};
        $self->{totalTime} = 0;
        $self->{currentTrack} = 0;
        $self->{firstTrack} = '';
    }

    sub addTrack
    {
        my ($self, $title, $time, $number) = @_;
        $self->{currentTrack}++;
        $self->{totalTime} += $time;
        $number = $self->{currentTrack} if !defined $number;
        push @{$self->{tracks}->{line}},
            {
                col => [$number, $title, $self->secondsToString($time)]
            };
    }

    sub getTracks
    {
        my $self = shift;
        return $self->{tracks};
    }

    sub secondsToString
    {
        my ($self, $time) = @_;
        return int($time / 60) .':'. sprintf '%02d', ($time %60);
    }

    sub getTotalTime
    {
        my $self = shift;
        return $self->secondsToString($self->{totalTime});
    }

    sub getM3UInfo
    {
        my ($self, $file) = shift;
        my $file = $self->{file};
        my $info = {};
        while (<$file>)
        {
            chomp;
            s/\r//;
            if (/^#/)
            {
                next if ! /^#EXTINF:(.*)/;
                my @values = split /,/, $1;
                $self->addTrack($values[1], $values[0]);
            }
            else
            {
                $self->{firstTrack} = $_
                    if !$self->{firstTrack};
            }
        }
        $info->{tracks} = $self->getTracks;
        $info->{running} = $self->getTotalTime;
        return $info;
    }

    sub getPLSInfo
    {
        my ($self, $file) = shift;
        my $file = $self->{file};
        my $info = {};
        my @tracks;
        while (<$file>)
        {
            chomp;
            s/\r//;
            next if ! /(File|Title|Length)(\d+)=(.*)$/;
            $tracks[$2]->{$1} = $3;
            $tracks[$2]->{Number} = $2;
        }
        foreach (@tracks)
        {
            next if !$_->{Title};
            $self->addTrack($_->{Title}, $_->{Length}, $_->{Number});
        }
            
        $info->{tracks} = $self->getTracks;
        $info->{running} = $self->getTotalTime;
        $self->{firstTrack} = $tracks[1]->{File};
        return $info;
    }

    sub addFirstTrackInfo
    {
        my ($self, $info) = @_;
 
        if ($^O =~ /win32/i)
        {
            $self->{firstTrack} =~ s|\\|/|g;
            $self->{fileName} =~ /^(.{2})/;
            my $drive = $1;
            $self->{firstTrack} = $drive.$self->{firstTrack}
                if $self->{firstTrack} =~ m|^/|;
        }

        if ($self->{firstTrack} =~ /mp3$/i)
        {
            if ($self->{hasMP3Info})
            {
                print "using MP3::Info\n";
                MP3::Info::use_mp3_utf8(1);
                my $song = MP3::Info::get_mp3tag($self->{firstTrack});
                $info->{title} = $song->{ALBUM};
                $info->{artist} = $song->{ARTIST};
                $info->{release} = $song->{YEAR};
                $info->{genre} = $song->{GENRE};
            }
            elsif ($self->{hasMP3Tag})
            {
                print "using MP3::Tag\n";
                my $song = MP3::Tag->new($self->{firstTrack});
                (undef, undef, $info->{artist}, $info->{title}) = $song->autoinfo;
            }
        }
        elsif ($self->{firstTrack} =~ /ogg$/i)
        {
            if ($self->{hasOggVorbisHeader})
            {
                print "using Ogg::Vorbis::Header::PurePerl\n";
                my $song = Ogg::Vorbis::Header::PurePerl->new($self->{firstTrack});
                $info->{title} = ($song->comment('album'))[0];
                $info->{artist} .= $_.', ' foreach $song->comment('artist');
                $info->{artist} =~ s/, $//;
                ($info->{release} = ($song->comment('date'))[0]) =~ s|^(\d{4})-(\d{2})-(\d{2}).*$|$3/$2/$1|;
                $info->{genre} .= $_.', ' foreach $song->comment('genre');
                $info->{genre} =~ s/, $//;
            }
        }
    }

    sub getInfo
    {
        my $self = shift;
        open FILE, '<'.$self->{fileName};
        binmode FILE;

        my $info = {};
        $self->resetTracks;

        $self->{file} = \*FILE;
        my $header = <FILE>;
        
        $info = $self->getM3UInfo
            if ($self->{fileName} =~ /m3u$/) || ($header =~ /^#EXTM3U/);
        $info = $self->getPLSInfo
            if ($self->{fileName} =~ /pls$/) || ($header =~ /^\[playlist\]/);
        $self->addFirstTrackInfo($info);

        close FILE;
        
        my $result;
        my $firstTrackName = $info->{tracks}->{line}->[0]->{col}->[1];
        $result->{tracks} = {displayed => $firstTrackName, value => $info->{tracks}};
        foreach (@{$self->{fields}})
        {
            next if /^tracks$/;
            $result->{$_} = {displayed => $info->{$_}, value => $info->{$_}};
        }
        return $result;
    }
    
    sub getFields
    {
        my $self = shift;
        return $self->{fields};
    }
}


1;
