package GCItemsLists;

###################################################
#
#  Copyright 2005-2006 Tian
#
#  This file is part of GCstar.
#
#  GCstar is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCstar is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCstar; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################

use strict;

{
    package GCTextList;
    
    use Gtk2::SimpleList;
    use base "Gtk2::ScrolledWindow";

    sub new
    {
        my ($proto, $parent, $title) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;
        bless ($self, $class);

        $self->{parent} = $parent;
        $self->{titleField} = $parent->{model}->{commonFields}->{title};
        
        $self->set_policy ('automatic', 'automatic');
        $self->set_shadow_type('none');
        
        $self->{list} = new Gtk2::SimpleList($title => 'text');
        $self->{list}->set_name('GCItemsTextList');
        $self->{list}->set_headers_clickable(1);
        $self->{list}->set_rules_hint(1);

        $self->{list}->get_column(0)->set_sort_indicator(1);
        $self->{list}->get_column(0)->signal_connect('clicked' => sub {
            $parent->{items}->changeOrder;
            $self->setSortOrder;
        });
#        $self->{list}->get_column(0)->set_sort_column_id(0);
#        $self->{list}->get_model->set_sort_column_id(0, 'ascending');
        
        $self->{list}->signal_connect (cursor_changed => sub {
            my ($sl, $path, $column) = @_;
            my @idx = $sl->get_selected_indices;
            $parent->display($idx[0]);
            $self->{currentIdx} = $idx[0];
        });

        $self->{list}->signal_connect ('row-activated' => sub {
           $parent->displayInWindow;
        });

        $self->add($self->{list});
        
        $self->{list}->signal_connect('button_press_event' => sub {
                my ($widget, $event) = @_;
                return 0 if $event->button ne 3;
                $self->{context}->popup(undef, undef, undef, undef, $event->button, $event->time);
                return 0;
        });
        
        $self->{list}->signal_connect('key-press-event' => sub {
                my ($widget, $event) = @_;
                my $key = Gtk2::Gdk->keyval_name($event->keyval);
                if ($key eq 'Delete')
                {
                    $self->{parent}->deleteCurrentItem;
                    return 1;
                }
                return 0;
        });
        
        $self->{currentIdx} = 0;

        return $self;
    }

    sub savePreferences
    {
    }

    sub couldExpandAll
    {
        my $self = shift;
        
        return 0;
    }

    sub setSortOrder
    {
        my $self = shift;
        my ($order, $field) = $self->{parent}->{items}->getSortOrder;
        $self->{list}->get_column(0)->set_sort_order($order ? 'ascending' : 'descending');
    }

    sub clearCache
    {
        my $self = shift;
    }

    sub reset
    {
        my $self = shift;
        @{$self->{list}->{data}} = ();
    }
    
    sub done
    {
        my $self = shift;
    }
    
    sub addItem
    {
        my ($self, $info) = @_;
        push @{$self->{list}->{data}}, $self->{parent}->transformTitle($info->{$self->{titleField}});
    }
    
    sub removeItem
    {
        my ($self, $number) = @_;
        
        splice @{$self->{list}->{data}}, $number, 1;
        $number-- if $number >= scalar @{$self->{list}->{data}};
        $self->select($number);
        
        #Reload not needed
        return (0, $number);
    }
    
    sub select
    {
        my ($self, $idx, $id, $init) = @_;

        $idx = 0 if $idx == -1;
        $self->{list}->select($idx);
        $self->{currentIdx} = $idx;
        return $idx;
    }
    
    sub showCurrent
    {
        my $self = shift;
        
   		my $path = $self->{list}->get_selection->get_selected_rows;
		$self->{list}->scroll_to_cell($path) if $path;
    }

    sub changeCurrent
    {
        #my ($self, $image, $borrower) = @_;
        my ($self, $previous, $new) = @_;

        $self->{list}->{data}->[$self->{currentIdx}] = $self->{parent}->transformTitle($new->{$self->{titleField}});

        #No reload needed
        return 0;
    }
    
}

{
    package GCImageList;
    
    use File::Basename;
    use GCUtils;
    use GCStyle;
    use base "Gtk2::VBox";
    use File::Temp qw/ tempfile /;
    
    sub new
    {
        my ($proto, $parent, $columns) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;
        bless ($self, $class);
        
        $self->{withImage} = $parent->{options}->listBgPicture;
        
        $self->{coverField} = $parent->{model}->{commonFields}->{cover};
        $self->{titleField} = $parent->{model}->{commonFields}->{title};
        $self->{idField} = $parent->{model}->{commonFields}->{id};
        $self->{borrowerField} = $parent->{model}->{commonFields}->{borrower}->{name};
        
        $parent->{options}->listImgSkin($GCStyle::defaultList) if ! $parent->{options}->exists('listImgSkin');
        $self->{skin} = $parent->{options}->listImgSkin;

        $parent->{options}->listImgSize(2) if ! $parent->{options}->exists('listImgSize');
        $self->{size} = $parent->{options}->listImgSize;

        #Default value (also for size = 2)
        ($self->{imgWidth}, $self->{imgHeight}) = (120, 160);
        $self->{vboxWidth} = $self->{imgWidth} + 10;
        $self->{vboxHeight} = $self->{imgHeight} + 10;
        $self->{vboxHeight} += 20 if $self->{withImage};

        $self->{factor} = 1;

        if ($self->{size} == 0)
        {
            $self->{factor} = 0.5;
        }
        elsif ($self->{size} == 1)
        {
             $self->{factor} = 0.8;
        }
        elsif ($self->{size} == 3)  
        {
            $self->{factor} = 1.5;
        }
        elsif($self->{size} == 4)
        {
            $self->{factor} = 2;
        }
        
        $self->{imgWidth} *= $self->{factor};
        $self->{imgHeight} *= $self->{factor};
        $self->{vboxWidth} = $self->{imgWidth} + (10 * $self->{factor});
        $self->{vboxHeight} = $self->{imgHeight} + (10 * $self->{factor});
        $self->{vboxHeight} += (20 * $self->{factor}) if $self->{withImage};
        
        $self->{scroll} = new Gtk2::ScrolledWindow;
        $self->{scroll}->set_policy ('automatic', 'automatic');
        $self->{scroll}->set_shadow_type('none');
        
        $self->{parent} = $parent;
        
        $self->{sortButton} = Gtk2::Button->new_from_stock('gtk-sort-descending');
        
        #$self->{list} = new Gtk2::Table(0,$columns,1);
        $self->{list} = new Gtk2::VBox(0,0);
        $self->{tooltips} = Gtk2::Tooltips->new();
        
        $self->{columns} = $columns;
        $self->reset;
        $self->clearCache;

        $self->{lendPicture} = $parent->{style}->{dir}.'/lend.png';
        $self->{lendPixbuf} = Gtk2::Gdk::Pixbuf->new_from_file($self->{lendPicture});
        $self->{lendPixbuf} = GCUtils::scaleMaxPixbuf($self->{lendPixbuf},
                                                      $self->{imgWidth},
                                                      $self->{imgHeight},
                                                      1);
        
        $self->{defaultImage} = $parent->{defaultImage};
        
        #$self->{list}->set_row_spacings(0);
        #$self->{list}->set_col_spacings(0);
        $self->{list}->set_border_width(0);
        $self->{list}->set_border_width(5) if ! $self->{withImage};
        
        $self->{scroll}->add_with_viewport($self->{list});
    
        if ($self->{withImage})
        {
            my $bgdir = $ENV{GCS_SHARE_DIR}.'/list_bg/'.$self->{skin};

#            $self->{selectedPixmap} = $bgdir.'/list_bg_selected.png';
#            $self->{unselectedPixmap} = $bgdir.'/list_bg.png';
            $self->{bgPixmap} = $bgdir.'/list_bg.png';

            my $tmpPixbuf = Gtk2::Gdk::Pixbuf->new_from_file($self->{bgPixmap});
            $tmpPixbuf = GCUtils::scaleMaxPixbuf($tmpPixbuf,
                                                 $self->{vboxWidth},
                                                 $self->{vboxHeight},
                                                 1);
            (my $fh, $self->{tmpBgPixmap}) = tempfile;
            close $fh;
            $tmpPixbuf->save($self->{tmpBgPixmap}, 'png');
            GCUtils::setWidgetPixmap($self->{list}->parent, $self->{tmpBgPixmap});
            
            $self->{backgroundPixbuf} = Gtk2::Gdk::Pixbuf->new_from_file($self->{bgPixmap});
            $self->{backgroundPixbuf} = GCUtils::scaleMaxPixbuf($self->{backgroundPixbuf},
                                                                $self->{vboxWidth},
                                                                $self->{vboxHeight},
                                                                1);
#            $self->{selectedPixbuf} = Gtk2::Gdk::Pixbuf->new_from_file($self->{selectedPixmap});
#            $self->{selectedPixbuf} = GCUtils::scaleMaxPixbuf($self->{selectedPixbuf},
#                                                              $self->{vboxWidth},
#                                                              $self->{vboxHeight},
#                                                              1);
        }
        else
        {
            my @colors = split m/,/, $parent->{options}->listBgColor;
            ($colors[0], $colors[1], $colors[2]) = (65535, 65535, 65535) if !@colors;
            $self->{inactiveBg} = new Gtk2::Gdk::Color($colors[0], $colors[1], $colors[2]);
            @colors = split m/,/, $parent->{options}->listFgColor;
            ($colors[0], $colors[1], $colors[2]) = (0, 0, 0) if !@colors;
            $self->{activeBg} = new Gtk2::Gdk::Color($colors[0], $colors[1], $colors[2]);

            $self->{list}->parent->modify_bg('normal', $self->{inactiveBg});
            $self->{list}->parent->modify_bg('active', $self->{inactiveBg});
            $self->{list}->parent->modify_bg('prelight', $self->{inactiveBg});
            $self->{list}->parent->modify_bg('selected', $self->{inactiveBg});
            $self->{list}->parent->modify_bg('insensitive', $self->{inactiveBg});
        }

        $self->pack_start($self->{sortButton},0,0,0);
        $self->pack_start($self->{scroll},1,1,0);
        
        $self->{list}->can_focus(1);
        
        $self->{sortButton}->signal_connect('clicked' => sub {
            $parent->{items}->changeOrder;
            $self->setSortOrder;
        });
        
        
        return $self;
    }

    sub savePreferences
    {
    }

    sub couldExpandAll
    {
        my $self = shift;
        
        return 0;
    }

    sub DESTROY
    {
        my $self = shift;
        
        unlink $self->{tmpBgPixmap};
        $self->SUPER::DESTROY;
    }

    sub setSortOrder
    {
        my $self = shift;
        my ($order, $field) = $self->{parent}->{items}->getSortOrder;
        
        $self->remove($self->{sortButton});
        $self->{sortButton}->destroy;
        $self->{sortButton} = Gtk2::Button->new_from_stock($order ? 'gtk-sort-descending' : 'gtk-sort-ascending');
        $self->pack_start($self->{sortButton},0,0,0);
        $self->reorder_child($self->{sortButton},0);
        $self->{sortButton}->show_all;
        
        $self->{sortButton}->signal_connect('clicked' => sub {
            $self->{parent}->{items}->changeOrder;
            $self->setSortOrder;
        });
        
    }

    sub clearCache
    {
        my $self = shift;
        
        if ($self->{cache})
        {
            foreach (values %{$self->{cache}})
            {
                $_->{eventBox}->destroy;
            }
        }
        
        $self->{cache} = {};
    }

    sub reset
    {
        my $self = shift;
        
        #Restore current picture if modified
        $self->{boxes}->[$self->{current}]->child->set_from_pixbuf($self->{previousPixbuf})
            if ($self->{previousPixbuf} && $self->{boxes}->[$self->{current}]->child);
        
        $self->{boxes}->[$self->{current}]->modify_bg('normal', $self->{inactiveBg})
            if (! $self->{withImage}) && $self->{boxes}->[$self->{current}];
        
        my @array = ();
        $self->{boxes} = \@array;
        $self->{number} = 0;
        $self->{current} = 0;
        $self->{previous} = 0;
        $self->{previousPixbuf} = undef;
        $self->{list}->hide;
        my @children = $self->{list}->get_children;
        foreach (@children)
        {
            my @children2 = $_->get_children;
            foreach my $child(@children2)
            {
                $_->remove($child);
            }
            $self->{list}->remove($_);
        }
        $self->{enhanceInformation} = [];
        
        Glib::Timeout->add(200, sub {
            $self->{list}->show_all;
        });
        
        $self->{initializing} = 1;
    }
    
    sub done
    {
        my $self = shift;
        
        my $i = 1;
        
        $self->{list}->show_all;
        $self->{initializing} = 0;
        foreach (@{$self->{enhanceInformation}})
        {
            Glib::Timeout->add($i * 30,
                              \&enhancePicture,
                              $_);
                              
            $i++;
        }
    }
    
    sub createPixbuf
    {
        my ($self, $displayedImage, $borrower) = @_;

        my $pixbuf;
        eval {
            $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file($displayedImage);
        };
        $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file($self->{defaultImage})
            if $@;
        my $width = $self->{imgWidth};
        my $height = $self->{imgHeight};
        $pixbuf = GCUtils::scaleMaxPixbuf($pixbuf,
                                          $self->{imgWidth},
                                          $self->{imgHeight},
                                          0,
                                          $self->{initializing});
        $width = $pixbuf->get_width;
        $height = $pixbuf->get_height;

        if ($borrower && ($borrower ne 'none'))
        {
           $self->{lendPixbuf}->composite($pixbuf,
                                          0, 0, 
                                          $width, $height,
                                          0, 0,
                                          1, 1, 
                                          'nearest', 255);
        }
        
        if ($self->{withImage})
        {
            my $offsetX = 5 * $self->{factor} + (($self->{imgWidth} - $width) / 2);
            my $offsetY = 15 * $self->{factor} + ($self->{imgHeight} - $height);
            my $bgPixbuf = $self->{backgroundPixbuf}->copy;
            $pixbuf->composite($bgPixbuf,
                                  $offsetX, $offsetY, 
                                  $width, $height,
                                  $offsetX, $offsetY,
                                  1, 1, 
                                  'nearest', 255);
#            $bgPixbuf->composite($pixbuf,
#                                  0, 0, 
#                                  $width, $height,
#                                  $offsetX, $offsetY,
#                                  1, 1, 
#                                  'nearest', 255);

            $pixbuf = $bgPixbuf;
        }
        return $pixbuf;
    }
    
    sub enhancePicture
    {
        my $data = shift;
        my $eventBox = $data->[1];
        return if ! $eventBox->child;
        my $self = $data->[0];
        my $pixbuf = $self->createPixbuf($data->[2], $data->[3]);
        $eventBox->child->set_from_pixbuf($pixbuf);

        if ($self->{boxes}->[$self->{current}] == $eventBox)
        {
            $self->{previousPixbuf} = $pixbuf;
            $self->select($self->{current});
        }
        
        return 0;
    }
    
    sub createEventBox
    {
        my ($self, $info) = @_;
    
        my $eventBox = new Gtk2::EventBox;
        $eventBox->can_focus(1);
                
        my $image = new Gtk2::Image;

        my $displayedImage = GCUtils::getDisplayedImage($info->{$self->{coverField}},
                                                        $self->{defaultImage},
                                                        $self->{parent}->{options}->file);

        my $pixbuf = $self->createPixbuf($displayedImage, $info->{$self->{borrowerField}});

        $self->{tooltips}->set_tip($eventBox, $self->{parent}->transformTitle($info->{$self->{titleField}}), '');

        if (! $self->{withImage})
        {
            $eventBox->modify_bg('normal', $self->{inactiveBg});
        }

        $image->set_from_pixbuf($pixbuf);
        
        $eventBox->add($image);
        $eventBox->set_size_request($self->{vboxWidth},$self->{vboxHeight});

        if ($self->{initializing})
        {
            push @{$self->{enhanceInformation}}, [$self, $eventBox, $displayedImage, $info->{$self->{borrowerField}}];
        }
        else
        {
            $eventBox->show_all;
        }
        
        return $eventBox;
    }

    sub getFromCache
    {
        my ($self, $info) = @_;
        if (($info->{$self->{idField}} == '') || 
            (! $self->{cache}->{$info->{$self->{idField}}}))
        {
            my $item = {};
            $item->{eventBox} = $self->createEventBox($info);
            $item->{keyHandler} = 0;
            $item->{mouseHandler} = 0;
        
            $self->{cache}->{$info->{$self->{idField}}} = $item;
        }
        return $self->{cache}->{$info->{$self->{idField}}};
    }
    
    sub addItem
    {
        my ($self, $info) = @_;
        
        my $item = $self->getFromCache($info);
        my $eventBox = $item->{eventBox};
        my $i = $self->{number};
        
        $eventBox->signal_handler_disconnect($item->{mouseHandler})
            if $item->{mouseHandler};
        $item->{mouseHandler} = $eventBox->signal_connect('button_press_event' => sub {
                my ($widget, $event) = @_;
                $self->{parent}->display($i) unless $event->type eq '2button-press';
                $self->select($i);
                $self->{parent}->displayInWindow if $event->type eq '2button-press';
                $self->{context}->popup(undef, undef, undef, undef, $event->button, $event->time)
                    if $event->button eq 3;
                $widget->grab_focus;
            });

        $eventBox->signal_handler_disconnect($item->{keyHandler})
            if $item->{keyHandler};

        $item->{keyHandler} = $eventBox->signal_connect('key-press-event' => sub {
                my ($widget, $event) = @_;
                my $idx = $i;
                my $key = Gtk2::Gdk->keyval_name($event->keyval);
                if ($key eq 'Delete')
                {
                    $self->{parent}->deleteCurrentItem;
                    return 1;
                }
                if (($key eq 'Return') || ($key eq 'space'))
                {
                    $self->{parent}->displayInWindow;
                    return 1;
                }
                $idx++ if $key eq 'Right';
                $idx-- if $key eq 'Left';
                $idx += $self->{columns} if $key eq 'Down';
                $idx -= $self->{columns} if $key eq 'Up';
                return 1 if ($idx < 0) || ($idx >= scalar @{$self->{boxes}});
                my $column = $idx % $self->{columns};
                $self->select($idx);
                $self->{parent}->display($idx);
                $self->{boxes}->[$idx]->grab_focus;
                $self->showCurrent if $key eq 'Down'
                                   || $key eq 'Up'
                                   || (($key eq 'Left') && ($column == ($self->{columns} - 1)))
                                   || (($key eq 'Right') && ($column == 0));
                return 1;
                
            });

        if (($self->{number} % $self->{columns}) == 0)
        {
            #New row begin
            $self->{currentRow} = new Gtk2::HBox(0,0);
            $self->{list}->pack_start($self->{currentRow},0,0,0);
            $self->{currentRow}->show_all if ! $self->{initializing};
        }
        $self->{currentRow}->pack_start($eventBox,0,0,0);

        push @{$self->{boxes}}, $eventBox;
        
        $self->{number}++;
    }
    
    sub grab_focus
    {
        my $self = shift;
        $self->SUPER::grab_focus;
        $self->{boxes}->[$self->{current}]->grab_focus;
    }
    
    sub removeItem
    {
        my ($self, $number, $id) = @_;

        $self->{cache}->{$id}->{eventBox}->destroy;
        delete $self->{cache}->{$id};
        #$self->select($number);

        #Reload needed
        return (1, ($number < (scalar(@{$self->{boxes}}) - 1) ? $number : ($number - 1)));
    }
        
    sub select
    {
        my ($self, $idx, $id, $init) = @_;
        $idx = 0 if $idx == -1;
        my @boxes = @{$self->{boxes}};

        return $idx if ! scalar(@boxes);
        
        my $previous = $self->{current};
        $self->{current} = $idx;
        
        if (! $self->{withImage})
        {
            $boxes[$previous]->modify_bg('normal', $self->{inactiveBg});  
            $boxes[$self->{current}]->modify_bg('normal', $self->{activeBg});
        
        }
        $boxes[$previous]->child->set_from_pixbuf($self->{previousPixbuf})
            if $self->{previousPixbuf} && $boxes[$previous]->child;
        my $pixbuf = $boxes[$self->{current}]->child->get_pixbuf;
        $self->{previousPixbuf} = $pixbuf->copy;
        
        $pixbuf->saturate_and_pixelate($pixbuf, 1.8, 0);

#        $self->{selectedPixbuf}->composite($pixbuf,
#                           0, 0, 
#                           $self->{vboxWidth}, $self->{vboxHeight},
#                           0, 0,
#                           1, 1, 
#                           'nearest', 255)
#            if $self->{withImage};
            
        $boxes[$self->{current}]->child->set_from_pixbuf($pixbuf);
        
        $self->{previous} = $previous;

        return $idx;        
    }
    
    sub showCurrent
    {
        my $self = shift;
 
        if ($self->{initializing})
        {
            Glib::Timeout->add(100 ,\&showCurrent, $self);
            return;
        }
 
        my $adj = $self->{scroll}->get_vadjustment;
        my $totalRows = int $self->{number} / $self->{columns};
        my $row = (int $self->{current} / $self->{columns}) - 1;

        $adj->set_value((($adj->upper - $adj->lower) / $totalRows) * $row) if $totalRows > 0;
    }

    sub changeCurrent
    {
        my ($self, $previous, $new) = @_;
        #To ease comparison, do some modifications.
        #empty borrower is equivalent to 'none'.

        $previous->{$self->{borrowerField}} = 'none' if $previous->{$self->{borrowerField}} eq '';
        $new->{$self->{borrowerField}} = 'none' if $new->{$self->{borrowerField}} eq '';
        $self->{tooltips}->set_tip($self->{boxes}->[$self->{current}], $self->{parent}->transformTitle($new->{$self->{titleField}}), '')
            if ($new->{$self->{titleField}} ne $previous->{$self->{titleField}});
        return 0 if ($previous->{$self->{coverField}} eq $new->{$self->{coverField}}) && ($previous->{$self->{borrowerField}} eq $new->{$self->{borrowerField}});
        my ($image, $borrower) = ($new->{$self->{coverField}}, $new->{$self->{borrowerField}});

        my @boxes = @{$self->{boxes}};
        
        my $displayedImage = GCUtils::getDisplayedImage($image,
                                                        $self->{defaultImage},
                                                        $self->{parent}->{options}->file);
        
        my $pixbuf = $self->createPixbuf($displayedImage, $borrower);

        $self->{previousPixbuf} = $pixbuf->copy;
        $pixbuf->saturate_and_pixelate($pixbuf, 1.8, 0);
#        $self->{selectedPixbuf}->composite($pixbuf,
#                                           0, 0, 
#                                           $self->{vboxWidth}, $self->{vboxHeight},
#                                           0, 0,
#                                           1, 1, 
#                                           'nearest', 255)
#            if $self->{withImage};
        $boxes[$self->{current}]->child->set_from_pixbuf($pixbuf);

        #No reload needed
        return 0;
    }    
}

{
    package GCDetailedList;
    
    use File::Basename;
    use base "Gtk2::ScrolledWindow";

    sub new
    {
        my ($proto, $parent) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new;
        bless ($self, $class);

        $self->{parent} = $parent;
        
        $self->{multi} = 1;
        
        #$parent->{options}->groupItems(1) if ! $parent->{options}->exists('groupItems');
        #$self->{groupItems} = $parent->{options}->groupItems;
        
        $self->{imgWidth} = 60;
        $self->{imgHeight} = 80;
        
        $parent->{options}->listImgSize(2) if ! $parent->{options}->exists('listImgSize');
        my $size = $parent->{options}->listImgSize;
        $self->{factor} = 1;

        if ($size == 0)
        {
            $self->{factor} = 0.5;
        }
        elsif ($size == 1)
        {
             $self->{factor} = 0.8;
        }
        elsif ($size == 3)  
        {
            $self->{factor} = 1.5;
        }
        elsif($size == 4)
        {
            $self->{factor} = 2;
        }
        
        $self->{imgWidth} *= $self->{factor};
        $self->{imgHeight} *= $self->{factor};
        
        $self->{defaultImage} = $parent->{defaultImage};
        $self->{titleField} = $parent->{model}->{commonFields}->{title};
        $self->clearCache;
        
        $self->set_policy ('automatic', 'automatic');
        $self->set_shadow_type('none');

        $self->{preferences} = $parent->{model}->{preferences};
        $self->{preferences}->details($self->{titleField})
            if ! $self->{preferences}->details;
        my @tmpArray = split m/\|/, $self->{preferences}->details;
        $self->{fieldsArray} = \@tmpArray;
        
        $self->{imageIndex} = -1;
        my @columnsType;
        $self->{columnsArray} = [];
        $self->{columns} = {};
        my $col = 0;

        $self->{borrowerField} = $parent->{model}->{commonFields}->{borrower}->{name};
        
        $self->{collectionIndex} = -1;
        $self->{rankIndex} = -1;

        $self->{addCollection} = 0;        
        $self->{addRank} = 0;

        $self->setGroupingInformation;

        foreach my $field(@tmpArray)
        {
            my $title = $parent->{model}->{fieldsInfo}->{$field}->{displayed};
        
            my $renderer;
            my $attribute;
            if ($parent->{model}->{fieldsInfo}->{$field}->{type} eq 'image')
            {
                push @columnsType, 'Gtk2::Gdk::Pixbuf';
                $renderer = Gtk2::CellRendererPixbuf->new;
                $attribute = 'pixbuf';
                $self->{imageIndex} = $col;
            }
            elsif ($parent->{model}->{fieldsInfo}->{$field}->{type} eq 'yesno')
            {
                push @columnsType, 'Glib::Boolean';
                $renderer = Gtk2::CellRendererToggle->new;
                $attribute = 'active';
            }
            else
            {
                push @columnsType, 'Glib::String';
                $renderer = Gtk2::CellRendererText->new;
                $attribute = 'text';
                $self->{collectionIndex} = $col if ($field eq $self->{serieField});
                $self->{rankIndex} = $col if ($field eq $self->{rankField});
            }
            $self->{columns}->{$field} = Gtk2::TreeViewColumn->new_with_attributes($title, $renderer, 
                                                                                   ($attribute) ? ($attribute => $col) : ());
            $self->{columns}->{$field}->set_resizable(1);
            #$self->{columns}->{$field}->set_reorderable(1);
            if ($parent->{model}->{fieldsInfo}->{$field}->{type} eq 'image')
            {
                $self->{columns}->{$field}->set_clickable (0);
            }
            else
            {
                $self->{columns}->{$field}->signal_connect('clicked' => sub {
                    $parent->{items}->changeOrder($field);
                    $self->setSortOrder;
                });
#                $self->{columns}->{$field}->set_sort_column_id($col);
            }
            push @{$self->{columnsArray}}, $self->{columns}->{$field};
            $col++;
        }
        push @columnsType, 'Glib::Int';
        $self->{idxColumn} = $col;
        if ($self->{rankIndex} == -1)
        {
            if ($parent->{model}->{fieldsInfo}->{$self->{rankField}}->{type} eq 'number')
            {
                push @columnsType, 'Glib::Int';
            }
            else
            {
                push @columnsType, 'Glib::String';
            }
            $self->{addRank} = 1;
            $col++;
            $self->{rankIndex} = $col;
        }
        if ($self->{collectionIndex} == -1)
        {
            push @columnsType, 'Glib::String';
            $self->{addCollection} = 1;
            $col++;
            $self->{collectionIndex} = $col;
        }
        $self->{model} = new Gtk2::TreeStore(@columnsType);
        $self->{list} = Gtk2::TreeView->new_with_model($self->{model});
        $self->{list}->append_column($_) foreach (@{$self->{columnsArray}});
        
        $self->{list}->set_name('GCItemsDetailsList');
        $self->{list}->set_headers_clickable(1);
        $self->{list}->set_rules_hint(1);
        $self->{list}->set_reorderable(1);

        # Restore size of columns
        if ($self->{preferences}->exists('columnsWidths'))
        {
            my $i = 0;
            my @widths = split /\|/, $self->{preferences}->columnsWidths;
            foreach (@{$self->{columnsArray}})
            {
                $_->set_sizing('fixed');
                $_->set_resizable(1);
                $_->set_fixed_width($widths[$i] || 70);
                $i++;
            }
        }

#        $self->{list}->get_model->set_sort_column_id(0, 'ascending');

        if ($self->{generateMaster})
        {
            $self->{list}->set_expander_column($self->{columns}->{$self->{titleField}});
        }

        #$self->{list}->get_selection->set_mode ('multiple');
        
        $self->{list}->signal_connect (cursor_changed => sub {
            my ($sl, $path, $column) = @_;
            my $iter = $sl->get_selection->get_selected;
            $self->{selectedIterString} = $self->convertIterToString($iter);
            $self->{currentRemoved} = 0;
            $parent->display($self->convertIterToIdx($iter));
            $self->{currentIterString} = $self->{selectedIterString}
                if !$self->{currentRemoved};
            $self->checkLock;
        });

        $self->{list}->signal_connect ('row-activated' => sub {
           $parent->displayInWindow;
        });

        $self->add($self->{list});
        
        $self->{list}->signal_connect('button_press_event' => sub {
                my ($widget, $event) = @_;
                return 0 if $event->button ne 3;
                $self->{context}->popup(undef, undef, undef, undef, $event->button, $event->time);
                return 0;
        });

        $self->{list}->signal_connect('key-press-event' => sub {
                my ($widget, $event) = @_;
                my $key = Gtk2::Gdk->keyval_name($event->keyval);
                if ($key eq 'Delete')
                {
                    $self->{parent}->deleteCurrentItem;
                    return 1;
                }
                return 0;
        });

        if ($self->{groupItems})
        {
            my $targetEntryMove = {
                target => 'MY_ROW_TARGET',
                flags => ['same-widget'],
                info => 42,
            };
            
            $self->{list}->enable_model_drag_source('button1-mask','move', $targetEntryMove);
            $self->{list}->enable_model_drag_dest('move', $targetEntryMove);
    
            $self->{list}->signal_connect('drag_data_get' => sub {
                return 1;
            });
    
            $self->{list}->signal_connect('drag_data_received' => \&dropHandler, $self);
        }
        else
        {
            $self->{list}->unset_rows_drag_dest;
            $self->{list}->unset_rows_drag_source;
        }

        $self->reset;

        $self->show_all;
        return $self;
    }

    sub savePreferences
    {
        my ($self, $preferences) = @_;
        # We store it as a pipe separated list of pixel sizes
       
        my $value = '';
        $value .= $_->get_width.'|' foreach (@{$self->{columnsArray}});
        $preferences->columnsWidths($value);
    }

    sub couldExpandAll
    {
        my $self = shift;
        
        return 1;
    }

    sub expandAll
    {
        my $self = shift;
        
        $self->{list}->expand_all;
    }

    sub collapseAll
    {
        my $self = shift;
        
        $self->{list}->collapse_all;
    }

    sub setGroupingInformation
    {
        my $self = shift;

        $self->{serieField} = $self->{preferences}->groupBy;
        $self->{groupItems} = ($self->{serieField} ne '');
        $self->{generateMaster} = 0;
        return if !$self->{groupItems};
        if ($self->{parent}->{model}->{preferences}->masterType eq 'generated')
        {
            $self->{rankField} = $self->{parent}->{items}->{currentSortField};
            $self->{generateMaster} = 1;
        }
        else
        {
            $self->{rankField} = $self->{preferences}->orderBy;
            $self->{generateMaster} = 0;
        }
        if ($self->{parent}->{model}->{fieldsInfo}->{$self->{rankField}}->{type} eq 'number')
        {
            $self->{isRankNumeric} = 1;
        }
        else
        {
            $self->{isRankNumeric} = 0;
        }
        $self->{sortOrder} = $self->{parent}->{items}->{currentSortOrder};
    }

    sub dropHandler
    {
        my ($treeview, $context, $widget_x, $widget_y, $data, $info,$time, $self) = @_;
        my $source = $context->get_source_widget;
        return if ($source ne $treeview);
        my $model = $treeview->get_model;
        my ($targetPath, $targetPos) = $treeview->get_dest_row_at_pos($widget_x, $widget_y);
        if ($targetPath)
        {
            my $targetIter = $model->get_iter($targetPath);
            my $targetIdx = $self->convertIterToIdx($targetIter);
            my $origIter = $treeview->get_selection->get_selected;
            my $origIdx = $self->convertIterToIdx($origIter);
            #We cannot drop an item on itself
            if ($targetIter == $origIter)
            {
                $context->finish(1,0,$time);
                return;
            }
            my @origData;
            my $i = 0;
            foreach ($model->get_value($origIter))
            {
                push @origData, $i, $_;
                $i++;
            }
            if ($targetPos =~ /^into/)
            {
                if (($targetPath->get_depth > 1) || $model->iter_has_child($origIter))
                {
                    #We can't add an item to a collection item.
                    $context->finish(1,0,$time);
                }
                else
                {
                    #Creating a new collection item
                    my $newIter;
                    if ($self->{generateMaster})
                    {
                        # For auto-generated we do not allow user to drop on a single item
                        if (! $model->iter_has_child($targetIter))
                        {
                            $context->finish(1,0,$time);
                            return;
                        }
                        #return if ! $model->iter_has_child($origIter);
                        my $rank = $self->{parent}->{items}->getValue($origIdx, $self->{rankField});
                        $rank = $self->transformValue($rank, $self->{rankField});
                        $newIter = $self->insertAtPosition($targetIter, $rank, 1);
                        $self->removeParentIfNeeded($origIter);
                    }
                    else
                    {
                        $newIter = $model->append($targetIter);
                    }
                    my $collection = $self->{parent}->{items}->getValue($targetIdx, $self->{serieField});
                    #We use parent title as collection if there is none.
                    if (!$collection)
                    {
                        $collection = $self->{parent}->{items}->getValue($targetIdx, $self->{titleField});
                        $self->{parent}->{items}->setValue($targetIdx, $self->{serieField}, $collection);
                    }
                    $model->set($newIter,@origData);
                    $self->{parent}->{items}->setValue($origIdx, $self->{serieField}, $collection);
                    $self->updateRank($newIter, $self->{parent}->{items}->getMaxRank($collection) + 1);
                    $context->finish(1,1,$time);
                }
            }
            else
            {
                my $origPath = $model->get_path($origIter);
                if ($targetPath->get_depth == 1)
                {
                    if  ($origPath->get_depth == 1)
                    {
                        #Just moving a master item is not allowed
                        $context->finish(1,0,$time);
                    }
                    else
                    {
                        #We get an item out of a collection
                        #We change its values and requires a reload to put it as needed by sort.
                        $self->{parent}->{items}->setValue($origIdx, $self->{serieField}, '');
                        $self->updateRank($origIter, 0);
                        $self->{parent}->{items}->reloadList;
                    }
                }
                else
                {
                    #We are placing a collection item
                    my $newIter;
                    my $newRank = $self->getIterRank($targetIter);
                    if ($targetPos eq 'before')
                    {
                        $newIter = $model->insert_before($model->iter_parent($targetIter),
                                                         $targetIter);
                    }
                    else
                    {
                        $newIter = $model->insert_after($model->iter_parent($targetIter),
                                                         $targetIter);
                        $newRank++;
                    }
                    my $collection = $self->{parent}->{items}->getValue($targetIdx, $self->{serieField});
                    $model->set($newIter,@origData);
                    $self->{parent}->{items}->setValue($origIdx, $self->{serieField}, $collection);
                    $self->updateRank($newIter, $newRank);
                    $self->checkRanks($newIter);
                    $context->finish(1,1,$time);                    
                }
            }
            $self->select($origIdx);
            $self->{parent}->markAsUpdated;
            #$context->abort($time);
            #return 1;
        }
    }

    sub removeParentIfNeeded
    {
        my ($self, $iter) = @_;
        #Destroy the previous auto-generated item if there was only one child
        my $parentIter = $self->{model}->iter_parent($iter);
        if ($parentIter && ($self->{model}->iter_n_children($parentIter) <= 1))
        {
            $self->{model}->remove($parentIter);
        }
    }

    sub removeFromModel
    {
        my ($self, $iter) = @_;
        my $parentToRemove = undef;
        if ($self->{generateMaster})
        {
            my $parentIter = $self->{model}->iter_parent($iter);
            if ($parentIter)
            {
                if ($self->{model}->iter_n_children($parentIter) <= 1)
                {
                    $parentToRemove = $parentIter;
                }
                else
                {
                    # If we removed the 1st one, we should change the index of the master
                    # to be the new 1st
                    my $removedIdx = $self->convertIterToIdx($iter);
                    my $firstIdx = $self->convertIterToIdx($self->{model}->iter_children($parentIter));
                    if ($firstIdx == $removedIdx)
                    {
                        my $newIdx = $self->convertIterToIdx($self->{model}->iter_nth_child($parentIter, 1));
                        $self->{model}->set($parentIter, $self->{idxColumn}, $newIdx);
                    }
                }
            }
        }
        $self->{model}->remove($iter);
        $self->{model}->remove($parentToRemove) if $parentToRemove;        
    }

    sub updateRank
    {
        my ($self, $iter, $rank) = @_;
        
        return if ($self->{generateMaster});
        
        my $idx = $self->convertIterToIdx($iter);
        
        $self->{model}->set($iter, $self->{rankIndex}, $rank);
        $self->{parent}->{items}->setValue($idx, $self->{rankField}, $rank);
    }

    sub checkRanks
    {
        my ($self, $iter) = @_;
        
        my $previousRank = $self->getIterRank($iter);
        my $updatedIdx = $self->convertIterToIdx($iter);
        while ($iter = $self->{model}->iter_next($iter))
        {
            my $currentIdx = $self->convertIterToIdx($iter);
            next if $currentIdx == $updatedIdx;
            my $currentRank = $self->getIterRank($iter);
            if ($currentRank <= $previousRank)
            {
                $self->updateRank($iter, $previousRank + 1);
            }
            $previousRank = $self->getIterRank($iter);
        }
    }

    sub setSortOrder
    {
        my $self = shift;
        ($self->{sortOrder}, $self->{sortField}) = $self->{parent}->{items}->getSortOrder;

        foreach (keys %{$self->{columns}})
        {
            if ($_ eq $self->{sortField})
            {
                $self->{columns}->{$_}->set_sort_indicator(1);
                $self->{columns}->{$_}->set_sort_order($self->{sortOrder} ? 'ascending' : 'descending');
            }
            else
            {
                $self->{columns}->{$_}->set_sort_indicator(0);
            }
        }
    }

    sub getPixbufFromCache
    {
        my ($self, $path) = @_;
        
        my $realPath = GCUtils::getDisplayedImage($path,
                                                  $self->{defaultImage},
                                                  $self->{parent}->{options}->file);
        
        if (! $self->{cache}->{$realPath})
        {
            my $pixbuf;
            eval {
                $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file($realPath);
            };
            $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file($self->{defaultImage})
                if $@;

            $pixbuf = GCUtils::scaleMaxPixbuf($pixbuf,
                                              $self->{imgWidth},
                                              $self->{imgHeight},
                                              $self->{withImage});
            $self->{cache}->{$realPath} = $pixbuf;
        }
        return $self->{cache}->{$realPath};
    }

    sub clearCache
    {
        my $self = shift;
        $self->{cache} = {};
    }

    sub reset
    {
        my $self = shift;
        $self->{list}->set_model(undef);
        $self->{model}->clear;
        $self->{alreadyInserted} = {};
        $self->{waiting} = undef;
        $self->{currentIdx} = 0;
        $self->{nextItemIdx} = -1;

        $self->setGroupingInformation;
    }

    sub done
    {
        my $self = shift;

        $self->{list}->set_model($self->{model});
        # Add all the items without master
        if ($self->{generateMaster})
        {
            foreach my $childInfo(@{$self->getWaiting})
            {
                my $iter = $self->{model}->append(undef);
                $self->{model}->set($iter, @{$childInfo->[1]});
            }
        }
        else
        {
            foreach my $collection(keys %{$self->{waiting}})
            {
                foreach my $childInfo(@{$self->getWaiting($collection)})
                {
                    my $iter = $self->{model}->append(undef);
                    $self->{model}->set($iter, @{$childInfo->[1]});
                }
            }
        }
        delete $self->{waiting};
    }

    sub transformValue
    {
        my ($self, $value, $field, $isGeneratedMaster, $multiAllowed) = @_;
        
        my $type = $self->{parent}->{model}->{fieldsInfo}->{$field}->{type};

        
        if ($type eq 'image')
        {
            $value = ($isGeneratedMaster ? undef : $self->getPixbufFromCache($value));
        }
        else
        {
            if ($field eq $self->{borrowerField})
            {
                $value = $self->{parent}->{lang}->{PanelNobody} if (! $value) || ($value eq 'none');
            }
            else
            {
                if ($type =~ /list$/o)
                {
                    if ($multiAllowed && $self->{multi})
                    {
                        $value = GCPreProcess::multipleListToArray($value);
                    }
                    else
                    {
                        $value = GCPreProcess::multipleList($value, $type);
                    }
                }
                if ($field eq $self->{titleField})
                {
                   $value = $self->{parent}->transformTitle($value);
                }
            }
        }
        return $value;
    }

    sub getIterRank
    {
        my ($self, $iter) = @_;
        return ($self->{model}->get($iter))[$self->{rankIndex}];
    }

    sub getIterCollection
    {
        my ($self, $iter) = @_;
        return ($self->{model}->get($iter))[$self->{collectionIndex}];
    }
    
    sub convertIterToString
    {
        my ($self, $iter) = @_;
        return '' if ! $iter;

        return $self->{model}->get_string_from_iter($iter);
    }
    
    sub convertIterToIdx
    {
        my ($self, $iter) = @_;
        return 0 if ! $iter;
        
        return ($self->{model}->get($iter))[$self->{idxColumn}];
    }
    
    sub convertIdxToIter
    {
        my ($self, $idx) = @_;
        $self->{model}->foreach(sub {
            my ($model, $path, $iter) = @_;
            $self->{currentIterString} = $model->get_path($iter)->to_string;
            my $gotIt = (($model->get($iter))[$self->{idxColumn}] == $idx);
            return 1 if (
                          (($model->get($iter))[$self->{idxColumn}] == $idx)
                          &&
                            (
                              (!$self->{generateMaster})
                              ||
                              ($self->{generateMaster} && !$model->iter_n_children($iter))
                            )
                         );
        });
    }
    
    sub getCurrentId
    {
        my $self = shift;
        
        return $self->convertIterToIdx($self->getCurrentIter);
    }
    
    sub getCurrentIter
    {
        my $self = shift;

        return $self->{model}->get_iter_first if ! $self->{currentIterString};
        return $self->{model}->get_iter_from_string($self->{currentIterString});
    }    

    sub insertAtPosition
    {
        my ($self, $master, $rank, $forceSort) = @_;
        my $childIter;
        if ($master && ($self->{generateMaster}) && !$forceSort)
        {
            # Already sorted by data container
            $childIter = $self->{model}->append($master);
        }
        else
        {
            my $nextIter = $self->{model}->iter_nth_child($master, 0);
            if ($nextIter && $self->isRankAfter($rank, $self->getIterRank($nextIter)))
            {
                while ($nextIter)
                {
                    last if ($self->isRankAfter($self->getIterRank($nextIter), $rank));
                    $nextIter = $self->{model}->iter_next($nextIter);
                }
            }
            if ($nextIter)
            {
                $childIter = $self->{model}->insert_before($master, $nextIter);                 
            }
            else
            {
                $childIter = $self->{model}->append($master);
            }
        }
        return $childIter;
    }

    sub findMaster
    {
        my ($self, $collection) = @_;
        my $master = $self->{model}->get_iter_first;
        
        while ($master)
        {
            last if $self->getIterCollection($master) eq $collection;
            $master = $self->{model}->iter_next($master);
        }
        return $master;
    }

    sub removeInCollection
    {
        my ($self, $collec, $idx) = @_;
        my $master = $self->findMaster($collec);
        my $iter = $self->{model}->iter_nth_child($master, 0);
        while ($iter)
        {
            last if $idx == $self->convertIterToIdx($iter);
            $iter = $self->{model}->iter_next($iter);
        }
        if ($iter)
        {
            my $removedCurrent = ($self->{selectedIterString} eq $self->{currentIterString});
            $self->removeFromModel($iter);
            $self->{currentRemoved} = $removedCurrent;
        }
    }

    sub isRankAfter
    {
        my ($self, $rank1, $rank2) = @_;
        use locale;
        if ($self->{isRankNumeric})
        {
            if ($self->{generateMaster} && ! $self->{sortOrder})
            {
                return 1 if $rank1 < $rank2;
            }
            else
            {
                return 1 if $rank1 > $rank2;
            }
        }
        else
        {
            if ($self->{generateMaster} && ! $self->{sortOrder})
            {
                return 1 if $rank1 lt $rank2;
            }
            else
            {
                return 1 if $rank1 gt $rank2;
            }
        }
        return 0;
    }

    sub addToWaiting
    {
        my ($self, $data, $collection, $rank) = @_;
        if ($self->{generateMaster})
        {
            push @{$self->{waiting}}, $data;
        }
        else
        {
            push @{$self->{waiting}->{$collection}->{$rank}}, $data;
        }
    }

    sub getWaiting
    {
        my ($self, $collection) = @_;
        if ($self->{generateMaster})
        {
            return $self->{waiting} if $self->{waiting};
            return [];
        }
        else
        {
            my @waitings;
            sub compare
            {
                use locale;
                return 1 if ($self->{generateMaster});
                return ($self->{isRankNumeric} ? ($a <=> $b) : ($a cmp $b));
            }
            foreach my $childRank(sort compare keys %{$self->{waiting}->{$collection}})
            {
                push @waitings, @{$self->{waiting}->{$collection}->{$childRank}};
            }
            return \@waitings;
        }
        
    }

    sub removeWaiting
    {
        my ($self, $collection) = @_;
        delete $self->{waiting}->{$collection};
    }

    sub createRowsData
    {
        my ($self, $info, $collection, $rank, $idx) = @_;
        my @data;
        my $col = 0;
        
        foreach my $field(@{$self->{fieldsArray}})
        {
            my $value = $self->transformValue($info->{$field}, $field, $info->{isGeneratedMaster});
            push @data, $col, $value;
            $col++;
        }
        push @data, $col, $idx;
        $col++;
        if ($self->{addRank})
        {
            push @data, $col, $rank;
            $col++;
        }
        push @data, $col, $collection if $self->{addCollection};
        return @data;
    }

    sub addItem
    {
        my ($self, $info, $immediate) = @_;

        $self->{nextItemIdx}++;

        my $collection = $self->transformValue($info->{$self->{serieField}}, $self->{serieField},
                                               0, $self->{groupItems});
        my $rank = $self->transformValue($info->{$self->{rankField}}, $self->{rankField});

        #Creating data;
        my @data = $self->createRowsData($info, $collection, $rank, $self->{nextItemIdx});

        if (($collection eq '') || (! defined $collection) 
         || (!$self->{groupItems}) || ((ref($collection) eq 'ARRAY') && (! scalar @$collection)))
        {
            #Simple entry
            if ($self->{generateMaster})
            {
                #All the simple entry are added on the end
                if ($immediate)
                {
                    $self->{model}->set($self->{model}->append(undef), @data);
                }
                else
                {
                    $self->addToWaiting([$self->{nextItemIdx}, \@data]);
                }
            }
            else
            {
                $self->{model}->set($self->{model}->append(undef), @data);
            }
            return;
        }
        
        if (ref($collection) ne 'ARRAY')
        {
            my @array = ($collection);
            $collection = \@array;
        }
        foreach (@$collection)
        {
            if (exists $self->{alreadyInserted}->{$_})
            {
                #Master already exists
                my $master;
                if ($self->{generateMaster})
                {
                    $master = $self->findMaster($_);
                }
                else
                {
                    $master = $self->{model}->get_iter_from_string($self->{alreadyInserted}->{$_});
                }
                my $childIter = $self->insertAtPosition($master, $rank);
                $self->{model}->set($childIter, @data);
            }
            elsif (!$self->{parent}->{items}->isMaster($info))
            {
                #No master and we are a child;
                if ($self->{generateMaster})
                {
                    #Create the master
                    my $master = $self->insertAtPosition(undef, $_);
                    $self->{alreadyInserted}->{$_} = 1;
                    my $masterName = $_;
                    my %info = (
                                   $self->{titleField} => $masterName,
                                   $self->{serieField} => $_,
                                   isGeneratedMaster => 1
                               );
                    my @masterData = $self->createRowsData(\%info, $_, $rank, $self->{nextItemIdx});
                    $self->{model}->set($master, @masterData);
                    #Insert the child
                    my $childIter = $self->insertAtPosition($master, $rank);
                    $self->{model}->set($childIter, @data);
                }
                else
                {
                    #Wait for a real master
                    $self->addToWaiting([$self->{nextItemIdx}, \@data], $_, $rank);
                }
            }
            else
            {
                #A new master
                my $iter = $self->{model}->append(undef);
                $self->{alreadyInserted}->{$_} = $self->{model}->get_path($iter)->to_string;
                $self->{model}->set($iter, @data);
                foreach my $childInfo(@{$self->getWaiting($_)})
                {
                    $self->{model}->set($self->{model}->append($iter), @{$childInfo->[1]});
                }
                $self->removeWaiting($_);
            }
        }
    }
    
    sub removeItem
    {
        my ($self, $number, $prevId, $info) = @_;
       
        if ($self->{cache}->{$prevId})
        {
            $self->{cache}->{$prevId}->destroy;
            delete $self->{cache}->{$prevId};
        }

        my $parentToRemove = undef;
        my $currentIter = $self->getCurrentIter;
        if (! $self->{generateMaster})
        {
            #Do a reload in some situations
            return 1
                ## We have a collection 
                if ($info->{$self->{serieField}})
                &&
                # And it is the previous master
                $self->{parent}->{items}->isMaster($info);
        }
        
        #Shift backward all following items.
        $self->{model}->foreach(sub {
            my ($model, $path, $iter) = @_;
            my $currentIdx = ($model->get($iter))[$self->{idxColumn}];
            if ($currentIdx > $number)
            {
                $model->set($iter, $self->{idxColumn}, $currentIdx - 1);
            }
        });

        my $nextIter = $self->{model}->iter_next($currentIter);
        if (!$nextIter)
        {
            my $parentIter = $self->{model}->iter_parent($currentIter);
            if ($parentIter)
            {
                $nextIter = $self->{model}->iter_next($parentIter);
            }
            if (!$nextIter)
            {
                $nextIter = $self->{model}->iter_nth_child($parentIter,
                                                           $self->{model}->iter_n_children($parentIter) - 2);
            }
        }
        $self->removeFromModel($currentIter);

        my $newIdx = $self->convertIterToIdx($nextIter);
        $self->select($newIdx);
        
        #Reload not needed
        return (0, $newIdx);
    }

    sub select
    {
        my ($self, $idx, $id, $init) = @_;
        my $currentIter;
        if ($idx == -1)
        {
            $self->{currentIterString} = '0';
            $currentIter = $self->{model}->get_iter_first;
            $idx = $self->convertIterToIdx($currentIter);
        }
        else
        {
            $self->convertIdxToIter($idx);
            $self->{currentIterString} = '0' if ! $self->{currentIterString};
            $currentIter = $self->getCurrentIter;
        }
        #$self->{currentIter} = $self->{model}->get_iter_from_string($self->{currentIterString});
        return if !$currentIter;
        if ($init)
        {
            my $parent = $self->{model}->iter_parent($currentIter);
            if ($parent)
            {
                my $treePath = $self->{model}->get_path($parent);
                if (!$self->{list}->row_expanded($treePath))
                {
                    $self->{currentIterString} = $self->convertIterToString($parent);
                    $idx = $self->getCurrentId;
                }
            }    
        }
        else
        {
            $self->{list}->expand_to_path(Gtk2::TreePath->new($self->{currentIterString}))
                if $self->{currentIterString} =~ /:/;
        }
        #$self->showCurrent;
        #Lock panel if we are on a master
        $self->checkLock($currentIter);
        
        $self->{list}->get_selection->select_iter($currentIter) if ($currentIter);
        
        return $idx;
    }
    
    sub checkLock
    {
        my ($self, $iter) = @_;
        $iter = $self->getCurrentIter if !$iter;
        if (($self->{model}->iter_n_children($iter) > 0)
         && ($self->{generateMaster}))
        {
            $self->{parent}->{panel}->lock(1);
        }
        else
        {
            $self->{parent}->{panel}->lock(0);
        }
    }
    
    sub showCurrent
    {
        my $self = shift;
        
   		my $path = $self->{list}->get_selection->get_selected_rows;
		$self->{list}->scroll_to_cell($path) if $path;
    }
    
    sub changeCurrent
    {
        my ($self, $previous, $new) = @_;
        my $hasChanged = 0;
        my $reloadNeeded = 0;
        my @data = ();
        my $col = 0;
        my $currentIter = $self->getCurrentIter;
        return if !$currentIter;
        my $currentDepth = $self->{model}->get_path($currentIter)->get_depth;
        
        if (($self->{generateMaster})
         && ($currentDepth == 1)
         && $self->{model}->iter_has_child($currentIter))
        {
            # We do nothing for generated masters
            return 0;
        }
        
        foreach my $field(@{$self->{fieldsArray}})
        {
            my $newValue = $self->transformValue($new->{$field}, $field);
            my $oldValue = $self->transformValue($previous->{$field}, $field);
            if ($newValue ne $oldValue)
            {
                $hasChanged = 1;
                $reloadNeeded = 1 if ($field eq $self->{sortField});
            }
            push @data, $col, $newValue;
            $col++;
        }
        my $idx = $self->getCurrentId;
        push @data, $col, $idx;
        $col++;
        my $previousRank = $self->transformValue($previous->{$self->{rankField}}, $self->{rankField});
        my $newRank = $self->transformValue($new->{$self->{rankField}}, $self->{rankField});
        if ($self->{addRank})
        {
            push @data, $col, $newRank;
            $col++;
            $hasChanged = 1 if ($newRank != $previousRank);
        }
        my $previousCollection = $self->transformValue($previous->{$self->{serieField}}, $self->{serieField});
        my $newCollection = $self->transformValue($new->{$self->{serieField}}, $self->{serieField});
        if ($self->{addCollection})
        {
            push @data, $col, $newCollection;
            $col++;
            $hasChanged = 1 if ($newCollection !~ /^$previousCollection$/i);
        }
        return 0 if !$hasChanged;

        if ($self->{groupItems})
        {
            if ($previousCollection && (!$newCollection))
            {
                #We get an item out of a collection. It needs to be sorted
                $reloadNeeded = 1;
            }
            elsif (
                   (! $self->{generateMaster})
                   &&
                   $newCollection
                   &&
                   (
                    (
                     ($currentDepth == 1) &&
                     (! $self->{parent}->{items}->isMaster($new))
                    )
                    ||
                    (
                     ($currentDepth == 2) &&
                     ($self->{parent}->{items}->isMaster($new)) &&
                     ($newRank != $previousRank)
                    )
                   )
                  )
            {
                #Master has been changed
                $reloadNeeded = 1;
            }
            elsif (($previousCollection ne $newCollection)
                || ($previousRank != $newRank))
            {
                my $previousCollectionArray;
                my $newCollectionArray;
                #An item is integrated or moved into a collection
                #First we find its master
                my @parents;
                if ($previousCollection eq $newCollection)
                {
                    #Just moving
                    $parents[0] = $self->{model}->iter_parent($currentIter);
                }
                else
                {
                    #Changing collection
                    $previousCollectionArray = $self->transformValue($previous->{$self->{serieField}}, $self->{serieField}, 0, $self->{groupItems});
                    $newCollectionArray = $self->transformValue($new->{$self->{serieField}}, $self->{serieField}, 0, $self->{groupItems});
                    if (ref($previousCollectionArray) ne 'ARRAY')
                    {
                        my @array = ($previousCollectionArray);
                        $previousCollectionArray = \@array;
                    }
                    if (ref($newCollectionArray) ne 'ARRAY')
                    {
                        my @array = ($newCollectionArray);
                        $newCollectionArray = \@array;
                    }
                    
                    foreach my $collec(@$newCollectionArray)
                    {
                        push @parents, $self->findMaster($collec);
                        if ($self->{generateMaster})
                        {
                            if (!$parents[-1])
                            {
                                #We have to create a new parent
                                #Create the master
                                $parents[-1] = $self->insertAtPosition(undef, $newCollection);
                                $self->{alreadyInserted}->{$newCollection} = 1;
                                my %info = (
                                       $self->{titleField} => $collec,
                                       $self->{serieField} => $newCollection,
                                       isGeneratedMaster => 1
                                );
                                my @masterData = $self->createRowsData(\%info, $newCollection, '', $self->convertIterToIdx($currentIter));
                                $self->{model}->set($parents[-1], @masterData);
                            }
                            else
                            {
                                # The parent already exists
                                # Check if the child were already there

                                # If it was, we remove it from parents
                                pop @parents if GCUtils::inArrayTest($collec, @$previousCollectionArray);
                            }
                        }
                    }
                    
                    if ($self->{generateMaster})
                    {
                    }
                    
                }

                my $childIter = 0;
                foreach my $parent(@parents)
                {
                    #First we insert it at correct position
                    my $cIter = $self->insertAtPosition($parent, $newRank, 1);
                    $self->{model}->set($cIter, @data);
                    # We point to the 1st one
                    $childIter = $cIter if !$childIter;
                }
                
                my $removedCurrent = 0;
                # When master is generated, we already managed this
                if ($self->{generateMaster})
                {
                    # Now we remove the previous ones for generated master
                    foreach my $collec(@$previousCollectionArray)
                    {
                        $self->removeInCollection($collec, $idx)
                            if !GCUtils::inArrayTest($collec, @$newCollectionArray);
                    }
                }
                else
                {
                    #First we store if we are removing the one that has been selected
                    $removedCurrent = ($self->{selectedIterString} eq $self->{currentIterString});
                    $self->removeFromModel($currentIter);
                    $self->{currentRemoved} = $removedCurrent;
                }
                if ($childIter)
                {
                    my $childPath = $self->{model}->get_path($childIter);
                    my $rowReference = new Gtk2::TreeRowReference($self->{model}, $childPath);
                    $self->{currentIterString} = $rowReference->get_path->to_string;
                    $self->{list}->expand_to_path($childPath);
                    $self->{list}->get_selection->select_iter($childIter) if $removedCurrent;
                }
                return 0;
            }
        }
        $self->{model}->set($currentIter, @data) if ! $reloadNeeded;
        
        return $reloadNeeded;
    }
    
}


1;