#!/usr/bin/env python2.5
# -*- Mode: python; coding: utf-8 -*-
# 
# Copyright (C) 2007  Kjell Braden
#
# This file is part of GDecrypt.
#
# GDecrypt is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# any later version.
#
# GDecrypt is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with GDecrypt.  If not, see <http://www.gnu.org/licenses/>.

import sys, os, stat
import re
import dbus, dbus.mainloop.glib
import gobject
gobject.threads_init()
import pygtk
pygtk.require('2.0')
import gksu2
import gtk, gtk.glade
import gettext
import createui


LOG_ANY = -1
LOG_MOUNT = 0
LOG_DISMOUNT = 1


class gdecrypt:
	devices = {}
	udis = {}

	def hal_add_handler(self, udi, *args):
		volume = self.bus.get_object('org.freedesktop.Hal', udi)
		volume_iface = dbus.Interface( volume, "org.freedesktop.Hal.Device")
		# only care if we got a volume
		if volume_iface.QueryCapability("volume"):
			# don't care if the volume is mounted since it can't be encrypted in this case
			if volume_iface.GetProperty("volume.is_mounted"):
				return

			# add the device do the list
			device_file = volume_iface.GetProperty ('block.device')
			self.udis[udi] = "%s" % device_file
			size = volume_iface.GetProperty('volume.size')
			label = volume_iface.GetProperty('volume.label')
			if label:
				self.devices["%s" % device_file] = "%.2f GiB — %s"%((size/(1024*1024*1024.0)),label)
			else:
				self.devices["%s" % device_file] = "%.2f GiB"%((size/(1024*1024*1024.0)))

			self.partitionmodel.prepend(("%s" % device_file ,self.devices["%s" % device_file], None))

	def show_aboutdialog(self,*args):
		self.aboutdialog.show_all()

	def hal_rem_handler(self, udi, *args):
		if udi in self.udis:
			# it's a device in our list, let's kill it...
			model = self.wTree.get_widget("partitionboxentry").get_model()
			iter = model.get_iter_first()
			text = self.udis[udi]
			# iterate through the list and remove it
			while iter != None:
				if text == model.get_value(iter,0).split()[0]:
					self.wTree.get_widget("partitionboxentry").remove_text(int(model.get_string_from_iter(iter)))

					# assuming every device is listed only once we can safely break here
					break

				iter = model.iter_next(iter)
			del self.devices[self.udis[udi]]
			del self.udis[udi]

	def open_createui(self, widget):
		cui = createui.CreateUI()

	def __init__(self):
		gettext.install('gdecrypt')
		gtk.glade.bindtextdomain('gdecrypt')
		gtk.glade.textdomain('gdecrypt')

		self.gladefile = "main.glade"
		self.wTree = gtk.glade.XML(self.gladefile, "window1")
		self.aTree = gtk.glade.XML(self.gladefile, "aboutdialog1")
				
		self.aTree.signal_autoconnect({"gtk_widget_hide":gtk.Widget.hide_on_delete})
		self.wTree.signal_autoconnect({"on_window1_destroy":gtk.main_quit, 'on_mountbutton_clicked':self.mount_block, 'on_umountbutton_clicked':self.dismount_wrapper, 'infobutton1_activate_cb':self.show_aboutdialog, "kfexp_activate_cb":self.kfexp_activate_cb, "partitionboxentry_changed_cb":self.partitionboxentry_changed_cb, 'on_umountallbutton_clicked':self.dismount_all_wrapper, 'newbutton_activate_cb':self.open_createui})
		self.aboutdialog = self.aTree.get_widget("aboutdialog1")
		self.window = self.wTree.get_widget("window1")
		try:
			self.aboutdialog.set_logo(gtk.icon_theme_get_default().load_icon("gdecrypt",48,0))
			self.aboutdialog.set_icon(gtk.icon_theme_get_default().load_icon("gdecrypt",16,0))
			self.window.set_icon(gtk.icon_theme_get_default().load_icon("gdecrypt",16,0))
		except gobject.GError:
			pass

		# Creating several pixbufs instead of self.window.set_icon_from_file for suppressing size-warning
		pixbuf128 = gtk.icon_theme_get_default().load_icon("gdecrypt",128,0)
		pixbuf64 = pixbuf128.scale_simple(64,64,gtk.gdk.INTERP_BILINEAR)
		pixbuf48 = pixbuf128.scale_simple(48,48,gtk.gdk.INTERP_BILINEAR)
		pixbuf32 = pixbuf128.scale_simple(32,32,gtk.gdk.INTERP_BILINEAR)
		pixbuf16 = pixbuf128.scale_simple(16,16,gtk.gdk.INTERP_BILINEAR)
		self.window.set_icon_list(pixbuf16,pixbuf32,pixbuf48,pixbuf64,pixbuf128)

		if os.getuid() != 0:
			sys.stderr.write(_("Need root permissions for decrypting devices!\n"))
			dialog = gtk.Dialog(title=_("Root permissions needed!"), parent=self.window, flags=gtk.DIALOG_MODAL|gtk.DIALOG_NO_SEPARATOR, buttons=(gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
			dialog.connect("destroy",gtk.main_quit)
			dialog.connect("response",gtk.main_quit)
			dialog.set_resizable(False)
			permlabel = gtk.Label(_("Need root permissions for decrypting devices!\n")+_("Aborting now."))
			permlabel.set_padding(20,20)
			dialog.vbox.pack_start(permlabel)
			dialog.show_all()
			return

		# initialize dbus
		self.bus = dbus.SystemBus()
		self.hal_manager = self.bus.get_object('org.freedesktop.Hal', '/org/freedesktop/Hal/Manager')
		self.hal_man_iface = dbus.Interface(self.hal_manager, dbus_interface="org.freedesktop.Hal.Manager")
		# we want to know about all devices added or removed
		self.bus.add_signal_receiver(self.hal_add_handler, "DeviceAdded", "org.freedesktop.Hal.Manager", None, "/org/freedesktop/Hal/Manager")
		self.bus.add_signal_receiver(self.hal_rem_handler, "DeviceRemoved", "org.freedesktop.Hal.Manager", None, "/org/freedesktop/Hal/Manager")


		# make the widgets known
		self.mountbutton = self.wTree.get_widget("mountbutton")
		self.umountbutton = self.wTree.get_widget("umountbutton")

		# set combobox up
		self.partitionmodel = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING, gtk.gdk.Pixbuf)
		self.wTree.get_widget("partitionboxentry").set_model(self.partitionmodel)

		pbcell = gtk.CellRendererPixbuf()
		cell = gtk.CellRendererText()
		cell.set_property('xalign', 1)
		cell.set_property('xpad', 15)
		self.wTree.get_widget("partitionboxentry").pack_end(cell, True)
		self.wTree.get_widget("partitionboxentry").pack_end(pbcell, False)
		self.wTree.get_widget("partitionboxentry").add_attribute(cell, 'text', 1)
		self.wTree.get_widget("partitionboxentry").add_attribute(pbcell, 'pixbuf', 2)


		# initialize treestore
		self.umount_tree_store = gtk.TreeStore( gobject.TYPE_STRING, gobject.TYPE_STRING, gobject.TYPE_STRING, gobject.TYPE_STRING,
						gobject.TYPE_BOOLEAN )
		self.umount_view = self.wTree.get_widget("umount_tree")
		self.umount_view.set_model(self.umount_tree_store)
		
		self.umount_renderer = gtk.CellRendererText()
		self.umount_renderer_check = gtk.CellRendererToggle()
		self.umount_renderer_check.set_property('activatable', True)
		self.umount_renderer_check.connect( 'toggled', self.col1_toggled_cb, self.umount_tree_store )

		self.umount_view.insert_column_with_attributes(-1,_("ID"), self.umount_renderer, text=0)
		self.umount_view.insert_column_with_attributes(-1,_("Device"), self.umount_renderer, text=1)
		self.umount_view.insert_column_with_attributes(-1,_("Mountpoint"), self.umount_renderer, text=2)
		self.umount_view.insert_column_with_attributes(-1,_("Size"), self.umount_renderer, text=3)
		self.umount_view.insert_column_with_attributes(-1,_("Dismount"), self.umount_renderer_check, active=4)

		self.buffers = { LOG_DISMOUNT:self.wTree.get_widget('ulog'), LOG_MOUNT:self.wTree.get_widget('mlog')}

		self.window.show_all()
		
		self.has_tc_support = False
		self.has_luks_support = False
		
		for i in  os.environ['PATH'].split(":"):
			if os.access(os.path.join(i,"truecrypt"),os.X_OK):
				self.has_tc_support = True

		for i in  os.environ['PATH'].split(":"):
			if os.access(os.path.join(i,"cryptsetup"),os.X_OK):
				self.has_luks_support = True

		if not self.has_tc_support:
			self.log(_("No truecrypt support available (not found)!")+"\n", LOG_ANY)


		if not self.has_luks_support:
			self.log(_("No luks support available (cryptsetup not found)!")+"\n", LOG_ANY)
		

		self.partitionmodel.append(("", _("Select container file"), gtk.icon_theme_get_default().load_icon("gtk-open",24,0)))

		self.fill_devices()
		for k in sorted(self.devices, reverse=True):
			self.partitionmodel.prepend((k,self.devices[k], None))

		self.fill_mapped_list()

		# set the default mountpoint directory to /media since most devices should get mounted there
		self.wTree.get_widget("mpchoose").set_current_folder("/media")

		self.log("\n",LOG_ANY)

	def fill_devices(self):
		''' gets all volumes from HAL and lists them in the devices list '''
		volume_udi_list = self.hal_man_iface.FindDeviceByCapability ('volume')
		for udi in volume_udi_list:
			volume = self.bus.get_object ('org.freedesktop.Hal', udi)
			volume_iface = dbus.Interface( volume, "org.freedesktop.Hal.Device")
			# don't care if the volume is mounted since it can't be encrypted in this case
			if volume_iface.GetProperty("volume.is_mounted"):
				continue
			device_file = volume_iface.GetProperty ('block.device')
			size = volume_iface.GetProperty('volume.size')
			label = volume_iface.GetProperty('volume.label')
			if label:
				self.devices["%s" % device_file] = "%.2f MiB — %s"%((size/(1024*1024.0)),label)
			else:
				self.devices["%s" % device_file] = "%.2f MiB"%((size/(1024*1024.0)))

	def col1_toggled_cb( self, cell, path, model ):
		model[path][4] = not model[path][4]
		return

	def done(self, t, msg=""):
		if len(msg) > 0:
			self.log(_("done.")+" "+msg+"\n", t)
		else:
			self.log(_("done.")+"\n", t)
	def failed(self, t, msg=""):
		if len(msg) > 0:
			self.log(_("failed")+": "+msg+"\n", t)
		else:
			self.log(_("failed")+"."+"\n", t)

	def mount_block(self,*args):
		self.mountbutton.set_sensitive(False)
		self.umountbutton.set_sensitive(False)
		self.mount()
		self.mountbutton.set_sensitive(True)
		self.umountbutton.set_sensitive(True)
		
		self.wTree.get_widget("partitionboxentry").grab_focus()
		
		self.fill_mapped_list()

	def dismount_wrapper(self, *args):
		self.dismount()

	def dismount_all_wrapper(self, *args):
		self.dismount(all=True)

	def dismount(self, all=False):
		self.mountbutton.set_sensitive(False)
		self.umountbutton.set_sensitive(False)

		for i in range(len(self.umount_tree_store)):
			if all or self.umount_tree_store[i][4]:
				n = self.umount_tree_store[i][0]
				self.log(_("Dismounting %s...")%("/dev/mapper/"+n), LOG_DISMOUNT)
				if self.umount_tree_store[i][3] != "unknown":
					umount_err = os.popen3("umount /dev/mapper/"+n)[2].readlines()
					if len(umount_err) != 0:
						self.failed(LOG_DISMOUNT)
						for i in umount_err:
							self.log("\t"+i[:-1]+"\n", LOG_DISMOUNT)
						continue
				backing = self.umount_tree_store[i][1]
				
				is_luks = (os.popen3('vol_id -t '+backing)[1].readline() == 'crypto_LUKS\n')
				
				if is_luks:
					if self.has_luks_support:
						luks_err = os.popen3("cryptsetup luksClose "+n)[2].readlines()
						if len(luks_err) != 0:
							self.failed(LOG_DISMOUNT)
							for i in luks_err:
								self.log("\t"+i[:1]+"\n", LOG_DISMOUNT)
						else:
							self.done(LOG_DISMOUNT)
					else:
						self.failed(LOG_DISMOUNT,_("luks device but no luks support available!"))
				else:
					if self.has_tc_support:
						tc_err = os.popen3("truecrypt -d "+n[9:])[2].readlines()
						if len(tc_err) != 0:
							self.failed(LOG_DISMOUNT)
							for i in tc_err:
								self.log("\t"+i[:-1]+"\n", LOG_DISMOUNT)
						else:
							self.done(LOG_DISMOUNT)
					else:
						self.failed(LOG_DISMOUNT,_("Probably truecrypt device but no truecrypt support available!"))
		
		self.mountbutton.set_sensitive(True)
		self.umountbutton.set_sensitive(True)
		
		self.fill_mapped_list()

		
	def fill_mapped_list(self):
		self.umount_tree_store.clear()
		# get the mapped devices
		if self.has_tc_support:
			tc_list = os.popen3("truecrypt -l")[1].readlines()
		else:
			tc_list = []
		if self.has_luks_support:
			dm_table = os.popen("dmsetup table")
			dm_list = []
			for i in dm_table:
				ilist = i.split()
				if len(ilist) < 9:
					continue
				if i.split()[3] == "crypt": # is probably a luks device
					dm_list.append("/dev/mapper/"+i.split()[0][:-1])
		else:
			dm_list = []

		if len(tc_list) == 0 and len(dm_list) == 0:
			return
		
	
		# find the matching mountpoints
		mounted_list = os.popen("mount").readlines()
		if self.has_tc_support:
			for line in tc_list:
				tc_dev = line.split()
				# check if /dev/mapper/truecryptX is a link and if so, resolve it
				tc_num = tc_dev[0][12:]
				if os.path.islink(tc_dev[0]):
					result = os.readlink(tc_dev[0])
					tc_dev[0] = os.path.abspath(os.path.join(os.path.dirname(tc_dev[0]), result))
				
				# resolve udev symlinks
				if os.path.islink(tc_dev[1]):
					result = os.readlink(tc_dev[1])
					tc_dev[1] = os.path.abspath(os.path.join(os.path.dirname(tc_dev[1]), result))
			
				pat = re.compile("^"+re.escape(tc_dev[0])+" on ([^ ]+)", re.M)
				mounted_on = "unknown"
				for line in mounted_list:
					mo  = pat.search(line)
					if mo:
						mounted_on = mo.group(1)
						break
				size = None
				if tc_dev[1] in self.devices:
					size = self.devices[tc_dev[1]]
				else:
					print repr(os.stat(tc_dev[1]).st_size)
					size = '%.2f MiB'%((os.stat(tc_dev[1]).st_size)/(1024*1024.0))
				self.umount_tree_store.append(None, (tc_num,tc_dev[1],mounted_on,size, None) )
		if self.has_luks_support:
			for luks_dev in dm_list:
				luks_name = luks_dev[12:]
				
				volume_udi_list = self.hal_man_iface.FindDeviceStringMatch('block.device','/dev/mapper/'+luks_name)
				if len(volume_udi_list) == 0:
					continue

				volume = self.bus.get_object ('org.freedesktop.Hal', volume_udi_list[0])
				volume_iface = dbus.Interface( volume, "org.freedesktop.Hal.Device" )
				real_luks_device = volume_iface.GetProperty('block.device')
				udi_backing = volume_iface.GetProperty ('volume.crypto_luks.clear.backing_volume')
				
	
				backing = self.bus.get_object ('org.freedesktop.Hal', udi_backing)
				backing_path = backing.GetProperty("block.device", dbus_interface="org.freedesktop.Hal.Device")
			
				pat1 = re.compile("^"+re.escape(luks_dev)+" on ([^ ]+)", re.M)
				pat2 = re.compile("^"+re.escape(real_luks_device)+" on ([^ ]+)", re.M)
				mounted_on = "unknown"
				for line in mounted_list:
					mo1  = pat1.search(line)
					mo2  = pat2.search(line)
					if mo1:
						mounted_on = mo1.group(1)
						break
					elif mo2:
						mounted_on = mo2.group(1)
				if backing_path:
					self.umount_tree_store.append(None, (luks_name,backing_path,mounted_on,self.devices[backing_path], None) )
			
			
		return

	def mount(self):	
		try:
			part = self.wTree.get_widget("partitionboxentry").child.get_text()
			stat_result = os.stat(part)
		except IndexError: 
			self.log(_("No valid partition specified!")+"\n", LOG_MOUNT)
			return 1
		except OSError, inst: # for the stat() call
			if inst.__str__()[0:35] == "[Errno 2] No such file or directory":
				self.log(_("Partition not found")+inst.__str__()[35:]+"\n", LOG_MOUNT)
				return 1
			else:
				raise
		
		# check if mountpoint is available
		if not os.path.isdir(self.wTree.get_widget("mpchoose").get_filename()):
			os.mkdir(self.wTree.get_widget("mpchoose").get_filename())
		elif len(os.listdir(self.wTree.get_widget("mpchoose").get_filename())) != 0:
			self.log(_("Mountpoint %s is not empty!")%self.wTree.get_widget("mpchoose").get_filename()+"\n", LOG_MOUNT)
			return 1

		is_luks = (os.popen3('vol_id -t '+part)[1].readline() == 'crypto_LUKS\n')
		if is_luks:
			# luks needs a block-device
			if not stat.S_ISBLK(stat_result[stat.ST_MODE]):
				self.log(_("%s is not a block device!")%part+"\n", LOG_MOUNT)
				return 1
		elif not stat.S_ISBLK(stat_result[stat.ST_MODE]) and not stat.S_ISREG(stat_result[stat.ST_MODE]):
			self.log(_("%s is neither a block device nor a regular file!")%part+"\n", LOG_MOUNT)
			return 1


		# let's do the mapping
		self.log(_("Decrypting %s")%part+"...", LOG_MOUNT)

		if is_luks and not self.has_luks_support:
			self.failed(LOG_MOUNT,_("luks device but no luks support available!"))
			return 1
		if not is_luks and not self.has_tc_support:
			self.failed(LOG_MOUNT,_("Probably truecrypt device but no truecrypt support available!"))
			return 1

		kf_path = None
		kfexp = self.wTree.get_widget("kfexp")
		if kfexp.get_expanded():
			kf_path = self.wTree.get_widget("kfchoose").get_filename()
			if kf_path == None or not os.access(kf_path, os.R_OK):
				self.failed(LOG_MOUNT, _("Can't access keyfile %s!")%kf_path)
				return 1

		if not is_luks or not kfexp.get_expanded():
			gksu_context = gksu2.Context()
			gksu_context.set_message(_("Please enter the password for the encrypted partition %s:")%part)
			gksu_context.set_grab(True)
			try:
				password = gksu2.ask_password_full(gksu_context,_("Enter password"))
			except gobject.GError, e:
				if e.domain == "libgksu" and e.code == 11:
					self.log(_("aborted.")+"\n", LOG_MOUNT)
					pass
					return 1
				else:
					raise
		else:
			password = None
		if not is_luks:
			mapped_dev = self.tc_decrypt(part,password, kf_path)
		else:
			mapped_dev = self.luks_decrypt(part,password, kf_path)

		if mapped_dev == 1:
			# 1 means they had errors...
			return 1
		
		
		# mapping done, mount now
		# filesystem detection via vol_id --type
		self.log(_("Detecting filesystem..."), LOG_MOUNT)
		fs = os.popen3("vol_id --type "+mapped_dev)[1].readline()[:-1]
		self.done(LOG_MOUNT,_("Result:")+" "+fs)
		
		self.log(_("Mounting..."), LOG_MOUNT)
		if fs == "vfat":
			mount_fd = os.popen("mount -t vfat -o uid=0,gid=46,umask=007 "+mapped_dev+" "+self.wTree.get_widget("mpchoose").get_filename())
		elif fs == "ntfs":
			mount_fd = os.popen("mount -t ntfs-3g -o locale=de_DE.utf8,uid=0,gid=46,umask=007,force "+mapped_dev+" "+self.wTree.get_widget("mpchoose").get_filename())
		else:
			mount_fd = os.popen("mount "+mapped_dev+" "+self.wTree.get_widget("mpchoose").get_filename())
		
		mount_err = mount_fd.readlines()
		if len(mount_err) != 0:
			self.failed(LOG_MOUNT)
			for i in mount_err:
				self.log("\t"+i[:-1]+"\n", LOG_MOUNT)
			self.log(_("Cleaning up..."), LOG_MOUNT)
			if is_luks:
				os.popen("cryptsetup luksClose "+name.split("/")[-1])
			else:
				os.popen("truecrypt -d "+mapped_dev)
			self.done(LOG_MOUNT)
			return 1
		self.done(LOG_MOUNT)
		return 0


	def tc_decrypt(self, part, password, kf_path=None):
		''' Decrypts the given partition using the given password and truecrypt
		Returns the path to the mapped device'''
		kf_s = ""
		if kf_path != None:
			kf_s = _("and a keyfile")
		
		self.log(_("using truecrypt")+" "+kf_s+"...", LOG_MOUNT)
		if kf_path != None:
			(tc_pipe_in,tc_pipe_out) = os.popen4("truecrypt -k '"+kf_path+"' "+part)
		else:
			(tc_pipe_in,tc_pipe_out) = os.popen4("truecrypt "+part)
		tc_pipe_in.write(password+"\n")
		tc_pipe_in.close()

		# error detection. Not really the best way since we assume truecrypt (only) fails if it writes anything to stdout/stderr
		tc_err = tc_pipe_out.readlines()
		if len(tc_err) != 0:
			self.failed(LOG_MOUNT)
			for i in tc_err:
				if i != "\n":
					self.log("\t"+i[:-1]+"\n", LOG_MOUNT)
			return 1
		self.done(LOG_MOUNT)
		
		return os.popen("truecrypt -l | grep -- ' "+part+"$'").readline().split()[0]

	def luks_decrypt(self,part,password, kf_path=None):
		''' Decrypts the given partition using the given password and luks
		Returns the path to the mapped device'''
		name = part.split("/")[-1]
		kf_s = ""
		if kf_path != None:
			kf_s = _("and a keyfile")
		
		self.log(_("using luks")+" "+kf_s+"...", LOG_MOUNT)
		if kf_path != None:
			(luks_pipe_in,luks_pipe_out) = os.popen4("cryptsetup luksOpen --key-file '"+kf_path+"' "+part+" "+name)
		else:
			(luks_pipe_in,luks_pipe_out) = os.popen4("cryptsetup luksOpen "+part+" "+name)
			luks_pipe_in.write(password+"\n")
			luks_pipe_in.close()

		# error detection. Not really the best way since we assume cryptsetup (only) fails if it writes anything to stdout/stderr
		luks_err = luks_pipe_out.readlines()
		if luks_err[0] != "Command successful.\n":
			self.failed(LOG_MOUNT)
			for i in luks_err:
				if i != "\n":
					self.log("\t"+i[:-1]+"\n", LOG_MOUNT)
			return 1
		self.done(LOG_MOUNT)

		return "/dev/mapper/"+name

	def appendlog(self,logbuffer,text):
		logbuffer.insert(logbuffer.get_end_iter(), text)
	def scroll_to_end(self, logview):
		logview.scroll_to_mark(logview.get_buffer().get_insert(), 0)

	def log(self,text,t):
		lbufs = []
		if t == LOG_ANY:
			lbufs = self.buffers.values()
		else:
			lbufs = (self.buffers[t],)
		[ self.appendlog(b.get_buffer(), text) for b in lbufs ]
		[ self.scroll_to_end(b) for b in lbufs ]
			
	def kfexp_activate_cb(self, exp):
		if not exp.get_expanded(): # was just opened
			self.wTree.get_widget("kfl").set_text(_("Use keyfile"))
		else: # was just closed
			self.wTree.get_widget("kfl").set_text(_("Don't use keyfile"))
	def partitionboxentry_changed_cb(self, box):
		if box.get_active() < len(box.get_model()) -1:
			return
		dlg = gtk.FileChooserDialog(title=_("Select container file"), buttons=(gtk.STOCK_CANCEL,gtk.RESPONSE_CANCEL,gtk.STOCK_OPEN,gtk.RESPONSE_OK))
		if dlg.run() == gtk.RESPONSE_OK:
			box.child.set_text(dlg.get_filename())
		else:
			box.set_active(-1)
		dlg.destroy()
	
if __name__ == "__main__":
	# set the dbus mainloop
	dbus.mainloop.glib.DBusGMainLoop(set_as_default=True)

	gdc = gdecrypt()

	# parse the arguments
	if len(sys.argv) != 1:
		if len(sys.argv) != 3:
			sys.exit(1)
		gdc.wTree.get_widget("partitionboxentry").child.set_text(sys.argv[1])
		gdc.wTree.get_widget("mpchoose").set_filename(sys.argv[2])
		gdc.mountbutton.grab_focus()

	# run!
	gtk.main()
