from DataTarget import DataTarget
from utils.datatypes import *
from urllib import unquote # for filename conversion
from main import UNSET_COORD
import gtk



#
# Abstract class for DisplayTargets.
#
class DisplayTarget(gtk.HBox, DataTarget):

    # observer actions
    OBS_GEOMETRY = 0

    # user actions
    ACTION_CLICK = 0
    ACTION_DOUBLECLICK = 1
    ACTION_PRESS = 2
    ACTION_RELEASE = 3
    ACTION_MENU = 4
    ACTION_SCROLL = 5
    ACTION_ENTER = 6
    ACTION_LEAVE = 7
    ACTION_MOTION = 8
    ACTION_FILE_DROP = 9
    ACTION_LINK = 10

    # placement anchors
    ANCHOR_NW = "nw"
    ANCHOR_N = "n"
    ANCHOR_NE = "ne"
    ANCHOR_E = "e"
    ANCHOR_SE = "se"
    ANCHOR_S = "s"
    ANCHOR_SW = "sw"
    ANCHOR_W = "w"
    ANCHOR_CENTER = "center"


    # what we accept for drag & drop
    __DND_FILE = [('text/uri-list', 0, 0)]


    def __init__(self, parent, display):

        # the real coordinates and size of this widget
        self.__geometry = (0, 0, 0, 0)
        self.__old_geometry = (UNSET_COORD, UNSET_COORD, 0, 0)

        # coordinates and size of this widget as set by the user
        self.__user_geometry = (UNSET_COORD, UNSET_COORD,
                                UNSET_COORD, UNSET_COORD)

        # the widget to which this one is positioned relatively
        self.__relative = None

        # relative settings (is_rel_x, is_rel_y)
        self.__is_relative = (0, 0)

        # whether the coord values are percents
        self.__is_percent_coords = (0, 0)

        # whether the size values are percents
        self.__is_percent_size = (0, 0)

        # the placement anchor
        self.__anchor = self.ANCHOR_NW

        # the value of the last notify_handle_action call
        self.__had_action = 0


        DataTarget.__init__(self, parent, display)
        gtk.HBox.__init__(self)

        self._set_property_type("x", TYPE_COORD_SIZE)
        self._set_property_type("y", TYPE_COORD_SIZE)
        self._set_property_type("width", TYPE_COORD_SIZE)
        self._set_property_type("height", TYPE_COORD_SIZE)
        self._set_property_type("relative-to", TYPE_LIST)

        self._set_property_type("visible", TYPE_BOOL)

        self._set_property_type("on-enter", TYPE_STRING)
        self._set_property_type("on-leave", TYPE_STRING)
        self._set_property_type("on-click", TYPE_STRING)
        self._set_property_type("on-press", TYPE_STRING)
        self._set_property_type("on-release", TYPE_STRING)
        self._set_property_type("on-scroll", TYPE_STRING)
        self._set_property_type("on-file-drop", TYPE_STRING)
        self._set_property_type("on-doublelick", TYPE_STRING)
        self._set_property_type("on-motion", TYPE_STRING)
        self._set_property_type("on-menu", TYPE_STRING)

        self.show()




    #
    # Reacts on notifications by the relative of this widget.
    #
    def __on_observe_relative(self, src, cmd, *args):

        if (cmd == src.OBS_GEOMETRY):
            self.adjust_geometry()



    def __on_file_drop( self, widget, context, x, y, data, info, time ):
        ''' catch DND events and process them to send them to them
        main display, which forwards them to the sensor '''

        # get the display
        display = self._get_display()

        # tell the main display to send files and coordinates to the sensor
        files = [unquote(uri) for uri in data.data.split("\r\n") if uri != '']
        print "DROP", files, self
        display.send_action(self.ACTION_FILE_DROP, files)
        


    def set_config(self, key, value):

        if (key == "x"):
            v, is_percent = value
            self.set_user_geometry(x = v)
            self.set_percentual_state(x = is_percent)
            self.adjust_geometry()
            
        elif (key == "y"):
            v, is_percent = value
            self.set_user_geometry(y = v)
            self.set_percentual_state(y = is_percent)
            self.adjust_geometry()

        elif (key == "width"):
            v, is_percent = value
            self.set_user_geometry(width = v)
            self.set_percentual_state(width = is_percent)
            self.adjust_geometry()

        elif (key == "height"):
            v, is_percent = value
            self.set_user_geometry(height = v)
            self.set_percentual_state(height = is_percent)
            self.adjust_geometry()

        elif (key == "relative-to"):
            name, mode = value
            if (mode == "x"): self.__is_relative = (1, 0)
            elif (mode == "y"): self.__is_relative = (0, 1)
            elif (mode == "xy"): self.__is_relative = (1, 1)

            # no longer watch the old relative
            if (self.__relative):
                self.__relative.remove_observer(self.__on_observe_relative)

            self.__relative = self._get_parent().get_child_by_id(name)
            self.__relative.add_observer(self.__on_observe_relative)
            self.adjust_geometry()

        elif (key == "anchor"):
            x, y = self.get_user_geometry()[:2]
            self.__anchor = value
            self.adjust_geometry()

        elif (key == "visible"):
            if (value): self.show()
            else: self.hide()
            self.adjust_geometry()

        elif (key == "link"):
            self.set_action_call(self.ACTION_LINK, value)

        elif (key == "on-enter"):
            self.set_action_call(self.ACTION_ENTER, value)
        elif (key == "on-leave"):
            self.set_action_call(self.ACTION_LEAVE, value)
        elif (key == "on-click"):
            self.set_action_call(self.ACTION_CLICK, value)
        elif (key == "on-file-drop"):
            self.set_action_call(self.ACTION_FILE_DROP, value)
            self.drag_dest_set(gtk.DEST_DEFAULT_ALL, self.__DND_FILE,
                               gtk.gdk.ACTION_COPY)
            self.connect("drag_data_received", self.__on_file_drop)
        elif (key == "on-press"):
            self.set_action_call(self.ACTION_PRESS, value)
        elif (key == "on-release"):
            self.set_action_call(self.ACTION_RELEASE, value)
        elif (key == "on-doubleclick"):
            self.set_action_call(self.ACTION_DOUBLECLICK, value)
        elif (key == "on-motion"):
            self.set_action_call(self.ACTION_MOTION, value)
        elif (key == "on-scroll"):
            self.set_action_call(self.ACTION_SCROLL, value)
        elif (key == "on-menu"):
            self.set_action_call(self.ACTION_MENU, value)

        else:
            DataTarget.set_config(self, key, value)



    #
    # Returns the true coordinates of this target when the given coordinates
    # are the hotspot.
    #
    def get_anchored_coords(self, x, y, w, h):

        if (x == y == UNSET_COORD): return (x, y)

        anchor = self.__anchor
        if (anchor in [self.ANCHOR_NW, self.ANCHOR_W, self.ANCHOR_SW]):
            ax = x
        elif (anchor in [self.ANCHOR_N, self.ANCHOR_CENTER, self.ANCHOR_S]):
            ax = x - (w / 2)
        else:
            ax = x - w

        if (anchor in [self.ANCHOR_NW, self.ANCHOR_N, self.ANCHOR_NE]):
            ay = y
        elif (anchor in [self.ANCHOR_W, self.ANCHOR_CENTER, self.ANCHOR_E]):
            ay = y - (h / 2)
        else:
            ay = y - h

        return (ax, ay)


    #
    # Returns the position of the anchor in this target. Origin is the top
    # left corner.
    #
    def get_anchor(self):

        x, y, w, h = self.get_geometry()
        ax, ay = self.get_anchored_coords(x, y, w, h)
        dx = (x - ax)
        dy = (y - ay)

        return (dx, dy)


    #
    # Returns the target ID to which this target is placed relatively.
    #
    def get_relative_to(self):

        return self.__relative



    #
    # Returns the geometry (coordinates and size) of this target.
    #
    def get_geometry(self):

        x, y, w, h = self.__geometry
        if (self.get_property("visible")):
            return (x, y, w, h)
        else:
            return (x, y, 0, 0)


    #
    # Sets the user geometry of this target.
    #
    def set_user_geometry(self, x = None, y = None,
                          width = None, height = None):

        ox, oy, ow, oh = self.__user_geometry
        if (x == None): x = ox
        if (y == None): y = oy
        if (width == None): width = ow
        if (height == None): height = oh
        self.__user_geometry = (x, y, width, height)



    #
    # Returns the geometry from the user's point of view.
    #
    def get_user_geometry(self):

        return self.__user_geometry



    #
    # Sets the percentual value state of this target.
    #
    def set_percentual_state(self, x = None, y = None,
                             width = None, height = None):

        opx, opy = self.__is_percent_coords
        opw, oph = self.__is_percent_size
        if (x == None): x = opx
        if (y == None): y = opy
        if (width == None): width = opw
        if (height == None): height = oph
        self.__is_percent_coords = (x, y)
        self.__is_percent_size = (width, height)



    #
    # Returns the percentual state of this target, i.e. whether coordinates
    # or size values are percentual.
    #
    def get_percentual_state(self):

        is_px, is_py = self.__is_percent_coords
        is_pw, is_ph = self.__is_percent_size
        return (is_px, is_py, is_pw, is_ph)
        


    #
    # Adjusts the geometry of this target.
    #
    def adjust_geometry(self):

        #print "adjust", self
        x, y, w, h = self.__geometry #get_geometry()
        ux, uy, uw, uh = self.get_user_geometry()
        px, py, pw, ph = self.get_percentual_state()
        anchor_x, anchor_y = self.get_anchor()
        parent = self._get_parent()
        if (parent):
            parent_width, parent_height = \
                 self._get_parent().get_container_geometry()[2:]
        else:
            parent_width = gtk.gdk.screen_width()
            parent_height = gtk.gdk.screen_height()

        if (ux == UNSET_COORD): new_x = x + anchor_x #self._get_value_for_unset(x = x)
        elif (px): new_x = int(ux * (parent_width / 100.0))
        else: new_x = ux

        if (uy == UNSET_COORD): new_y = y + anchor_y #self._get_value_for_unset(y = y)
        elif (py): new_y = int(uy * (parent_height / 100.0))
        else: new_y = uy

        if (uw == UNSET_COORD): new_w = self._get_value_for_unset(width = w)
        elif (pw): new_w = int(uw * (parent_width / 100.0))
        else: new_w = uw

        if (uh == UNSET_COORD): new_h = self._get_value_for_unset(height = h)
        elif (ph): new_h = int(uh * (parent_height / 100.0))
        else: new_h = uh

        if (self.__relative):
            is_rel_x, is_rel_y = self.__is_relative

            # get the coords of the anchor of the relative
            anchor_x, anchor_y = self.__relative.get_anchor()
            rx, ry, rw, rh = self.__relative.get_geometry()

            # get the remaining amount of pixels from the anchor to the
            # bottom/right edge of the relative
            tw = rw - anchor_x
            th = rh - anchor_y

            if (ux != UNSET_COORD): new_x = rx + anchor_x + ux
            else: new_x = rx + anchor_x
            if (uy != UNSET_COORD): new_y = ry + anchor_y + uy
            else: new_y = ry + anchor_y
            if (is_rel_x): new_x += tw
            if (is_rel_y): new_y += th
        #end if

        # get coords of the top left corner
        new_x, new_y = self.get_anchored_coords(new_x, new_y, new_w, new_h)
        if ((not parent and self.window)
              or (new_x, new_y, new_w, new_h) != self.__old_geometry):
            self.__geometry = (new_x, new_y, new_w, new_h)
            self.__old_geometry = self.__geometry
            self.set_size_request(new_w, new_h)
            # there's no need to propagate the GEOMETRY of widgets with
            # width = 0 or height = 0
            #if (new_w != 0 and new_h != 0):
            self._propagate_geometry()


    #
    # Returns the geometry value that should be used for unset values.
    #
    def _get_value_for_unset(self, x = None, y = None,
                             width = None, height = None):

        if (x != None): return x
        elif (y != None): return y
        elif (width != None): return width
        elif (height != None): return height


    #
    # Tells other targets about geometry changes.
    #
    def _propagate_geometry(self):

        parent = self._get_parent()
        if (parent): gtk.idle_add(parent.adjust_geometry)
        self.update_observer(self.OBS_GEOMETRY)


    #
    # Sets the position of this target.
    #
    def set_position(self, x, y):

        ox, oy, w, h = self.__geometry
        if ((x, y) != (ox, oy)):
            self.__geometry = (x, y, w, h)
            self.adjust_geometry()

            
    #
    # Sets the size of this target. Use this instead of set_size_request() in
    # targets to set the size manually.
    #
    def set_size(self, width, height):

        x, y, w, h = self.__geometry
        if ((w, h) != (width, height)):
            self.__geometry = (x, y, width, height)
            self.adjust_geometry()


    def handle_action(self, action, px, py, path, args):

        if (action in [self.ACTION_MOTION]):
            x, y = self.get_pointer()
            args = [x, y] + list(args[2:])

        DataTarget.handle_action(self, action, px, py, path, args)


    #
    # Notifies the target whether the action occured on it or not.
    # This detects ENTER and LEAVE events.
    #
    def notify_handle_action(self, value, path):

        if (self.get_index() != -1): path.append(self.get_index())
        
        # enter notify
        if (not self.__had_action and value):
            action = self.ACTION_ENTER
            if (self.has_action(action)):
                call = self.get_action_call(action)
                self._get_display().call_sensor(* [call, path])
            
        # leave notify
        elif (self.__had_action and not value):
            action = self.ACTION_LEAVE
            if (self.has_action(action)):
                call = self.get_action_call(action)
                self._get_display().call_sensor(* [call, path])

        self.__had_action = value
