from ContainerTarget import ContainerTarget
from utils.Hash2D import Hash2D
from utils.datatypes import *
import layouters

import gtk



#
# Class for arrays of TargetDisplays.
#
class TargetArray(ContainerTarget):


    def __init__(self, parent, display):

        self.__zones = Hash2D()

        # the current layouter function
        self.__layouter = ["horizontal", "48"]

        # the class of the array elements
        self.__elementclass = None

        # the default settings of the array elements
        self.__elementsettings = None

        # children data of the array elements
        self.__elementchildren = []

        # the current elements of this array
        self.__children = []

        # mapping between unique IDs and targets
        self.__targets = {}

        
        ContainerTarget.__init__(self, parent, display)
        self.__layout = gtk.Fixed()
        self.__layout.show()
        self.add(self.__layout)

        self._set_property_type("length", TYPE_INT)
        self._set_property_type("layout", TYPE_LIST)


    def add_children(self, childrendata):

        name, self.__elementclass, self.__elementsettings, \
              self.__elementchildren = \
              childrendata[0]

        # the array should be empty at the beginning but we need to create a
        # child so the array can connect to sensor output
        self.__add_child()
        self.__remove_child()



    def __on_observe_children(self, src, cmd, *args):
            
        if (cmd == src.OBS_GEOMETRY and self.__children):
            x, y, w, h = src.get_geometry()
            self.__layout.move(src, x, y)
            self.__zones.set(x, y, x + w, y + h, src)



    #
    # Adds an element to this array.
    #
    def __add_child(self):

        child = self.__elementclass(self, self._get_display())
        self.__children.append(child)
        child.set_index(len(self.__children) - 1)

        if (self.__elementchildren):
            child.add_children(self.__elementchildren)
        
        if (self.__elementsettings):
            for key, value in self.__elementsettings.get_entries():
                child.set_config(key, value)

        cid = str(child)

        self.__layout.put(child, 0, 0)
        child.add_observer(self.__on_observe_children)

        self._register_child(child, cid)
        


    #
    # Removes an element from this array.
    #
    def __remove_child(self):
        assert(self.__children)

        child = self.__children.pop()
        self.__layout.remove(child)
        child.remove_observer(self.__on_observe_children)
        self.__zones.delete(child)

        self._unregister_child(child)



    #
    # Positions all elements of this array.
    #
    def __place_children(self):

        parts = self.__layouter
        layout = parts[0]
        args = parts[1:]

        # get the layouter function
        layouter = layouters.LAYOUTERS[layout]


        cnt = 0
        previous_id = ""
        for child in self.__children:
            x, y, w, h = child.get_geometry()
            cx, cy, relative = layouter(cnt, args)

            cx, cy = child.get_anchored_coords(cx, cy, w, h)

            # handle relative positioning for dynamic sized children
            rel = ""
            if (relative == layouters.RELATIVE_X): rel = "x"
            elif (relative == layouters.RELATIVE_Y): rel = "y"
            elif (relative == layouters.RELATIVE_XY): rel = "xy"

            if (not rel): child.set_position(cx, cy)

            if (previous_id and rel):
                child.set_config("relative-to", previous_id + ", " + rel)

            previous_id = self.get_id_by_child(child)
            cnt += 1
        #end for

        self.adjust_geometry()
        

        

    def set_config(self, key, value):

        value = self._convert_type(key, value)

        if (key == "layout"):
            self.__layouter = value
            self.__place_children()
            
        elif (key == "length"):
            length = int(value)
            if (len(self.__children) != length):
                while (length > len(self.__children)): self.__add_child()
                while (length < len(self.__children)): self.__remove_child()
            self.__place_children()
            
        else:
            ContainerTarget.set_config(self, key, value)




    def distribute_sensor_output(self, sensor, indexes, key, value):

        index = int(indexes.pop(0))
            
        # add children, if necessary
        if (index >= len(self.__children)):
            while (index >= len(self.__children)): self.__add_child()
            self.__place_children()

        entries = self.__targets.get((sensor, index, key), [])
        for target, prop in entries:
            if (indexes):
                target.distribute_sensor_output(sensor, indexes[:],
                                                key, value)
            else:
                target.set_config(prop, value)
        #end for


    def add_mapping(self, sensorplug, target, prop):

        ident, port = sensorplug.split(":")
        index = len(self.__children) - 1

        key = (ident, index, port)
        self.__targets.setdefault(key, []).append((target, prop))
        self._get_propagator().add_mapping(sensorplug, self, prop)
