from desktop.GlassWindow import GlassWindow
from DisplayContainer import DisplayContainer
from WindowSnapper import WindowSnapper
from main import ICON, UNSET_COORD

import gtk
import gobject


#
# Class for display windows.
#
class Window(GlassWindow, DisplayContainer):

    # mapping: str -> window type
    __WINDOW_FLAGS = {"below": GlassWindow.TYPE_KEEP_BELOW,
                      "above": GlassWindow.TYPE_KEEP_ABOVE,
                      "sticky": GlassWindow.TYPE_STICKY,
                      "managed": GlassWindow.TYPE_MANAGED}

    __window_snapper = WindowSnapper()


    def __init__(self, display):

        self.__display = display

        # ID of the idle handler for initial placement of the window
        self.__init_drag_handler = 0

        # window bbox as it was stored in the window snapper
        self.__window_bbox = (0, 0, 0, 0)

        # window position for detecting moves
        self.__window_pos = (UNSET_COORD, UNSET_COORD)
        self.__is_placed = 0

        # window size for detecting resizing
        self.__window_size = (0, 0)

        # temporary data used for dragging windows
        self.__is_dragging = 0
        self.__drag_offset = (0, 0)
        self.__dont_snap = 0

        self.__reentrance_lock = False
        self.__positioning_lock = False

        self.__shape = None


        GlassWindow.__init__(self, gtk.WINDOW_TOPLEVEL)
        DisplayContainer.__init__(self, display)
        
        self.set_size_request(-1, -1)
        self.set_default_size(10, 10)
        self.add(display)
        display.set_container(self)

        # set the icon
        self.set_icon(gtk.gdk.pixbuf_new_from_file(ICON))

        # set up event handlers
        self.connect("key-press-event", self.__on_key, 0)
        self.connect("key-release-event", self.__on_key, 1)
        self.connect("button-press-event", self.__on_button, 0)
        self.connect("button-release-event", self.__on_button, 1)
        self.add_events(gtk.gdk.BUTTON_PRESS_MASK |
                        gtk.gdk.BUTTON_RELEASE_MASK)

        # sawfish needs this
        self.realize()
        self.set_property("skip-taskbar-hint", 1)
        self.set_property("skip-pager-hint", 1)

        self._set_window_type(self.TYPE_STICKY) # | self.TYPE_KEEP_BELOW)

        # get into drag mode if the position won't be set before
        self.__init_drag_handler = gtk.idle_add(self.__begin_dragging)

        gtk.timeout_add(3000, self.__save_position)


    #
    # Periodically checks the window position and saves it if it has changed.
    # That way, users may drag the window with alt+drag on window managers
    # that support it.
    #
    def __save_position(self):

        if (self.__positioning_lock):
            self.__positioning_lock = False
            return True
        
        x, y = self.get_position()
        if (not self.__is_dragging and
            (x, y) != self.__window_pos != (UNSET_COORD, UNSET_COORD)):
            bx, by, bw, bh = self.__window_bbox
            self.__window_snapper.remove(bx, by, bw, bh)

            w, h = self.__window_size
            self.__window_snapper.insert(x, y, w, h)
            self.__window_bbox = (x, y, w, h)
            
            self.__window_pos = (x, y)
            self.__reentrance_lock = 1
            self.__display.save_position(x, y)
            self.__reentrance_lock = 0

        return True
            

    #
    # Begins a window dragging operation.
    #
    def __begin_dragging(self):

        assert(self.window)

        self.__is_dragging = 1
        x, y = self.get_pointer()
        w, h = self.__window_size
        if (not (0 <= x <= w)): x = w / 2
        if (not (0 <= y <= h)): y = h / 2
        self.__drag_offset = (x, y)
        self.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.FLEUR))

        bx, by, bw, bh = self.__window_bbox
        self.__window_snapper.remove(bx, by, bw, bh)

        gtk.timeout_add(20, self.__do_dragging)
        self.__display.set_sensitive(gtk.FALSE)

        # dig a hole into the display where the mouse pointer is, so that the
        # window will receive button events (what a funny hack! :D )
        if (w != 0 and h != 0):
            mask = gtk.gdk.Pixmap(None, w, h, 1)
            gc = mask.new_gc()
            gc.foreground = gtk.gdk.Color(0, 0, 0, 1)
            mask.draw_rectangle(gc, gtk.TRUE, 0, 0, w, h)
            gc.foreground = gtk.gdk.Color(0, 0, 0, 0)
            mask.draw_rectangle(gc, gtk.TRUE, x - 4 , y - 4, 8, 8)
            self.__display.shape_combine_mask(mask, 0, 0)
        
        return gtk.FALSE


    #
    # Ends a window dragging operation.
    #
    def __end_dragging(self):

        assert(self.window)

        if (self.__display.window):
            self.__display.window.shape_combine_mask(None, 0, 0)
        self.__is_dragging = 0
        self.window.set_cursor(None)
        self.__display.set_sensitive(gtk.TRUE)

        x, y = self.get_position() #self.__window_pos
        w, h = self.__window_size
        self.__window_size = (w, h)
        self.__window_snapper.insert(x, y, w, h)
        self.__is_placed = 1
        self.__window_bbox = (x, y, w, h)

        self.__reentrance_lock = 1
        self.__display.save_position(x, y)
        self.__reentrance_lock = 0

        return gtk.FALSE


    #
    # Drags the window.
    #
    def __do_dragging(self):

        if (self.__is_dragging):
            offx, offy = self.__drag_offset
            winx, winy = self.get_position()
            x, y = self.get_pointer()
            x += winx; y += winy

            new_x = x - offx
            new_y = y - offy
            if (self.__window_pos != (new_x, new_y)):
                self.__window_pos = (new_x, new_y)
                if (not self.__dont_snap):
                    w, h = self.__window_size
                    new_x, new_y = self.__window_snapper.snap(new_x, new_y,
                                                              w, h)
                self.move(new_x, new_y)

            return gtk.TRUE

        else:
            return gtk.FALSE



    def __on_key(self, src, event, is_release):
        
        if (not is_release):
            # don't snap while one of the SHIFT keys is pressed
            self.__dont_snap = (event.keyval in (0xffe1, 0xffe2))
        else:
            self.__dont_snap = 0



    def __on_button(self, src, event, is_release):

        if (not is_release and event.button == 2):
            self.__begin_dragging()

        elif (is_release and self.__is_dragging):
            self.__end_dragging()

            # set window shape after the window has been placed
            if (self.__shape):
                self.set_shape(self.__shape)
                self.__shape = None
            



    def close(self):

        wx, wy, w, h = self.__window_bbox
        self.__window_snapper.remove(wx, wy, w, h)
        self.destroy()


    def set_position(self, x, y):

        # don't run the initial positioning handler if a position has been
        # set
        if (self.__init_drag_handler):
            self.__is_dragging = False
            gobject.source_remove(self.__init_drag_handler)
            self.__init_drag_handler = 0
            self.__end_dragging()

        # break cycles
        if ((x, y) == self.__window_pos or self.__reentrance_lock): return

        ox, oy = self.__window_pos
        if ((x, y) != self.__window_pos):
            self.__positioning_lock = True
            gtk.idle_add(self.move, x, y)
            self.__window_pos = (x, y)
            bx, by, bw, bh = self.__window_bbox
            self.__window_snapper.remove(bx, by, bw, bh)
            w, h = self.__window_size
            self.__window_snapper.insert(x, y, w, h)
            self.__is_placed = 1
            self.__window_bbox = (x, y, w, h)
            self.__display.save_position(x, y)


    def is_placed(self):

        return self.__is_placed
    

    def set_size(self, width, height):

        bx, by, bw, bh = self.__window_bbox
        self.__window_snapper.remove(bx, by, bw, bh)
        self.__window_size = (width, height)
        self.resize(width, height)
        self.__window_snapper.insert(bx, by, width, height)
        self.__window_bbox = (bx, by, width, height)

        

    def set_window_flags(self, value):

        flags = 0
        value = value.split(",")
        for p in value:
            if (p.strip() in self.__WINDOW_FLAGS):
                flags |= self.__WINDOW_FLAGS[p.strip()]
            else:
                print p.strip(), "is not a valid window flag"
        self._set_window_type(flags)


    def set_shape(self, mask):

        if (self.__window_pos == (UNSET_COORD, UNSET_COORD)):
            self.__shape = mask
        else:
            self.shape_combine_mask(mask, 0, 0)
