#:section The [cmd:admin] module
#:para    The [cmd:admin] module provides several functions for remote controlling [app:gDesklets].
#         It's intended for use by 3rd party software.



from config.ConfigManager import ConfigManager
from utils import singleton
from utils import vfs

import random, time


DISPLAY_TYPE_WINDOW = "window"
DISPLAY_TYPE_PLUG = "plug"


#
#:function get_displays | profile = "" |
#          Returns a hashtable (id --> path) of all displays in the given profile.
#          If no parameter is passed, all displays of the current profile are returned.
#          If the profile isn't valid no displays at all are returned.
#:  param  profile | string | The name of the profile.
#:  return hashtable
#:/function
#
def get_displays(profile = ""):

    config = singleton.get(ConfigManager)

    if (profile == ""):
        displays = config.get("profiles", get_profile(), "main", "displays")
    else:
        if (profile in get_profiles()):
            displays = config.get("profiles", profile, "main", "displays")
        else:
            return {}

    ret = {}

    parts = displays.split(",")
    for p in parts:
        if (not p): continue

        try:
            id, path = p.split("::")
        except ValueError:
            # keep backwards compatibility
            id, path = p.split(":")
            
        if (vfs.exists(path) and not vfs.isdir(path)):
            ret[id] = path
    #end for

    return ret



#
#:function add_display | file, id = "" |
#          Adds the given display file.
#:  param file | string | File, i.e. [cmd:/path/to/file.display].
#:  param id   | string | ID of the file, if it's not given a new one will be created
#                         automatically.
#:  param displaytype | enum | The type of the display. One of
#                              DISPLAY_TYPE_WINDOW (the default) or
#                              DISPLAY_TYPE_PLUG (for displays embeddable
#                              using the XEMBED protocol).
#:/function
#
def add_display(file, id = "", displaytype = DISPLAY_TYPE_WINDOW):

    displays = get_displays()
    if (not id): id = new_id()
    if (vfs.exists(file) and not vfs.isdir(file)):
        displays[id] = file
        
    __set_displays(displays)
    config = singleton.get(ConfigManager)
    config.set("profiles", get_profile(), id, "type", displaytype)



#
#:function remove_display | id |
#          Removes display with the given ID.
#:  param id | string | ID for the display which has to be removed.
#:/function
#
def remove_display(id):

    displays = get_displays()
    del displays[id]
    __set_displays(displays)



#
#:function restart_display | id |
#          Used to restart the display with the given ID. It's deprecated and
#          no longer works. Use remove_display(id) + add_display(file, id) instead.
#:  param id | string | ID for the display which has to be restarted.
#:/function
#
def restart_display(id):

    print "restart_display() is deprecated and no longer works"



#
#:function get_display_type | ident |
#          Returns the display type of the given display.
#:  param id | string | ID of display.
#:  return display type
#:/function
#
def get_display_type(ident):

    config = singleton.get(ConfigManager)
    dtype = config.get("profiles", get_profile(), ident, "type")
    if (not dtype): dtype = DISPLAY_TYPE_WINDOW

    return dtype



#
#:function new_id | |
#          Returns a new unique ID.
#:  return string
#:/function
#
def new_id():

    return (str(time.time()) + \
            str(random.random())).replace(".", "").replace(",", "")



#
#:function get_geometry | id, for_nw = 0 |
#          Returns the geometry of a display as a (x, y, width, height) tuple.
#:  param id     | string | ID of display whose geometry should be returned.
#:  param for_nw | bool   | If True, then the coordinates of the topleft
#                           corner of the window are returned. Otherwise it
#                           will be the coordinates of the anchor.
#:  return (int, int, int, int)
#:/function
#
def get_geometry(id, for_nw = 0):

    key = id + "_default_"
    
    config = singleton.get(ConfigManager)
    if (not for_nw):
        x = int(config.get("profiles", get_profile(), key, "x"))
        y = int(config.get("profiles", get_profile(), key, "y"))
    else:
        x = int(config.get("profiles", get_profile(), key, "real-x") or -1)
        y = int(config.get("profiles", get_profile(), key, "real-y") or -1)

    w = int(config.get("profiles", get_profile(), key, "width"))
    h = int(config.get("profiles", get_profile(), key, "height"))
        
    return (x, y, w, h)



#
#:function set_position | |
#          Sets the position of a display.
#:  param id | string | ID of display whose position should be set.
#:  param x  | int    | x coordinate for the display
#:  param y  | int    | y coordinate for the display
#:/function
#
def set_position(id, x, y):

    key = id + "_default_"

    config = singleton.get(ConfigManager)
    config.set("profiles", get_profile(), key, "x", str(x))
    config.set("profiles", get_profile(), key, "y", str(y))



#
#:function get_profile | |
#          Returns the name of current profile.
#:  return string
#:/function
#
def get_profile():

    config = singleton.get(ConfigManager)
    profile = config.get("profile") or "default"

    return profile



#
#:function get_profiles | |
#          Returns a list of the available profiles.
#:  return string list
#:/function
#
def get_profiles():

    config = singleton.get(ConfigManager)
    profiles = config.list("profiles")

    return profiles



#
#:function set_profile | profile |
#          Sets the profile.
#:  param profile | string | Profile which should be set.
#:/function
#
def set_profile(profile):

    config = singleton.get(ConfigManager)
    config.set("profile", profile)



# private method for add_display and remove_display
def __set_displays(displays):

    config = singleton.get(ConfigManager)
    parts = [id + "::" + path for id, path in displays.items()]
    config.set("profiles", get_profile(), "main", "displays", ",".join(parts))



# initialize ConfigManager
#singleton.get(ConfigManager)
