#
# Wrapper for GNOME VFS. If VFS is not available, the real filesystem will be
# used as a fallback.
#

# try loading the VFS
try:
    import gnome.vfs as _vfs
    _MODES = {"r": _vfs.OPEN_READ,
              "w": _vfs.OPEN_WRITE}
    HAVE_VFS = 1

except ImportError:
    _MODES = {"r": "r",
              "w": "w"}
    HAVE_VFS = 0

import os

# store the current working directory
_cwd = os.getcwd()

# we are going to overwrite "open"; save the built-in "open"
_open = open



def open(uri, mode = "r"):

    if (not exists(uri)): uri = os.path.join(_cwd, uri)
    if (HAVE_VFS):
        vfsuri = _vfs.URI(uri)
        if (_vfs.exists(vfsuri)):
            fd = _vfs.open(vfsuri, _MODES[mode])
        elif (mode == "w"):
            fd = _vfs.create(vfsuri, _MODES[mode])
        else:
            raise IOError("Could not open file %s." % (uri))

    else:
        fd = _open(uri, _MODES[mode])

    return fd



def read_all(fd):

    data = ""
    while (1):
        try:
            d = fd.read(1024)
        except:
            break
        if (not d): break
        else: data += d
    #end while
        
    return data



def exists(uri):

    if (HAVE_VFS): return _vfs.exists(_vfs.URI(uri))
    else: return os.path.exists(uri)



def isdir(uri):

    if (not exists(uri)): uri = os.path.join(_cwd, uri)
    if (HAVE_VFS):
        return (_vfs.get_file_info(_vfs.URI(uri)).type == \
                _vfs.FILE_TYPE_DIRECTORY)
    else:
        return os.path.isdir(uri)



def chdir(uri):

    global _cwd
    if (exists(uri)):
        _cwd = uri



def getcwd():

    return _cwd



def join(*paths):

    paths = list(paths)
    if (not paths): return ""
    elif (len(paths) == 1): return paths[0]
    else:
        p2 = paths.pop()
        p1 = paths.pop()

        if (p2.find("://") != -1):
            newpath = p2
        else:
            newpath = os.path.join(p1, p2)

        if (paths): return join( * paths + [newpath] )
        if (paths): return join(paths + [newpath])
        else: return newpath
