from config import settings
from config.DaemonConfigger import DaemonConfigger
from display.Display import Display
from display.Window import Window
from display.Plug import Plug
from factory.DisplayFactory import DisplayFactory
from main import COPYRIGHT, HOME, NAME, USERHOME, VERSION, POSITIONSFILE
from main.AboutDialog import AboutDialog
from main.remotecommands import *
from main.RemoteSocket import RemoteSocket
from utils import dialog
from utils import vfs

import gobject
import gtk
import os
import sys


# This class starts, restarts and kills Displays
class Starter:

    __DISPLAY_TYPE_WINDOW = "window"
    __DISPLAY_TYPE_PLUG = "plug"


    def __init__(self):

        # the communication socket
        self.__socket = RemoteSocket()

        # the factory which reads the display's XML
        self.__factory = DisplayFactory()

        # the About dialog
        self.__about_dialog = AboutDialog(os.path.join(HOME, "data"))

        # the configuration dialog
        self.__configger = DaemonConfigger()

        # the command which gets called when the user removes a display
        self.__remove_command = ""

        # the set of open displays as a hashtable "id -> display"
        self.__open_displays = {}
        # the paths of the display files "id -> path"
        self.__display_paths = {}

        # set the message handlers for the socket
        self.__socket.add_message_handler(COMMAND_OPEN_DISPLAY,
                                          self.__handle_open_display)
        self.__socket.add_message_handler(COMMAND_OPEN_DISPLAY_WITH_ID,
                                          self.__handle_open_display_with_id)
        self.__socket.add_message_handler(COMMAND_OPEN_PLUG,
                                          self.__handle_open_plug)
        self.__socket.add_message_handler(COMMAND_CLOSE_DISPLAY,
                                          self.__handle_close_display)
        self.__socket.add_message_handler(COMMAND_SHUTDOWN,
                                          self.__handle_shutdown)
        self.__socket.add_message_handler(COMMAND_DISPLAYS,
                                          self.__handle_displays)
        self.__socket.add_message_handler(COMMAND_DISPLAY_LIST,
                                          self.__handle_display_list)
        self.__socket.add_message_handler(COMMAND_GEOMETRY,
                                          self.__handle_geometry)
        self.__socket.add_message_handler(COMMAND_VERSION,
                                          self.__handle_version)
        self.__socket.add_message_handler(COMMAND_ABOUT,
                                          self.__handle_about)
        self.__socket.add_message_handler(COMMAND_CONFIGURE,
                                          self.__handle_configger)
        self.__socket.add_message_handler(COMMAND_SET_REMOVE_COMMAND,
                                          self.__handle_set_remove_command)


        # socket ready, start handling requests
        self.__socket.start()

        # setup a nice systray icon
        if (not HAVE_WIN32 and settings.show_tray_icon):
            from main.TrayIcon import TrayIcon
            trayicon = TrayIcon()
            trayicon.set_menu([(None, _("_Manage desklets"),
                                self.__handle_manage),
                               (),
                               (gtk.STOCK_PROPERTIES, _("_Configuration"),
                                 self.__handle_configger),
                               (None, _("_View log"),
                                self.__handle_show_log),
                                (),
                               (None, _("_About..."),
                                self.__handle_about_dialog),
                               (),
                               (gtk.STOCK_QUIT, _("_Stop daemon"),
                                self.__handle_shutdown)])




    #
    # Reacts on observer messages from the display.
    #
    def __on_display_action(self, src, cmd, *args):

        if (cmd == src.OBS_CLOSE):
            ident = args[0]
            self.__close_display(ident)
            del src

        elif (cmd == src.OBS_RESTART):
            ident = args[0]
            xcoord, ycoord = src.get_geometry()[0:2]
            xcoord = xcoord.as_px()
            ycoord = ycoord.as_px()
            path = self.__display_paths[ident]
            self.__remove_display(ident)
            gobject.timeout_add(250, self.__add_display, ident, path,
                                self.__DISPLAY_TYPE_WINDOW)
            del src



    #
    # Adds the given display.
    #
    def __add_display(self, ident, path, displaytype):

        dsp = None
        container = None

        if (ident not in self.__open_displays):
            try:
                dsp = self.__create_display(ident, path)
            except (IOError, SyntaxError), exc:
                log("Warning: Couldn't add desklet \"%s\".\n%s"  % (path, exc))
                return

            self.__open_displays[ident] = dsp
            self.__display_paths[ident] = path

            log("Adding \"%s\" with ID \"%s\" to the desklet list."
                % (path, ident))
            dsp.add_observer(self.__on_display_action)

            if (displaytype == self.__DISPLAY_TYPE_WINDOW):
                container = Window(dsp)

            elif (displaytype == self.__DISPLAY_TYPE_PLUG):
                container = Plug()
                dsp.set_container(container)
                container.show()

            dsp.initialize()

        return container



    #
    # Creates and returns a new display from the given data.
    #
    def __create_display(self, ident, path):

        if (not vfs.exists(path)):
            raise IOError("Warning: The file \"%s\" doesn't exist." % (path,))

        try:
            data = vfs.read_entire_file(path)

        except:
            log("Could not open desklet file \"%s\"." % (path,))
            dialog.warning(_("Could not open desklet file \"%s\"") % (path,),
                           _("The desklet file could not be opened because "
                             "the file was not readable."))
            self.__close_display(ident)
            return

        display = self.__factory.create_display(ident, data, path)

        if (not display):
            dialog.warning(_("Invalid desklet file \"%s\"") % (path,),
                           _("The desklet file contains invalid data and "
                             "could not be loaded."))
            self.__close_display(ident)
            raise SyntaxError("Warning: Invalid desklet file \"%s\"" % (path,))

        return display



    #
    # Removes the given display.
    #
    def __remove_display(self, ident):

        try:
            display = self.__open_displays[ident]
        except KeyError, exc:
            log("Warning: Couldn't remove desklet with ID \"%s\".\n%s" \
                % (ident, exc))
            return

        log("Removing \"%s\" with ID \"%s\" from the desklet list."
            % (self.__display_paths[ident], ident))
        display.remove_display()

        try:
            del self.__open_displays[ident]
            del self.__display_paths[ident]
        except StandardError:
            pass

        import gc
        # gc.set_debug(gc.DEBUG_STATS)
        gc.collect()
        #print len(gc.garbage)


    #
    # Closes the given display.
    #
    def __close_display(self, ident):

        if (ident in self.__open_displays):
            display = self.__open_displays[ident]
            display.purge_display()

        self.__remove_display(ident)
        if (self.__remove_command):
            os.system("%s %s &" % (self.__remove_command, ident))



    def __handle_open_display(self, path):

        ident = Display.make_id()
        self.__handle_open_display_with_id(path, ident)

        return (ident,)



    def __handle_open_display_with_id(self, path, ident):

        self.__add_display(ident, path, self.__DISPLAY_TYPE_WINDOW)



    def __handle_open_plug(self, path):

        import time
        ident = str(time.time())
        plug = self.__add_display(ident, path, self.__DISPLAY_TYPE_PLUG)
        xid = plug.get_xembed_id()

        return (ident, xid)



    def __handle_close_display(self, ident):

        self.__remove_display(ident)



    def __handle_version(self):

        return (NAME, VERSION)



    def __handle_about(self):

        return ("%s %s" % (NAME, VERSION),
                "%s" % (COPYRIGHT.replace(u"\xa9", u"(C)"),),
                "This software is licensed under the terms of the GNU GPL.")



    def __handle_shutdown(self, *args):

        #self.__save_positions()

        for ident, display in self.__open_displays.items():
            try:
                display.remove_display()
            # what kind of exception are we expecting here?
            except:
                log("Could not remove desklet \"%s\"!" % (display,))
        gtk.main_quit()



    def __handle_manage(self, *args):

        cmd = os.path.join(HOME, "gdesklets-shell")
        os.system(cmd + " &")


    def __handle_configger(self, *args):

        self.__configger.show()



    def __handle_about_dialog(self, *args):

        self.__about_dialog.show()



    def __handle_show_log(self, *args):

        cmd = os.path.join(HOME, "gdesklets-logview")
        os.system(cmd + " &")


    def __handle_displays(self):

        return self.__open_displays.keys()



    def __handle_display_list(self):

        return self.__open_displays.values()



    def __handle_geometry(self, ident):

        display = self.__open_displays[ident]

        return display.get_geometry()



    def __handle_set_remove_command(self, command):

        # only allow setting the remove command once for security reasons
        if (not self.__remove_command):
            self.__remove_command = command


